package jp.co.sra.jun.geometry.surfaces;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBalls;
import jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes;
import jp.co.sra.jun.geometry.pluralities.Jun3dTriangles;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;

/**
 * Jun3dTriangle class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2004/12/14 (by Mitsuhiro Asada)
 *  @updated   2006/09/28 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun692 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dTriangle.java,v 8.20 2008/02/20 06:30:58 nisinaka Exp $
 */
public class Jun3dTriangle extends JunTriangle {
	protected Jun3dPoint p1;
	protected Jun3dPoint p2;
	protected Jun3dPoint p3;

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 * 
	 * @category Instance creation
	 */
	private Jun3dTriangle() {
		super();
	}

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 *
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public Jun3dTriangle(Jun3dPoint aPoint1, Jun3dPoint aPoint2, Jun3dPoint aPoint3) {
		this.setP1_(aPoint1);
		this.setP2_(aPoint2);
		this.setP3_(aPoint3);
	}

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public static Jun3dTriangle First_second_third_(JunPoint aPoint1, JunPoint aPoint2, JunPoint aPoint3) {
		return Jun3dTriangle.On_on_on_(aPoint1, aPoint2, aPoint3);
	}

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public static Jun3dTriangle On_on_on_(JunPoint aPoint1, JunPoint aPoint2, JunPoint aPoint3) {
		Jun3dPoint thePoint1 = Jun3dPoint.Coerce_(aPoint1);
		Jun3dPoint thePoint2 = Jun3dPoint.Coerce_(aPoint2);
		Jun3dPoint thePoint3 = Jun3dPoint.Coerce_(aPoint3);
		if (thePoint1.equals(thePoint2) || thePoint1.equals(thePoint3) || thePoint2.equals(thePoint3)) {
			return null;
		}

		return new Jun3dTriangle(thePoint1, thePoint2, thePoint3);
	}

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static Jun3dTriangle On_normalVector_(Jun3dPoint centerPoint, Jun3dPoint aVector) {
		return Jun3dTriangle.On_normalVector_distance_(centerPoint, aVector, 1);
	}

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber double
	 * @category Instance creation
	 */
	public static Jun3dTriangle On_normalVector_distance_(Jun3dPoint centerPoint, Jun3dPoint aVector, double aNumber) {
		return Jun3dTriangle.On_normalVector_distance_rightVector_(centerPoint, aVector, aNumber, new Jun3dPoint(1, 0, 0));
	}

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param normalVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber double
	 * @param rightVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static Jun3dTriangle On_normalVector_distance_rightVector_(Jun3dPoint centerPoint, Jun3dPoint normalVector, double aNumber, Jun3dPoint rightVector) {
		JunPlane aPlane = JunPlane.On_normalVector_(centerPoint, normalVector);
		Jun3dLine aroundLine = centerPoint.to_(centerPoint.plus_(normalVector));
		Jun3dLine aLine = aroundLine.translatedBy_(rightVector);
		Jun3dPoint aPoint = aPlane.intersectingPointWithLine_(aLine);

		if (Math.abs(centerPoint.x() - aPoint.x()) < Jun3dTriangle.Accuracy() && Math.abs(centerPoint.y() - aPoint.y()) < Jun3dTriangle.Accuracy() && Math.abs(centerPoint.z() - aPoint.z()) < Jun3dTriangle.Accuracy()) {
			Jun3dPoint[] points = new Jun3dPoint[] { aPlane.p1(), aPlane.p2(), aPlane.p3() };
			TreeMap map = new TreeMap();
			for (int i = 0; i < points.length; i++) {
				map.put(new Double(points[i].distance_(centerPoint)), points[i]);
			}
			aPoint = (Jun3dPoint) map.get(map.lastKey());
		}
		aLine = centerPoint.to_(aPoint).normalized();
		aPoint = aLine.atT_(aNumber);
		return Jun3dTriangle.On_on_on_(aPoint, aPoint.transform_(Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(120), aroundLine)), aPoint.transform_(Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(240), aroundLine)));
	}

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category Instance creation
	 */
	public static Jun3dTriangle On_vertical_(Jun3dPoint centerPoint, Jun3dLine aLine) {
		return Jun3dTriangle.On_vertical_distance_(centerPoint, aLine, 1);
	}

	/**
	 * Create a new instance of Jun3dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @param aNumber double
	 * @category Instance creation
	 */
	public static Jun3dTriangle On_vertical_distance_(Jun3dPoint centerPoint, Jun3dLine aLine, double aNumber) {
		return Jun3dTriangle.On_normalVector_distance_(centerPoint, aLine.normalVector(), aNumber);
	}

	/**
	 * Answer the receiver's bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category accessing
	 */
	public Jun3dBoundingBall boundingBall() {
		return this.asBoundingBall();
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		return this.asBoundingBox();
	}

	/**
	 * Get the first point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint first() {
		return this.p1();
	}

	/**
	 * Get the middle point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint middle() {
		return this.p2();
	}

	/**
	 * Get the last point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint last() {
		return this.p3();
	}

	/**
	 * Get the p1 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p1() {
		return p1;
	}

	/**
	 * Answer my p1 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#_p1()
	 * @category accessing
	 */
	protected JunPoint _p1() {
		return this.p1();
	}

	/**
	 * Get the p2 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p2() {
		return p2;
	}

	/**
	 * Answer my p2 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#_p2()
	 * @category accessing
	 */
	protected JunPoint _p2() {
		return this.p2();
	}

	/**
	 * Get the p3 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p3() {
		return p3;
	}

	/**
	 * Answer the receiver's three points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] points() {
		return new Jun3dPoint[] { this.p1(), this.p2(), this.p3() };
	}

	/**
	 * Answer my p3 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#_p3()
	 * @category accessing
	 */
	protected JunPoint _p3() {
		return this.p3();
	}

	/**
	 * Get the second point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint second() {
		return this.p2();
	}

	/**
	 * Get the third point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint third() {
		return this.p3();
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dTriangle aTriangle = (Jun3dTriangle) anObject;
		return this.first().equal_(aTriangle.first()) && this.second().equal_(aTriangle.second()) && this.third().equal_(aTriangle.third());
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dTriangle aTriangle = (Jun3dTriangle) anObject;
		return this.first().equals(aTriangle.first()) && this.second().equals(aTriangle.second()) && this.third().equals(aTriangle.third());
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @category converting
	 */
	public Jun3dLine[] asArrayOfLines() {
		return this.asArrayOf3dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dLines()
	 * @category converting
	 */
	public Jun3dLine[] asArrayOf3dLines() {
		return new Jun3dLine[] {};
	}

	/**
	 * Convert to an array of point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[3]
	 * @category converting
	 */
	public Jun3dPoint[] asArrayOfPoints() {
		return new Jun3dPoint[] { this.first(), this.second(), this.third() };
	}

	/**
	 * Convert to an array of a <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category converting
	 */
	public Jun3dTriangle[] asArrayOfTriangles() {
		return this.asArrayOf3dTriangles();
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dTriangles()
	 * @category converting 
	 */
	public Jun3dTriangle[] asArrayOf3dTriangles() {
		return new Jun3dTriangle[] { this };
	}

	/**
	 * Convert to a <code>Jun3dBoundingBall</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category converting
	 */
	public Jun3dBoundingBall asBoundingBall() {
		return new Jun3dBoundingBall(new Jun3dPoint[] { this.p1(), this.p2(), this.p3() });
	}

	/**
	 * Convert to a <code>Jun3dBoundingBalls</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBalls
	 * @category converting
	 */
	public Jun3dBoundingBalls asBoundingBalls() {
		Jun3dBoundingBalls boundingBalls = new Jun3dBoundingBalls();
		Collection aCollection = new ArrayList();
		aCollection.add(this.asBoundingBall());
		boundingBalls.boundingBalls_(aCollection);
		return boundingBalls;
	}

	/**
	 * Convert to a <code>Jun3dBoundingBox</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category converting
	 */
	public Jun3dBoundingBox asBoundingBox() {
		return new Jun3dBoundingBox(new Jun3dPoint[] { this.p1(), this.p2(), this.p3() });
	}

	/**
	 * Convert to a <code>Jun3dBoundingBoxes</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category converting
	 */
	public Jun3dBoundingBoxes asBoundingBoxes() {
		Jun3dBoundingBoxes boundingBoxes = new Jun3dBoundingBoxes();
		Collection aCollection = new ArrayList();
		aCollection.add(this.asBoundingBox());
		boundingBoxes.boundingBoxes_(aCollection);
		return boundingBoxes;
	}

	/**
	 * Convert to a <code>Jun3dCircle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dCircle
	 * @category converting
	 */
	public Jun3dCircle asCircle() {
		Jun3dPoint centerPoint = this.centerOfCircumcircle();
		return new Jun3dCircle(centerPoint, centerPoint.distance_(this.p1()), this.normalVector());
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dPolygon aPolygon = new JunOpenGL3dPolygon(new Jun3dPoint[] { this.p1(), this.p2(), this.p3() });
		aPolygon.paint_(this.defaultColor());
		return aPolygon;
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code> with points.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObjectWithPoints()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectWithPoints() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.add_(this.asJunOpenGL3dObject());
		JunOpenGL3dObject theP1 = this.p1().asJunOpenGL3dObject();
		theP1.paint_(Color.red);
		compoundObject.add_(theP1);
		JunOpenGL3dObject theP2 = this.p2().asJunOpenGL3dObject();
		theP2.paint_(Color.green);
		compoundObject.add_(theP2);
		JunOpenGL3dObject theP3 = this.p3().asJunOpenGL3dObject();
		theP3.paint_(Color.blue);
		compoundObject.add_(theP3);
		return compoundObject;
	}

	/**
	 * Convert to a <code>JunPlane</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#asPlane()
	 * @category converting
	 */
	public JunPlane asPlane() {
		return new JunPlane(this.p1(), this.p2(), this.p3());
	}

	/**
	 * Convert to an array of point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[4]
	 * @category converting
	 */
	public Jun3dPoint[] asPointArray() {
		return new Jun3dPoint[] { this.first(), this.second(), this.third(), this.first() };
	}

	/**
	 * Convert to a <code>JunTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category converting
	 */
	public Jun3dTriangle asTriangle() {
		return this;
	}

	/**
	 * Convert to a <code>Jun3dTriangles</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dTriangles
	 * @category converting
	 */
	public Jun3dTriangles asTriangles() {
		Collection aCollection = new ArrayList();
		aCollection.add(this);
		return new Jun3dTriangles(aCollection);
	}

	/**
	 * Answer the array of the intersecting points.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category dividing
	 */
	public Jun3dPoint[] pointsDividedBy_(JunPlane aPlane) {
		Collection aCollection = (Collection) this.tableDividedBy_(aPlane).get($("points"));
		return (Jun3dPoint[]) aCollection.toArray(new Jun3dPoint[aCollection.size()]);
	}

	/**
	 * Answer the array of the triangles in the positive side of the plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category dividing
	 */
	public Jun3dTriangle[] positivesDividedBy_(JunPlane aPlane) {
		Collection aCollection = (Collection) this.tableDividedBy_(aPlane).get($("positives"));
		return (Jun3dTriangle[]) aCollection.toArray(new Jun3dTriangle[aCollection.size()]);
	}

	/**
	 * Answer the array of the triangles in the negative side of the plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category dividing
	 */
	public Jun3dTriangle[] negativesDividedBy_(JunPlane aPlane) {
		Collection aCollection = (Collection) this.tableDividedBy_(aPlane).get($("negatives"));
		return (Jun3dTriangle[]) aCollection.toArray(new Jun3dTriangle[aCollection.size()]);
	}

	/**
	 * Answer the array of the triangles on the plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category dividing
	 */
	public Jun3dTriangle[] zerosDividedBy_(JunPlane aPlane) {
		Collection aCollection = (Collection) this.tableDividedBy_(aPlane).get($("zeros"));
		return (Jun3dTriangle[]) aCollection.toArray(new Jun3dTriangle[aCollection.size()]);
	}

	/**
	 * Answer the arrayof the triangles on and in the positive side of the plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category dividing
	 */
	public Jun3dTriangle[] diviedBy_(JunPlane aPlane) {
		Map aMap = this.tableDividedBy_(aPlane);
		ArrayList aList = new ArrayList();
		aList.addAll((Collection) aMap.get($("zeros")));
		aList.addAll((Collection) aMap.get($("positives")));
		return (Jun3dTriangle[]) aList.toArray(new Jun3dTriangle[aList.size()]);
	}

	/**
	 * Answer the table which contains all dividing information.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return java.util.Map
	 * @category dividing
	 */
	public Map tableDividedBy_(JunPlane aPlane) {
		ArrayList positives = new ArrayList(3);
		ArrayList negatives = new ArrayList(3);
		ArrayList zeros = new ArrayList(3);
		ArrayList points = new ArrayList(3);

		JunPlane trianglePlane = this.asPlane();
		Jun3dPoint[] pointArray = new Jun3dPoint[] { this.p1(), this.p2(), this.p3() };
		int[] signArray = new int[pointArray.length];
		boolean hasPositive = false;
		boolean hasNegative = false;
		boolean hasZero = false;
		for (int i = 0; i < pointArray.length; i++) {
			signArray[i] = pointArray[i].whichSideOf_(aPlane);
			if (signArray[i] > 0) {
				hasPositive = true;
			} else if (signArray[i] < 0) {
				hasNegative = true;
			} else {
				hasZero = true;
			}
		}

		if (hasZero) {
			if (!hasPositive && !hasNegative) {
				zeros.add(this);
				for (int i = 0; i < pointArray.length; i++) {
					points.add(pointArray[i]);
				}
			} else if (hasPositive && hasNegative) {
				Jun3dPoint fromPoint = null;
				ArrayList toPoints = new ArrayList(2);
				for (int i = 0; i < pointArray.length; i++) {
					Jun3dPoint aPoint = pointArray[i];
					int sign = signArray[i];
					if (sign == 0) {
						points.add(aPoint);
						fromPoint = aPoint;
					} else {
						toPoints.add(aPoint);
					}
				}
				Jun3dPoint toPoints_first = (Jun3dPoint) toPoints.get(0);
				Jun3dPoint toPoints_last = (Jun3dPoint) toPoints.get(toPoints.size() - 1);
				Jun3dLine aLine = new Jun3dLine(toPoints_first, toPoints_last);
				Jun3dPoint intersectingPoint = aPlane.intersectingPointWithLine_(aLine);
				if (intersectingPoint != null) {
					points.add(intersectingPoint);
					Jun3dTriangle aTriangle = Jun3dTriangle.On_on_on_(fromPoint, intersectingPoint, toPoints_first);
					if (aTriangle != null) {
						if (fromPoint.plus_(aTriangle.normalVector()).whichSideOf_(trianglePlane) < 0) {
							aTriangle = aTriangle.reversed();
						}
						if (signArray[Arrays.asList(pointArray).indexOf(toPoints_first)] > 0) {
							positives.add(aTriangle);
						} else {
							negatives.add(aTriangle);
						}
					}
					aTriangle = Jun3dTriangle.On_on_on_(fromPoint, intersectingPoint, toPoints_last);
					if (aTriangle != null) {
						if (fromPoint.plus_(aTriangle.normalVector()).whichSideOf_(trianglePlane) < 0) {
							aTriangle = aTriangle.reversed();
						}
						if (signArray[Arrays.asList(pointArray).indexOf(toPoints_last)] > 0) {
							positives.add(aTriangle);
						} else {
							negatives.add(aTriangle);
						}
					}
				} else {
					// do nothing
				}
			} else {
				if (hasPositive) {
					positives.add(this);
				}
				if (hasNegative) {
					negatives.add(this);
				}
				for (int i = 0; i < pointArray.length; i++) {
					if (signArray[i] == 0) {
						points.add(pointArray[i]);
					}
				}
			}
		} else {
			if (hasPositive && hasNegative) {
				ArrayList positiveCollection = new ArrayList(2);
				ArrayList negativeCollection = new ArrayList(2);
				for (int i = 0; i < pointArray.length; i++) {
					Jun3dPoint aPoint = pointArray[i];
					int sign = signArray[i];
					if (sign > 0) {
						positiveCollection.add(aPoint);
					} else {
						negativeCollection.add(aPoint);
					}
				}
				Jun3dPoint fromPoint, firstToPoint, lastToPoint;
				int fromSign;
				if (positiveCollection.size() == 1) {
					fromPoint = (Jun3dPoint) positiveCollection.get(0);
					fromSign = 1;
					firstToPoint = (Jun3dPoint) negativeCollection.get(0);
					lastToPoint = (Jun3dPoint) negativeCollection.get(negativeCollection.size() - 1);
				} else {
					fromPoint = (Jun3dPoint) negativeCollection.get(0);
					fromSign = -1;
					firstToPoint = (Jun3dPoint) positiveCollection.get(0);
					lastToPoint = (Jun3dPoint) positiveCollection.get(positiveCollection.size() - 1);
				}
				Jun3dLine aLine = new Jun3dLine(fromPoint, firstToPoint);
				Jun3dPoint firstIntersectingPoint = aPlane.intersectingPointWithLine_(aLine);
				aLine = new Jun3dLine(fromPoint, lastToPoint);
				Jun3dPoint lastIntersectingPoint = aPlane.intersectingPointWithLine_(aLine);
				if (firstIntersectingPoint != null && lastIntersectingPoint != null) {
					points.add(firstIntersectingPoint);
					points.add(lastIntersectingPoint);
					Jun3dTriangle aTriangle = Jun3dTriangle.On_on_on_(fromPoint, firstIntersectingPoint, lastIntersectingPoint);
					if (aTriangle != null) {
						if (fromPoint.plus_(aTriangle.normalVector()).whichSideOf_(trianglePlane) < 0) {
							aTriangle = aTriangle.reversed();
						}
						if (fromSign > 0) {
							positives.add(aTriangle);
						} else {
							negatives.add(aTriangle);
						}
					}
					aTriangle = Jun3dTriangle.On_on_on_(firstToPoint, firstIntersectingPoint, lastToPoint);
					Jun3dTriangle anotherTriangle = Jun3dTriangle.On_on_on_(firstToPoint, lastIntersectingPoint, lastToPoint);
					if (aTriangle != null && anotherTriangle != null) {
						if (aTriangle.area() <= anotherTriangle.area()) {
							anotherTriangle = Jun3dTriangle.On_on_on_(firstIntersectingPoint, lastToPoint, lastIntersectingPoint);
						} else {
							aTriangle = Jun3dTriangle.On_on_on_(firstIntersectingPoint, firstToPoint, lastIntersectingPoint);
						}
					}
					if (aTriangle != null) {
						if (aTriangle.p1().plus_(aTriangle.normalVector()).whichSideOf_(trianglePlane) < 0) {
							aTriangle = aTriangle.reversed();
						}
						if (fromSign > 0) {
							negatives.add(aTriangle);
						} else {
							positives.add(aTriangle);
						}
					}
					if (anotherTriangle != null) {
						if (anotherTriangle.p1().plus_(anotherTriangle.normalVector()).whichSideOf_(trianglePlane) < 0) {
							anotherTriangle = anotherTriangle.reversed();
						}
						if (fromSign > 0) {
							negatives.add(anotherTriangle);
						} else {
							positives.add(anotherTriangle);
						}
					}
				} else {
					if (fromSign > 0) {
						negatives.add(this);
					} else {
						positives.add(this);
					}
				}
			} else {
				if (hasPositive) {
					positives.add(this);
				}
				if (hasNegative) {
					negatives.add(this);
				}
			}
		}

		HashMap aTable = new HashMap();
		aTable.put($("positives"), positives);
		aTable.put($("negatives"), negatives);
		aTable.put($("zeros"), zeros);
		aTable.put($("points"), points);
		return aTable;
	}

	/**
	 * Answer the receiver's area size
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category functions
	 */
	public double area() {
		if (this.hasArea() == false) {
			return 0.0d;
		}
		double a = this.p1().distance_(this.p2());
		double b = this.p2().distance_(this.p3());
		double c = this.p3().distance_(this.p1());
		double s = (a + b + c) / 2;
		double area = Math.sqrt(s * (s - a) * (s - b) * (s - c));
		return area;
	}

	/**
	 * Answer the receiver's area with sign.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#areaWithSign()
	 * @category functions
	 */
	public double areaWithSign() {
		return this.areaWithSignFromPoint_(new Jun3dPoint(0, 0, Math.max(Math.max(this.p1().z(), this.p2().z()), this.p3().z()) + 1));
	}

	/**
	 * Answer the receiver's area with sign from the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category functions
	 */
	public double areaWithSignFromPoint_(Jun3dPoint aPoint) {
		double areaWithSign = this.area() * this.asPlane().whichSide_(aPoint);
		return areaWithSign;
	}

	/**
	 * Answer the center of circumcircle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint centerOfCircumcircle() {
		return this.circumcircle().center();
	}

	/**
	 * Answer the center of incircle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint centerOfIncircle() {
		return this.incircle().center();
	}

	/**
	 * Answer the circum circle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @category functions
	 */
	public Jun3dCircle circumcircle() {
		Jun3dPoint unitVector = this.asPlane().normalUnitVector();
		Jun3dPoint middlePoint = this.p1().center_(this.p2());
		Jun3dLine firstLine = new Jun3dLine(middlePoint, middlePoint.plus_(unitVector));
		Jun3dTransformation aTransformation = Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(90), this.p1().to_(this.p2()));
		firstLine = firstLine.transform_(aTransformation);
		middlePoint = this.p1().center_(this.p3());
		Jun3dLine secondLine = new Jun3dLine(middlePoint, middlePoint.plus_(unitVector));
		aTransformation = Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(90), this.p1().to_(p3));
		secondLine = secondLine.transform_(aTransformation);
		Jun3dPoint centerPoint = firstLine.intersectingPointWithLine_(secondLine);
		double radiusValue = centerPoint.distance_(this.p1());
		return new Jun3dCircle(centerPoint, radiusValue, this.normalVector());
	}

	/**
	 * Answer the receiver's detailed triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category functions
	 */
	public Jun3dTriangle[] detailedTriangles() {
		Jun3dTriangle[] detailedTriangles = new Jun3dTriangle[4];
		Jun3dPoint[] pointArray = new Jun3dPoint[6];
		pointArray[0] = this.p1();
		pointArray[1] = this.p2();
		pointArray[2] = this.p3();
		pointArray[3] = pointArray[0].center_(pointArray[1]);
		pointArray[4] = pointArray[1].center_(pointArray[2]);
		pointArray[5] = pointArray[2].center_(pointArray[0]);
		detailedTriangles[0] = new Jun3dTriangle(pointArray[0], pointArray[3], pointArray[5]);
		detailedTriangles[1] = new Jun3dTriangle(pointArray[3], pointArray[1], pointArray[4]);
		detailedTriangles[2] = new Jun3dTriangle(pointArray[4], pointArray[2], pointArray[5]);
		detailedTriangles[3] = new Jun3dTriangle(pointArray[3], pointArray[4], pointArray[5]);
		return detailedTriangles;
	}

	/**
	 * Answer the receiver's detailed triangles with the specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category functions
	 */
	public Jun3dTriangle[] detailedTrianglesLevel_(int levelNumber) {
		if (levelNumber <= 0) {
			return new Jun3dTriangle[] { this };
		}

		Collection aCollection = new ArrayList();
		Jun3dTriangle[] detailedTriangles = this.detailedTriangles();
		for (int i = 0; i < detailedTriangles.length; i++) {
			Jun3dTriangle[] triangles = detailedTriangles[i].detailedTrianglesLevel_(levelNumber - 1);
			for (int j = 0; j < triangles.length; j++) {
				aCollection.add(triangles[j]);
			}
		}
		return (Jun3dTriangle[]) aCollection.toArray(new Jun3dTriangle[aCollection.size()]);
	}

	/**
	 * Answer the receiver's in circle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @throws java.lang.IllegalStateException
	 * @category functions
	 */
	public Jun3dCircle incircle() {
		double t = this.peripheryOfTriangle();
		if (t < this.Accuracy()) {
			throw new IllegalStateException("unexpected triangle");
		}
		double d12 = this.p1().distance_(this.p2());
		double d23 = this.p2().distance_(this.p3());
		double d31 = this.p3().distance_(this.p1());
		double x = (d23 * this.p1().x() + d31 * this.p2().x() + d12 * this.p3().x()) / t;
		double y = (d23 * this.p1().y() + d31 * this.p2().y() + d12 * this.p3().y()) / t;
		double z = (d23 * this.p1().z() + d31 * this.p2().z() + d12 * this.p3().z()) / t;
		Jun3dPoint centerPoint = new Jun3dPoint(x, y, z);
		double s = this.peripheryOfTriangle() / 2;
		if (s < this.Accuracy()) {
			throw new IllegalStateException("unexpected triangle");
		}
		double radiusValue = Math.sqrt((s - this.p1().distance_(this.p2())) * (s - this.p2().distance_(this.p3())) * (s - this.p3().distance_(this.p1())) / s);
		return new Jun3dCircle(centerPoint, radiusValue, this.normalVector());
	}

	/**
	 * Answer the normal unit vector of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint normalUnitVector() {
		return this.asPlane().normalUnitVector();
	}

	/**
	 * Answer the normal vector of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint normalVector() {
		return this.asPlane().normalVector();
	}

	/**
	 * Answer the periphery of triangle.
	 * 
	 * @return double
	 * @category functions
	 */
	public double peripheryOfTriangle() {
		double perimeter = 0.0d;
		perimeter = perimeter + this.p1().distance_(this.p2());
		perimeter = perimeter + this.p2().distance_(this.p3());
		perimeter = perimeter + this.p3().distance_(this.p1());
		return perimeter;
	}

	/**
	 * Answer the three lines of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[3]
	 * @category functions
	 */
	public JunPlane[] threePlanes() {
		JunPlane[] threePlanes = new JunPlane[3];

		Jun3dLine aLine = this.p1().to_(this.p2());
		Jun3dPoint aPoint = aLine.nearestPointFromPoint_(this.p3());
		JunPlane aPlane = aPoint.plane_(aPoint.to_(this.p3).normalVector());
		threePlanes[0] = aPlane;

		aLine = this.p1().to_(this.p3());
		aPoint = aLine.nearestPointFromPoint_(this.p2());
		aPlane = aPoint.plane_(aPoint.to_(this.p2).normalVector());
		threePlanes[1] = aPlane;

		aLine = this.p2().to_(this.p3());
		aPoint = aLine.nearestPointFromPoint_(this.p1());
		aPlane = aPoint.plane_(aPoint.to_(this.p1).normalVector());
		threePlanes[2] = aPlane;

		return threePlanes;
	}

	/**
	 * Answer the reversed triangle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category functions
	 */
	public Jun3dTriangle reversed() {
		return Jun3dTriangle.On_on_on_(this.p3(), this.p2(), this.p1());
	}

	/**
	 * Answer the receiver's subdivide triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dTriangles
	 * @category subdividing
	 */
	public Jun3dTriangles subdivide() {
		return this.subdivide4();
	}

	/**
	 * Answer the receiver's subdivide triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dTriangles
	 * @category subdividing
	 */
	public Jun3dTriangles subdivide4() {
		return new Jun3dTriangles(this.detailedTriangles());
	}

	/**
	 * Answer the receiver's subdivide triangles with specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dTriangles
	 * @category subdividing
	 */
	public Jun3dTriangles subdivide4Level_(int levelNumber) {
		return new Jun3dTriangles(this.detailedTrianglesLevel_(levelNumber));
	}

	/**
	 * Answer the receiver's subdivide triangles with specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dTriangles
	 * @category subdividing
	 */
	public Jun3dTriangles subdivideLevel_(int levelNumber) {
		return this.subdivide4Level_(levelNumber);
	}

	/**
	 * Answer true if the receiver contains aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun3dPoint aPoint) {
		JunPlane[] planes = this.threePlanes();
		for (int i = 0; i < planes.length; i++) {
			if (aPoint.whichSideOf_(planes[i]) < 0) {
				return false;
			}
		}
		return this.asPlane().distanceFromPoint_(aPoint) < this.Accuracy();
	}

	/**
	 * Answer true if the receiver is a 3d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Answer the new Jun3dTriangle which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category transforming
	 */
	public Jun3dTriangle rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun3dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new Jun3dTriangle which is scaled by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category transforming
	 */
	public Jun3dTriangle scaledBy_(double aNumber) {
		return this.transform_(Jun3dTransformation.Scale_(aNumber));
	}

	/**
	 * Answer the new Jun3dTriangle which is scaled by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category transforming
	 */
	public Jun3dTriangle scaledBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply aJunTransformation to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTriangle transform_(Jun3dTransformation aTransformation) {
		return Jun3dTriangle.On_on_on_(this.p1().transform_(aTransformation), this.p2().transform_(aTransformation), this.p3().transform_(aTransformation));
	}

	/**
	 * Answer the copy of the receiver which is translated by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category transforming
	 */
	public Jun3dTriangle translatedBy_(double aNumber) {
		return this.transform_(Jun3dTransformation.Translate_(aNumber));
	}

	/**
	 * Answer the copy of the receiver which is translated by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category transforming
	 */
	public Jun3dTriangle translatedBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Translate_(aPoint));
	}

	/**
	 * Set the p1 point.
	 * 
	 * @param aPpoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP1_(JunPoint aPpoint) {
		p1 = Jun3dPoint.Coerce_(aPpoint);
	}

	/**
	 * Set the p2 point.
	 * 
	 * @param aPpoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP2_(JunPoint aPpoint) {
		p2 = Jun3dPoint.Coerce_(aPpoint);
	}

	/**
	 * Set the p3 point.
	 * 
	 * @param aPpoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP3_(JunPoint aPpoint) {
		p3 = Jun3dPoint.Coerce_(aPpoint);
	}
}
