package jp.co.sra.jun.geometry.surfaces;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.curves.Jun3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;

/**
 * Jun3dPolygon class
 * 
 *  @author    nisinaka
 *  @created   1998/11/09 (by nisinaka)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2004/09/30 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun691 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dPolygon.java,v 8.19 2008/02/20 06:30:58 nisinaka Exp $
 */
public class Jun3dPolygon extends JunPlane {

	/** The accuracy number for Jun3dPolygon. */
	public static final double ACCURACY = JunGeometry.ACCURACY * Math.pow(10, Math.max(0, Math.round(Math.log(1 / JunGeometry.ACCURACY) / Math.log(10) / 2 - 2)));

	/** The parameter polygon. */
	protected Jun2dPolygon parameterPolygon = null;

	/** The bounding box. */
	protected Jun3dBoundingBox boundingBox = null;

	/** The normal vector. */
	protected Jun3dPoint normalVector = null;

	/**
	 * Create a new instance of Jun3dPolygon with three Jun3dPoints.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static Jun3dPolygon _On_on_on_(Jun3dPoint aPoint1, Jun3dPoint aPoint2, Jun3dPoint aPoint3) {
		if (aPoint1.equals(aPoint2)) {
			return null;
		}
		if (aPoint1.equals(aPoint3)) {
			return null;
		}
		if (aPoint2.equals(aPoint3)) {
			return null;
		}

		Jun3dPolygon polygon = new Jun3dPolygon();
		polygon._initialize(aPoint1, aPoint2, aPoint3);
		return polygon;
	}

	/**
	 * Create a new instance of Jun3dPolygon with three Jun3dPoints.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static Jun3dPolygon On_on_on_(Jun3dPoint aPoint1, Jun3dPoint aPoint2, Jun3dPoint aPoint3) {
		return Vertexes_(new Jun3dPoint[] { aPoint1, aPoint2, aPoint3 });
	}

	/**
	 * Create a new instance of Jun3dPolygon.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun2dPolygon jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category Instance creation
	 */
	public static Jun3dPolygon Origin_uVector_vVector_parameterPolygon_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, Jun3dPoint aJun3dPoint3, Jun2dPolygon aJun2dPolygon) {
		Jun3dPoint wVector = (Jun3dPoint) aJun3dPoint2.vectorProduct_(aJun3dPoint3).unitVector();
		Jun3dPoint vVector = (Jun3dPoint) wVector.vectorProduct_(aJun3dPoint2).unitVector();
		Jun3dPolygon polygon = Jun3dPolygon._On_on_on_(aJun3dPoint1, aJun3dPoint1.plus_(aJun3dPoint2), aJun3dPoint1.plus_(vVector));
		if (polygon == null) {
			return null;
		}

		polygon.setParameterPolygon_(aJun2dPolygon);
		return polygon;
	}

	/**
	 * Create a new instance of Jun3dPolygon.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param anArrayOfJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public static Jun3dPolygon Origin_uVector_vVector_points_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, Jun3dPoint aJun3dPoint3, Jun3dPoint[] anArrayOfJun3dPoint) {
		Jun3dPoint wVector = aJun3dPoint2.vectorProduct_(aJun3dPoint3).unitVector();
		Jun3dPoint vVector = wVector.vectorProduct_(aJun3dPoint2).unitVector();
		Jun3dPolygon polygon = Jun3dPolygon._On_on_on_(aJun3dPoint1, aJun3dPoint1.plus_(aJun3dPoint2), aJun3dPoint1.plus_(vVector));
		if (polygon == null) {
			return null;
		}

		Jun2dPoint[] parameterPoints = new Jun2dPoint[anArrayOfJun3dPoint.length];
		for (int i = 0; i < anArrayOfJun3dPoint.length; i++) {
			parameterPoints[i] = polygon.parameterPointAtPoint_(anArrayOfJun3dPoint[i]);
		}
		polygon.setParameterPolygon_(new Jun2dPolygon(parameterPoints));
		return polygon;
	}

	/**
	 * Create a new instance of Jun3dPolygon with an array of Jun3dPoint. This
	 * corresponds to the instance creation method "vertexes:".
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param anArrayOfJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public static Jun3dPolygon Points_(Jun3dPoint[] anArrayOfJun3dPoint) {
		return Vertexes_(anArrayOfJun3dPoint);
	}

	/**
	 * Create a new instance of Jun3dPolygon with an array of Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param anArrayOfJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public static Jun3dPolygon Vertexes_(Jun3dPoint[] anArrayOfJun3dPoint) {
		Jun3dPoint[] points = anArrayOfJun3dPoint;
		int size = points.length;
		if (size < 3) {
			return null;
		}

		Jun3dPoint origin = points[0];
		int uIndex = 0;
		double maxDistance = Double.NEGATIVE_INFINITY;
		for (int i = 1; i < points.length; i++) {
			double distance = points[i].distance_(origin);
			if (maxDistance < distance) {
				uIndex = i;
				maxDistance = distance;
			}
		}

		Jun3dPoint uPoint = points[uIndex];
		Jun3dPoint uVector = (Jun3dPoint) uPoint.minus_(origin).unitVector();
		Jun3dPoint wVector = Jun3dPoint.Zero();
		for (int i = 0; i < size; i++) {
			Jun3dPoint point1 = points[i];
			Jun3dPoint point2 = points[(i + 1) % size];
			Jun3dPoint point3 = points[(i + 2) % size];
			wVector = (Jun3dPoint) wVector.plus_(((Jun3dPoint) point2.minus_(point1)).vectorProduct_((Jun3dPoint) point3.minus_(point2)));
		}

		Jun3dPoint vVector = (Jun3dPoint) wVector.vectorProduct_(uVector).unitVector();
		return Origin_uVector_vVector_points_(origin, uVector, vVector, points);
	}

	/**
	 * Create a new instance of Jun3dPolygon with an Vector of Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param aCollectionOfJun3dPoint java.util.Collection of
	 *        jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public static Jun3dPolygon Vertexes_(Collection aCollectionOfJun3dPoint) {
		Jun3dPoint[] anArrayOfJun3dPoint = new Jun3dPoint[aCollectionOfJun3dPoint.size()];
		aCollectionOfJun3dPoint.toArray(anArrayOfJun3dPoint);
		return Vertexes_(anArrayOfJun3dPoint);
	}

	/**
	 * Create a new instance of Jun3dPolygon with an array of Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param anArrayOfJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 * @deprecated since Jun628, use Vertexes_(Jun3dPoint[])
	 */
	public static Jun3dPolygon Vertices_(Jun3dPoint[] anArrayOfJun3dPoint) {
		return Vertexes_(anArrayOfJun3dPoint);
	}

	/**
	 * Create a new instance of Jun3dPolygon with three Jun3dPoints.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 * @deprecated since Jun628, use On_on_on_(Jun3dPoint, Jun3dPoint, Jun3dPoint)
	 */
	public static Jun3dPolygon With_with_with_(Jun3dPoint aPoint1, Jun3dPoint aPoint2, Jun3dPoint aPoint3) {
		return Vertexes_(new Jun3dPoint[] { aPoint1, aPoint2, aPoint3 });
	}

	/**
	 * Answer my area.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return this.parameterPolygon().area() * this.uVector().length() * this.vVector().length();
	}

	/**
	 * Answer my normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#normalVector()
	 * @category accessing
	 */
	public Jun3dPoint normalVector() {
		if (normalVector == null) {
			normalVector = this.preferredNormalVector();
		}

		return normalVector;
	}

	/**
	 * Answer the total number of points.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfPoints() {
		return this.pointsSize();
	}

	/**
	 * Answer my origin point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint origin() {
		return p1;
	}

	/**
	 * Answer the parameter at the specified index as a Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param anInteger int
	 * @category accessing
	 */
	public Jun2dPoint parameterPointAt_(int anInteger) {
		return this.parameterPolygon().pointAt_(anInteger);
	}

	/**
	 * Answer the parameter at the specified point as a Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun2dPoint parameterPointAtPoint_(Jun3dPoint aPoint) {
		Jun3dPoint thePoint = (Jun3dPoint) aPoint.minus_(p1);
		Jun3dPoint uVector = this.uVector();
		Jun3dPoint vVector = this.vVector();
		double u = uVector.dotProduct_(thePoint) / uVector.dotProduct_(uVector);
		double v = vVector.dotProduct_(thePoint) / vVector.dotProduct_(vVector);
		return new Jun2dPoint(u, v);
	}

	/**
	 * Answer my parameter polygon.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category accessing
	 */
	public final Jun2dPolygon parameterPolygon() {
		return parameterPolygon;
	}

	/**
	 * Answer the Jun3dPoint specified with the index number.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param anInteger int
	 * @category accessing
	 */
	public Jun3dPoint pointAt_(int anInteger) {
		return this.pointAtParameterPoint_(this.parameterPolygon().pointAt_(anInteger));
	}

	/**
	 * Answer the Jun3dPoint specified with the parameter point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	private Jun3dPoint pointAtParameterPoint_(Jun2dPoint aPoint) {
		return this.pointAtU_v_(aPoint.x(), aPoint.y());
	}

	/**
	 * Answer the Jun3dPoint specified with "u" and "v".
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param u double
	 * @param v double
	 * @category accessing
	 */
	private Jun3dPoint pointAtU_v_(double u, double v) {
		return p1.plus_(this.uVector().multipliedBy_(u)).plus_(this.vVector().multipliedBy_(v));
	}

	/**
	 * Answer the array of all points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public final Jun3dPoint[] points() {
		int size = this.pointsSize();
		Jun3dPoint[] points = new Jun3dPoint[size];
		for (int i = 0; i < size; i++) {
			points[i] = this.pointAt_(i);
		}
		return points;
	}

	/**
	 * Answer the number of points.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int pointsSize() {
		return this.parameterPolygon().pointsSize();
	}

	/**
	 * Answer the U vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint uVector() {
		return p2.minus_(p1);
	}

	/**
	 * Answer the V vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint vVector() {
		return p3.minus_(p1);
	}

	/**
	 * Answer the bounding box of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		if (boundingBox == null) {
			boundingBox = this.preferredBoundingBox();
		}
		return boundingBox;
	}

	/**
	 * Answer the preferable bounding box of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox preferredBoundingBox() {
		Jun3dPoint[] points = this.points();
		double minX = Double.POSITIVE_INFINITY;
		double minY = Double.POSITIVE_INFINITY;
		double minZ = Double.POSITIVE_INFINITY;
		double maxX = Double.NEGATIVE_INFINITY;
		double maxY = Double.NEGATIVE_INFINITY;
		double maxZ = Double.NEGATIVE_INFINITY;

		for (int i = 0; i < points.length; i++) {
			Jun3dPoint p = points[i];
			minX = Math.min(minX, p.x());
			minY = Math.min(minY, p.y());
			minZ = Math.min(minZ, p.z());
			maxX = Math.max(maxX, p.x());
			maxY = Math.max(maxY, p.y());
			maxZ = Math.max(maxZ, p.z());
		}

		return Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(minX, minY, minZ), new Jun3dPoint(maxX, maxY, maxZ));
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (super.equal_(anObject) == false) {
			return false;
		}

		Jun3dPolygon aPolygon = (Jun3dPolygon) anObject;
		return this.parameterPolygon().equal_(aPolygon.parameterPolygon());
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (super.equals(anObject) == false) {
			return false;
		}

		Jun3dPolygon aPolygon = (Jun3dPolygon) anObject;
		return this.parameterPolygon().equals(aPolygon.parameterPolygon());
	}

	/**
	 * Convert the receiver as an array of the convex polygons.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon[]
	 * @category converting
	 */
	public Jun3dPolygon[] asArrayOfConvexPolygons() {
		final ArrayList convexPolygons = new ArrayList();
		this.asConvexPolygonsDo_(new StBlockClosure() {
			public Object value_(Object polygon) {
				convexPolygons.add(polygon);
				return null;
			}
		});
		return (Jun3dPolygon[]) convexPolygons.toArray(new Jun3dPolygon[convexPolygons.size()]);
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @category converting
	 */
	public Jun3dLine[] asArrayOfLines() {
		return this.asArrayOf3dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dLines()
	 * @category converting
	 */
	public Jun3dLine[] asArrayOf3dLines() {
		Jun3dPoint[] points = this.points();
		boolean isLoop = points[0].equal_(points[points.length - 1]);
		Jun3dPoint[] collection = new Jun3dPoint[isLoop ? points.length : points.length + 1];
		for (int i = 0; i < points.length; i++) {
			collection[i] = new Jun3dPoint(points[i]);
		}
		if (isLoop == false) {
			collection[collection.length - 1] = new Jun3dPoint(points[0]);
		}
		Jun3dPolyline polyline = new Jun3dPolyline(collection);
		return polyline.asArrayOfLines();
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category converting
	 */
	public Jun3dPoint[] asArrayOfPoints() {
		return this.points();
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dTriangles()
	 * @category converting
	 */
	public Jun3dTriangle[] asArrayOf3dTriangles() {
		int numberOfPoints = this.numberOfPoints();
		if (numberOfPoints == 3) {
			Jun3dPoint[] points = this.points();
			return new Jun3dTriangle[] { Jun3dTriangle.On_on_on_(points[0], points[1], points[2]) };
		}

		Jun3dPolygon[] convexPolygons = this.asArrayOfConvexPolygons();
		ArrayList aList = new ArrayList(numberOfPoints - 2);
		for (int i = 0; i < convexPolygons.length; i++) {
			Jun3dPolygon convexPolygon = convexPolygons[i];
			for (int index = 1; index < (numberOfPoints - 1); index++) {
				aList.add(Jun3dTriangle.On_on_on_(convexPolygon.pointAt_(0), convexPolygon.pointAt_(index), convexPolygon.pointAt_(index + 1)));
			}
		}
		return (Jun3dTriangle[]) aList.toArray(new Jun3dTriangle[aList.size()]);
	}

	/**
	 * Convert the receiver as an array of triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon[]
	 * @category converting
	 */
	public Jun3dPolygon[] asArrayOfTrianglePolygons() {
		int numberOfPoints = this.numberOfPoints();
		if (numberOfPoints == 3) {
			return new Jun3dPolygon[] { this };
		}

		Jun3dPolygon[] convexPolygons = this.asArrayOfConvexPolygons();
		ArrayList aList = new ArrayList(numberOfPoints - 2);
		for (int i = 0; i < convexPolygons.length; i++) {
			Jun3dPolygon convexPolygon = convexPolygons[i];
			for (int index = 1; index < (numberOfPoints - 1); index++) {
				aList.add(Jun3dPolygon.Vertexes_(new Jun3dPoint[] { convexPolygon.pointAt_(0), convexPolygon.pointAt_(index), convexPolygon.pointAt_(index + 1) }));
			}
		}
		return (Jun3dPolygon[]) aList.toArray(new Jun3dPolygon[aList.size()]);
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon[]
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#asArrayOfTriangles()
	 * @category converting
	 */
	public Jun3dTriangle[] asArrayOfTriangles() {
		if (this.numberOfPoints() == 3) {
			return new Jun3dTriangle[] { new Jun3dTriangle(this.p1(), this.p2(), this.p3()) };
		}
		Jun3dPolygon[] convexPolygons = this.asArrayOfConvexPolygons();
		ArrayList triangles = new ArrayList(this.numberOfPoints() - 2);
		for (int i = 0; i < convexPolygons.length; i++) {
			for (int j = 1; j < convexPolygons[i].numberOfPoints() - 1; j++) {
				triangles.add(new Jun3dTriangle(convexPolygons[i].pointAt_(0), convexPolygons[i].pointAt_(j), convexPolygons[i].pointAt_(j + 1)));
			}
		}
		return (Jun3dTriangle[]) triangles.toArray(new Jun3dTriangle[triangles.size()]);
	}

	/**
	 * Convert the receiver as a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dObject aBody = null;
		if (this.isConvex()) {
			aBody = new JunOpenGL3dPolygon(this.points());
		} else {
			JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
			Jun3dPolygon[] polygons = this.asArrayOfConvexPolygons();
			for (int i = 0; i < polygons.length; i++) {
				compoundObject.add_(new JunOpenGL3dPolygon(polygons[i].points()));
			}
			aBody = compoundObject;
		}
		aBody.objectsDo_(new StBlockClosure() {
			public Object value_(Object object) {
				((JunOpenGL3dObject) object).paint_alpha_(Jun3dPolygon.this.defaultColor(), Float.NaN);
				return null;
			}
		});
		return aBody;
	}

	/**
	 * Convert the receiver as the <code>Jun3dPolyline</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category converting
	 */
	public Jun3dPolyline asPolyline() {
		return new Jun3dPolyline(this.points());
	}

	/**
	 * Convert the receiver as reversed.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#reversed()
	 * @category converting
	 */
	public JunPlane reversed() {
		Jun3dPoint[] points = this.points();
		int size = points.length;
		Jun3dPoint[] rPoints = new Jun3dPoint[size];
		for (int i = 0; i < size; i++) {
			rPoints[i] = points[size - i - 1];
		}
		return (Jun3dPolygon) _New(this.getClass(), rPoints);
	}

	/**
	 * Convert myself as a convex polygons, enumerate each polygons, and
	 * evaluate the block.
	 * 
	 * @return java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public Object asConvexPolygonsDo_(final StBlockClosure aBlock) {
		return this.parameterPolygon().asConvexPolygonsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun2dPolygon pPolygon = (Jun2dPolygon) anObject;
				Jun3dPolygon convexPolygon = (Jun3dPolygon) Jun3dPolygon.this.copy();
				convexPolygon.setParameterPolygon_(pPolygon);
				return aBlock.value_(convexPolygon);
			}
		});
	}

	/**
	 * Enumerate every line segments and evaluate the StBlockClosre.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void lineSegmentsDo_(StBlockClosure aBlock) {
		int numberOfPoints = this.numberOfPoints();
		for (int index = 0; index < numberOfPoints; index++) {
			aBlock.value_(new Jun3dLine(this.pointAt_(index), this.pointAt_((index + 1) % numberOfPoints)));
		}
	}

	/**
	 * Enumerate every parameter line segments and evaluate the StBlockClosre.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void parameterLineSegmentsDo_(StBlockClosure aBlock) {
		int numberOfPoints = this.numberOfPoints();
		for (int index = 0; index < numberOfPoints; index++) {
			aBlock.value_(new Jun2dLine(this.parameterPointAt_(index), this.parameterPointAt_((index + 1) % numberOfPoints)));
		}
	}

	/**
	 * Enumerate parameter points and evaluate the block.
	 * 
	 * @return java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public Object parameterPointsDo_(StBlockClosure aBlock) {
		for (int i = 0; i < this.numberOfPoints(); i++) {
			Object result = aBlock.value_(this.parameterPointAt_(i));
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Enumerate every points and evaluate the block.
	 * 
	 * @return java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public Object pointsDo_(StBlockClosure aBlock) {
		for (int i = 0; i < this.pointsSize(); i++) {
			Object result = aBlock.value_(this.pointAt_(i));
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Answer the Point which intersects with the Line.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#intersectingPointWithLine_(jp.co.sra.jun.geometry.curves.Jun3dLine)
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithLine_(Jun3dLine aLine) {
		Jun3dPoint intersectingPoint = super.intersectingPointWithLine_(aLine);
		if (intersectingPoint == null) {
			return null;
		}
		if (this.containsPoint_(intersectingPoint)) {
			return intersectingPoint;
		}
		return null;
	}

	/**
	 * Answer the intersecting point with the line segment Jun3dLine.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#intersectingPointWithLineSegment_(jp.co.sra.jun.geometry.curves.Jun3dLine)
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithLineSegment_(Jun3dLine aLine) {
		if (this.boundingBox().intersectsOrTouches_(aLine.boundingBox())) {
			Jun3dPoint intersectingPoint = super.intersectingPointWithLineSegment_(aLine);
			if (intersectingPoint == null) {
				return null;
			}
			if (this.containsPoint_(intersectingPoint)) {
				return intersectingPoint;
			} else {
				return null;
			}
		} else {
			return null;
		}
	}

	/**
	 * Answer the nearest point on the receiver from the Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#nearestPointFromPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category functions
	 */
	public Jun3dPoint nearestPointFromPoint_(Jun3dPoint aPoint) {
		return this.pointAtParameterPoint_(this.parameterPolygon().asPositivePolygon().nearestPointFromPoint_(this.parameterPointAtPoint_(aPoint)));
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write(this._className().toString());
		aWriter.write('(');
		Jun3dPoint[] points = this.points();
		for (int i = 0; i < points.length; i++) {
			points[i].printOn_(aWriter);
			aWriter.write(' ');
		}
		aWriter.write(')');
	}

	/**
	 * Answer true if the receiver contains the Jun3dLine as a line segment,
	 * otherwise false.
	 * 
	 * @return boolean
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category testing
	 */
	public boolean containsLineSegment_(Jun3dLine aLine) {
		return this.containsLineSegmentFrom_to_((Jun3dPoint) aLine.from(), (Jun3dPoint) aLine.to());
	}

	/**
	 * Answer true if the receiver contains the Jun3dLine as a line segment,
	 * otherwise false.
	 * 
	 * @return boolean
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category testing
	 */
	public boolean containsLineSegmentFrom_to_(Jun3dPoint aPoint1, Jun3dPoint aPoint2) {
		if ((this.containsPoint_(aPoint1) && this.containsPoint_(aPoint2)) == false) {
			return false;
		}
		return this.parameterPolygon().absContainsLineSegmentFrom_to_(this.parameterPointAtPoint_(aPoint1), this.parameterPointAtPoint_(aPoint2));
	}

	/**
	 * Answer true if the receiver contains the Jun3dPoint, otherwise false.
	 * 
	 * @return boolean
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunPlane#containsPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category testing
	 */
	public boolean containsPoint_(Jun3dPoint aPoint) {
		if (!super.containsPoint_(aPoint)) {
			return false;
		}
		return this.parameterPolygon().absContainsPoint_(this.parameterPointAtPoint_(aPoint));
	}

	/**
	 * Answer true if the receiver is consistent, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConsistent() {
		return this.parameterPolygon().isConsistent();
	}

	/**
	 * Answer true if the receiver is convex, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConvex() {
		return this.parameterPolygon().absIsConvex();
	}

	/**
	 * Answer the sign value of which side of the plane the receiver is located.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return int
	 * @category testing
	 */
	public int whichSideOf_(JunPlane aPlane) {
		boolean hasPlus = false;
		boolean hasMinus = false;
		boolean hasZero = false;

		Jun3dPoint[] points = this.points();
		for (int i = 0; i < points.length; i++) {
			int sign = points[i].whichSideOf_(aPlane);
			if (sign > 0) {
				hasPlus = true;
			} else if (sign < 0) {
				hasMinus = true;
			} else {
				hasZero = true;
			}
		}

		if (hasPlus && hasMinus) {
			return 0;
		}
		if (hasZero) {
			return 0;
		}
		if (hasMinus) {
			return -1;
		}
		return 1;
	}

	/**
	 * Answer the area of my parameter polygon.
	 * 
	 * @return double
	 * @category private
	 */
	protected double parameterArea() {
		return this.parameterPolygon().area();
	}

	/**
	 * Answer my preferable normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint preferredNormalVector() {
		Jun3dPoint vector = Jun3dPoint.Zero();
		int size = this.pointsSize();
		for (int i = 0; i < size; i++) {
			Jun3dPoint point1 = this.pointAt_(i);
			Jun3dPoint point2 = this.pointAt_((i + 1) % size);
			Jun3dPoint point3 = this.pointAt_((i + 2) % size);
			vector = vector.plus_((point2.minus_(point1)).vectorProduct_(point3.minus_(point2)));
		}
		return vector.unitVector();
	}

	/**
	 * Set the Jun2dPolygon as my parameter polygon.
	 * 
	 * @param aJun2dPolygon jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category private
	 */
	protected void setParameterPolygon_(Jun2dPolygon aJun2dPolygon) {
		if (aJun2dPolygon.isPositive()) {
			parameterPolygon = aJun2dPolygon;
		} else {
			parameterPolygon = aJun2dPolygon.transposed();

			Jun3dPoint oldP2 = p2;
			Jun3dPoint oldP3 = p3;
			p2 = oldP3;
			p3 = oldP2;
		}
	}

}
