package jp.co.sra.jun.geometry.solid;

import java.io.IOException;
import java.io.Writer;
import java.util.Arrays;

import jp.co.sra.jun.geometry.abstracts.JunSolid;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall;
import jp.co.sra.jun.geometry.surfaces.Jun3dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * JunTetrahedron class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/07/02 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun607 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTetrahedron.java,v 8.5 2008/02/20 06:30:58 nisinaka Exp $
 */
public class JunTetrahedron extends JunSolid {
	protected Jun3dPoint p1;
	protected Jun3dPoint p2;
	protected Jun3dPoint p3;
	protected Jun3dPoint p4;

	/**
	 * Create a new instance of <code>JunTetrahedron</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunTetrahedron() {
		super();
	}

	/**
	 * Create a new instance of <code>JunTetrahedron</code> and initialize it.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint4 jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public JunTetrahedron(JunPoint aPoint1, JunPoint aPoint2, JunPoint aPoint3, JunPoint aPoint4) {
		super();
		this.setP1_(aPoint1);
		this.setP2_(aPoint2);
		this.setP3_(aPoint3);
		this.setP4_(aPoint4);
	}

	/**
	 * Answer the receiver's area.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		double area = 0d;
		Jun3dTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			area = area + triangles[i].area();
		}
		return area;
	}

	/**
	 * Answer the receiver's area with sign.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double areaWithSign() {
		double areaWithSign = 0d;
		Jun3dTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			areaWithSign = areaWithSign + triangles[i].areaWithSign();
		}
		return areaWithSign;
	}

	/**
	 * Answer the receiver's p1.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p1() {
		return p1;
	}

	/**
	 * Answer the receiver's p2.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p2() {
		return p2;
	}

	/**
	 * Answer the receiver's p3.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p3() {
		return p3;
	}

	/**
	 * Answer the receiver's p4.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p4() {
		return p4;
	}

	/**
	 * Answer the receiver's triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category accessing 
	 */
	public Jun3dTriangle[] triangles() {
		return new Jun3dTriangle[] { p1.triangle_and_(p2, p3), p1.triangle_and_(p3, p4), p1.triangle_and_(p4, p2), p2.triangle_and_(p4, p3) };
	}

	/**
	 * Answer the receiver's volume.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#volume()
	 * @category accessing
	 */
	public double volume() {
		return Math.abs(this.volumeWithSign());
	}

	/**
	 * Answer the receiver's volume with sign.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double volumeWithSign() {
		double x21 = this.p2().x() - this.p1().x();
		double y21 = this.p2().y() - this.p1().y();
		double z21 = this.p2().z() - this.p1().z();
		double x31 = this.p3().x() - this.p1().x();
		double y31 = this.p3().y() - this.p1().y();
		double z31 = this.p3().z() - this.p1().z();
		double x41 = this.p4().x() - this.p1().x();
		double y41 = this.p4().y() - this.p1().y();
		double z41 = this.p4().z() - this.p1().z();
		double d34 = y31 * z41 - z31 * y41;
		double d24 = y21 * z41 - z21 * y41;
		double d23 = y21 * z31 - z21 * y31;
		double volumeWithSign = (x21 * d34 - x31 * d24 + x41 * d23) / 6;
		return volumeWithSign;
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || anObject.getClass() != this.getClass()) {
			return false;
		}

		JunTetrahedron aTetrahedron = (JunTetrahedron) anObject;
		return this.p1().equal_(aTetrahedron.p1()) && this.p2().equal_(aTetrahedron.p2()) && this.p3().equal_(aTetrahedron.p3()) && this.p4().equal_(aTetrahedron.p4());
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object.
	 * 
	 * @return boolean
	 * @param anObject java.lang.Object
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || anObject.getClass() != this.getClass()) {
			return false;
		}

		JunTetrahedron aTetrahedron = (JunTetrahedron) anObject;
		return this.p1().equals(aTetrahedron.p1()) && this.p2().equals(aTetrahedron.p2()) && this.p3().equals(aTetrahedron.p3()) && this.p4().equals(aTetrahedron.p4());
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		Jun3dTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			compoundObject.add_(triangles[i].asJunOpenGL3dObject());
		}
		return compoundObject;
	}

	/**
	 * Answer the receiver's bary center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint barycenter() {
		return this.centerOfGravity();
	}

	/**
	 * Answer the receiver's center point of circumcircle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint centerOfCircumcircle() {
		return this.circumsphere().center();
	}

	/**
	 * Answer the receiver's center point of gravity.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint centerOfGravity() {
		return new Jun3dPoint((this.p1().x() + this.p2().x() + this.p3().x() + this.p4().x()) / 4, (this.p1().y() + this.p2().y() + this.p3().y() + this.p4().y()) / 4, (this.p1().z() + this.p2().z() + this.p3().z() + this.p4().z()) / 4);
	}

	/**
	 * Answer the receiver's circum ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category functions
	 */
	public Jun3dBoundingBall circumball() {
		JunSphere aSphere = this.circumsphere();
		return new Jun3dBoundingBall(aSphere.center(), aSphere.radius());
	}

	/**
	 * Answer the receiver's circum center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint circumcenter() {
		return this.centerOfCircumcircle();
	}

	/**
	 * Answer the receiver's circum sphere.
	 * 
	 * @return jp.co.sra.jun.geometry.solid.JunSphere
	 * @category functions
	 */
	public JunSphere circumsphere() {
		JunPlane firstPlane = this.p1().bisector_(this.p2());
		JunPlane secondPlane = this.p2().bisector_(this.p3());
		JunPlane thirdPlane = this.p3().bisector_(this.p4());
		Jun3dPoint centerPoint = firstPlane.intersectingPointWithPlane_wihtPlane_(secondPlane, thirdPlane);
		double radiusValue = centerPoint.distance_(p1);
		return new JunSphere(centerPoint, radiusValue);
	}

	/**
	 * Answer the receiver's normal unit vectors.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category functions
	 */
	public Jun3dPoint[] normalUnitVectors() {
		Jun3dTriangle[] triangles = this.triangles();
		Jun3dPoint[] normalUnitVectors = new Jun3dPoint[triangles.length];
		for (int i = 0; i < triangles.length; i++) {
			normalUnitVectors[i] = triangles[i].normalUnitVector();
		}
		return normalUnitVectors;
	}

	/**
	 * Answer the receiver's normal vectors.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category functions
	 */
	public Jun3dPoint[] normalVectors() {
		Jun3dTriangle[] triangles = this.triangles();
		Jun3dPoint[] normalVectors = new Jun3dPoint[triangles.length];
		for (int i = 0; i < triangles.length; i++) {
			normalVectors[i] = triangles[i].normalVector();
		}
		return normalVectors;
	}

	/**
	 * Answer the receiver's radius of circum circle.
	 * 
	 * @return double
	 * @category functions
	 */
	public double radiusOfCircumcircle() {
		return this.circumsphere().radius();
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		this.p1().printOn_(aWriter);
		aWriter.write(" tetrahedron: ");
		this.p2().printOn_(aWriter);
		aWriter.write(" and: ");
		this.p3().printOn_(aWriter);
		aWriter.write(" and: ");
		this.p4().printOn_(aWriter);
		aWriter.write(')');
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		aWriter.write(" on: ");
		this.p1().printOn_(aWriter);
		aWriter.write(" on: ");
		this.p2().printOn_(aWriter);
		aWriter.write(" on: ");
		this.p3().printOn_(aWriter);
		aWriter.write(" on: ");
		this.p4().printOn_(aWriter);
		aWriter.write(')');
	}

	/**
	 * Answer <code>true</code> if the receiver is a tetrahedron, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isTetrahedron()
	 * @category testing
	 */
	public boolean isTetrahedron() {
		return true;
	}

	/**
	 * Answer the value which side of on a plane.
	 * 
	 * @param bisector jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return int
	 * @category testing
	 */
	public int whichSideOf_(JunPlane bisector) {
		Jun3dTriangle[] triangles = this.triangles();
		int[] anArray = new int[triangles.length];
		for (int i = 0; i < triangles.length; i++) {
			anArray[i] = triangles[i].whichSideOf_(bisector);
		}
		boolean hasPlus = Arrays.binarySearch(anArray, 1) != -1;
		boolean hasMinus = Arrays.binarySearch(anArray, -1) != -1;
		if (hasPlus && hasMinus) {
			return 0;
		}
		boolean hasZero = Arrays.binarySearch(anArray, 0) != -1;
		if (hasZero) {
			return 0;
		}
		if (hasMinus && hasPlus == false) {
			return -1;
		}
		return 1;
	}

	/**
	 * Set the receiver's p1.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP1_(JunPoint aPoint) {
		p1 = Jun3dPoint.Coerce_(aPoint);
	}

	/**
	 * Set the receiver's p2.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP2_(JunPoint aPoint) {
		p2 = Jun3dPoint.Coerce_(aPoint);
	}

	/**
	 * Set the receiver's p3.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP3_(JunPoint aPoint) {
		p3 = Jun3dPoint.Coerce_(aPoint);
	}

	/**
	 * Set the receiver's p4.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP4_(JunPoint aPoint) {
		p4 = Jun3dPoint.Coerce_(aPoint);
	}
}
