package jp.co.sra.jun.geometry.solid;

import java.io.IOException;
import java.io.Writer;
import java.util.Arrays;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.surfaces.Jun3dCircle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * JunCircularCylinder class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/29 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun610 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCircularCylinder.java,v 8.6 2008/02/20 06:30:58 nisinaka Exp $
 */
public class JunCircularCylinder extends JunSphere {
	protected double height;

	/**
	 * Create a new instance of <code>JunCircularCylinder</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunCircularCylinder() {
		super();
	}

	/**
	 * Create a new instance of <code>JunCircularCylinder</code> and initialize it.
	 * 
	 * @param centerPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param radiusValue double
	 * @param heightValue double
	 * @category Instance creation
	 */
	public JunCircularCylinder(JunPoint centerPoint, double radiusValue, double heightValue) {
		super();
		this.setCenter_(centerPoint);
		this.setRadius_(radiusValue);
		this.setHeight_(heightValue);
	}

	/**
	 * Answer my area.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return 2 * Math.PI * this.radius() * (this.height() + this.radius());
	}

	/**
	 * Answer the receiver's height.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double height() {
		return height;
	}

	/**
	 * Answer my volume.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#volume()
	 * @category accessing
	 */
	public double volume() {
		return Math.PI * Math.pow(this.radius(), 2) * this.height();
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (super.equal_(anObject) == false) {
			return false;
		}

		JunCircularCylinder aCircularCylinder = (JunCircularCylinder) anObject;
		return this.isEqualNumber_to_(this.h(), aCircularCylinder.h());
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object.
	 * 
	 * @return boolean
	 * @param anObject java.lang.Object
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (super.equals(anObject) == false) {
			return false;
		}

		JunCircularCylinder aCircularCylinder = (JunCircularCylinder) anObject;
		return this.h() == aCircularCylinder.h();
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dObject aCylinder = JunOpenGL3dObject.CylinderBy_radius_height_(10, this.radius(), this.height()).translatedBy_(this.center());
		aCylinder.objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunOpenGL3dObject) each).paint_(JunCircularCylinder.this.defaultColor());
				return null;
			}
		});
		return aCylinder;
	}

	/**
	 * Answer the receiver's h.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double h() {
		return this.height();
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		this.center().printOn_(aWriter);
		aWriter.write(" circularCylinder: ");
		aWriter.write(Double.toString(this.radius()));
		aWriter.write(" height: ");
		aWriter.write(Double.toString(this.height()));
		aWriter.write(')');
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		aWriter.write(" center: ");
		this.center().storeOn_(aWriter);
		aWriter.write(" radius: ");
		aWriter.write(Double.toString(this.radius()));
		aWriter.write(" height: ");
		aWriter.write(Double.toString(this.height()));
		aWriter.write(')');
	}

	/**
	 * Answer <code>true</code> if the receiver is a circular cylinder, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isCircularCylinder()
	 * @category testing
	 */
	public boolean isCircularCylinder() {
		return true;
	}

	/**
	 * Answer <code>true</code> if the receiver is a sphere, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isSphere()
	 * @category testing
	 */
	public boolean isSphere() {
		return false;
	}

	/**
	 * Answer the value which side of on a plane.
	 * 
	 * @param bisector jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return int
	 * @category testing
	 */
	public int whichSideOf_(JunPlane bisector) {
		int[] anArray = new int[] { new Jun3dCircle(this.center(), this.radius()).whichSideOf_(bisector), new Jun3dCircle(this.center().plus_(new Jun3dPoint(0, 0, this.height())), this.radius()).whichSideOf_(bisector) };
		boolean hasPlus = Arrays.binarySearch(anArray, 1) != -1;
		boolean hasMinus = Arrays.binarySearch(anArray, -1) != -1;
		if (hasPlus && hasMinus) {
			return 0;
		}
		boolean hasZero = Arrays.binarySearch(anArray, 0) != -1;
		if (hasZero) {
			return 0;
		}
		if (hasMinus && hasPlus == false) {
			return -1;
		}
		return 1;
	}

	/**
	 * Set the receiver's height.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	public void setHeight_(double aNumber) {
		height = aNumber;
	}
}
