package jp.co.sra.jun.geometry.forms;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;

import jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayProcessor;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;

/**
 * JunFormTriangulation3 class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/08 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun632 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFormTriangulation3.java,v 8.7 2008/02/20 06:30:57 nisinaka Exp $
 */
public class JunFormTriangulation3 extends JunAbstractFormTriangulation {
	protected Jun2dTriangle[] universalDelaunayTriangles;

	/**
	 * Create a new instance of <code>JunFormTriangulation3</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunFormTriangulation3() {
		super();
	}

	/**
	 * Create a new instance of <code>JunFormTriangulation3</code> and initialize it.
	 * 
	 * @param points
	 * @category Instance creation
	 */
	public JunFormTriangulation3(Jun2dPoint[] points) {
		super(points);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.forms.JunForm2dRegion#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		universalDelaunayTriangles = null;
	}

	/**
	 * Set the receiver's new points.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @see jp.co.sra.jun.geometry.forms.JunForm2dRegion#points_(jp.co.sra.jun.geometry.basic.Jun2dPoint[])
	 * @category accessing
	 */
	public void points_(Jun2dPoint[] pointCollection) {
		super.points_(pointCollection);
		universalDelaunayTriangles = null;
	}

	/**
	 * Answer the receiver's current triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @see jp.co.sra.jun.geometry.forms.JunAbstractFormTriangulation#triangles()
	 * @category accessing
	 */
	public Jun2dTriangle[] triangles() {
		return this.trianglesInterim_(null);
	}

	/**
	 * Answer my current triangles.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category accessing
	 */
	public Jun2dTriangle[] trianglesInterim_(StBlockClosure aBlock) {
		if (this.pointsWithoutLast().length < 3) {
			return new Jun2dTriangle[0];
		}

		return this.universalDelaunayTrianglesInterim_(aBlock);
	}

	/**
	 * Universal the constrained delaunay triangles.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category computing
	 */
	protected Jun2dTriangle[] universalDelaunayTrianglesInterim_(StBlockClosure aBlock) {
		if (universalDelaunayTriangles == null) {
			Collection triangleCollection = null;
			if (aBlock == null) {
				Jun2dDelaunayProcessor aProcessor = Jun2dDelaunayProcessor.FromPolygon_(this.pointsWithoutLast());
				JunPoint[][] trianglePoints = aProcessor.insideTriangles();
				triangleCollection = new ArrayList();
				for (int i = 0; i < trianglePoints.length; i++) {
					Jun2dTriangle triangle = new Jun2dTriangle(trianglePoints[i][0].as2dPoint(), trianglePoints[i][1].as2dPoint(), trianglePoints[i][2].as2dPoint());
					triangleCollection.add(triangle);
				}
			} else {
				aBlock.value_(new Jun2dTriangle[0]);
				Jun2dPoint[] points = this.pointsWithoutLast();
				Collection aStream = new ArrayList(points.length);
				for (int i = 0; i < points.length; i++) {
					aStream.add(points[i]);
					if (i >= 2) {
						Jun2dDelaunayProcessor aProcessor = Jun2dDelaunayProcessor.FromPolygon_((Jun2dPoint[]) aStream.toArray(new Jun2dPoint[aStream.size()]));
						JunPoint[][] trianglePoints = aProcessor.insideTriangles();
						triangleCollection = new ArrayList();
						for (int j = 0; j < trianglePoints.length; j++) {
							Jun2dTriangle triangle = new Jun2dTriangle(trianglePoints[j][0].as2dPoint(), trianglePoints[j][1].as2dPoint(), trianglePoints[j][2].as2dPoint());
							triangleCollection.add(triangle);
						}
						aBlock.value_(triangleCollection.toArray(new Jun2dTriangle[triangleCollection.size()]));
					}
				}
			}
			universalDelaunayTriangles = (Jun2dTriangle[]) triangleCollection.toArray(new Jun2dTriangle[triangleCollection.size()]);
		}
		return universalDelaunayTriangles;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		this.displayOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayOn_at_(Graphics graphicsContext, Point aPoint) {
		this.displayOn_at_triangles_color_(graphicsContext, aPoint, this.triangles(), Color.red);
	}

	/**
	 * Display the receiver on the graphics with the specified attributes.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param triangleCollection jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param colorValue java.awt.Color
	 * @category displaying
	 */
	public void displayOn_at_triangles_color_(Graphics graphicsContext, Point aPoint, Jun2dTriangle[] triangleCollection, Color colorValue) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();

		try {
			Rectangle boundingBox = this.boundingBox().asRectangle();
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);

			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.fill(boundingBox);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			for (int i = 0; i < triangleCollection.length; i++) {
				JunForm2dRegion aPolyline;
				if (triangleCollection[i].first().equals(triangleCollection[i].last())) {
					aPolyline = new JunForm2dRegion(triangleCollection[i].asPointArray());
				} else {
					Jun2dPoint[] points = triangleCollection[i].asPointArray();
					Jun2dPoint[] newPoints = new Jun2dPoint[points.length + 1];
					for (int j = 0; j < points.length; j++) {
						newPoints[j] = points[j];
					}
					newPoints[newPoints.length - 1] = points[0];
					aPolyline = new JunForm2dRegion(newPoints);
				}
				GeneralPath aGeneralPath = aPolyline.toGeneralPath();
				aGraphics.setColor(StColorValue.Blend(colorValue, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(colorValue);
				aGraphics.draw(aGeneralPath);
			}
			aGraphics.setColor(Color.black);
			aGraphics.draw(this.toGeneralPath());
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Display the receiver on the graphics with the specified attributes.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param triangleCollection jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category displaying
	 */
	public void displayOn_triangles_(Graphics graphicsContext, Jun2dTriangle[] triangleCollection) {
		this.displayOn_at_triangles_color_(graphicsContext, new Point(0, 0), triangleCollection, Color.red);
	}
}
