package jp.co.sra.jun.geometry.forms;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;

/**
 * JunFormTriangulation class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/07 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun666 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFormTriangulation.java,v 8.5 2008/02/20 06:30:57 nisinaka Exp $
 */
public class JunFormTriangulation extends JunFormOperation implements StDisplayable {
	protected JunAbstractFormTriangulation formTriangulation;
	protected Jun2dTriangle[] formTriangles;
	protected double formArea;

	/**
	 * Answer the default triangulation class.
	 * 
	 * @return java.lang.Class
	 * @category Defaults
	 */
	public static Class DefaultTriangulationClass() {
		return JunFormTriangulation3.class;
	}

	/**
	 * Create a new instance of <code>JunFormTriangulation</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunFormTriangulation() {
		super();
	}

	/**
	 * Create a new instance of <code>JunFormTriangulation</code> and initialize it.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunFormTriangulation(Jun2dPoint[] pointCollection) {
		this(pointCollection, DefaultTriangulationClass());
	}

	/**
	 * Create a new instance of <code>JunFormTriangulation</code> and initialize it.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunFormTriangulation(Jun2dPoint[] pointCollection, Class formTriangulationByWhom) {
		this();
		JunAbstractFormTriangulation aTriangulation = null;
		try {
			aTriangulation = (JunAbstractFormTriangulation) formTriangulationByWhom.newInstance();
		} catch (Exception e) {
			System.err.println(e.getStackTrace());
		}
		this.triangulation_(aTriangulation);
		this.points_(pointCollection);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.forms.JunFormOperation#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		formTriangulation = null;
		formTriangles = null;
		formArea = Double.NaN;
	}

	/**
	 * Answer the receiver's area value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double area() {
		double area = 0.0d;
		Jun2dTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			area = area + this.areaOfTriangle_(triangles[i].points());
		}
		return area;
	}

	/**
	 * Answer the receiver's area value with sign.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double areaWithSign() {
		double area = 0.0d;
		Jun2dTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			area = area + this.areaWithSignOfTriangle_(triangles[i].points());
		}
		return area;
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		return this.triangulation().boundingBox();
	}

	/**
	 * Answer the receiver's bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category accessing
	 */
	public Rectangle bounds() {
		return this.triangulation().bounds();
	}

	/**
	 * Answer the receiver's point collection.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		return this.triangulation().points();
	}

	/**
	 * Set the receiver's point collection.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category accessing
	 */
	public void points_(Jun2dPoint[] pointCollection) {
		this.triangulation().points_(pointCollection);
		formTriangles = null;
		formArea = Double.NaN;
	}

	/**
	 * Answer the receiver's segments as an array of <code>Jun2dLine</code>
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dLine[] segments() {
		return this.triangulation().segments();
	}

	/**
	 * Answer the receiver's triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @throws java.lang.IllegalStateException
	 * @category accessing
	 */
	public Jun2dTriangle[] triangles() {
		if (formTriangles == null) {
			Jun2dTriangle[] triangleCollection = this.triangulation().triangles();
			Collection aStream = new ArrayList(triangleCollection.length);
			double accuracy = this.defaultAccuracy();
			for (int i = 0; i < triangleCollection.length; i++) {
				Jun2dPoint[] points = triangleCollection[i].points();
				Collection collection = new ArrayList();
				for (int j = 0; j < points.length; j++) {
					int index = -1;
					Jun2dPoint[] myPoints = this.points();
					for (int k = 0; j < myPoints.length; k++) {
						if (myPoints[k].distance_(points[j]) < accuracy) {
							index = k;
							break;
						}
					}
					if (index < 0) {
						throw new IllegalStateException("unexpected error");
					}
					collection.add(this.points()[index]);
				}
				aStream.add((Jun2dPoint[]) collection.toArray(new Jun2dPoint[collection.size()]));
			}
			Jun2dPoint[][] pointCollection = (Jun2dPoint[][]) aStream.toArray(new Jun2dPoint[aStream.size()][]);
			aStream = new ArrayList(pointCollection.length);
			for (int i = 0; i < pointCollection.length; i++) {
				aStream.add(this.rightHandTriangle_(pointCollection[i]));
			}
			formTriangles = (Jun2dTriangle[]) aStream.toArray(new Jun2dTriangle[aStream.size()]);
		}
		return formTriangles;
	}

	/**
	 * Answer the receiver as <code>StImage</code>.
	 *
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		StImage anImage = new StImage(this.bounds().getSize());
		Graphics aGraphics = anImage.image().getGraphics();
		try {
			this.displayOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
		return anImage;
	}

	/**
	 * Convert to GeneralPath.
	 * 
	 * @return java.awt.geom.GeneralPath
	 * @category converting
	 */
	public GeneralPath toGeneralPath() {
		Jun2dPoint[] points = this.points();
		if (points == null || points.length == 0) {
			return null;
		}

		GeneralPath aGeneralPath = new GeneralPath(GeneralPath.WIND_EVEN_ODD, points.length);
		aGeneralPath.moveTo((float) points[0].x(), (float) points[0].y());
		for (int i = 1; i < points.length; i++) {
			aGeneralPath.lineTo((float) points[i].x(), (float) points[i].y());
		}
		return aGeneralPath;
	}

	/**
	 * Answer the default accuracy.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultAccuracy() {
		return this.triangulation().defaultAccuracy();
	}

	/**
	 * Answer the default so far z.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultSoFarZ() {
		return 29999;
	}

	/**
	 * Answer the default triangulation class.
	 * 
	 * @return java.lang.Class
	 * @category defaults
	 */
	public Class defaultTriangulationClass() {
		return DefaultTriangulationClass();
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		this.displayOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics graphicsContext, Point aPoint) {
		this.displayOn_at_triangles_color_(graphicsContext, aPoint, this.triangles(), Color.red);
	}

	/**
	 * Display the receiver on the graphics with the specified triangles and color at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param triangleCollection jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param colorValue java.awt.Color
	 * @category displaying
	 */
	public void displayOn_at_triangles_color_(Graphics graphicsContext, Point aPoint, Jun2dTriangle[] triangleCollection, Color colorValue) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();
		try {
			Point boundingBoxOrigin = this.boundingBox().origin()._toPoint();
			Point boundingBoxExtent = this.boundingBox().extent()._toPoint();
			Rectangle boundingBox = new Rectangle(boundingBoxOrigin.x, boundingBoxOrigin.y, boundingBoxExtent.x, boundingBoxExtent.y);
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.fillRect(boundingBox.x, boundingBox.y, boundingBox.width, boundingBox.height);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			for (int i = 0; i < triangleCollection.length; i++) {
				JunForm2dRegion aPolyline;
				if (triangleCollection[i].first().equals(triangleCollection[i].last())) {
					aPolyline = new JunForm2dRegion(triangleCollection[i].asPointArray());
				} else {
					Jun2dPoint[] points = triangleCollection[i].asPointArray();
					Jun2dPoint[] newPoints = new Jun2dPoint[points.length + 1];
					for (int j = 0; j < points.length; j++) {
						newPoints[j] = points[j];
					}
					newPoints[newPoints.length - 1] = points[0];
					aPolyline = new JunForm2dRegion(newPoints);
				}
				GeneralPath aGeneralPath = aPolyline.toGeneralPath();
				aGraphics.setColor(StColorValue.Blend(colorValue, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(colorValue);
				aGraphics.draw(aGeneralPath);
			}
			aGraphics.setColor(Color.black);
			aGraphics.draw(this.toGeneralPath());
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
	}

	/**
	 * Display the receiver on the graphics with the specified triangles at the specified point.s
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param triangleCollection jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category displaying
	 */
	public void displayOn_at_triangles_(Graphics graphicsContext, Point aPoint, Jun2dTriangle[] triangleCollection) {
		this.displayOn_at_triangles_color_(graphicsContext, aPoint, triangleCollection, Color.red);
	}

	/**
	 * Answer <code>true</code> if the receiver is contains the specified line segment, otherwise <code>false</code>.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return boolean
	 * @category testing
	 */
	public boolean containsLineSegment_(Jun2dLine aLine) {
		return this.triangulation().containsLineSegment_(aLine);
	}

	/**
	 * Answer <code>true</code> if the receiver is contains the specified point, otherwise <code>false</code>.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun2dPoint aPoint) {
		return this.triangulation().containsPoint_(aPoint);
	}

	/**
	 * Answer the area of triangle with the specified three points.
	 * 
	 * @param threePoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return double
	 * @category private
	 */
	protected double areaOfTriangle_(Jun2dPoint[] threePoints) {
		return Math.abs(this.areaWithSignOfTriangle_(threePoints));
	}

	/**
	 * Answer the area with sign of triangle with the specified three points.
	 * 
	 * @param threePoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return double
	 * @category private
	 */
	protected double areaWithSignOfTriangle_(Jun2dPoint[] threePoints) {
		Jun2dPoint p1 = threePoints[0];
		Jun2dPoint p2 = threePoints[1];
		Jun2dPoint p3 = threePoints[2];
		double x1 = p1.x();
		double y1 = p1.y();
		double x2 = p2.x();
		double y2 = p2.y();
		double x3 = p3.x();
		double y3 = p3.y();
		double area = ((x1 - x2) * (y1 + y2) + (x2 - x3) * (y2 + y3) + (x3 - x1) * (y3 + y1)) * 0.5d;
		return area;
	}

	/**
	 * Answer right hand triangle.
	 * 
	 * @param threePoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category private
	 */
	protected Jun2dTriangle rightHandTriangle_(Jun2dPoint[] threePoints) {
		Jun2dTriangle aTriangle = null;
		Jun3dPoint[] trianglePoints = new Jun3dPoint[threePoints.length];
		for (int i = 0; i < trianglePoints.length; i++) {
			trianglePoints[i] = new Jun3dPoint(threePoints[i].x(), threePoints[i].y(), 0);
		}
		JunPlane aPlane = new JunPlane(trianglePoints[0], trianglePoints[1], trianglePoints[2]);
		if (0d <= aPlane.valueF_(new Jun3dPoint(0, 0, this.defaultSoFarZ()))) {
			aTriangle = new Jun2dTriangle(threePoints[0], threePoints[1], threePoints[2]);
		} else {
			aTriangle = new Jun2dTriangle(threePoints[2], threePoints[1], threePoints[0]);
		}
		return aTriangle;
	}

	/**
	 * Answer the receiver's triangulation.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunAbstractFormTriangulation
	 * @category private
	 */
	protected JunAbstractFormTriangulation triangulation() {
		if (formTriangulation == null) {
			formTriangulation = (JunAbstractFormTriangulation) _New(this.defaultTriangulationClass());
		}
		return formTriangulation;
	}

	/**
	 * Set the receiver's triangulation.
	 * 
	 * @param formTriangulationByWhom jp.co.sra.jun.geometry.forms.JunAbstractFormTriangulation
	 * @category private
	 */
	protected void triangulation_(JunAbstractFormTriangulation formTriangulationByWhom) {
		formTriangulation = formTriangulationByWhom;
	}
}
