package jp.co.sra.jun.geometry.forms;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;

/**
 * JunFormTriangleNode class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/14 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun500 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFormTriangleNode.java,v 8.5 2008/02/20 06:30:57 nisinaka Exp $
 */
public class JunFormTriangleNode extends JunFormTriangle {
	protected Jun2dPoint m1;
	protected Jun2dPoint m2;
	protected Jun2dPoint m3;
	protected Jun2dPoint sp1;
	protected Jun2dPoint sp2;
	protected Jun2dPoint sp3;
	protected JunFormTriangleNode tn1;
	protected JunFormTriangleNode tn2;
	protected JunFormTriangleNode tn3;

	/**
	 * Create a new instance of <code>JunFormTriangleNode</code> and initialize it.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public JunFormTriangleNode(Jun2dPoint aPoint1, Jun2dPoint aPoint2, Jun2dPoint aPoint3) {
		super(aPoint1, aPoint2, aPoint3);
	}

	/**
	 * Create a new instance of <code>JunFormTriangleNode</code> and initialize it.
	 * 
	 * @param pointArray jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunFormTriangleNode(Jun2dPoint[] pointArray) {
		super(pointArray);
	}

	/**
	 * Create a new instance of <code>JunFormTriangleNode</code> and initialize it.
	 * 
	 * @param pointArray jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param spinePoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunFormTriangleNode(Jun2dPoint[] pointArray, Jun2dPoint[] spinePoints) {
		super(pointArray);

		for (int i = 0; i < spinePoints.length; i++) {
			if (spinePoints[i].equals(this.m1())) {
				this.setSp1_(spinePoints[i]);
				if (this.sp2() != null && this.sp3() != null) {
					break;
				}
			}
			if (spinePoints[i].equals(this.m2())) {
				this.setSp2_(spinePoints[i]);
				if (this.sp1() != null && this.sp3() != null) {
					break;
				}
			}
			if (spinePoints[i].equals(this.m3())) {
				this.setSp3_(spinePoints[i]);
				if (this.sp1() != null && this.sp2() != null) {
					break;
				}
			}
		}
	}

	/**
	 * Create a new instance of <code>JunFormTriangleNode</code> and initialize it.
	 * 
	 * @param aTriangle jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category Instance creation
	 */
	public JunFormTriangleNode(Jun2dTriangle aTriangle) {
		super(aTriangle);
	}

	/**
	 * Answer the against point at the specified mid point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint againstPointAtMidPoint_(Jun2dPoint aPoint) {
		if (this.m1().equals(aPoint)) {
			return p3;
		}
		if (this.m2().equals(aPoint)) {
			return p1;
		}
		if (this.m3().equals(aPoint)) {
			return p2;
		}
		return null;
	}

	/**
	 * Answer the receiver's m1.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint 
	 * @category accessing
	 */
	public Jun2dPoint m1() {
		if (m1 == null) {
			m1 = this.p1().plus_(this.p2().minus_(this.p1()).dividedBy_(2));
		}
		return m1;
	}

	/**
	 * Answer the receiver's m2.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint m2() {
		if (m2 == null) {
			m2 = this.p2().plus_(this.p3().minus_(this.p2()).dividedBy_(2));
		}
		return m2;
	}

	/**
	 * Answer the receiver's m3.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint m3() {
		if (m3 == null) {
			m3 = this.p3().plus_(this.p1().minus_(this.p3()).dividedBy_(2));
		}
		return m3;
	}

	/**
	 * Answer the receiver's points at the specified mid point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] pointsAtMidPoint_(Jun2dPoint aPoint) {
		if (this.m1().equals(aPoint)) {
			return new Jun2dPoint[] { this.p1(), this.p2() };
		}
		if (this.m2().equals(aPoint)) {
			return new Jun2dPoint[] { this.p2(), this.p3() };
		}
		if (this.m3().equals(aPoint)) {
			return new Jun2dPoint[] { this.p3(), this.p1() };
		}
		return null;
	}

	/**
	 * Answer the receiver's sp1.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint sp1() {
		return sp1;
	}

	/**
	 * Answer the receiver's sp2.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint sp2() {
		return sp2;
	}

	/**
	 * Answer the receiver's sp3.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint sp3() {
		return sp3;
	}

	/**
	 * Answer the receiver's spine points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] spinePoints() {
		Collection aStream = new ArrayList();
		if (this.sp1() != null) {
			aStream.add(this.sp1());
		}
		if (this.sp2() != null) {
			aStream.add(this.sp2());
		}
		if (this.sp3() != null) {
			aStream.add(this.sp3());
		}
		return (Jun2dPoint[]) aStream.toArray(new Jun2dPoint[aStream.size()]);
	}

	/**
	 * Enumerate each objects and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object do_(StBlockClosure aBlock) {
		return this.do_visited_(aBlock, new HashSet());
	}

	/**
	 * Enumerate each objects and evaluate the block with visited set.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aSet java.util.Set
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object do_visited_(StBlockClosure aBlock, Set aSet) {
		aSet.add(this);
		aBlock.value_(this);
		if (this.tn1() != null && aSet.contains(this.tn1) == false) {
			this.tn1().do_visited_(aBlock, aSet);
		}
		if (this.tn2() != null && aSet.contains(this.tn2) == false) {
			this.tn2().do_visited_(aBlock, aSet);
		}
		if (this.tn3() != null && aSet.contains(this.tn3) == false) {
			this.tn3().do_visited_(aBlock, aSet);
		}
		return null;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		if (this.isTerminal()) {
			aWriter.write("terminal");
		}
		if (this.isSleeve()) {
			aWriter.write("sleeve");
		}
		if (this.isJunction()) {
			aWriter.write("junction");
		}
		aWriter.write("Triangle (");
		Jun2dPoint[] points = this.points();
		for (int i = 0; i < points.length; i++) {
			points[i].printOn_(aWriter);
			if (i < points.length - 1) {
				aWriter.write(", ");
			}
		}
		aWriter.write(")");
	}

	/**
	 * Answer <code>true</code> if the receiver is junction, otherwise <code>false</code>
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isJunction() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is sleeve, otherwise <code>false</code>
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isSleeve() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is terminal, otherwise <code>false</code>
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isTerminal() {
		return false;
	}

	/**
	 * Flush the receiver's triangle nodes.
	 * 
	 * @category triangle accessing
	 */
	public void flushTriangleNodes() {
		this.tn1_(null);
		this.tn2_(null);
		this.tn3_(null);
	}

	/**
	 * Answer the three triangles with two points.
	 * 
	 * @param midPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param midPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category triangle accessing
	 */
	public Jun2dPoint[][] threeTriangles_with_(Jun2dPoint midPoint1, Jun2dPoint midPoint2) {
		Jun2dPoint againstPoint1 = this.againstPointAtMidPoint_(midPoint1);
		Jun2dPoint againstPoint2 = this.againstPointAtMidPoint_(midPoint2);
		Jun2dPoint[] points = this.points();
		Jun2dPoint otherPoint = null;
		for (int i = 0; i < points.length; i++) {
			if (points[i].equals(againstPoint1) == false && points[i].equals(againstPoint2) == false) {
				otherPoint = points[i];
				break;
			}
		}

		Jun2dPoint[][] arrayOfTrianglePoints = new Jun2dPoint[3][];

		Jun2dPoint[] trianglePoints = new Jun2dPoint[] { midPoint1, midPoint2, otherPoint };
		arrayOfTrianglePoints[0] = trianglePoints;
		if (midPoint1.distance_(againstPoint1) < midPoint2.distance_(againstPoint2)) {
			trianglePoints = new Jun2dPoint[] { midPoint1, midPoint2, againstPoint1 };
			arrayOfTrianglePoints[1] = trianglePoints;
			trianglePoints = new Jun2dPoint[] { midPoint1, againstPoint1, againstPoint2 };
			arrayOfTrianglePoints[2] = trianglePoints;
		} else {
			trianglePoints = new Jun2dPoint[] { midPoint1, midPoint2, againstPoint2 };
			arrayOfTrianglePoints[1] = trianglePoints;
			trianglePoints = new Jun2dPoint[] { midPoint2, againstPoint2, againstPoint1 };
			arrayOfTrianglePoints[2] = trianglePoints;
		}
		return arrayOfTrianglePoints;
	}

	/**
	 * Answer the receiver's tn1.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode
	 * @category triangle accessing
	 */
	public JunFormTriangleNode tn1() {
		return tn1;
	}

	/**
	 * Set the receiver's tn1.
	 * 
	 * @param triangleNode jp.co.sra.jun.geometry.forms.JunFormTriangleNode
	 * @category triangle accessing
	 */
	public void tn1_(JunFormTriangleNode triangleNode) {
		tn1 = triangleNode;
	}

	/**
	 * Answer the receiver's tn2.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode
	 * @category triangle accessing
	 */
	public JunFormTriangleNode tn2() {
		return tn2;
	}

	/**
	 * Set the receiver's tn2.
	 * 
	 * @param triangleNode jp.co.sra.jun.geometry.forms.JunFormTriangleNode
	 * @category triangle accessing
	 */
	public void tn2_(JunFormTriangleNode triangleNode) {
		tn2 = triangleNode;
	}

	/**
	 * Answer the receiver's tn3.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode
	 * @category triangle accessing
	 */
	public JunFormTriangleNode tn3() {
		return tn3;
	}

	/**
	 * Set the receiver's tn3.
	 * 
	 * @param triangleNode jp.co.sra.jun.geometry.forms.JunFormTriangleNode
	 * @category triangle accessing
	 */
	public void tn3_(JunFormTriangleNode triangleNode) {
		tn3 = triangleNode;
	}

	/**
	 * Answer the receiver's triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]
	 * @category triangle accessing
	 */
	public JunFormTriangleNode[] triangleNodes() {
		Collection aStream = new ArrayList();
		if (this.tn1() != null) {
			aStream.add(this.tn1());
		}
		if (this.tn2() != null) {
			aStream.add(this.tn2());
		}
		if (this.tn3() != null) {
			aStream.add(this.tn3());
		}
		return (JunFormTriangleNode[]) aStream.toArray(new JunFormTriangleNode[aStream.size()]);
	}

	/**
	 * Set the receiver's p1.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.geometry.forms.JunFormTriangle#setP1_(jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category private
	 */
	protected void setP1_(Jun2dPoint aPoint) {
		super.setP1_(aPoint);
		m3 = null;
		m1 = null;
	}

	/**
	 * Set the receiver's p2.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.geometry.forms.JunFormTriangle#setP2_(jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category private
	 */
	protected void setP2_(Jun2dPoint aPoint) {
		super.setP2_(aPoint);
		m1 = null;
		m2 = null;
	}

	/**
	 * Set the receiver's p3.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.geometry.forms.JunFormTriangle#setP3_(jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category private
	 */
	protected void setP3_(Jun2dPoint aPoint) {
		super.setP3_(aPoint);
		m2 = null;
		m3 = null;
	}

	/**
	 * Set the receiver's sp1.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected void setSp1_(Jun2dPoint aPoint) {
		sp1 = aPoint;
	}

	/**
	 * Set the receiver's sp2.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected void setSp2_(Jun2dPoint aPoint) {
		sp2 = aPoint;
	}

	/**
	 * Set the receiver's sp3.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected void setSp3_(Jun2dPoint aPoint) {
		sp3 = aPoint;
	}
}
