package jp.co.sra.jun.geometry.forms;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolylineLoop;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex;

/**
 * JunFormCreation class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/06 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun668 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFormCreation.java,v 8.6 2008/02/20 06:30:57 nisinaka Exp $
 */
public class JunFormCreation extends JunFormOperation {
	protected JunFormTriangulation formTriangulation;
	protected JunFormTriangleNode[] draftTriangleNodes;
	protected JunFormTriangleNode[] triangleNodes;
	protected Map chordalAxesTable;
	protected Jun2dPoint[][][] chordalAxesInformation;
	protected Collection fanTriangles;
	protected Object[][][][] spinePolylineInformation;
	protected JunFormTriangle[] finalTriangles;
	protected Jun2dPoint[][] finalSpines;
	protected JunOpenGL3dCompoundObject finalBody;

	/**
	 * Create a new instance of <code>JunFormCreation</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunFormCreation() {
		super();
	}

	/**
	 * Create a new instance of <code>JunFormCreation</code> and initialize it.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunFormCreation(Jun2dPoint[] points) {
		this();
		this.points_(points);
	}

	/**
	 * Initialize the receiver
	 * 
	 * @see jp.co.sra.jun.geometry.forms.JunFormOperation#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		formTriangulation = null;
		draftTriangleNodes = null;
		triangleNodes = null;
		chordalAxesTable = null;
		chordalAxesInformation = null;
		fanTriangles = null;
		spinePolylineInformation = null;
		finalTriangles = null;
		finalSpines = null;
		finalBody = null;
		this.triangulation();
	}

	/**
	 * Answer the receiver's body as <code>JunOpenGL3dCompoundObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category accessing
	 */
	public JunOpenGL3dCompoundObject body() {
		if (finalBody == null) {
			finalBody = this.computeFinalBody();
		}
		return finalBody;
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return java.awt.Rectangle
	 * @category accessing
	 */
	public Rectangle bounds() {
		return formTriangulation.bounds();
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		return formTriangulation.boundingBox();
	}

	/**
	 * Amswer the receiver's points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		return formTriangulation.points();
	}

	/**
	 * Set the receiver's points.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public void points_(Jun2dPoint[] pointCollection) {
		Collection aStream = new ArrayList(pointCollection.length);
		for (int i = 0; i < pointCollection.length; i++) {
			aStream.add(pointCollection[i].rounded());
		}
		List aCollection = new ArrayList(aStream);
		if (aCollection.isEmpty() == false && aCollection.get(0).equals(aCollection.get(aCollection.size() - 1)) == false) {
			aStream.add(aCollection.get(0));
		}

		formTriangulation = this.triangulation();
		formTriangulation.points_((Jun2dPoint[]) aStream.toArray(new Jun2dPoint[aStream.size()]));
		draftTriangleNodes = null;
		triangleNodes = null;
		chordalAxesTable = null;
		chordalAxesInformation = null;
		fanTriangles = null;
		spinePolylineInformation = null;
		finalTriangles = null;
		finalSpines = null;
		finalBody = null;
	}

	/**
	 * Answer the receiver's spines.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category accessing
	 */
	public Jun2dPoint[][] spines() {
		if (finalSpines == null) {
			finalSpines = this.computeFinalSpines();
		}
		return finalSpines;
	}

	/**
	 * Answer the receiver's triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangle[]
	 * @category accessing
	 */
	public JunFormTriangle[] triangles() {
		if (finalTriangles == null) {
			finalTriangles = this.computeFinalTriangles();
		}
		return finalTriangles;
	}

	/**
	 * Answer the receiver's triangulated triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[];
	 * @category accessing
	 */
	public Jun2dTriangle[] triangulatedTriangles() {
		return this.triangulation().triangles();
	}

	/**
	 * Answer the receiver's triangulation.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangulation
	 * @category accessing
	 */
	public JunFormTriangulation triangulation() {
		if (formTriangulation == null) {
			formTriangulation = new JunFormTriangulation();
		}
		return formTriangulation;
	}

	/**
	 * Complex body and answer it.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject complexBody() {
		final JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		this.complexBody1().do_(new StBlockClosure() {
			public Object value_(Object each) {
				body.add_((JunOpenGL3dObject) each);
				return null;
			}
		});
		this.complexBody2().do_(new StBlockClosure() {
			public Object value_(Object each) {
				body.add_((JunOpenGL3dObject) each);
				return null;
			}
		});
		body.paint_(this.defaultPaint());
		return body;
	}

	/**
	 * Complex body 0.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject complexBody0() {
		return this.simpleBody0();
	}

	/**
	 * Complex body 1.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject complexBody1() {
		Map table = this.tableForSpinePointToLiftUpAmount();
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		JunFormTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint[] points = triangles[i].points();
			Jun3dPoint[] trianglePoints = new Jun3dPoint[points.length];
			for (int j = 0; j < points.length; j++) {
				Number aNumber = (Number) table.get(points[j]);
				trianglePoints[j] = new Jun3dPoint(points[j].x(), points[j].y(), (aNumber != null ? 0 - aNumber.doubleValue() : 0d));
			}
			List spinePoints = new ArrayList();
			for (int j = 0; j < points.length; j++) {
				if (table.containsKey(points[j])) {
					spinePoints.add(new Jun3dPoint(points[j].x(), points[j].y(), 0));
				}
			}
			if (spinePoints.size() < 1 || 2 < spinePoints.size()) {
				throw new IllegalStateException("unexpected error");
			}
			if (spinePoints.size() == 1) {
				Jun3dPoint sp1 = (Jun3dPoint) spinePoints.get(0);
				Jun3dPoint p1 = null;
				Jun3dPoint p2 = null;
				Jun3dPoint p3 = null;
				for (int j = 0; j < trianglePoints.length; j++) {
					if (trianglePoints[j].z() == 0d) {
						if (p1 == null) {
							p1 = trianglePoints[j];
						}
						p2 = trianglePoints[j];
					} else if (p3 == null) {
						p3 = trianglePoints[j];
					}
				}
				Jun3dPoint[] points1 = this.pointsOval_horizontal_vertical_(sp1, p1, p3);
				Jun3dPoint[] points2 = this.pointsOval_horizontal_vertical_(sp1, p2, p3);
				JunOpenGL3dObject object = this.sewPoints_withPoints_reverseFlag_(points1, points2, false);
				body.add_(object);
			}
			if (spinePoints.size() == 2) {
				Jun3dPoint sp1 = (Jun3dPoint) spinePoints.get(0);
				Jun3dPoint sp2 = (Jun3dPoint) spinePoints.get(spinePoints.size() - 1);
				Jun3dPoint p1 = null;
				Jun3dPoint p2 = null;
				Jun3dPoint p3 = null;
				for (int j = 0; j < trianglePoints.length; j++) {
					if (trianglePoints[j].z() != 0d) {
						if (p1 == null) {
							p1 = trianglePoints[j];
						}
						p2 = trianglePoints[j];
					} else if (p3 == null) {
						p3 = trianglePoints[j];
					}
				}
				Jun3dPoint[] points1 = this.pointsOval_horizontal_vertical_(sp1, p3, p1);
				Jun3dPoint[] points2 = this.pointsOval_horizontal_vertical_(sp2, p3, p2);
				JunOpenGL3dObject object = this.sewPoints_withPoints_reverseFlag_(points1, points2, false);
				body.add_(object);
			}
		}
		return body;
	}

	/**
	 * Complex body 2.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject complexBody2() {
		Map table = this.tableForSpinePointToLiftUpAmount();
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		JunFormTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint[] points = triangles[i].points();
			Jun3dPoint[] trianglePoints = new Jun3dPoint[points.length];
			for (int j = 0; j < points.length; j++) {
				Number aNumber = (Number) table.get(points[j]);
				trianglePoints[j] = new Jun3dPoint(points[j].x(), points[j].y(), (aNumber != null ? aNumber.doubleValue() : 0d));
			}
			Collection aCollection = new ArrayList();
			for (int j = 0; j < points.length; j++) {
				if (table.containsKey(points[j])) {
					aCollection.add(new Jun3dPoint(points[j].x(), points[j].y(), 0));
				}
			}
			Jun3dPoint[] spinePoints = (Jun3dPoint[]) aCollection.toArray(new Jun3dPoint[aCollection.size()]);
			if (spinePoints.length < 1 || 2 < spinePoints.length) {
				throw new IllegalStateException("unexpected error");
			}
			if (spinePoints.length == 1) {
				Jun3dPoint sp1 = spinePoints[0];
				Jun3dPoint p1 = null;
				Jun3dPoint p2 = null;
				Jun3dPoint p3 = null;
				for (int j = 0; j < trianglePoints.length; j++) {
					if (trianglePoints[j].z() == 0d) {
						if (p1 == null) {
							p1 = trianglePoints[j];
						}
						p2 = trianglePoints[j];
					} else if (p3 == null) {
						p3 = trianglePoints[j];
					}
				}
				Jun3dPoint[] points1 = this.pointsOval_horizontal_vertical_(sp1, p1, p3);
				Jun3dPoint[] points2 = this.pointsOval_horizontal_vertical_(sp1, p2, p3);
				JunOpenGL3dObject object = this.sewPoints_withPoints_reverseFlag_(points1, points2, true);
				body.add_(object);
			}
			if (spinePoints.length == 2) {
				Jun3dPoint sp1 = spinePoints[0];
				Jun3dPoint sp2 = spinePoints[spinePoints.length - 1];
				Jun3dPoint p1 = null;
				Jun3dPoint p2 = null;
				Jun3dPoint p3 = null;
				for (int j = 0; j < trianglePoints.length; j++) {
					if (trianglePoints[j].z() != 0d) {
						if (p1 == null) {
							p1 = trianglePoints[j];
						}
						p2 = trianglePoints[j];
					} else if (p3 == null) {
						p3 = trianglePoints[j];
					}
				}
				Jun3dPoint[] points1 = this.pointsOval_horizontal_vertical_(sp1, p3, p1);
				Jun3dPoint[] points2 = this.pointsOval_horizontal_vertical_(sp2, p3, p2);
				JunOpenGL3dObject object = this.sewPoints_withPoints_reverseFlag_(points1, points2, true);
				body.add_(object);
			}
		}
		return body;
	}

	/**
	 * Answer the receiver's oval factors.
	 * 
	 * @return double[][]
	 * @category body creation
	 */
	public double[][] ovalFactors() {
		double[] factors = this.defaultOvalFactors();
		double[][] stream = new double[factors.length][];
		for (int i = 0; i < factors.length; i++) {
			double degrees = factors[i];
			double radians = Math.toRadians(degrees);
			double x = Math.cos(radians);
			double y = Math.sin(radians);
			if (x < 1.0e-12d) {
				x = 0.0d;
			}
			stream[i] = new double[] { x, y };
		}
		return stream;
	}

	/**
	 * Answer the oval points with the specified points.
	 * 
	 * @param cPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param hPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param vPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category body creation
	 */
	public Jun3dPoint[] pointsOval_horizontal_vertical_(Jun3dPoint cPoint, Jun3dPoint hPoint, Jun3dPoint vPoint) {
		double[][] factors = this.ovalFactors();
		Collection stream = new ArrayList();
		for (int i = 0; i < factors.length; i++) {
			Jun3dPoint point = hPoint.minus_(cPoint).multipliedBy_(factors[i][0]).plus_(cPoint);
			point = new Jun3dPoint(point.x(), point.y(), (vPoint.z() - cPoint.z()) * factors[i][1]);
			stream.add(point);
		}
		return (Jun3dPoint[]) stream.toArray(new Jun3dPoint[stream.size() - 1]);
	}

	/**
	 * Sew the specified points with another points with reverse flag.
	 * 
	 * @param pointCollection1 jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param pointCollection2 jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aBoolean boolean
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject sewPoints_withPoints_reverseFlag_(Jun3dPoint[] pointCollection1, Jun3dPoint[] pointCollection2, boolean aBoolean) {
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		Jun3dPoint[] points1 = null;
		Jun3dPoint[] points2 = null;
		if (pointCollection1[0].equals(pointCollection2[0])) {
			points1 = new Jun3dPoint[pointCollection1.length];
			for (int i = 0; i < pointCollection1.length; i++) {
				points1[i] = pointCollection1[pointCollection1.length - 1 - i];
			}
			points2 = new Jun3dPoint[pointCollection2.length];
			for (int i = 0; i < pointCollection2.length; i++) {
				points2[i] = pointCollection2[pointCollection2.length - 1 - i];
			}
		} else {
			points1 = pointCollection1;
			points2 = pointCollection2;
		}
		for (int index = 0; index < points1.length - 1; index++) {
			Jun3dPoint p1 = points1[index];
			Jun3dPoint p2 = points2[index];
			Jun3dPoint p3 = points1[index + 1];
			Jun3dPoint p4 = points2[index + 1];
			Jun3dPoint[] points = this.sortTrianglePoints_(new Jun3dPoint[] { p1, p2, p3 });
			if (aBoolean == true) {
				Jun3dPoint[] reverse = new Jun3dPoint[points.length];
				for (int i = 0; i < points.length; i++) {
					reverse[i] = points[points.length - 1 - i];
				}
				points = reverse;
			}
			JunOpenGL3dPolygon object = new JunOpenGL3dPolygon(points);
			body.add_(object);
			if (p3.equals(p4) == false) {
				points = this.sortTrianglePoints_(new Jun3dPoint[] { p2, p3, p4 });
				if (aBoolean == true) {
					Jun3dPoint[] reverse = new Jun3dPoint[points.length];
					for (int i = 0; i < points.length; i++) {
						reverse[i] = points[points.length - 1 - i];
					}
					points = reverse;
				}
				JunOpenGL3dPolygon object2 = new JunOpenGL3dPolygon(points);
				body.add_(object2);
			}
		}
		return body;
	}

	/**
	 * Answer the simple body.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject simpleBody() {
		final JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		this.simpleBody1().objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				if (each instanceof JunOpenGL3dPolygon) {
					body.add_((JunOpenGL3dPolygon) each);
				}
				return null;
			}
		});
		this.simpleBody2().objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				if (each instanceof JunOpenGL3dPolygon) {
					body.add_((JunOpenGL3dPolygon) each);
				}
				return null;
			}
		});
		body.paint_(this.defaultPaint());
		return body;
	}

	/**
	 * Answer the simple body 0.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject simpleBody0() {
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		JunFormTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint[] points = triangles[i].points();
			Jun3dPoint[] vertexes = new Jun3dPoint[points.length];
			for (int j = 0; j < vertexes.length; j++) {
				vertexes[j] = new Jun3dPoint(points[j].x(), points[j].y(), 0);
			}
			JunOpenGL3dPolylineLoop polyline = new JunOpenGL3dPolylineLoop(vertexes);
			JunOpenGL3dPolygon polygon = new JunOpenGL3dPolygon(vertexes);
			body.add_(new JunOpenGL3dCompoundObject(polyline, polygon));
		}
		return body;
	}

	/**
	 * Answer the simple body 1.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject simpleBody1() {
		Map table = this.tableForSpinePointToLiftUpAmount();
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		JunFormTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint[] points = triangles[i].points();
			Jun3dPoint[] vertexes = new Jun3dPoint[points.length];
			for (int j = 0; j < vertexes.length; j++) {
				Number aNumber = (Number) table.get(points[j]);
				double z = (aNumber != null) ? 0d - aNumber.doubleValue() : 0d;
				vertexes[j] = new Jun3dPoint(points[j].x(), points[j].y(), z);
			}
			JunOpenGL3dPolylineLoop polyline = new JunOpenGL3dPolylineLoop(vertexes);
			JunOpenGL3dPolygon polygon = new JunOpenGL3dPolygon(vertexes);
			body.add_(new JunOpenGL3dCompoundObject(polyline, polygon));
		}
		return body;
	}

	/**
	 * Answer the simple body 2.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject simpleBody2() {
		Map table = this.tableForSpinePointToLiftUpAmount();
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		JunFormTriangle[] triangles = this.triangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint[] points = triangles[i].points();
			Jun3dPoint[] vertexes = new Jun3dPoint[points.length];
			for (int j = 0; j < vertexes.length; j++) {
				Number aNumber = (Number) table.get(points[j]);
				double z = (aNumber != null) ? aNumber.doubleValue() : 0d;
				vertexes[vertexes.length - 1 - j] = new Jun3dPoint(points[j].x(), points[j].y(), z);
			}
			JunOpenGL3dPolylineLoop polyline = new JunOpenGL3dPolylineLoop(vertexes);
			JunOpenGL3dPolygon polygon = new JunOpenGL3dPolygon(vertexes);
			body.add_(new JunOpenGL3dCompoundObject(polyline, polygon));
		}
		return body;
	}

	/**
	 * Answer the spine body 0.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject spineBody0() {
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		Jun2dPoint[][] spines = this.spines();
		for (int i = 0; i < spines.length; i++) {
			Jun2dPoint[] spine = spines[i];
			if (spine.length == 1) {
				Jun3dPoint point = new Jun3dPoint(spine[0].x(), spine[0].y(), 0);
				JunOpenGL3dVertex vertex = new JunOpenGL3dVertex(point);
				body.add_(vertex);
			}
			if (spine.length > 1) {
				Jun3dPoint[] points = new Jun3dPoint[spine.length];
				for (int j = 0; j < points.length; j++) {
					points[j] = new Jun3dPoint(spine[j].x(), spine[j].y(), 0);
				}
				JunOpenGL3dPolyline polyline = new JunOpenGL3dPolyline(points);
				body.add_(polyline);
			}
		}
		return body;
	}

	/**
	 * Answer the spine body 1.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject spineBody1() {
		Map table = this.tableForSpinePointToLiftUpAmount();
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		Jun2dPoint[][] spines = this.spines();
		for (int i = 0; i < spines.length; i++) {
			Jun2dPoint[] spine = spines[i];
			if (spine.length == 1) {
				Number aNumber = (Number) table.get(spine[0]);
				double z = (aNumber != null) ? 0d - aNumber.doubleValue() : 0d;
				Jun3dPoint point = new Jun3dPoint(spine[0].x(), spine[0].y(), z);
				JunOpenGL3dVertex vertex = new JunOpenGL3dVertex(point);
				body.add_(vertex);
			}
			if (spine.length > 1) {
				Jun3dPoint[] points = new Jun3dPoint[spine.length];
				for (int j = 0; j < points.length; j++) {
					Number aNumber = (Number) table.get(spine[j]);
					double z = (aNumber != null) ? 0d - aNumber.doubleValue() : 0d;
					points[j] = new Jun3dPoint(spine[j].x(), spine[j].y(), z);
				}
				JunOpenGL3dPolyline polyline = new JunOpenGL3dPolyline(points);
				body.add_(polyline);
			}
		}
		return body;
	}

	/**
	 * Answer the spine body 2.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @throws java.lang.IllegalStateException
	 * @category body creation
	 */
	public JunOpenGL3dCompoundObject spineBody2() {
		Map table = this.tableForSpinePointToLiftUpAmount();
		JunOpenGL3dCompoundObject body = new JunOpenGL3dCompoundObject();
		Jun2dPoint[][] spines = this.spines();
		for (int i = 0; i < spines.length; i++) {
			Jun2dPoint[] spine = spines[i];
			if (spine.length == 1) {
				Number aNumber = (Number) table.get(spine[0]);
				double z = (aNumber != null) ? aNumber.doubleValue() : 0d;
				Jun3dPoint point = new Jun3dPoint(spine[0].x(), spine[0].y(), z);
				JunOpenGL3dVertex vertex = new JunOpenGL3dVertex(point);
				body.add_(vertex);
			}
			if (spine.length > 1) {
				Jun3dPoint[] points = new Jun3dPoint[spine.length];
				for (int j = 0; j < points.length; j++) {
					Number aNumber = (Number) table.get(spine[j]);
					double z = (aNumber != null) ? aNumber.doubleValue() : 0d;
					points[j] = new Jun3dPoint(spine[j].x(), spine[j].y(), z);
				}
				JunOpenGL3dPolyline polyline = new JunOpenGL3dPolyline(points);
				body.add_(polyline);
			}
		}
		return body;
	}

	/**
	 * Answer the receiver's map table for spine point to lift up amount.
	 * 
	 * @return java.util.Map
	 * @category body creation
	 */
	public Map tableForSpinePointToLiftUpAmount() {
		Map table = this.tableForSpinePointToTriangles();
		Set points = table.keySet();
		Map dictionary = new HashMap(points.size());
		Iterator iterator = table.keySet().iterator();
		while (iterator.hasNext()) {
			Jun2dPoint point = (Jun2dPoint) iterator.next();
			JunFormTriangle[] triangles = (JunFormTriangle[]) table.get(point);
			Set set = new HashSet();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] trianglePoints = triangles[i].points();
				for (int j = 0; j < trianglePoints.length; j++) {
					if (points.contains(trianglePoints[j]) == false) {
						set.add(trianglePoints[j]);
					}
				}
			}
			double total = 0.0d;
			Jun2dPoint[] setPoints = (Jun2dPoint[]) set.toArray(new Jun2dPoint[set.size()]);
			for (int i = 0; i < setPoints.length; i++) {
				total = total + point.distance_(setPoints[i]);
			}
			double amount = total / set.size();
			dictionary.put(point, new Double(amount));
		}

		Jun2dPoint[][] spines = this.spines();
		for (int i = 0; i < spines.length; i++) {
			for (int index = 1; index < spines[i].length - 1; index++) {
				double amount1 = ((Number) dictionary.get(spines[i][index - 1])).doubleValue();
				double amount2 = ((Number) dictionary.get(spines[i][index])).doubleValue();
				double amount3 = ((Number) dictionary.get(spines[i][index + 1])).doubleValue();
				double amount = (amount1 + amount2 + amount3) / 3;
				dictionary.put(spines[i][index], new Double(amount));
			}
		}
		Iterator keys = dictionary.keySet().iterator();
		while (keys.hasNext()) {
			Jun2dPoint key = (Jun2dPoint) keys.next();
			double value = ((Number) dictionary.get(key)).doubleValue();
			dictionary.put(key, new Double(value * this.defaultLiftUpFactor()));
		}

		return dictionary;
	}

	/**
	 * Answer the receiver's map table for spine point to triangles.
	 * 
	 * @return java.util.Map
	 * @category body creation
	 */
	public Map tableForSpinePointToTriangles() {
		Jun2dPoint[][] spines = this.spines();
		JunFormTriangle[] triangles = this.triangles();
		Map table = new HashMap();
		for (int i = 0; i < spines.length; i++) {
			Jun2dPoint[] points = spines[i];
			for (int j = 0; j < points.length; j++) {
				table.put(points[j], new ArrayList(5));
			}
		}
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint[] points = triangles[i].points();
			for (int j = 0; j < points.length; j++) {
				Collection collection = (Collection) table.get(points[j]);
				if (collection != null) {
					collection.add(triangles[i]);
				}
			}
		}

		Map newTable = new HashMap();
		Iterator iterator = table.keySet().iterator();
		while (iterator.hasNext()) {
			Jun2dPoint key = (Jun2dPoint) iterator.next();
			Collection aCollection = (Collection) table.get(key);
			newTable.put(key, aCollection.toArray(new JunFormTriangleNode[aCollection.size()]));
		}
		table = newTable;

		return table;
	}

	/**
	 * Answer the chordal axes.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category chordal axes
	 */
	public Jun2dPoint[][] chordalAxes() {
		Collection aStream = new ArrayList();
		Jun2dPoint[][] chordalAxesTerminalToTerminal = this.chordalAxesTerminalToTerminal();
		for (int i = 0; i < chordalAxesTerminalToTerminal.length; i++) {
			aStream.add(chordalAxesTerminalToTerminal[i]);
		}
		Jun2dPoint[][] chordalAxesTerminalToJunction = this.chordalAxesTerminalToJunction();
		for (int i = 0; i < chordalAxesTerminalToJunction.length; i++) {
			aStream.add(chordalAxesTerminalToJunction[i]);
		}
		Jun2dPoint[][] chordalAxesJunctionToJunction = this.chordalAxesJunctionToJunction();
		for (int i = 0; i < chordalAxesJunctionToJunction.length; i++) {
			aStream.add(chordalAxesJunctionToJunction[i]);
		}
		return (Jun2dPoint[][]) aStream.toArray(new Jun2dPoint[aStream.size()][]);
	}

	/**
	 * Answer the chordal axes junction to junction.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category chordal axes
	 */
	public Jun2dPoint[][] chordalAxesJunctionToJunction() {
		if (chordalAxesInformation == null) {
			chordalAxesInformation = this.computeChordalAxes();
		}
		return chordalAxesInformation[2];
	}

	/**
	 * Answer the chordal axes table.
	 * 
	 * @return java.util.Map
	 * @category chordal axes
	 */
	public Map chordalAxesTable() {
		if (chordalAxesTable == null) {
			chordalAxesTable = this.computeChordalAxesTable();
		}
		return chordalAxesTable;
	}

	/**
	 * Answer the chordal axes terminal to junction.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category chordal axes
	 */
	public Jun2dPoint[][] chordalAxesTerminalToJunction() {
		if (chordalAxesInformation == null) {
			chordalAxesInformation = this.computeChordalAxes();
		}
		return chordalAxesInformation[1];
	}

	/**
	 * Answer the chordal axes terminal to terminal.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category chordal axes
	 */
	public Jun2dPoint[][] chordalAxesTerminalToTerminal() {
		if (chordalAxesInformation == null) {
			chordalAxesInformation = this.computeChordalAxes();
		}
		return chordalAxesInformation[0];
	}

	/**
	 * Enumerate each chordal axis and evaluate the block.
	 * 
	 * @param chordalAxis jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category chordal axes
	 */
	public Object chordalAxis_do_(Jun2dPoint[] chordalAxis, StBlockClosure aBlock) {
		if (chordalAxis.length < 2) {
			return this;
		}
		Jun2dPoint specialPoint = chordalAxis[0];
		Jun2dPoint aPoint = chordalAxis[1];
		JunFormTriangleNode[] nodes = (JunFormTriangleNode[]) this.chordalAxesTable().get(aPoint);
		JunFormTriangleNode currentNode = null;
		for (int i = 0; i < nodes.length; i++) {
			if ((nodes[i].isTerminal() && ((JunFormTriangleTerminal) nodes[i]).tp().equals(specialPoint)) || (nodes[i].isJunction() && ((JunFormTriangleJunction) nodes[i]).jp().equals(specialPoint))) {
				currentNode = nodes[i];
				break;
			}
		}
		Object result = aBlock.value_value_value_(null, specialPoint, currentNode);
		if (result != null) {
			return result;
		}
		for (int index = 1; index < chordalAxis.length; index++) {
			aPoint = chordalAxis[index];
			JunFormTriangleNode nextNode = null;
			JunFormTriangleNode[] triangles = (JunFormTriangleNode[]) this.chordalAxesTable().get(aPoint);
			for (int i = 0; i < triangles.length; i++) {
				if (triangles[i].equals(currentNode) == false) {
					nextNode = triangles[i];
					break;
				}
			}
			result = aBlock.value_value_value_(currentNode, aPoint, nextNode);
			if (result != null) {
				return result;
			}
			if (nextNode.isTerminal()) {
				return aBlock.value_value_value_(nextNode, ((JunFormTriangleTerminal) nextNode).tp(), null);
			}
			if (nextNode.isJunction()) {
				return aBlock.value_value_value_(nextNode, ((JunFormTriangleJunction) nextNode).jp(), null);
			}
			currentNode = nextNode;
		}
		return null;
	}

	/**
	 * Compute the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category computing
	 */
	public JunOpenGL3dObject compute() {
		this.draftTriangleNodes();
		this.triangleNodes();
		this.chordalAxesTable();
		this.computeChordalAxes();
		this.fanTriangles();
		this.prunChordalAxes();
		this.spines();
		this.triangles();
		return this.body();
	}

	/**
	 * Compute chordal axes.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][][]
	 * @throws java.lang.IllegalStateException
	 * @category computing
	 */
	public Jun2dPoint[][][] computeChordalAxes() {
		Set visitedTriangles = new HashSet();
		Collection streamChordalAxesTtoT = new ArrayList();
		Collection streamChordalAxesTtoJ = new ArrayList();

		JunFormTriangleTerminal[] terminalTriangles = this.terminalTriangleNodes();
		for (int i = 0; i < terminalTriangles.length; i++) {
			if (visitedTriangles.contains(terminalTriangles[i]) == false) {
				JunFormTriangleTerminal terminalTriangle = terminalTriangles[i];
				Collection stream = new ArrayList();
				stream.add(terminalTriangle.tp());
				Jun2dPoint sp = terminalTriangle.spinePoints()[0];
				stream.add(sp);
				visitedTriangles.add(terminalTriangle);
				JunFormTriangleNode[] triangles = (JunFormTriangleNode[]) this.chordalAxesTable().get(sp);
				Collection collection = new ArrayList();
				for (int j = 0; j < triangles.length; j++) {
					if (triangles[j].equals(terminalTriangle) == false) {
						collection.add(triangles[j]);
					}
				}
				triangles = (JunFormTriangleNode[]) collection.toArray(new JunFormTriangleNode[collection.size()]);
				JunFormTriangleNode triangle = triangles[0];
				while (triangle.isSleeve()) {
					collection = new ArrayList();
					Jun2dPoint[] points = triangle.spinePoints();
					for (int j = 0; j < points.length; j++) {
						if (points[j].equals(sp) == false) {
							collection.add(points[j]);
						}
					}
					points = (Jun2dPoint[]) collection.toArray(new Jun2dPoint[collection.size()]);
					sp = points[0];
					stream.add(sp);
					visitedTriangles.add(triangle);
					triangles = (JunFormTriangleNode[]) this.chordalAxesTable().get(sp);
					collection = new ArrayList();
					for (int j = 0; j < triangles.length; j++) {
						if (triangles[j].equals(triangle) == false) {
							collection.add(triangles[j]);
						}
					}
					triangles = (JunFormTriangleNode[]) collection.toArray(new JunFormTriangleNode[collection.size()]);
					triangle = triangles[0];
				}
				if (triangle.isJunction()) {
					stream.add(((JunFormTriangleJunction) triangle).jp());
					streamChordalAxesTtoJ.add((Jun2dPoint[]) stream.toArray(new Jun2dPoint[stream.size()]));
				}
				if (triangle.isTerminal()) {
					stream.add(((JunFormTriangleTerminal) triangle).tp());
					visitedTriangles.add(triangle);
					streamChordalAxesTtoT.add((Jun2dPoint[]) stream.toArray(new Jun2dPoint[stream.size()]));
				}
			}
		}

		Collection streamChordalAxesJtoJ = new ArrayList();
		JunFormTriangleJunction[] junctionTriangles = this.junctionTriangleNodes();
		for (int i = 0; i < junctionTriangles.length; i++) {
			JunFormTriangleJunction junctionTriangle = junctionTriangles[i];
			Collection collection = new ArrayList();
			JunFormTriangleNode[] nodes = junctionTriangle.triangleNodes();
			for (int j = 0; j < nodes.length; j++) {
				if (visitedTriangles.contains(nodes[j]) == false) {
					collection.add(nodes[j]);
				}
			}
			nodes = (JunFormTriangleNode[]) collection.toArray(new JunFormTriangleNode[collection.size()]);
			for (int j = 0; j < nodes.length; j++) {
				JunFormTriangleNode triangleNode = nodes[j];
				if (visitedTriangles.contains(triangleNode) == false) {
					Collection stream = new ArrayList();
					stream.add(junctionTriangle.jp());
					Jun2dPoint sp = null;
					Jun2dPoint[] spinePoints = triangleNode.spinePoints();
					for (int k = 0; k < spinePoints.length; k++) {
						JunFormTriangleNode[] triangles = (JunFormTriangleNode[]) this.chordalAxesTable().get(spinePoints[k]);
						for (int l = 0; l < triangles.length; l++) {
							if (triangles[l].equals(junctionTriangle)) {
								sp = spinePoints[k];
								break;
							}
						}
						if (sp != null) {
							break;
						}
					}
					stream.add(sp);
					visitedTriangles.add(junctionTriangle);

					JunFormTriangleNode[] triangles = (JunFormTriangleNode[]) this.chordalAxesTable().get(sp);
					collection = new ArrayList();
					for (int k = 0; k < triangles.length; k++) {
						if (triangles[k].equals(junctionTriangle) == false) {
							collection.add(triangles[k]);
						}
					}
					triangles = (JunFormTriangleNode[]) collection.toArray(new JunFormTriangleNode[collection.size()]);
					JunFormTriangleNode triangle = triangles[0];
					while (triangle.isSleeve()) {
						collection = new ArrayList();
						Jun2dPoint[] points = triangle.spinePoints();
						for (int k = 0; k < points.length; k++) {
							if (points[k].equals(sp) == false) {
								collection.add(points[k]);
							}
						}
						points = (Jun2dPoint[]) collection.toArray(new Jun2dPoint[collection.size()]);
						sp = points[0];
						stream.add(sp);
						visitedTriangles.add(triangle);

						triangles = (JunFormTriangleNode[]) this.chordalAxesTable().get(sp);
						collection = new ArrayList();
						for (int k = 0; k < triangles.length; k++) {
							if (triangles[k].equals(triangle) == false) {
								collection.add(triangles[k]);
							}
						}
						triangles = (JunFormTriangleNode[]) collection.toArray(new JunFormTriangleNode[collection.size()]);
						triangle = triangles[0];
					}
					if (triangle.isJunction()) {
						stream.add(((JunFormTriangleJunction) triangle).jp());
						visitedTriangles.add(triangle);
					}
					if (triangle.isTerminal()) {
						throw new IllegalStateException("unexpected error");
					}
					streamChordalAxesJtoJ.add((Jun2dPoint[]) stream.toArray(new Jun2dPoint[stream.size()]));
				}
			}
		}
		return new Jun2dPoint[][][] {
				(Jun2dPoint[][]) streamChordalAxesTtoT.toArray(new Jun2dPoint[streamChordalAxesTtoT.size()][]),
				(Jun2dPoint[][]) streamChordalAxesTtoJ.toArray(new Jun2dPoint[streamChordalAxesTtoJ.size()][]),
				(Jun2dPoint[][]) streamChordalAxesJtoJ.toArray(new Jun2dPoint[streamChordalAxesJtoJ.size()][]) };
	}

	/**
	 * Compute chordal axes table.
	 * 
	 * @return java.util.Map
	 * @throws java.lang.IllegalStateException
	 * @category computing
	 */
	public Map computeChordalAxesTable() {
		if (this.isEmpty()) {
			return new HashMap();
		}

		Map aDictionary = new HashMap();
		JunFormTriangleNode[] triangleNodeCollection = this.triangleNodes();
		for (int i = 0; i < triangleNodeCollection.length; i++) {
			Jun2dPoint[] spinePoints = triangleNodeCollection[i].spinePoints();
			for (int j = 0; j < spinePoints.length; j++) {
				if (aDictionary.containsKey(spinePoints[j]) == false) {
					aDictionary.put(spinePoints[j], new ArrayList());
				}
				Collection aCollection = (Collection) aDictionary.get(spinePoints[j]);
				aCollection.add(triangleNodeCollection[i]);
			}
		}

		for (int i = 0; i < triangleNodeCollection.length; i++) {
			StSymbol[] msg1s = new StSymbol[] { $("sp1"), $("sp2"), $("sp3"), };
			StSymbol[] msg2s = new StSymbol[] { $("tn1_"), $("tn2_"), $("tn3_"), };
			for (int j = 0; j < msg1s.length; j++) {
				Jun2dPoint spinePoint = null;
				try {
					spinePoint = (Jun2dPoint) triangleNodeCollection[i].perform_(msg1s[j].toString());
				} catch (Exception e) {
					System.out.println(e.getMessage());
					e.printStackTrace();
				}
				if (spinePoint != null) {
					Collection nodeCollection = (Collection) aDictionary.get(spinePoint);
					JunFormTriangleNode[] nodes = (JunFormTriangleNode[]) nodeCollection.toArray(new JunFormTriangleNode[nodeCollection.size()]);
					List aCollection = new ArrayList();
					for (int k = 0; k < nodes.length; k++) {
						if (nodes[k].equals(triangleNodeCollection[i]) == false) {
							aCollection.add(nodes[k]);
						}
					}
					if (aCollection.size() > 1) {
						throw new IllegalStateException("unexpected error");
					}
					if (aCollection.size() == 1) {
						try {
							triangleNodeCollection[i].perform_with_(msg2s[j].toString(), aCollection.get(0));
						} catch (Exception e) {
							System.out.println(e.getMessage());
							e.printStackTrace();
						}
					} else {
						throw new IllegalStateException("unexpected error");
					}
				}
			}
		}

		Map newDicitionary = new HashMap();
		Iterator iterator = aDictionary.keySet().iterator();
		while (iterator.hasNext()) {
			Jun2dPoint key = (Jun2dPoint) iterator.next();
			Collection aCollection = (Collection) aDictionary.get(key);
			newDicitionary.put(key, aCollection.toArray(new JunFormTriangleNode[aCollection.size()]));
		}
		aDictionary = newDicitionary;

		return aDictionary;
	}

	/**
	 * Compute draft triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]
	 * @category computing
	 */
	public JunFormTriangleNode[] computeDraftTriangleNodes() {
		Jun2dTriangle[] triangleCollection = this.triangulatedTriangles();
		Jun2dPoint[] points = this.points();
		int pointsSize = points.length;
		Collection aStream = new ArrayList(pointsSize);
		for (int i = 0; i < pointsSize - 1; i++) {
			Jun2dPoint p1 = points[i];
			Jun2dPoint p2 = points[i + 1];
			aStream.add(p2.minus_(p1).dividedBy_(2.0d).plus_(p1));
		}
		Collection midPoints = new ArrayList(aStream);
		aStream = new ArrayList(triangleCollection.length);
		for (int i = 0; i < triangleCollection.length; i++) {
			JunFormTriangleNode triangleNode = new JunFormTriangleNode(triangleCollection[i]);
			Collection collection = new ArrayList();
			if (midPoints.contains(triangleNode.m1()) == false) {
				collection.add(triangleNode.m1());
			}
			if (midPoints.contains(triangleNode.m2()) == false) {
				collection.add(triangleNode.m2());
			}
			if (midPoints.contains(triangleNode.m3()) == false) {
				collection.add(triangleNode.m3());
			}
			triangleNode = null;
			if (collection.size() == 1) {
				triangleNode = new JunFormTriangleTerminal(triangleCollection[i].points(), (Jun2dPoint[]) collection.toArray(new Jun2dPoint[collection.size()]));
			}
			if (collection.size() == 2) {
				triangleNode = new JunFormTriangleSleeve(triangleCollection[i].points(), (Jun2dPoint[]) collection.toArray(new Jun2dPoint[collection.size()]));
			}
			if (collection.size() == 3) {
				triangleNode = new JunFormTriangleJunction(triangleCollection[i].points(), (Jun2dPoint[]) collection.toArray(new Jun2dPoint[collection.size()]));
			}
			aStream.add(triangleNode);
		}
		return (JunFormTriangleNode[]) aStream.toArray(new JunFormTriangleNode[aStream.size()]);
	}

	/**
	 * Compute final body.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category computing
	 */
	public JunOpenGL3dCompoundObject computeFinalBody() {
		return this.complexBody();
	}

	/**
	 * Compute final spines.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category computing
	 */
	public Jun2dPoint[][] computeFinalSpines() {
		return this.spinePolylines();
	}

	/**
	 * Compute final triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]
	 * @category computing
	 */
	public JunFormTriangleNode[] computeFinalTriangles() {
		Collection aStream = new ArrayList();

		Object[][][] spinePolylineInformation = this.spinePolylineInformationJunctionToJunction();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			Object[][] each = spinePolylineInformation[i];
			for (int j = 0; j < each.length; j++) {
				Object[] array = each[j];
				JunFormTriangleNode[] triangleCollection = (JunFormTriangleNode[]) array[1];
				for (int k = 0; k < triangleCollection.length; k++) {
					aStream.add(triangleCollection[k]);
				}
			}
		}

		spinePolylineInformation = this.spinePolylineInformationTerminalToJunction();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			Object[][] each = spinePolylineInformation[i];
			for (int j = 0; j < each.length; j++) {
				Object[] array = each[j];
				JunFormTriangleNode[] triangleCollection = (JunFormTriangleNode[]) array[1];
				for (int k = 0; k < triangleCollection.length; k++) {
					aStream.add(triangleCollection[k]);
				}
			}
		}

		spinePolylineInformation = this.spinePolylineInformationTerminalToTerminal();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			Object[][] each = spinePolylineInformation[i];
			for (int j = 0; j < each.length; j++) {
				Object[] array = each[j];
				JunFormTriangleNode[] triangleCollection = (JunFormTriangleNode[]) array[1];
				for (int k = 0; k < triangleCollection.length; k++) {
					aStream.add(triangleCollection[k]);
				}
			}
		}

		return (JunFormTriangleNode[]) aStream.toArray(new JunFormTriangleNode[aStream.size()]);
	}

	/**
	 * Compute triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]
	 * @throws java.lang.IllegalStateException
	 * @category computing
	 */
	public JunFormTriangleNode[] computeTriangleNodes() {
		if (this.isEmpty()) {
			return new JunFormTriangleNode[0];
		}

		Map aDictionary = new HashMap();
		JunFormTriangleNode[] triangleNodeCollection = this.draftTriangleNodes();
		for (int i = 0; i < triangleNodeCollection.length; i++) {
			Jun2dPoint[] spinePoints = triangleNodeCollection[i].spinePoints();
			for (int j = 0; j < spinePoints.length; j++) {
				if (aDictionary.containsKey(spinePoints[j]) == false) {
					aDictionary.put(spinePoints[j], new ArrayList(2));
				}
				Collection aCollection = (Collection) aDictionary.get(spinePoints[j]);
				aCollection.add(triangleNodeCollection[i]);
			}
		}

		final Collection removingTriangleNodes = new ArrayList();
		for (int i = 0; i < triangleNodeCollection.length; i++) {
			StSymbol[] msg1s = new StSymbol[] { $("sp1"), $("sp2"), $("sp3") };
			StSymbol[] msg2s = new StSymbol[] { $("tn1_"), $("tn2_"), $("tn3_") };
			for (int j = 0; j < msg1s.length; j++) {
				StSymbol msg1 = msg1s[j];
				StSymbol msg2 = msg2s[j];
				Jun2dPoint spinePoint = null;
				try {
					spinePoint = (Jun2dPoint) triangleNodeCollection[i].perform_(msg1.toString());
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
				if (spinePoint != null) {
					List aCollection = (List) aDictionary.get(spinePoint);
					JunFormTriangleNode[] triangleNodes = (JunFormTriangleNode[]) aCollection.toArray(new JunFormTriangleNode[aCollection.size()]);
					aCollection = new ArrayList();
					for (int k = 0; k < triangleNodes.length; k++) {
						if (triangleNodes[k] != triangleNodeCollection[i]) {
							aCollection.add(triangleNodes[k]);
						}
					}
					if (aCollection.size() > 1) {
						throw new IllegalStateException("unexpected error");
					}
					if (aCollection.size() == 1) {
						try {
							triangleNodeCollection[i].perform_with_(msg2.toString(), aCollection.get(0));
						} catch (Exception e) {
							throw new RuntimeException(e);
						}
					} else {
						removingTriangleNodes.add(triangleNodeCollection[i]);
					}
				}
			}
		}

		JunFormTriangleNode rootTriangleNode = null;
		for (int i = 0; i < triangleNodeCollection.length; i++) {
			boolean aBoolean = removingTriangleNodes.contains(triangleNodeCollection[i]) == false;
			if (aBoolean) {
				JunFormTriangleNode[] triangleNodes = triangleNodeCollection[i].triangleNodes();
				boolean isNil = true;
				for (int j = 0; j < triangleNodes.length; j++) {
					if (removingTriangleNodes.contains(triangleNodes[j])) {
						isNil = false;
						break;
					}
				}
				if (isNil) {
					rootTriangleNode = triangleNodeCollection[i];
					break;
				}
			}
		}
		if (rootTriangleNode == null) {
			throw new IllegalStateException("unexpected error");
		}

		final Collection aStream = new ArrayList(triangleNodeCollection.length);
		rootTriangleNode.do_(new StBlockClosure() {
			public Object value_(Object triangleNode) {
				if (removingTriangleNodes.contains(triangleNode) == false) {
					aStream.add(triangleNode);
				}
				return null;
			}
		});
		if (aStream.size() < 2) {
			throw new IllegalStateException("unexpected error");
		}

		triangleNodeCollection = (JunFormTriangleNode[]) aStream.toArray(new JunFormTriangleNode[aStream.size()]);
		for (int i = 0; i < triangleNodeCollection.length; i++) {
			triangleNodeCollection[i].flushTriangleNodes();
		}

		aDictionary = new HashMap();
		for (int i = 0; i < triangleNodeCollection.length; i++) {
			Jun2dPoint[] spinePoints = triangleNodeCollection[i].spinePoints();
			for (int j = 0; j < spinePoints.length; j++) {
				if (aDictionary.containsKey(spinePoints[j]) == false) {
					aDictionary.put(spinePoints[j], new ArrayList(2));
				}
				Collection aCollection = (Collection) aDictionary.get(spinePoints[j]);
				aCollection.add(triangleNodeCollection[i]);
			}
		}

		Collection aStream2 = new ArrayList(triangleNodeCollection.length);
		for (int i = 0; i < triangleNodeCollection.length; i++) {
			Jun2dPoint spinePoint = null;
			Jun2dPoint[] spinePoints = triangleNodeCollection[i].spinePoints();
			for (int j = 0; j < spinePoints.length; j++) {
				if (((Collection) aDictionary.get(spinePoints[j])).size() != 2) {
					spinePoint = spinePoints[j];
					break;
				}
			}
			if (spinePoint == null) {
				aStream2.add(triangleNodeCollection[i]);
			} else {
				Jun2dPoint[] points = triangleNodeCollection[i].points();
				Collection collection = new ArrayList();
				for (int j = 0; j < spinePoints.length; j++) {
					if (spinePoints[j] != spinePoint) {
						collection.add(spinePoints[j]);
					}
				}
				JunFormTriangleNode triangle = new JunFormTriangleTerminal(points, (Jun2dPoint[]) collection.toArray(new Jun2dPoint[collection.size()]));
				aStream2.add(triangle);
			}
		}

		triangleNodeCollection = (JunFormTriangleNode[]) aStream2.toArray(new JunFormTriangleNode[aStream2.size()]);
		return triangleNodeCollection;
	}

	/**
	 /**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		return this.body();
	}

	/**
	 * Convert to GeneralPath.
	 * 
	 * @return java.awt.geom.GeneralPath
	 * @category converting
	 */
	public GeneralPath toGeneralPath() {
		return this._createGeneralPath_(this.points());
	}

	/**
	 * Answer the default lift up factor.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultLiftUpFactor() {
		return 0.8d;
	}

	/**
	 * Answer the default oval factor.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultOvalFactor() {
		return 4;
	}

	/**
	 * Answer the default oval factors.
	 * 
	 * @return int
	 * @category defaults
	 */
	public double[] defaultOvalFactors() {
		double step = 90d / this.defaultOvalFactor();
		double[] factors = new double[this.defaultOvalFactor() + 1];
		for (int i = 0; i < factors.length; i++) {
			factors[i] = step * i;
		}
		return factors;
	}

	/**
	 * Answer the default paint color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	public Color defaultPaint() {
		return Color.gray;
	}

	/**
	 * Answer the default so far.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultSoFar() {
		return 29999;
	}

	/**
	 * Display the receiver's chordal axes on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayChordalAxesOn_(Graphics graphicsContext) {
		this.displayChordalAxesOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver's chordal axes on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayChordalAxesOn_at_(Graphics graphicsContext, Point aPoint) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();

		try {
			Rectangle boundingBox = this.boundingBox().asRectangle();
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);

			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.fill(boundingBox);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			JunFormTriangleNode[] triangles = this.sleeveTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.green, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.green);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.terminalTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.red, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.red);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.junctionTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.blue, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.blue);
				aGraphics.draw(aGeneralPath);
			}

			aGraphics.setStroke(new BasicStroke(2, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));

			Jun2dPoint[][] chordalAxes = this.chordalAxesTerminalToTerminal();
			for (int i = 0; i < chordalAxes.length; i++) {
				Jun2dPoint[] aPolyline = chordalAxes[i];
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolyline);
				aGraphics.setColor(Color.cyan);
				aGraphics.draw(aGeneralPath);
			}

			chordalAxes = this.chordalAxesTerminalToJunction();
			for (int i = 0; i < chordalAxes.length; i++) {
				Jun2dPoint[] aPolyline = chordalAxes[i];
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolyline);
				aGraphics.setColor(Color.magenta);
				aGraphics.draw(aGeneralPath);
			}

			chordalAxes = this.chordalAxesJunctionToJunction();
			for (int i = 0; i < chordalAxes.length; i++) {
				Jun2dPoint[] aPolyline = chordalAxes[i];
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolyline);
				aGraphics.setColor(Color.yellow);
				aGraphics.draw(aGeneralPath);
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Display the receiver's draft triangle nodes on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayDraftTriangleNodesOn_(Graphics graphicsContext) {
		this.displayDraftTriangleNodesOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver's draft triangle nodes on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayDraftTriangleNodesOn_at_(Graphics graphicsContext, Point aPoint) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();

		try {
			Rectangle boundingBox = this.boundingBox().asRectangle();
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);

			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.fill(boundingBox);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			JunFormTriangleNode[] triangles = this.draftSleeveTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.green, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.green);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.draftTerminalTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.red, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.red);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.draftJunctionTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.blue, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.blue);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.draftTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] points = triangles[i].spinePoints();
				for (int j = 0; j < points.length; j++) {
					Rectangle box = points[j].asBoundingBox().asRectangle();
					box.grow(1, 1);
					aGraphics.setColor(Color.black);
					aGraphics.fill(box);
				}
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		this.displayOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayOn_at_(Graphics graphicsContext, Point aPoint) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();

		try {
			Rectangle boundingBox = this.boundingBox().asRectangle();
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);

			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.fill(boundingBox);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			JunFormTriangle[] triangles = this.triangles();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.cyan, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.blue);
				aGraphics.draw(aGeneralPath);
			}

			Jun2dPoint[][] spines = this.spines();
			for (int i = 0; i < spines.length; i++) {
				Jun2dPoint[] spinePolyline = spines[i];
				aGraphics.setStroke(new BasicStroke(2, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
				aGraphics.setColor(Color.red);
				if (spinePolyline.length == 1) {
					Point point = spinePolyline[0].asPoint();
					aGraphics.drawRect(point.x - 1, point.y - 1, 2, 2);
				} else {
					GeneralPath aGeneralPath = this._createGeneralPath_(spinePolyline);
					aGraphics.draw(aGeneralPath);
				}
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Display the receiver's polyline on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayPolylineOn_(Graphics graphicsContext) {
		this.displayPolylineOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver's polyline on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayPolylineOn_at_(Graphics graphicsContext, Point aPoint) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();

		try {
			Rectangle boundingBox = this.boundingBox().asRectangle();
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);

			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.fill(boundingBox);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			GeneralPath aGeneralPath = this.toGeneralPath();
			aGraphics.setColor(Color.black);
			aGraphics.draw(aGeneralPath);
			aGraphics.setColor(Color.green);

			Jun2dPoint[] points = this.points();
			for (int i = 0; i < points.length; i++) {
				Point point = points[i].asPoint();
				aGraphics.drawRect(point.x - 1, point.y - 1, 2, 2);
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Display the receiver's spine polylines on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displaySpinePolylinesOn_(Graphics graphicsContext) {
		this.displaySpinePolylinesOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver's spine polylines on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displaySpinePolylinesOn_at_(Graphics graphicsContext, Point aPoint) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();

		try {
			Rectangle boundingBox = this.boundingBox().asRectangle();
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);

			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.fill(boundingBox);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			JunFormTriangleNode[] triangles = this.sleeveTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.green, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.green);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.terminalTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.red, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.red);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.junctionTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.blue, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.blue);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.fanTriangles();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.magenta, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.magenta);
				aGraphics.draw(aGeneralPath);
			}

			aGraphics.setStroke(new BasicStroke(2, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			Jun2dPoint[][] spinePolylines = this.spinePolylines();
			for (int i = 0; i < spinePolylines.length; i++) {
				aGraphics.setColor(Color.yellow);
				if (spinePolylines[i].length == 1) {
					Rectangle spineBox = spinePolylines[i][0].boundingBox().asRectangle();
					spineBox.grow(1, 1);
					aGraphics.fill(spineBox);
				} else {
					GeneralPath aGeneralPath = this._createGeneralPath_(spinePolylines[i]);
					aGraphics.draw(aGeneralPath);
				}
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Display the receiver's triangle nodes on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayTriangleNodesOn_(Graphics graphicsContext) {
		this.displayTriangleNodesOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver's triangle nodes on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayTriangleNodesOn_at_(Graphics graphicsContext, Point aPoint) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();

		try {
			Rectangle boundingBox = this.boundingBox().asRectangle();
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);

			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.fill(boundingBox);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			JunFormTriangleNode[] triangles = this.sleeveTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.green, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.green);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.terminalTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.red, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.red);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.junctionTriangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] aPolylineLoop = triangles[i].asPolylineLoop();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.blue, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.blue);
				aGraphics.draw(aGeneralPath);
			}

			triangles = this.triangleNodes();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] points = triangles[i].spinePoints();
				for (int j = 0; j < points.length; j++) {
					Rectangle box = points[j].asBoundingBox().asRectangle();
					box.grow(1, 1);
					aGraphics.setColor(Color.black);
					aGraphics.fill(box);
				}
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Display the receiver's triangulated triangles on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayTriangulatedTrianglesOn_(Graphics graphicsContext) {
		this.displayTriangulatedTrianglesOn_at_(graphicsContext, new Point(0, 0));
	}

	/**
	 * Display the receiver's triangulated triangles on the graphics at the specified point.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayTriangulatedTrianglesOn_at_(Graphics graphicsContext, Point aPoint) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();

		try {
			Rectangle boundingBox = this.boundingBox().asRectangle();
			Point displayPoint = new Point(0 - boundingBox.x + aPoint.x, 0 - boundingBox.y + aPoint.y);

			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.setColor(Color.white);
			aGraphics.fill(boundingBox);
			aGraphics.setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
			aGraphics.setColor(Color.gray);
			aGraphics.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			Jun2dTriangle[] triangles = this.triangulation().triangles();
			for (int i = 0; i < triangles.length; i++) {
				Jun2dTriangle triangle = triangles[i];
				Jun2dPoint[] aPolylineLoop = (triangle.first().equals(triangle.last())) ? triangle.asArrayOfPoints() : triangle.asPointArray();
				GeneralPath aGeneralPath = this._createGeneralPath_(aPolylineLoop);
				aGraphics.setColor(StColorValue.Blend(Color.red, Color.white));
				aGraphics.fill(aGeneralPath);
				aGraphics.setColor(Color.red);
				aGraphics.draw(aGeneralPath);
			}

			GeneralPath aGeneralPath = this.toGeneralPath();
			aGraphics.setColor(Color.black);
			aGraphics.draw(aGeneralPath);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Answer fan center and points with the specified chordal axis points.
	 * 
	 * @param chordalAxis jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return java.lang.Object[2] { jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.basic.Jun2dPoint[] };
	 * @throws java.lang.IllegalStateException
	 * @category pruning
	 */
	public Object[] fanCenterAndPoints_(Jun2dPoint[] chordalAxis) {
		final Set fanPoints = new HashSet();
		Object result = this.chordalAxis_do_(chordalAxis, new StBlockClosure() {
			public Object value_value_value_(Object obj1, Object obj2, Object obj3) {
				JunFormTriangleNode from = (JunFormTriangleNode) obj1;
				Jun2dPoint sp = (Jun2dPoint) obj2;
				JunFormTriangleNode to = (JunFormTriangleNode) obj3;

				if (from == null) {
					fanPoints.add(((JunFormTriangleTerminal) to).tp());
				}

				if (from != null && to != null) {
					boolean aBoolean = JunFormCreation.this.insideFan_from_to_((Jun2dPoint[]) fanPoints.toArray(new Jun2dPoint[fanPoints.size()]), sp, to.pointsAtMidPoint_(sp)[0]);
					Jun2dPoint[] points = from.points();
					for (int i = 0; i < points.length; i++) {
						fanPoints.add(points[i]);
					}
					if (aBoolean == false) {
						return new Object[] { sp, fanPoints.toArray(new Jun2dPoint[fanPoints.size()]) };
					}
				}

				if (to == null) {
					return new Object[] { sp, fanPoints.toArray(new Jun2dPoint[fanPoints.size()]) };
				}

				return null;
			}
		});
		if (result != null) {
			return (Object[]) result;
		}

		throw new IllegalStateException("unexpected error");
	}

	/**
	 * Answer the fan triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]
	 * @category pruning
	 */
	public JunFormTriangleNode[] fanTriangles() {
		if (fanTriangles == null) {
			this.prunChordalAxes();
		}
		return (JunFormTriangleNode[]) fanTriangles.toArray(new JunFormTriangleNode[fanTriangles.size()]);
	}

	/**
	 * Answer the fan triangles with the specified wedge points and center point.
	 * 
	 * @param wedgePoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]
	 * @category pruning
	 */
	public JunFormTriangleNode[] fanTriangles_centerPoint_(Jun2dPoint[] wedgePoints, Jun2dPoint centerPoint) {
		Collection aStream = new ArrayList(wedgePoints.length);
		for (int i = 0; i < wedgePoints.length - 1; i++) {
			Jun2dPoint[] trianglePoints = new Jun2dPoint[] { wedgePoints[i], wedgePoints[i + 1], centerPoint };
			aStream.add(this.createTriangle_(trianglePoints));
		}
		return (JunFormTriangleNode[]) aStream.toArray(new JunFormTriangleNode[aStream.size()]);
	}

	/**
	 * Answer <code>true</code> if the specified points inside fan with two point, otherwise <code>false</code>.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param fromPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param toPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category pruning
	 */
	public boolean insideFan_from_to_(Jun2dPoint[] pointCollection, Jun2dPoint fromPoint, Jun2dPoint toPoint) {
		double radius = toPoint.distance_(fromPoint);
		for (int i = 0; i < pointCollection.length; i++) {
			double distance = pointCollection[i].distance_(fromPoint);
			if (distance > radius) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Prun chordal axes.
	 * 
	 * @return java.lang.Object[][][][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category pruning
	 */
	public Object[][][][] prunChordalAxes() {
		fanTriangles = new ArrayList();
		Collection aStream = new ArrayList();
		aStream.add(this.prunChordalAxesTerminalToTerminal());
		aStream.add(this.prunChordalAxesTerminalToJunction());
		aStream.add(this.prunChordalAxesJunctionToJunction());
		return (Object[][][][]) aStream.toArray(new Object[aStream.size()][][][]);
	}

	/**
	 * Prun chordal axis junction to junction.
	 * 
	 * @return java.lang.Object[][][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category pruning
	 */
	public Object[][][] prunChordalAxesJunctionToJunction() {
		Jun2dPoint[][] chordalAxes = this.chordalAxesJunctionToJunction();
		Collection aStream = new ArrayList(chordalAxes.length);
		for (int i = 0; i < chordalAxes.length; i++) {
			Object[][] spineInformation = this.prunChordalAxisJunctionToJunction_(chordalAxes[i]);
			aStream.add(spineInformation);
		}
		return (Object[][][]) aStream.toArray(new Object[aStream.size()][][]);
	}

	/**
	 * Prun chordal axis terminal to junction.
	 * 
	 * @return java.lang.Object[][][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category pruning
	 */
	public Object[][][] prunChordalAxesTerminalToJunction() {
		Jun2dPoint[][] chordalAxes = this.chordalAxesTerminalToJunction();
		Collection aStream = new ArrayList(chordalAxes.length);
		for (int i = 0; i < chordalAxes.length; i++) {
			Object[][] spineInformation = this.prunChordalAxisTerminalToJunction_(chordalAxes[i]);
			aStream.add(spineInformation);
		}
		return (Object[][][]) aStream.toArray(new Object[aStream.size()][][]);
	}

	/**
	 * Prun chordal axis terminal to terminal.
	 * 
	 * @return java.lang.Object[][][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category pruning
	 */
	public Object[][][] prunChordalAxesTerminalToTerminal() {
		Jun2dPoint[][] chordalAxes = this.chordalAxesTerminalToTerminal();
		Collection aStream = new ArrayList(chordalAxes.length);
		for (int i = 0; i < chordalAxes.length; i++) {
			Object[][] spineInformation = this.prunChordalAxisTerminalToTerminal_(chordalAxes[i]);
			aStream.add(spineInformation);
		}
		return (Object[][][]) aStream.toArray(new Object[aStream.size()][][]);
	}

	/**
	 * Prun chordal axis junction to junction with the specified points.
	 * 
	 * @param chordalAxisJunctionToJunction jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return java.lang.Object[][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category pruning
	 */
	public Object[][] prunChordalAxisJunctionToJunction_(Jun2dPoint[] chordalAxisJunctionToJunction) {
		final Collection aStream = new ArrayList(chordalAxisJunctionToJunction.length);
		final StValueHolder previousSpHolder = new StValueHolder(null);
		this.chordalAxis_do_(chordalAxisJunctionToJunction, new StBlockClosure() {
			public Object value_value_value_(Object obj1, Object obj2, Object obj3) {
				JunFormTriangleNode from = (JunFormTriangleNode) obj1;
				Jun2dPoint sp = (Jun2dPoint) obj2;
				JunFormTriangleNode to = (JunFormTriangleNode) obj3;
				Jun2dPoint previousSp = (Jun2dPoint) previousSpHolder.value();

				if (from == null) {
					previousSp = sp;
					JunFormTriangleNode[] triangleCollection = new JunFormTriangleNode[0];
					aStream.add(new Object[] { sp, triangleCollection });
				}

				if (from != null && to != null) {
					if (from.isJunction()) {
						JunFormTriangleNode[] triangleCollection = new JunFormTriangleNode[2];
						Jun2dPoint[] points = from.pointsAtMidPoint_(sp);
						triangleCollection[0] = JunFormCreation.this.createTriangle_(new Jun2dPoint[] { previousSp, sp, points[0] });
						triangleCollection[1] = JunFormCreation.this.createTriangle_(new Jun2dPoint[] { previousSp, sp, points[points.length - 1] });
						aStream.add(new Object[] { sp, triangleCollection });
					}
					if (from.isJunction() == false && to.isJunction() == false) {
						JunFormTriangleNode[] triangleCollection = new JunFormTriangleNode[3];
						Jun2dPoint[][] threeTrianglePoints = from.threeTriangles_with_(sp, previousSp);
						for (int i = 0; i < threeTrianglePoints.length; i++) {
							triangleCollection[i] = JunFormCreation.this.createTriangle_(threeTrianglePoints[i]);
						}
						aStream.add(new Object[] { sp, triangleCollection });
					}
					if (to.isJunction()) {
						Collection triangleCollection = new ArrayList(5);
						if (from.isJunction() == false) {
							Jun2dPoint[][] threeTrianglePoints = from.threeTriangles_with_(sp, previousSp);
							for (int i = 0; i < threeTrianglePoints.length; i++) {
								triangleCollection.add(JunFormCreation.this.createTriangle_(threeTrianglePoints[i]));
							}
						}
						Jun2dPoint[] points = to.pointsAtMidPoint_(sp);
						triangleCollection.add(JunFormCreation.this.createTriangle_(new Jun2dPoint[] { sp, ((JunFormTriangleJunction) to).jp(), points[0] }));
						triangleCollection.add(JunFormCreation.this.createTriangle_(new Jun2dPoint[] { sp, ((JunFormTriangleJunction) to).jp(), points[points.length - 1] }));
						aStream.add(new Object[] { sp, triangleCollection.toArray(new JunFormTriangleNode[triangleCollection.size()]) });
					}
				}

				if (to == null) {
					JunFormTriangleNode[] triangleCollection = new JunFormTriangleNode[0];
					aStream.add(new Object[] { sp, triangleCollection });
				}
				previousSpHolder.value_(sp);

				return null;
			}
		});

		return (Object[][]) aStream.toArray(new Object[aStream.size()][]);
	}

	/**
	 * Prun chordal axis terminal to junction with the specified points.
	 * 
	 * @param chordalAxisTerminalToJunction jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return java.lang.Object[][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category pruning
	 */
	public Object[][] prunChordalAxisTerminalToJunction_(Jun2dPoint[] chordalAxisTerminalToJunction) {
		final Collection aStream = new ArrayList(chordalAxisTerminalToJunction.length);
		Object[] anArray = this.fanCenterAndPoints_(chordalAxisTerminalToJunction);
		Jun2dPoint centerPoint = (Jun2dPoint) anArray[0];
		Jun2dPoint[] wedgePoints = (Jun2dPoint[]) anArray[1];
		final StValueHolder centerIndexHolder = new StValueHolder(this._indexOf_within_(centerPoint, chordalAxisTerminalToJunction));
		wedgePoints = this.sortWedgePoints_centerPoint_terminalPoint_(wedgePoints, centerPoint, chordalAxisTerminalToJunction[0]);
		JunFormTriangleNode[] triangleCollection = this.fanTriangles_centerPoint_(wedgePoints, centerPoint);
		for (int i = 0; i < triangleCollection.length; i++) {
			fanTriangles.add(triangleCollection[i]);
		}
		aStream.add(new Object[] { centerPoint, triangleCollection });
		final StValueHolder indexHolder = new StValueHolder(0);
		final StValueHolder previousSpHolder = new StValueHolder(null);
		this.chordalAxis_do_(chordalAxisTerminalToJunction, new StBlockClosure() {
			public Object value_value_value_(Object obj1, Object obj2, Object obj3) {
				JunFormTriangleNode from = (JunFormTriangleNode) obj1;
				Jun2dPoint sp = (Jun2dPoint) obj2;
				JunFormTriangleNode to = (JunFormTriangleNode) obj3;
				Jun2dPoint previousSp = (Jun2dPoint) previousSpHolder.value();
				int index = indexHolder._intValue();
				int centerIndex = centerIndexHolder._intValue();

				if (index > centerIndex) {
					if (from != null && to != null) {
						if (to.isJunction() == false) {
							JunFormTriangleNode[] triangleCollection = new JunFormTriangleNode[3];
							Jun2dPoint[][] threeTrianglePoints = from.threeTriangles_with_(sp, previousSp);
							for (int i = 0; i < threeTrianglePoints.length; i++) {
								triangleCollection[i] = JunFormCreation.this.createTriangle_(threeTrianglePoints[i]);
							}
							aStream.add(new Object[] { sp, triangleCollection });
						}
						if (to.isJunction()) {
							Collection triangleCollection = new ArrayList(5);
							if (from.isJunction() == false) {
								Jun2dPoint[][] threeTrianglePoints = from.threeTriangles_with_(sp, previousSp);
								for (int i = 0; i < threeTrianglePoints.length; i++) {
									triangleCollection.add(JunFormCreation.this.createTriangle_(threeTrianglePoints[i]));
								}
							}
							Jun2dPoint[] points = to.pointsAtMidPoint_(sp);
							triangleCollection.add(JunFormCreation.this.createTriangle_(new Jun2dPoint[] { sp, ((JunFormTriangleJunction) to).jp(), points[0] }));
							triangleCollection.add(JunFormCreation.this.createTriangle_(new Jun2dPoint[] { sp, ((JunFormTriangleJunction) to).jp(), points[points.length - 1] }));
							aStream.add(new Object[] { sp, triangleCollection.toArray(new JunFormTriangleNode[triangleCollection.size()]) });
						}
					}
					if (to == null) {
						JunFormTriangleNode[] triangleCollection = new JunFormTriangleNode[0];
						aStream.add(new Object[] { sp, triangleCollection });
					}
				}

				if (index == centerIndex + 1 && from.isJunction() && to == null) {
					JunFormTriangleNode[] triangleCollection = new JunFormTriangleNode[2];
					Jun2dPoint[] points = from.pointsAtMidPoint_(previousSp);
					triangleCollection[0] = JunFormCreation.this.createTriangle_(new Jun2dPoint[] { previousSp, ((JunFormTriangleJunction) from).jp(), points[0] });
					triangleCollection[1] = JunFormCreation.this.createTriangle_(new Jun2dPoint[] { previousSp, ((JunFormTriangleJunction) from).jp(), points[points.length - 1] });
					aStream.add(new Object[] { sp, triangleCollection });
				}

				indexHolder.value_(index + 1);
				previousSpHolder.value_(sp);

				return null;
			}
		});

		return (Object[][]) aStream.toArray(new Object[aStream.size()][]);
	}

	/**
	 * Prun chordal axis terminal to terminal with the specified points.
	 * 
	 * @param chordalAxisTerminalToTerminal jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return java.lang.Object[][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category pruning
	 */
	public Object[][] prunChordalAxisTerminalToTerminal_(Jun2dPoint[] chordalAxisTerminalToTerminal) {
		final Collection aStream = new ArrayList(chordalAxisTerminalToTerminal.length);
		if (chordalAxisTerminalToTerminal.length == 3) {
			Jun2dPoint centerPoint1 = chordalAxisTerminalToTerminal[1];
			final Set wedgePointSet = new HashSet();
			this.chordalAxis_do_(chordalAxisTerminalToTerminal, new StBlockClosure() {
				public Object value_value_value_(Object obj1, Object obj2, Object obj3) {
					JunFormTriangleNode from = (JunFormTriangleNode) obj1;
					// Jun2dPoint sp = (Jun2dPoint) obj2;
					JunFormTriangleNode to = (JunFormTriangleNode) obj3;

					if (from == null) {
						Jun2dPoint[] points = to.points();
						for (int i = 0; i < points.length; i++) {
							wedgePointSet.add(points[i]);
						}
					}
					if (to == null) {
						Jun2dPoint[] points = from.points();
						for (int i = 0; i < points.length; i++) {
							wedgePointSet.add(points[i]);
						}
					}
					return null;
				}
			});

			Jun2dPoint[] wedgePoints1 = (Jun2dPoint[]) wedgePointSet.toArray(new Jun2dPoint[wedgePointSet.size()]);
			wedgePoints1 = this.sortWedgePoints_centerPoint_terminalPoint_(wedgePoints1, centerPoint1, chordalAxisTerminalToTerminal[0]);
			Jun2dPoint[] tempPoints = new Jun2dPoint[wedgePoints1.length];
			for (int i = 0; i < wedgePoints1.length; i++) {
				tempPoints[i] = wedgePoints1[i];
			}
			tempPoints[tempPoints.length - 1] = wedgePoints1[0];
			wedgePoints1 = tempPoints;

			JunFormTriangleNode[] triangleCollection1 = this.fanTriangles_centerPoint_(wedgePoints1, centerPoint1);
			for (int i = 0; i < triangleCollection1.length; i++) {
				fanTriangles.add(triangleCollection1[i]);
			}
			aStream.add(new Object[] { centerPoint1, triangleCollection1 });
			return (Object[][]) aStream.toArray(new Object[aStream.size()][]);
		}

		Object[] anArray = this.fanCenterAndPoints_(chordalAxisTerminalToTerminal);
		Jun2dPoint centerPoint1 = (Jun2dPoint) anArray[0];
		Jun2dPoint[] wedgePoints1 = (Jun2dPoint[]) anArray[1];
		final int centerIndex1 = this._indexOf_within_(centerPoint1, chordalAxisTerminalToTerminal);
		wedgePoints1 = this.sortWedgePoints_centerPoint_terminalPoint_(wedgePoints1, centerPoint1, chordalAxisTerminalToTerminal[0]);
		JunFormTriangleNode[] triangleCollection1 = this.fanTriangles_centerPoint_(wedgePoints1, centerPoint1);
		for (int i = 0; i < triangleCollection1.length; i++) {
			fanTriangles.add(triangleCollection1[i]);
		}
		aStream.add(new Object[] { centerPoint1, triangleCollection1 });

		Jun2dPoint[] reverse = new Jun2dPoint[chordalAxisTerminalToTerminal.length];
		for (int i = 0; i < reverse.length; i++) {
			reverse[i] = chordalAxisTerminalToTerminal[chordalAxisTerminalToTerminal.length - 1 - i];
		}
		anArray = this.fanCenterAndPoints_(reverse);
		Jun2dPoint centerPoint2 = (Jun2dPoint) anArray[0];
		Jun2dPoint[] wedgePoints2 = (Jun2dPoint[]) anArray[1];
		final int centerIndex2 = chordalAxisTerminalToTerminal.length - this._indexOf_within_(centerPoint2, reverse) - 1;
		wedgePoints2 = this.sortWedgePoints_centerPoint_terminalPoint_(wedgePoints2, centerPoint2, chordalAxisTerminalToTerminal[chordalAxisTerminalToTerminal.length - 1]);
		JunFormTriangleNode[] triangleCollection2 = this.fanTriangles_centerPoint_(wedgePoints1, centerPoint2);
		for (int i = 0; i < triangleCollection2.length; i++) {
			fanTriangles.add(triangleCollection2[i]);
		}

		final StValueHolder indexHolder = new StValueHolder(0);
		final StValueHolder previousSpHolder = new StValueHolder(null);
		this.chordalAxis_do_(chordalAxisTerminalToTerminal, new StBlockClosure() {
			public Object value_value_value_(Object obj1, Object obj2, Object obj3) {
				JunFormTriangleNode from = (JunFormTriangleNode) obj1;
				Jun2dPoint sp = (Jun2dPoint) obj2;
				JunFormTriangleNode to = (JunFormTriangleNode) obj3;
				Jun2dPoint previousSp = (Jun2dPoint) previousSpHolder.value();
				int index = indexHolder._intValue();

				if (index > centerIndex1 && index <= centerIndex2) {
					if (from != null && to != null) {
						JunFormTriangleNode[] triangleCollection = new JunFormTriangleNode[3];
						Jun2dPoint[][] threeTrianglePoints = from.threeTriangles_with_(sp, previousSp);
						for (int i = 0; i < threeTrianglePoints.length; i++) {
							triangleCollection[i] = JunFormCreation.this.createTriangle_(threeTrianglePoints[i]);
						}
						aStream.add(new Object[] { sp, triangleCollection });
					}
				}

				indexHolder.value_(index + 1);
				previousSpHolder.value_(sp);

				return null;
			}
		});

		aStream.add(new Object[] { centerPoint2, triangleCollection2 });
		return (Object[][]) aStream.toArray(new Object[aStream.size()][]);
	}

	/**
	 * Sort wedge points with the specified center point and terminal point.
	 * 
	 * @param wedgePoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param terminalPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category pruning
	 */
	protected Jun2dPoint[] sortWedgePoints_centerPoint_terminalPoint_(Jun2dPoint[] wedgePoints, Jun2dPoint centerPoint, Jun2dPoint terminalPoint) {
		Collection positiveCollection = new ArrayList();
		Collection negativeCollection = new ArrayList();
		JunPlane aPlane = new JunPlane(new Jun3dPoint(terminalPoint.x(), terminalPoint.y(), 0), new Jun3dPoint(centerPoint.x(), centerPoint.y(), 0), new Jun3dPoint(centerPoint.x(), centerPoint.y(), 1));
		for (int i = 0; i < wedgePoints.length; i++) {
			double valueF = aPlane.valueF_(new Jun3dPoint(wedgePoints[i].x(), wedgePoints[i].y(), 0));
			JunAngle angle = new Jun2dLine(centerPoint, terminalPoint).angleWithLine_(new Jun2dLine(centerPoint, wedgePoints[i]));
			if (valueF >= 0) {
				positiveCollection.add(new Object[] { angle, wedgePoints[i] });
			} else {
				negativeCollection.add(new Object[] { angle, wedgePoints[i] });
			}
		}

		Object[][] positiveArray = (Object[][]) positiveCollection.toArray(new Object[positiveCollection.size()][]);
		Arrays.sort(positiveArray, new Comparator() {
			public int compare(Object o1, Object o2) {
				JunAngle a1 = (JunAngle) ((Object[]) o1)[0];
				JunAngle a2 = (JunAngle) ((Object[]) o2)[0];
				return (a1.rad() == a2.rad()) ? 0 : ((a1.rad() < a2.rad()) ? -1 : 1);
				// return (a1.rad() < a2.rad() ? 1 : 0);
			}
		});

		Object[][] negativeArray = (Object[][]) negativeCollection.toArray(new Object[negativeCollection.size()][]);
		Arrays.sort(negativeArray, new Comparator() {
			public int compare(Object o1, Object o2) {
				JunAngle a1 = (JunAngle) ((Object[]) o1)[0];
				JunAngle a2 = (JunAngle) ((Object[]) o2)[0];
				return (a1.rad() == a2.rad()) ? 0 : ((a1.rad() > a2.rad()) ? -1 : 1);
				// return (a1.rad() > a2.rad()) ? 1 : 0;
			}
		});

		Collection aStream = new ArrayList(positiveArray.length * 2);
		for (int i = 0; i < negativeArray.length; i++) {
			aStream.add(negativeArray[i][1]);
		}
		for (int i = 0; i < positiveArray.length; i++) {
			aStream.add(positiveArray[i][1]);
		}

		return (Jun2dPoint[]) aStream.toArray(new Jun2dPoint[aStream.size()]);
	}

	/**
	 * Answer the spine polyline information.
	 * 
	 * @return java.lang.Object[][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category spine polyline
	 */
	public Object spinePolylineInformation() {
		Collection aStream = new ArrayList();
		Object[][][] spinePolylineInformation = this.spinePolylineInformationTerminalToTerminal();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			aStream.add(spinePolylineInformation[i]);
		}
		spinePolylineInformation = this.spinePolylineInformationTerminalToJunction();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			aStream.add(spinePolylineInformation[i]);
		}
		spinePolylineInformation = this.spinePolylineInformationJunctionToJunction();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			aStream.add(spinePolylineInformation[i]);
		}
		return (Object[][]) aStream.toArray(new Object[aStream.size()][]);
	}

	/**
	 * Answer the spine polyline information junction to junction.
	 * 
	 * @return java.lang.Object[][][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category spine polyline
	 */
	public Object[][][] spinePolylineInformationJunctionToJunction() {
		if (spinePolylineInformation == null) {
			spinePolylineInformation = this.prunChordalAxes();
		}
		return spinePolylineInformation[2];
	}

	/**
	 * Answer the spine polyline information terminal to junction.
	 * 
	 * @return java.lang.Object[][][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category spine polyline
	 */
	public Object[][][] spinePolylineInformationTerminalToJunction() {
		if (spinePolylineInformation == null) {
			spinePolylineInformation = this.prunChordalAxes();
		}
		return spinePolylineInformation[1];
	}

	/**
	 * Answer the spine polyline information terminal to terminal.
	 * 
	 * @return java.lang.Object[][][jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]]
	 * @category spine polyline
	 */
	public Object[][][] spinePolylineInformationTerminalToTerminal() {
		if (spinePolylineInformation == null) {
			spinePolylineInformation = this.prunChordalAxes();
		}
		return spinePolylineInformation[0];
	}

	/**
	 * Answer the receiver's spine polylines.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category spine polyline
	 */
	public Jun2dPoint[][] spinePolylines() {
		Collection aStream = new ArrayList();
		Jun2dPoint[][] spinePolylines = this.spinePolylinesTerminalToTerminal();
		for (int i = 0; i < spinePolylines.length; i++) {
			aStream.add(spinePolylines[i]);
		}
		spinePolylines = this.spinePolylinesTerminalToJunction();
		for (int i = 0; i < spinePolylines.length; i++) {
			aStream.add(spinePolylines[i]);
		}
		spinePolylines = this.spinePolylinesJunctionToJunction();
		for (int i = 0; i < spinePolylines.length; i++) {
			aStream.add(spinePolylines[i]);
		}
		return (Jun2dPoint[][]) aStream.toArray(new Jun2dPoint[aStream.size()][]);
	}

	/**
	 * Answer the spine polylines junction to junction.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category spine polyline
	 */
	public Jun2dPoint[][] spinePolylinesJunctionToJunction() {
		Object[][][] spinePolylineInformation = this.spinePolylineInformationJunctionToJunction();
		Collection aCollection = new ArrayList();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			Object[][] each = spinePolylineInformation[i];
			Collection points = new ArrayList();
			for (int j = 0; j < each.length; j++) {
				points.add(each[j][0]);
			}
			Jun2dPoint[] aPolyline = (Jun2dPoint[]) points.toArray(new Jun2dPoint[points.size()]);
			aCollection.add(aPolyline);
		}
		return (Jun2dPoint[][]) aCollection.toArray(new Jun2dPoint[aCollection.size()][]);
	}

	/**
	 * Answer the spine polylines terminal to junction.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category spine polyline
	 */
	public Jun2dPoint[][] spinePolylinesTerminalToJunction() {
		Object[][][] spinePolylineInformation = this.spinePolylineInformationTerminalToJunction();
		Collection aCollection = new ArrayList();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			Object[][] each = spinePolylineInformation[i];
			Collection points = new ArrayList();
			for (int j = 0; j < each.length; j++) {
				points.add(each[j][0]);
			}
			Jun2dPoint[] aPolyline = (Jun2dPoint[]) points.toArray(new Jun2dPoint[points.size()]);
			aCollection.add(aPolyline);
		}
		return (Jun2dPoint[][]) aCollection.toArray(new Jun2dPoint[aCollection.size()][]);
	}

	/**
	 * Answer the spine polylines terminal to terminal.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category spine polyline
	 */
	public Jun2dPoint[][] spinePolylinesTerminalToTerminal() {
		Object[][][] spinePolylineInformation = this.spinePolylineInformationTerminalToTerminal();
		Collection aCollection = new ArrayList();
		for (int i = 0; i < spinePolylineInformation.length; i++) {
			Object[][] each = spinePolylineInformation[i];
			Collection points = new ArrayList();
			for (int j = 0; j < each.length; j++) {
				points.add(each[j][0]);
			}
			Jun2dPoint[] aPolyline = (Jun2dPoint[]) points.toArray(new Jun2dPoint[points.size()]);
			aCollection.add(aPolyline);
		}
		return (Jun2dPoint[][]) aCollection.toArray(new Jun2dPoint[aCollection.size()][]);
	}

	/**
	 * Answer <code>true</code> if the receiver is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.points().length == 0 || this.points().length < 5;
	}

	/**
	 * Answer the receiver's draft junction triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleJunction[]
	 * @category triangle node
	 */
	public JunFormTriangleJunction[] draftJunctionTriangleNodes() {
		Collection aCollection = new ArrayList();
		JunFormTriangleNode[] draftTriangleNodes = this.draftTriangleNodes();
		for (int i = 0; i < draftTriangleNodes.length; i++) {
			if (draftTriangleNodes[i].isJunction()) {
				aCollection.add(draftTriangleNodes[i]);
			}
		}
		return (JunFormTriangleJunction[]) aCollection.toArray(new JunFormTriangleJunction[aCollection.size()]);
	}

	/**
	 * Answer the receiver's draft sleeve triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleSleeve[]
	 * @category triangle node
	 */
	public JunFormTriangleSleeve[] draftSleeveTriangleNodes() {
		Collection aCollection = new ArrayList();
		JunFormTriangleNode[] draftTriangleNodes = this.draftTriangleNodes();
		for (int i = 0; i < draftTriangleNodes.length; i++) {
			if (draftTriangleNodes[i].isSleeve()) {
				aCollection.add(draftTriangleNodes[i]);
			}
		}
		return (JunFormTriangleSleeve[]) aCollection.toArray(new JunFormTriangleSleeve[aCollection.size()]);
	}

	/**
	 * Answer the receiver's draft terminal triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleTerminal[]
	 * @category triangle node
	 */
	public JunFormTriangleTerminal[] draftTerminalTriangleNodes() {
		Collection aCollection = new ArrayList();
		JunFormTriangleNode[] draftTriangleNodes = this.draftTriangleNodes();
		for (int i = 0; i < draftTriangleNodes.length; i++) {
			if (draftTriangleNodes[i].isTerminal()) {
				aCollection.add(draftTriangleNodes[i]);
			}
		}
		return (JunFormTriangleTerminal[]) aCollection.toArray(new JunFormTriangleTerminal[aCollection.size()]);
	}

	/**
	 * Answer the receiver's draft triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]
	 * @category triangle node
	 */
	public JunFormTriangleNode[] draftTriangleNodes() {
		if (draftTriangleNodes == null) {
			draftTriangleNodes = this.computeDraftTriangleNodes();
		}
		return draftTriangleNodes;
	}

	/**
	 * Answer the receiver's junction triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleJunction[]
	 * @category triangle node
	 */
	public JunFormTriangleJunction[] junctionTriangleNodes() {
		Collection aCollection = new ArrayList();
		JunFormTriangleNode[] triangleNodes = this.triangleNodes();
		for (int i = 0; i < triangleNodes.length; i++) {
			if (triangleNodes[i].isJunction()) {
				aCollection.add(triangleNodes[i]);
			}
		}
		return (JunFormTriangleJunction[]) aCollection.toArray(new JunFormTriangleJunction[aCollection.size()]);
	}

	/**
	 * Answer the receiver's sleeve triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleSleeve[]
	 * @category triangle node
	 */
	public JunFormTriangleSleeve[] sleeveTriangleNodes() {
		Collection aCollection = new ArrayList();
		JunFormTriangleNode[] triangleNodes = this.triangleNodes();
		for (int i = 0; i < triangleNodes.length; i++) {
			if (triangleNodes[i].isSleeve()) {
				aCollection.add(triangleNodes[i]);
			}
		}
		return (JunFormTriangleSleeve[]) aCollection.toArray(new JunFormTriangleSleeve[aCollection.size()]);
	}

	/**
	 * Answer the receiver's terminal triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleTerminal[]
	 * @category triangle node
	 */
	public JunFormTriangleTerminal[] terminalTriangleNodes() {
		Collection aCollection = new ArrayList();
		JunFormTriangleNode[] triangleNodes = this.triangleNodes();
		for (int i = 0; i < triangleNodes.length; i++) {
			if (triangleNodes[i].isTerminal()) {
				aCollection.add(triangleNodes[i]);
			}
		}
		return (JunFormTriangleTerminal[]) aCollection.toArray(new JunFormTriangleTerminal[aCollection.size()]);
	}

	/**
	 * Answer the receiver's triangle nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode[]
	 * @category triangle node
	 */
	public JunFormTriangleNode[] triangleNodes() {
		if (triangleNodes == null) {
			triangleNodes = this.computeTriangleNodes();
		}
		return triangleNodes;
	}

	/**
	 * Create to GeneralPath from the specified points..
	 * 
	 * @return java.awt.geom.GeneralPath
	 * @category private
	 */
	protected GeneralPath _createGeneralPath_(Jun2dPoint[] points) {
		if (points == null || points.length == 0) {
			return null;
		}

		GeneralPath aGeneralPath = new GeneralPath(GeneralPath.WIND_EVEN_ODD, points.length);
		aGeneralPath.moveTo((float) points[0].x(), (float) points[0].y());
		for (int i = 1; i < points.length; i++) {
			aGeneralPath.lineTo((float) points[i].x(), (float) points[i].y());
		}
		return aGeneralPath;
	}

	/**
	 * Answer the index of the first occurrence of the specified point within the specified points.
	 * If the specified points does	not contain the specified point, answer -1.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param point jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return int
	 * @category private
	 */
	protected int _indexOf_within_(Jun2dPoint point, Jun2dPoint[] points) {
		for (int i = 0; i < points.length; i++) {
			if (points[i] == point) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Create triangles with the specified three points, and answer it.
	 * 
	 * @param vertexPoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return jp.co.sra.jun.geometry.forms.JunFormTriangleNode
	 * @category private
	 */
	protected JunFormTriangleNode createTriangle_(Jun2dPoint[] vertexPoints) {
		Jun3dPoint[] trianglePoints = new Jun3dPoint[3];
		for (int i = 0; i < trianglePoints.length; i++) {
			trianglePoints[i] = new Jun3dPoint(vertexPoints[i].x(), vertexPoints[i].y(), 0);
		}
		JunPlane aPlane = new JunPlane(trianglePoints[0], trianglePoints[1], trianglePoints[2]);
		JunFormTriangleNode aTriangle;
		if (0.0d > aPlane.valueF_(new Jun3dPoint(0, 0, this.defaultSoFar()))) {
			aTriangle = new JunFormTriangleNode(vertexPoints);
		} else {
			Jun2dPoint[] reverse = new Jun2dPoint[vertexPoints.length];
			for (int i = 0; i < vertexPoints.length; i++) {
				reverse[i] = vertexPoints[vertexPoints.length - 1 - i];
			}
			aTriangle = new JunFormTriangleNode(reverse);
		}
		return aTriangle;
	}

	/**
	 * Anser the sorted triangle points.
	 * 
	 * @param vertexPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category private
	 */
	protected Jun3dPoint[] sortTrianglePoints_(Jun3dPoint[] vertexPoints) {
		JunPlane aPlane = new JunPlane(vertexPoints[0], vertexPoints[1], vertexPoints[2]);
		if (0 > aPlane.valueF_(new Jun3dPoint(0, 0, this.defaultSoFar()))) {
			return vertexPoints;
		} else {
			Jun3dPoint[] reverse = new Jun3dPoint[vertexPoints.length];
			for (int i = 0; i < vertexPoints.length; i++) {
				reverse[i] = vertexPoints[vertexPoints.length - 1 - i];
			}
			return reverse;
		}
	}
}
