package jp.co.sra.jun.geometry.curves;

import java.util.ArrayList;

import jp.co.sra.smalltalk.StObject;

import jp.co.sra.jun.geometry.abstracts.JunCurve;
import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.support.JunBSplineFunction;
import jp.co.sra.jun.geometry.transformations.JunTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsCurve;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * JunNurbsCurve class
 * 
 *  @author    nisinaka
 *  @created   1998/09/29 (by nisinaka)
 *  @updated   2004/11/16 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun692 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunNurbsCurve.java,v 8.12 2008/02/20 06:30:57 nisinaka Exp $
 */
public class JunNurbsCurve extends JunCurve {

	/** An array of control points. */
	protected Jun3dPoint[] controlPoints = null;

	/** An array of weights. */
	protected double[] weights;

	/** An array of knots. */
	protected double[] knotVector;

	/**
	 * Create a new JunNurbsCurve and initialize it.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @category Instance creation
	 */
	public static JunNurbsCurve BezierControlPoints_(Jun3dPoint[] points) {
		int numberOfPoints = points.length;
		double[] knotVector = new double[numberOfPoints * 2];
		for (int i = 0; i < numberOfPoints; i++) {
			knotVector[i] = 0;
			knotVector[i + numberOfPoints] = 1;
		}
		return BSplineControlPoints_knotVector_(points, knotVector);
	}

	/**
	 * Create a new JunNurbsCurve and initialize it.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param weights double[]
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @category Instance creation
	 */
	public static JunNurbsCurve BezierControlPoints_weights_(Jun3dPoint[] points, double[] weights) {
		int numberOfPoints = points.length;
		double[] knotVector = new double[numberOfPoints * 2];
		for (int i = 0; i < numberOfPoints; i++) {
			knotVector[i] = 0;
			knotVector[i + numberOfPoints] = 1;
		}
		return ControlPoints_weights_knotVector_(points, weights, knotVector);
	}

	/**
	 * Create a new JunNurbsCurve and initialize it.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfNumbers double[]
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @category Instance creation
	 */
	public static JunNurbsCurve BSplineControlPoints_knotVector_(Jun3dPoint[] anArrayOfPoints, double[] anArrayOfNumbers) {
		int numberOfPoints = anArrayOfPoints.length;
		double[] weights = new double[numberOfPoints];
		for (int i = 0; i < numberOfPoints; i++) {
			weights[i] = 1.0;
		}
		return ControlPoints_weights_knotVector_(anArrayOfPoints, weights, anArrayOfNumbers);
	}

	/**
	 * Create a new JunNurbsCurve and initialize it.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param weights double[]
	 * @param knotVector double[]
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @deprecated since Jun499, use the constructor.
	 * @category Instance creation
	 */
	public static JunNurbsCurve ControlPoints_weights_knotVector_(Jun3dPoint[] points, double[] weights, double[] knotVector) {
		return new JunNurbsCurve(points, weights, knotVector);
	}

	/**
	 * Create a new instance of JunNurbsCurve and initialize it.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param weights double[]
	 * @param knotVector double[]
	 * @category Instance creation
	 */
	public JunNurbsCurve(Jun3dPoint[] points, double[] weights, double[] knotVector) {
		this.controlPoints_weights_knotVector_(points, weights, knotVector);
	}

	/**
	 * Answer my current control points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] controlPoints() {
		return controlPoints;
	}

	/**
	 * Set my new control points.
	 * 
	 * @param anArray jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public void controlPoints_(Jun3dPoint[] anArray) {
		controlPoints = anArray;
	}

	/**
	 * Answer the control point at the specified index.
	 * 
	 * @param index int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint controlPointAt_(int index) {
		return controlPoints[index];
	}

	/**
	 * Put a Jun3dPoint as a control point at the specified index.
	 * 
	 * @param index int
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void controlPointAt_put_(int index, Jun3dPoint aPoint) {
		controlPoints[index] = aPoint;
	}

	/**
	 * Answer the number of the control points.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int controlPointSize() {
		return controlPoints.length;
	}

	/**
	 * Answer my current weights.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] weights() {
		return weights;
	}

	/**
	 * Set my new weights.
	 * 
	 * @param anArray double[]
	 * @category accessing
	 */
	public void weights_(double[] anArray) {
		weights = anArray;
	}

	/**
	 * Answer the weight at the specified index.
	 * 
	 * @param index int
	 * @return double
	 * @category accessing
	 */
	public double weightAt_(int index) {
		return weights[index];
	}

	/**
	 * Answer my current knot vector.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] knotVector() {
		return knotVector;
	}

	/**
	 * Answer the knot vector value at the specified index.
	 * 
	 * @param index int
	 * @return double
	 * @category accessing
	 */
	public double knotVectorAt_(int index) {
		return knotVector[index];
	}

	/**
	 * Answer the order number.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int order() {
		return knotVector.length - controlPoints.length;
	}

	/**
	 * Answer a regularized point of this curve at a specified number.
	 * 
	 * @param t double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atT_(double t) {
		if (t < 0 || 1 < 1) {
			return null;
		}

		JunBSplineFunction bspline = this.bspline();
		Jun3dPoint numerator = Jun3dPoint.Zero();
		double dominator = 0;
		for (int i = 0; i < controlPoints.length; i++) {
			double nw = bspline.i_t_(i, t) * weights[i];
			numerator = numerator.plus_(controlPoints[i].multipliedBy_(nw));
			dominator += nw;
		}

		return (dominator == 0) ? this.controlPointAt_(this.controlPointSize() - 1) : numerator.dividedBy_(dominator);
	}

	/**
	 * Answer the BSpline object.
	 * 
	 * @return jp.co.sra.jun.geometry.support.JunBSplineFunction
	 * @category accessing
	 */
	public JunBSplineFunction bspline() {
		return new JunBSplineFunction(this._copyKnotVector(), this.order());
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		JunNurbsCurve aNurbsCurve = (JunNurbsCurve) anObject;
		int size = this.controlPointSize();
		if (size != aNurbsCurve.controlPointSize()) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.controlPointAt_(i).equal_(aNurbsCurve.controlPointAt_(i)) == false) {
				return false;
			}
		}

		size = this.weights().length;
		if (size != aNurbsCurve.weights().length) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.isEqualNumber_to_(this.weightAt_(i), aNurbsCurve.weightAt_(i)) == false) {
				return false;
			}
		}

		size = this.knotVector().length;
		if (size != aNurbsCurve.knotVector().length) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.isEqualNumber_to_(this.knotVectorAt_(i), aNurbsCurve.knotVectorAt_(i)) == false) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		JunNurbsCurve aNurbsCurve = (JunNurbsCurve) anObject;
		int size = this.controlPointSize();
		if (size != aNurbsCurve.controlPointSize()) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.controlPointAt_(i).equals(aNurbsCurve.controlPointAt_(i)) == false) {
				return false;
			}
		}

		size = this.weights().length;
		if (size != aNurbsCurve.weights().length) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.weightAt_(i) != aNurbsCurve.weightAt_(i)) {
				return false;
			}
		}

		size = this.knotVector().length;
		if (size != aNurbsCurve.knotVector().length) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.knotVectorAt_(i) != aNurbsCurve.knotVectorAt_(i)) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @category converting
	 */
	public Jun3dLine[] asArrayOfLines() {
		return super.asArrayOf3dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dLines()
	 * @category converting
	 */
	public Jun3dLine[] asArrayOf3dLines() {
		ArrayList lines = new ArrayList();
		Jun3dPoint[] points = this.asPointArray();
		if (points.length > 1) {
			Jun3dPoint startPoint = points[0];
			for (int i = 1; i < points.length; i++) {
				Jun3dPoint endPoint = points[i];
				lines.add(startPoint.to_(endPoint));
				startPoint = endPoint;
			}
		}
		return (Jun3dLine[]) lines.toArray(new Jun3dLine[lines.size()]);
	}

	/**
	 * Convert the receiver as a JunOpenGL3dNurbsCurve.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsCurve
	 * @category converting
	 */
	public JunOpenGL3dNurbsCurve asJunOpenGL3dNurbsCurve() {
		return new JunOpenGL3dNurbsCurve(this._copyControlPoints(), this._copyWeights(), this._copyKnotVector());
	}

	/**
	 * Convert the receiver as a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		return this.asJunOpenGL3dNurbsCurve();
	}

	/**
	 * Convert the receiver as a nurbs curve.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @see jp.co.sra.jun.geometry.abstracts.JunCurve#asNurbsCurve()
	 * @category converting
	 */
	public JunNurbsCurve asNurbsCurve() {
		return this;
	}

	/**
	 * Convert the receiver as an array of points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category converting
	 */
	public Jun3dPoint[] asPointArray() {
		return this.asPointArray_(this.controlPointSize() * 4);
	}

	/**
	 * Convert the receiver as an array of points.
	 * 
	 * @param division int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category converting
	 */
	public Jun3dPoint[] asPointArray_(int divisions) {
		if (divisions < 0) {
			return null;
		}

		Jun3dPoint[] points = new Jun3dPoint[divisions];
		for (int i = 0; i <= divisions; i++) {
			points[i] = this.atT_((double) i / divisions);
		}
		return points;
	}

	/**
	 * Answer the reversed JunNurbsCurve.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @category converting
	 */
	public JunNurbsCurve reversed() {
		int size = controlPoints.length;
		Jun3dPoint[] reversedControlPoints = new Jun3dPoint[size];
		for (int i = 0; i < size; i++) {
			reversedControlPoints[i] = controlPoints[size - i - 1];
		}

		size = weights.length;
		double[] reversedWeights = new double[size];
		for (int i = 0; i < size; i++) {
			reversedWeights[i] = weights[size - i - 1];
		}

		size = knotVector.length;
		double[] reversedKnotVector = new double[size];
		for (int i = 0; i < size; i++) {
			reversedKnotVector[i] = knotVector[size - 1] - knotVector[size - i];
		}

		return new JunNurbsCurve(reversedControlPoints, reversedWeights, reversedKnotVector);
	}

	/**
	 * Do an extra copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();
		controlPoints = this._copyControlPoints();
		weights = this._copyWeights();
		knotVector = this._copyKnotVector();
		return this;
	}

	/**
	 * Answer the copy of the receiver which is applied the specified transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#transform_(jp.co.sra.jun.geometry.transformations.JunTransformation)
	 * @category transforming
	 */
	public JunGeometry transform_(JunTransformation aTransformation) {
		JunNurbsCurve theNurbsCurve = (JunNurbsCurve) this.copy();
		for (int i = 0; i < theNurbsCurve.controlPointSize(); i++) {
			Jun3dPoint aPoint = theNurbsCurve.controlPointAt_(i);
			Jun3dPoint transformedPoint = (Jun3dPoint) aPoint.transform_(aTransformation);
			theNurbsCurve.controlPointAt_put_(i, transformedPoint);
		}
		return theNurbsCurve;
	}

	/**
	 * Set the instance variables.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfNumber1 double[]
	 * @param anArrayOfNumber2 double[]
	 * @category private
	 */
	protected void controlPoints_weights_knotVector_(Jun3dPoint[] anArrayOfPoints, double[] anArrayOfNumber1, double[] anArrayOfNumber2) {
		controlPoints = anArrayOfPoints;
		weights = anArrayOfNumber1;
		knotVector = anArrayOfNumber2;
	}

	/**
	 * Answer the copy of the control points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category private
	 */
	protected Jun3dPoint[] _copyControlPoints() {
		if (controlPoints == null) {
			return null;
		}

		Jun3dPoint[] copyControlPoints = new Jun3dPoint[controlPoints.length];
		System.arraycopy(controlPoints, 0, copyControlPoints, 0, controlPoints.length);
		return copyControlPoints;
	}

	/**
	 * Answer the copy of the weights
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category private
	 */
	protected double[] _copyWeights() {
		if (weights == null) {
			return null;
		}

		double[] copyWeights = new double[weights.length];
		System.arraycopy(weights, 0, copyWeights, 0, weights.length);
		return copyWeights;
	}

	/**
	 * Answer the copy of the knot vector
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category private
	 */
	protected double[] _copyKnotVector() {
		if (knotVector == null) {
			return null;
		}

		double[] copyKnotVector = new double[knotVector.length];
		System.arraycopy(knotVector, 0, copyKnotVector, 0, knotVector.length);
		return copyKnotVector;
	}

}
