package jp.co.sra.jun.geometry.curves;

import java.awt.Color;
import java.awt.Point;
import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.SmalltalkException;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.geometry.transformations.JunTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;

/**
 * Jun3dLine class
 * 
 *  @author    nisinaka
 *  @created   1998/10/09 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/20 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dLine.java,v 8.23 2008/02/20 06:30:57 nisinaka Exp $
 */
public class Jun3dLine extends JunLine {

	protected double x0;
	protected double y0;
	protected double z0;
	protected double f;
	protected double g;
	protected double h;

	/**
	 * Construct a new instance of Jun3dLine with two points.
	 * 
	 * @param fromPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param toPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public Jun3dLine(JunPoint fromPoint, JunPoint toPoint) {
		super();
		this.from_to_(fromPoint, toPoint);
	}

	/**
	 * Answer a new Jun3dLine with unit values.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category Constants access
	 */
	public static Jun3dLine Unity() {
		Jun3dLine aLine = new Jun3dLine(Jun3dPoint.Zero(), Jun3dPoint.Unity());

		return aLine.normalizedLine();
	}

	/**
	 * Answer a new Jun3dLine with zero values.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category Constants access
	 */
	public static final Jun2dLine Zero() {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Answer my start point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.curves.JunLine#_from()
	 * @category accessing
	 */
	protected JunPoint _from() {
		return this.from();
	}

	/**
	 * Answer my start point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.curves.JunLine#_to()
	 * @category accessing
	 */
	protected JunPoint _to() {
		return this.to();
	}

	/**
	 * Answer a regularized point of this curve at a specified number.
	 * 
	 * @param t double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atT_(double t) {
		return new Jun3dPoint(x0 + (f * t), y0 + (g * t), z0 + (h * t));
	}

	/**
	 * Answer a point on the receiver at the specified x-coordinate.
	 * 
	 * @param x double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atX_(double x) {
		JunPlane aPlane = new JunPlane(new Jun3dPoint(x, 0, 0), new Jun3dPoint(x, 1, 0), new Jun3dPoint(x, 0, 1));
		return aPlane.intersectingPointWithLine_(this);
	}

	/**
	 * Answer a point on the receiver at the specified y-coordinate.
	 * 
	 * @param y double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atY_(double y) {
		JunPlane aPlane = new JunPlane(new Jun3dPoint(0, y, 0), new Jun3dPoint(0, y, 1), new Jun3dPoint(1, y, 0));
		return aPlane.intersectingPointWithLine_(this);
	}

	/**
	 * Answer a point on the receiver at the specified z-coordinate.
	 * 
	 * @param z double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atZ_(double z) {
		JunPlane aPlane = new JunPlane(new Jun3dPoint(0, 0, z), new Jun3dPoint(1, 0, z), new Jun3dPoint(0, 1, z));
		return aPlane.intersectingPointWithLine_(this);
	}

	/**
	 * Answer the bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category accessing
	 */
	public Jun3dBoundingBall boundingBall() {
		return this.asBoundingBall();
	}

	/**
	 * Answer the bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		return this.asBoundingBox();
	}

	/**
	 * Answer the center point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint center() {
		return this.atT_(0.5d);
	}

	/**
	 * Answer the start point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint first() {
		return this.from();
	}

	/**
	 * Answer the start point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint from() {
		return this.atT_(0.0d);
	}

	/**
	 * Initialize this line with two points.
	 * 
	 * @param fromPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param toPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category accessing
	 */
	public void from_to_(JunPoint fromPoint, JunPoint toPoint) {
		Jun3dPoint point1 = Jun3dPoint.Coerce_(fromPoint);
		Jun3dPoint point2 = Jun3dPoint.Coerce_(toPoint);
		x0 = point1.x();
		y0 = point1.y();
		z0 = point1.z();
		f = point2.x() - point1.x();
		g = point2.y() - point1.y();
		h = point2.z() - point1.z();
	}

	/**
	 * Answer the end point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint last() {
		return this.to();
	}

	/**
	 * Answer the length of this curve.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.curves.JunLine#length()
	 * @category accessing
	 */
	public double length() {
		return this.first().distance_(this.last());
	}

	/**
	 * Answer the normal unit vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint normalUnitVector() {
		return this.normalizedLine().normalVector();
	}

	/**
	 * Answer the normal vector of this.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint normalVector() {
		return this.to().minus_(this.from());
	}

	/**
	 * Answer the 't' value of the Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category accessing
	 */
	public double tAtPoint_(Jun3dPoint aPoint) {
		double denominator = (f * f) + (g * g) + (h * h);

		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}

		double x = aPoint.x() - x0;
		double y = aPoint.y() - y0;
		double z = aPoint.z() - z0;

		return ((f * x) + (g * y) + h * z) / denominator;
	}

	/**
	 * Answer the 't' value of the x value.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category accessing
	 */
	public double tAtX_(double aNumber) {
		if (Math.abs(this.f()) < this.Accuracy()) {
			return Double.NaN;
		}
		return (aNumber - this.x0()) / this.f();
	}

	/**
	 * Answer the 't' value of the y value.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category accessing
	 */
	public double tAtY_(double aNumber) {
		if (Math.abs(this.g()) < this.Accuracy()) {
			return Double.NaN;
		}
		return (aNumber - this.y0()) / this.g();
	}

	/**
	 * Answer the 't' value of the z value.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category accessing
	 */
	public double tAtZ_(double aNumber) {
		if (Math.abs(this.h()) < this.Accuracy()) {
			return Double.NaN;
		}
		return (aNumber - this.z0()) / this.h();
	}

	/**
	 * Answer the end point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint to() {
		return this.atT_(1.0d);
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dLine aLine = (Jun3dLine) anObject;
		return this.isEqualNumber_to_(x0, aLine.x0) && this.isEqualNumber_to_(y0, aLine.y0) && this.isEqualNumber_to_(z0, aLine.z0) && this.isEqualNumber_to_(f, aLine.f) && this.isEqualNumber_to_(g, aLine.g) && this.isEqualNumber_to_(h, aLine.h);
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dLine aLine = (Jun3dLine) anObject;
		return x0 == aLine.x0 && y0 == aLine.y0 && z0 == aLine.z0 && f == aLine.f && g == aLine.g && h == aLine.h;
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @category converting
	 */
	public Jun3dPoint[][] asArrays() {
		return new Jun3dPoint[][] { new Jun3dPoint[] { this.from(), this.to() } };
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @category converting
	 */
	public Jun3dLine[] asArrayOfLines() {
		return this.asArrayOf3dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dLines()
	 * @category converting
	 */
	public Jun3dLine[] asArrayOf3dLines() {
		return new Jun3dLine[] { this };
	}

	/**
	 * Convert the receiver as a Jun3dBoundingBall.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category converting
	 */
	public Jun3dBoundingBall asBoundingBall() {
		Jun3dPoint centerPoint = this.center();
		double aRadius = centerPoint.distance_(this.to());
		return new Jun3dBoundingBall(centerPoint, aRadius);
	}

	/**
	 * Convert the receiver as a Jun3dBoundingBox.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category converting
	 */
	public Jun3dBoundingBox asBoundingBox() {
		return Jun3dBoundingBox.Vertex_vertex_(this.from(), this.to());
	}

	/**
	 * Convert the receiver as a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		Jun3dPoint[] vertexes = { this.from(), this.to() };
		JunOpenGL3dPolyline aPolyline = new JunOpenGL3dPolyline(vertexes);
		aPolyline.lineWidth_(1);
		aPolyline.paint_(this.defaultColor());
		return aPolyline;
	}

	/**
	 * Convert the receiver as a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObjectWithPoints()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectWithPoints() {
		Jun3dPoint[] vertexes = { this.from(), this.to() };
		JunOpenGL3dPolyline aPolyline = new JunOpenGL3dPolyline(vertexes);
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.add_(aPolyline);
		JunOpenGL3dObject fromPoint = this.from().asJunOpenGL3dObject();
		fromPoint.paint_(Color.red);
		compoundObject.add_(fromPoint);
		JunOpenGL3dObject toPoint = this.to().asJunOpenGL3dObject();
		toPoint.paint_(Color.blue);
		compoundObject.add_(toPoint);
		return compoundObject;
	}

	/**
	 * Convert the receiver as a <code>JunNurbsCurve</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @see jp.co.sra.jun.geometry.abstracts.JunCurve#asNurbsCurve()
	 * @category converting
	 */
	public JunNurbsCurve asNurbsCurve() {
		return JunNurbsCurve.BezierControlPoints_(new Jun3dPoint[] { this.from(), this.center(), this.to() });
	}

	/**
	 * Convert the receiver as an arrays of <code>Point</code>.
	 * 
	 * @return java.awt.Point[][]
	 * @see jp.co.sra.jun.geometry.curves.JunLine#asPointArrays()
	 * @category converting
	 */
	public Point[][] asPointArrays() {
		Jun3dPoint[][] a3dPoints = this.asArrays();
		Point[][] points = new Point[a3dPoints.length][];
		for (int i = 0; i < a3dPoints.length; i++) {
			points[i] = new Point[a3dPoints[i].length];
			for (int j = 0; j < a3dPoints[i].length; j++) {
				points[i][j] = a3dPoints[i][j].asPoint();
			}
		}
		return points;
	}

	/**
	 * Convert the receiver as a Jun3dPolyline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category converting
	 */
	public Jun3dPolyline asPolyline() {
		return new Jun3dPolyline(new Jun3dPoint[] { this.from(), this.to() });
	}

	/**
	 * Convert the receiver as a Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category converting
	 */
	public Jun3dPoint asUnitVector() {
		return new Jun3dPoint(this.f(), this.g(), this.h()).unitVector();
	}

	/**
	 * Answer the angle between the receiver and aLine.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.JunLine
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.geometry.curves.JunLine#angleWithLine_(jp.co.sra.jun.geometry.curves.JunLine)
	 * @category functions
	 */
	public JunAngle angleWithLine_(JunLine aLine) {
		Jun3dLine theLine = (aLine instanceof Jun3dLine) ? (Jun3dLine) aLine : new Jun3dLine(aLine._from(), aLine._to());
		double denominator = (this.f() * this.f() + this.g() * this.g() + this.h() * this.h()) * (theLine.f() * theLine.f() + theLine.g() * theLine.g() + theLine.h() * theLine.h());
		denominator = Math.sqrt(denominator);
		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("unexpected line parameters");
		}

		double numerator = (this.f() * theLine.f()) + (this.g() * theLine.g()) + (this.h() * theLine.h());
		double gamma = Math.min(Math.max(numerator / denominator, -1.0d), 1.0d);
		return JunAngle.FromRad_(Math.acos(gamma));
	}

	/**
	 * Answer the distance between the Jun3dPoint and the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#distanceFromPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category functions
	 */
	public double distanceFromPoint_(Jun3dPoint aPoint) {
		double denominator = (f * f) + (g * g) + (h * h);

		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}

		double x = aPoint.x() - x0;
		double y = aPoint.y() - y0;
		double z = aPoint.z() - z0;
		double t = ((f * x) + (g * y) + h * z) / denominator;
		Jun3dPoint pointOnMe = this.atT_(t);

		return aPoint.distance_(pointOnMe);
	}

	/**
	 * Answer the intersecting point with the Jun3dLine.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithLine_(Jun3dLine aLine) {
		double yourF = aLine.f;
		double yourG = aLine.g;
		double yourH = aLine.h;
		double yourX0 = aLine.x0;
		double yourY0 = aLine.y0;
		double yourZ0 = aLine.z0;
		double denominatorXY = (yourF * g) - (f * yourG);
		double denominatorYZ = (yourG * h) - (g * yourH);
		double denominatorZX = (yourH * f) - (h * yourF);
		double denominatorXYabs = Math.abs(denominatorXY);
		double denominatorYZabs = Math.abs(denominatorYZ);
		double denominatorZXabs = Math.abs(denominatorZX);
		double t;
		double yourT;

		if (Math.max(Math.max(denominatorXYabs, denominatorYZabs), denominatorZXabs) < ACCURACY) {
			// These are parallel lines.
			return null;
		}

		if ((denominatorXYabs >= denominatorYZabs) && (denominatorXYabs >= denominatorZXabs)) {
			t = (yourF * (yourY0 - y0) - yourG * (yourX0 - x0)) / denominatorXY;
			if (Math.abs(yourF) > Math.abs(yourG)) {
				yourT = ((f * t) + x0 - yourX0) / yourF;
			} else {
				yourT = ((g * t) + y0 - yourY0) / yourG;
			}
		} else if (denominatorYZabs >= denominatorZXabs) {
			t = (yourG * (yourZ0 - z0) - yourH * (yourY0 - y0)) / denominatorYZ;
			if (Math.abs(yourG) > Math.abs(yourH)) {
				yourT = ((g * t) + y0 - yourY0) / yourG;
			} else {
				yourT = ((h * t) + z0 - yourZ0) / yourH;
			}
		} else {
			t = (yourH * (yourX0 - x0) - yourF * (yourZ0 - z0)) / denominatorZX;
			if (Math.abs(yourH) > Math.abs(yourF)) {
				yourT = ((h * t) + z0 - yourZ0) / yourH;
			} else {
				yourT = ((f * t) + x0 - yourX0) / yourF;
			}
		}

		double x = (f * t) + x0;
		double y = (g * t) + y0;
		double z = (h * t) + z0;
		double yourX = yourF * yourT + yourX0;
		double yourY = yourG * yourT + yourY0;
		double yourZ = yourH * yourT + yourZ0;
		if ((Math.abs(x - yourX) > this.Accuracy()) || (Math.abs(y - yourY) > this.Accuracy()) || (Math.abs(z - yourZ) > this.Accuracy())) {
			return null;
		}
		return new Jun3dPoint(x, y, z);
	}

	/**
	 * Answer the distance between the Jun3dPoint and the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category functions
	 */
	public double lineSegmentDistanceFromPoint_(Jun3dPoint aPoint) {
		Jun3dPoint thePoint = (Jun3dPoint) aPoint;
		double denominator = (f * f) + (g * g) + (h * h);
		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}
		double x = thePoint.x() - x0;
		double y = thePoint.y() - y0;
		double z = thePoint.z() - z0;
		double t = ((f * x) + (g * y) + h * z) / denominator;
		Jun3dPoint pointOnMe;
		if (t < 0) {
			pointOnMe = (Jun3dPoint) this.atT_(0);
		} else if (t > 1) {
			pointOnMe = (Jun3dPoint) this.atT_(1);
		} else {
			pointOnMe = (Jun3dPoint) this.atT_(t);
		}
		return thePoint.distance_(pointOnMe);
	}

	/**
	 * Answer the nearest point on the line segment from the Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint lineSegmentNearestPointFromPoint_(Jun3dPoint aPoint) {
		double denominator = (f * f) + (g * g) + (h * h);
		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}
		double x = aPoint.x() - x0;
		double y = aPoint.y() - y0;
		double z = aPoint.z() - z0;
		double t = ((f * x) + (g * y) + h * z) / denominator;
		if (t < 0) {
			t = 0;
		} else if (t > 1) {
			t = 1;
		}
		return (Jun3dPoint) this.atT_(t);
	}

	/**
	 * Answer the nearest point on the receiver from the Jun3dPoint.
	 * 
	 * @param aJun3dLine jp.co.sra.jun.geometry.curves.aJun3dLine
	 * @return jp.co.sra.jun.geometry.curves.aJun3dLine
	 * @category functions
	 */
	public Jun3dPoint nearestPointFromLine_(Jun3dLine aJun3dLine) {
		Jun3dPoint commonNormalVector = new Jun3dPoint(this.f(), this.g(), this.h()).vectorProduct_(new Jun3dPoint(aJun3dLine.f(), aJun3dLine.g(), aJun3dLine.h()));
		if (commonNormalVector.length() < this.Accuracy()) {
			return null;
		}
		JunPlane plane = JunPlane.On_normalVector_(this.from(), commonNormalVector);
		Jun3dLine theLineOnPlane = plane.projectionOfLine_(aJun3dLine);
		double f1 = theLineOnPlane.f();
		double g1 = theLineOnPlane.g();
		double x1 = theLineOnPlane.x0();
		double y1 = theLineOnPlane.y0();
		double t = ((x0 - x1) * g1 + (y1 - y0) * f1) / ((f1 * g) - (f * g1));
		return this.atT_(t);
	}

	/**
	 * Answer the nearest point on the receiver from the Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint nearestPointFromPoint_(Jun3dPoint aPoint) {
		double denominator = (f * f) + (g * g) + (h * h);
		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}
		double x = aPoint.x() - x0;
		double y = aPoint.y() - y0;
		double z = aPoint.z() - z0;
		return (Jun3dPoint) this.atT_(((f * x) + (g * y) + h * z) / denominator);
	}

	/**
	 * Create a normalized line of this.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dLine normalized() {
		return this.normalizedLine();
	}

	/**
	 * Create a normalized line of this.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dLine normalizedLine() {
		Jun3dLine aLine = (Jun3dLine) this.copy();
		double n = this.n();
		aLine.f = f * n;
		aLine.g = g * n;
		aLine.h = h * n;
		return aLine;
	}

	/**
	 * Answer the projection of the Jun3dPoint on the receiver.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint projectionOfPoint_(Jun3dPoint aJun3dPoint) {
		return this.atT_(this.tAtPoint_(aJun3dPoint));
	}

	/**
	 * Answer the reflecting line on the receiver with the specified plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dLine reflectingLineWithPlane_(JunPlane aPlane) {
		return aPlane.reflectingLineWithLine_(this);
	}

	/**
	 * Answer the reversed line of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dLine reversed() {
		return new Jun3dLine(this.to(), this.from());
	}

	/**
	 * Answer the parameter f.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double f() {
		return f;
	}

	/**
	 * Answer the parameter g.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double g() {
		return g;
	}

	/**
	 * Answer the parameter h.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double h() {
		return h;
	}

	/**
	 * Answer a coefficient for my normalization.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double n() {
		double denominator = (f * f) + (g * g) + (h * h);

		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}

		return 1.0d / Math.sqrt(denominator);
	}

	/**
	 * Answer the parameter x0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double x0() {
		return x0;
	}

	/**
	 * Answer the parameter y0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double y0() {
		return y0;
	}

	/**
	 * Answer the parameter z0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double z0() {
		return z0;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("3dLine (");
		this.from().printOn_(aWriter);
		aWriter.write(" , ");
		this.to().printOn_(aWriter);
		aWriter.write(")");
	}

	/**
	 * Answer the receiver's subdivide polyline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category subdividing
	 */
	public Jun3dPolyline subdivide() {
		return new Jun3dPolyline(new Jun3dPoint[] { this.from(), this.center(), this.to() });
	}

	/**
	 * Answer the receiver's subdivide polyline with specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category subdividing
	 */
	public Jun3dPolyline subdivideLevel_(int levelNumber) {
		Jun3dPolyline polyline = this.asPolyline();
		for (int i = 0; i < levelNumber; i++) {
			polyline = polyline.subdivide();
		}
		return polyline;
	}

	/**
	 * Answer true if the receiver contains aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun3dPoint aPoint) {
		return this.distanceFromPoint_(aPoint) < ACCURACY;
	}

	/**
	 * Answer true if the receiver is a 3d geometry element.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Answer true if the receiver is parallel with aLine, otherwise false.
	 * 
	 * @return boolean
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category testing
	 */
	public boolean isParallelWithLine_(Jun3dLine aLine) {
		double degrees = this.angleWithLine_(aLine).deg();
		return (Math.abs(0.0d - degrees) < this.Accuracy() || Math.abs(180.0d - degrees) < this.Accuracy());
	}

	/**
	 * Answer true if the receiver contains the Jun3dPoint, otherwise false.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean lineSegmentContainsPoint_(Jun3dPoint aPoint) {
		return this.lineSegmentDistanceFromPoint_(aPoint) < ACCURACY;
	}

	/**
	 * Answer the new Jun3dLine which is rotated by anAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category transforming
	 */
	public Jun3dLine rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun3dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new Jun3dLine which is scaled by aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category transforming
	 */
	public Jun3dLine scaledBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply aJunTransformation to the receiver.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category transforming
	 */
	public Jun3dLine transform_(Jun3dTransformation aTransformation) {
		return new Jun3dLine(this.from().transform_(aTransformation), this.to().transform_(aTransformation));
	}

	/**
	 * Answer the copy of the receiver which is applied the specified transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#transform_(jp.co.sra.jun.geometry.transformations.JunTransformation)
	 * @category transforming
	 */
	public JunGeometry transform_(JunTransformation aTransformation) {
		return this.transform_((Jun3dTransformation) aTransformation);
	}

	/**
	 * Answer the transformation to rotate the angle around the receiver.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTransformation transformationToRotate_(JunAngle anAngle) {
		return Jun3dTransformation.Rotate_around_(anAngle, this);
	}

	/**
	 * Answer the new Jun3dLine which is translated by aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.abstracts.Jun3dLine
	 * @category transforming
	 */
	public Jun3dLine translatedBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Translate_(aPoint));
	}
}
