package jp.co.sra.jun.geometry.coordinate;

import java.util.ArrayList;

import jp.co.sra.smalltalk.StObject;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.geometry.transformations.JunTransformation;

/**
 * Jun3dCoordinateSystem class
 * 
 *  @author    nisinaka
 *  @created   1998/10/05 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/20 (by Mitsuhiro Asada)
 *  @updated   2006/10/11 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun610 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dCoordinateSystem.java,v 8.12 2008/02/20 06:30:57 nisinaka Exp $
 */
public class Jun3dCoordinateSystem extends JunCoordinateSystem {

	protected Jun3dPoint origin;
	protected Jun3dTransformation transformation;
	protected Jun3dTransformation invTransformation;
	protected Jun3dTransformation globalTransformation;
	protected Jun3dTransformation globalInvTransformation;
	protected Jun3dCoordinateSystem outerCoordinateSystem;
	protected ArrayList localCoordinateSystems;
	protected Jun3dPoint xUnitVector;
	protected Jun3dPoint yUnitVector;
	protected Jun3dPoint zUnitVector;

	/**
	 * Create a new Jun3dCoordinateSystem with the specified scale.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category Typical coordinate system
	 */
	public static Jun3dCoordinateSystem Scale_(double aNumber) {
		return new Jun3dCoordinateSystem(Jun3dTransformation.Scale_(aNumber));
	}

	/**
	 * Create a new Jun3dCoordinateSystem with the specified scale.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category Typical coordinate system
	 */
	public static Jun3dCoordinateSystem Scale_(Jun3dPoint aPoint) {
		return new Jun3dCoordinateSystem(Jun3dTransformation.Scale_(aPoint));
	}

	/**
	 * Create a new Jun3dCoordinateSystem with the specified translation.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category Typical coordinate system
	 */
	public static Jun3dCoordinateSystem Translate_(double aNumber) {
		return new Jun3dCoordinateSystem(Jun3dTransformation.Translate_(aNumber));
	}

	/**
	 * Create a new Jun3dCoordinateSystem with the specified translation.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category Typical coordinate system
	 */
	public static Jun3dCoordinateSystem Translate_(Jun3dPoint aPoint) {
		return new Jun3dCoordinateSystem(Jun3dTransformation.Translate_(aPoint));
	}

	/**
	 * Create a new Jun3dCoordinateSystem with specified Jun3dTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category Instance creation
	 */
	public static Jun3dCoordinateSystem Transformation_(Jun3dTransformation aTransformation) {
		return new Jun3dCoordinateSystem(aTransformation);
	}

	/**
	 * Create a new instance of Jun3dCoordinateSystem and initialize it.
	 *
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category Instance creation
	 */
	public Jun3dCoordinateSystem(Jun3dTransformation aTransformation) {
		this.transformation_(aTransformation);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		origin = null;
		transformation = null;
		invTransformation = null;
		globalTransformation = null;
		globalInvTransformation = null;
		outerCoordinateSystem = null;
		localCoordinateSystems = null;
		xUnitVector = null;
		yUnitVector = null;
		zUnitVector = null;
	}

	/**
	 * Answer my current transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category accessing
	 */
	public Jun3dTransformation transformation() {
		if (transformation == null) {
			transformation = this.computeTransformation();
		}
		return transformation;
	}

	/**
	 * Answer my current transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @see jp.co.sra.jun.geometry.coordinate.JunCoordinateSystem#_transformation()
	 * @category accessing
	 */
	public JunTransformation _transformation() {
		return this.transformation();
	}

	/**
	 * Set my new transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category accessing
	 */
	public void transformation_(Jun3dTransformation aTransformation) {
		if (transformation == null) {
			if (aTransformation == null) {
				return;
			}
		} else {
			if (transformation.equals(aTransformation)) {
				return;
			}
		}

		transformation = aTransformation;
		invTransformation = null;
		this.clearGlobalTransformation();
		this.clearOriginAndVectors();
	}

	/**
	 * Answer my current inverse transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @category accessing
	 */
	public Jun3dTransformation invTransformation() {
		if (invTransformation == null) {
			invTransformation = this.transformation().inverse();
		}
		return invTransformation;
	}

	/**
	 * Set my new inverse transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @category accessing
	 */
	public void invTransformation_(Jun3dTransformation aTransformation) {
		invTransformation = aTransformation;
		transformation = aTransformation.inverse();
		this.clearOriginAndVectors();
	}

	/**
	 * Answer the global transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category accessing
	 */
	public Jun3dTransformation globalTransformation() {
		if (globalTransformation == null) {
			globalTransformation = (Jun3dTransformation) this.transformation();
			if (outerCoordinateSystem != null) {
				globalTransformation = globalTransformation.product_(outerCoordinateSystem.globalTransformation());
			}
		}
		return globalTransformation;
	}

	/**
	 * Answer the inverse of global transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category accessing
	 */
	public Jun3dTransformation globalInvTransformation() {
		if (globalInvTransformation == null) {
			globalInvTransformation = this.globalTransformation().inverse();
		}
		return globalInvTransformation;
	}

	/**
	 * Answer my global inverse transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @see jp.co.sra.jun.geometry.coordinate.JunCoordinateSystem#_globalInvTransformation()
	 * @category accessing
	 */
	public JunTransformation _globalInvTransformation() {
		return this.globalInvTransformation();
	}

	/**
	 * Answer my current outer coordinate system.
	 * 
	 * @return jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category accessing
	 */
	public Jun3dCoordinateSystem outerCoordinateSystem() {
		return outerCoordinateSystem;
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		return this == anObject;
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		return this == anObject;
	}

	/**
	 * Convert the receiver as a local coordinate system of aCoordinateSystem.
	 * 
	 * @param aCoordinateSystem jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @return jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category converting
	 */
	public Jun3dCoordinateSystem asLocalCoordinateSystemOf_(Jun3dCoordinateSystem aCoordinateSystem) {
		if (outerCoordinateSystem == null) {
			if (aCoordinateSystem == null) {
				return this;
			}
		} else {
			if (outerCoordinateSystem.equals(aCoordinateSystem)) {
				return this;
			}
		}

		if (aCoordinateSystem == null) {
			return new Jun3dCoordinateSystem(this.globalTransformation());
		}

		return aCoordinateSystem.newLocalCoordinateSystemTransformation_(this.globalTransformation().product_(aCoordinateSystem.globalInvTransformation()));
	}

	/**
	 * Do something after a shallow copy.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();
		localCoordinateSystems = null;
		if (outerCoordinateSystem != null) {
			outerCoordinateSystem.privateAddLocalCoordinateSystem_(this);
		}
		return this;
	}

	/**
	 * Create a new local coordinate system with a specified scale.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.coordinate.JunCoordinateSystem
	 * @category Instance creation
	 */
	public Jun3dCoordinateSystem newLocalCoordinateSystemScale_(double aNumber) {
		Jun3dCoordinateSystem aCoordinateSystem = Scale_(aNumber);
		this.privateAddLocalCoordinateSystem_(aCoordinateSystem);
		return aCoordinateSystem;
	}

	/**
	 * Create a new local coordinate system with a transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category Instance creation
	 */
	public Jun3dCoordinateSystem newLocalCoordinateSystemTransformation_(Jun3dTransformation aTransformation) {
		Jun3dCoordinateSystem aCoordinateSystem = Transformation_(aTransformation);
		this.privateAddLocalCoordinateSystem_(aCoordinateSystem);
		return aCoordinateSystem;
	}

	/**
	 * Answer true if the receiver is a 3d geometry element.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Clear the global transformation.
	 * 
	 * @category private
	 */
	protected void clearGlobalTransformation() {
		if (globalTransformation == null && globalInvTransformation == null) {
			return;
		}

		globalTransformation = null;
		globalInvTransformation = null;
		if (localCoordinateSystems != null) {
			for (int i = 0; i < localCoordinateSystems.size(); i++) {
				((Jun3dCoordinateSystem) localCoordinateSystems.get(i)).clearGlobalTransformation();
			}
		}
	}

	/**
	 * Clear the origin and vectors.
	 * 
	 * @category private
	 */
	protected void clearOriginAndVectors() {
		origin = null;
		xUnitVector = null;
		yUnitVector = null;
		zUnitVector = null;
	}

	/**
	 * Clear the transformations.
	 * 
	 * @category private
	 */
	protected void clearTransformations() {
		transformation = null;
		invTransformation = null;
		this.clearGlobalTransformation();
	}

	/**
	 * Compute the transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category private
	 */
	protected Jun3dTransformation computeTransformation() {
		double[] parameters = new double[16];
		parameters[0] = xUnitVector.x() - origin.x();
		parameters[1] = xUnitVector.y() - origin.x();
		parameters[2] = xUnitVector.z() - origin.x();
		parameters[3] = 0.0d;
		parameters[4] = xUnitVector.x() - origin.y();
		parameters[5] = xUnitVector.y() - origin.y();
		parameters[6] = xUnitVector.z() - origin.y();
		parameters[7] = 0.0d;
		parameters[8] = xUnitVector.x() - origin.z();
		parameters[9] = xUnitVector.y() - origin.z();
		parameters[10] = xUnitVector.z() - origin.z();
		parameters[11] = 0.0d;
		parameters[12] = origin.x();
		parameters[13] = origin.y();
		parameters[14] = origin.z();
		parameters[15] = 1.0d;
		return Jun3dTransformation.FromArray_(parameters);
	}

	/**
	 * Add aCoordinateSystem as a local coordinate system of the receiver.
	 * 
	 * @param aCoordinateSystem jp.co.sra.jun.geometry.coordinate.Jun3dCoordinateSystem
	 * @category private
	 */
	protected void privateAddLocalCoordinateSystem_(Jun3dCoordinateSystem aCoordinateSystem) {
		aCoordinateSystem.outerCoordinateSystem = this;

		if (localCoordinateSystems == null) {
			localCoordinateSystems = new ArrayList();
		}
		localCoordinateSystems.add(aCoordinateSystem);
	}

}
