package jp.co.sra.jun.geometry.boundaries;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBalls;
import jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes;
import jp.co.sra.jun.geometry.transformations.JunTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * Jun3dBoundingBall class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2000/01/24 (by NISHIHARA Satoshi)
 *  @updated   2004/10/19 (by Mitsuhiro Asada)
 *  @updated   2006/10/10 (by nisinaka)
 *  @updated   2007/05/29 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun666 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dBoundingBall.java,v 8.15 2008/02/20 06:30:55 nisinaka Exp $
 */
public class Jun3dBoundingBall extends JunBoundingBall {

	/**
	 * Create a new Jun3dBoundingBall and initialize it.
	 * 
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param radiusValue double
	 * @category Instance creation
	 */
	public Jun3dBoundingBall(Jun3dPoint centerPoint, double radiusValue) {
		this.center_(centerPoint);
		this.radius_(radiusValue);
	}

	/**
	 * Create a new instance of Jun3dBoundingBall from anArrayOfJunPoint
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category Instance creation
	 */
	public Jun3dBoundingBall(Jun3dPoint[] pointCollection) {
		if (pointCollection == null || pointCollection.length == 0) {
			return;
		}

		double x = 0;
		double y = 0;
		double z = 0;
		for (int i = 0; i < pointCollection.length; i++) {
			x += pointCollection[i].x();
			y += pointCollection[i].y();
			z += pointCollection[i].z();
		}
		Jun3dPoint centerPoint = new Jun3dPoint(x / pointCollection.length, y / pointCollection.length, z / pointCollection.length);

		double maxDistance = 0;
		for (int i = 0; i < pointCollection.length; i++) {
			double distance = centerPoint.distance_(pointCollection[i]);
			if (distance > maxDistance) {
				maxDistance = distance;
			}
		}

		this.center_(centerPoint);
		this.radius_(maxDistance);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		center = Jun3dPoint.Zero();
	}

	/**
	 * Answer my center point
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint center() {
		return (Jun3dPoint) this._center();
	}

	/**
	 * Set my center point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void center_(Jun3dPoint aPoint) {
		this._center(aPoint);
	}

	/**
	 * Answer my area.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return 4 * Math.PI * Math.pow(this.radius(), 2);
	}

	/**
	 * Answer my depth.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBall#depth()
	 * @category accessing
	 */
	public double depth() {
		return this.boundingBox().depth();
	}

	/**
	 * Answer the receiver's detailed bounding balls.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBalls
	 * @category accessing 
	 */
	public Jun3dBoundingBalls detailedBoundingBalls() {
		return this.octaBoundingBalls();
	}

	/**
	 * Answer the receiver's detailed bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category accessing 
	 */
	public Jun3dBoundingBoxes detailedBoundingBoxes() {
		return this.octaBoundingBoxes();
	}

	/**
	 * Answer the receiver's octa bounding balls.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBalls
	 * @category accessing 
	 */
	public Jun3dBoundingBalls octaBoundingBalls() {
		Jun3dBoundingBalls boundingBalls = new Jun3dBoundingBalls();
		boundingBalls.addAll_(this.boundingBox().octaBoundingBalls());
		return boundingBalls;
	}

	/**
	 * Answer the receiver's octa bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes
	 * @category accessing 
	 */
	public Jun3dBoundingBoxes octaBoundingBoxes() {
		Jun3dBoundingBoxes boundingBoxes = new Jun3dBoundingBoxes();
		boundingBoxes.addAll_(this.boundingBox().octaBoundingBoxes());
		return boundingBoxes;
	}

	/**
	 * Answer my volume.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#volume()
	 * @category accessing
	 */
	public double volume() {
		return 4 * Math.PI * Math.pow(this.radius(), 3) / 3;
	}

	/**
	 * Answer my bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category accessing
	 */
	public Jun3dBoundingBall boundingBall() {
		return this;
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		return Jun3dBoundingBox.Origin_extent_(this.center(), Jun3dPoint.Zero()).expandedBy_(this.radius());
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.JunBoundingBox
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBall#_boundingBox()
	 * @category accessing
	 */
	protected JunBoundingBox _boundingBox() {
		return this.boundingBox();
	}

	/**
	 * Add a point to the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category adding
	 */
	public void add_(Jun3dPoint aPoint) {
		if (this.center().distance_(aPoint) > this.radius()) {
			Jun3dBoundingBall ball = this.merge_(new Jun3dBoundingBall(aPoint, 0));
			this.center_(ball.center());
			this.radius_(ball.radius());
		}
	}

	/**
	 * Convert to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dObject aBall = JunOpenGL3dObject.Ball_radius_center_(3, this.radius(), this.center());
		aBall.objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunOpenGL3dObject) each).paint_alpha_(Jun3dBoundingBall.this.defaultColor(), Jun3dBoundingBall.this.defaultAlpha());
				return null;
			}
		});
		return aBall;
	}

	/**
	 * Expand the receiver by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category functions
	 */
	public Jun3dBoundingBall expandedBy_(double aNumber) {
		return new Jun3dBoundingBall(this.center(), this.radius() + aNumber);
	}

	/**
	 * Inset the receiver by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category functions
	 */
	public Jun3dBoundingBall insetBy_(double aNumber) {
		return new Jun3dBoundingBall(this.center(), this.radius() - aNumber);
	}

	/**
	 * Merge the receiver with the specified bounding ball.
	 * 
	 * @param aBoundingBall jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category functions
	 */
	public Jun3dBoundingBall merge_(Jun3dBoundingBall aBoundingBall) {
		double distance = this.center().distance_(aBoundingBall.center());
		if ((distance + aBoundingBall.radius()) < this.radius()) {
			return (Jun3dBoundingBall) this.copy();
		}
		if ((distance + this.radius()) < aBoundingBall.radius()) {
			return (Jun3dBoundingBall) aBoundingBall.copy();
		}

		Jun3dLine aLine = new Jun3dLine(this.center(), aBoundingBall.center());
		Jun3dPoint fromPoint = aLine.atT_(0 - this.radius() / distance);
		Jun3dPoint toPoint = aLine.atT_(1 + aBoundingBall.radius() / distance);
		return new Jun3dBoundingBall(fromPoint.center_(toPoint), fromPoint.distance_(toPoint) / 2);
	}

	/**
	 * Answer true if the receiver contains the specified bounding ball, otherwise false.
	 * 
	 * @param aBoundingBall jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @return boolean
	 * @category testing
	 */
	public boolean contains_(Jun3dBoundingBall aBoundingBall) {
		double delta = this.center().distance_(aBoundingBall.center());
		return delta <= this.radius() && delta + aBoundingBall.radius() <= this.radius();
	}

	/**
	 * Answer true if the receiver contains the point, otherwise false.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun3dPoint aPoint) {
		return this.center().distance_(aPoint) <= this.radius() || this.touchesPoint_(aPoint);
	}

	/**
	 * Answer true if the receiver intersects with the bounding ball, otherwise false.
	 * 
	 * @param aBoundingBall jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @return boolean
	 * @category testing
	 */
	public boolean intersects_(Jun3dBoundingBall aBoundingBall) {
		return this.center().distance_(aBoundingBall.center()) <= this.radius() + aBoundingBall.radius();
	}

	/**
	 * Answer true if the receiver touches the point, otherwise false.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean touchesPoint_(Jun3dPoint aPoint) {
		double distance = this.center().distance_(aPoint);
		return this.isEqualNumber_to_(distance, this.radius());
	}

	/**
	 * Answer true if the receiver is a 3d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Answer <code>true</code> if the receiver touches with the specified bounding ball, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category testing
	 */
	public boolean touches_(Jun3dBoundingBall aBoundingBall) {
		if (this.isEqualPoint_to_(this.center(), aBoundingBall.center()) && this.isEqualNumber_to_(this.radius(), aBoundingBall.radius())) {
			return true;
		}
		double distance = this.center().distance_(aBoundingBall.center());
		if (distance <= this.radius()) {
			return this.isEqualNumber_to_(this.radius(), distance + aBoundingBall.radius());
		}
		if (distance <= aBoundingBall.radius()) {
			return this.isEqualNumber_to_(aBoundingBall.radius(), distance + this.radius());
		}
		return this.isEqualNumber_to_(distance, this.radius() + aBoundingBall.radius());
	}

	/**
	 * Answer the copy of the receiver which is applied the specified transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#transform_(jp.co.sra.jun.geometry.transformations.JunTransformation)
	 * @category transforming
	 */
	public JunGeometry transform_(JunTransformation aTransformation) {
		Jun3dLine aLine = new Jun3dLine(this.center(), this.center().plus_(new Jun2dPoint(this.radius(), 0)));
		aLine = (Jun3dLine) aLine.transform_(aTransformation);
		return new Jun3dBoundingBall(aLine.from(), aLine.from().distance_(aLine.to()));
	}

	/**
	 * Answer an rounded copy of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category truncation and round off
	 */
	public Jun3dBoundingBall rounded() {
		return new Jun3dBoundingBall(this.center().rounded(), Math.round(this.radius()));
	}

	/**
	 * Answer an truncated copy of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category truncation and round off
	 */
	public Jun3dBoundingBall truncated() {
		double newRadius = (this.radius() >= 0) ? Math.ceil(this.radius()) : Math.floor(this.radius());
		return new Jun3dBoundingBall(this.center().truncated(), newRadius);
	}

}
