package jp.co.sra.jun.geometry.basic;

import java.awt.Point;
import java.util.ArrayList;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.surfaces.JunPlane;

/**
 * JunPoint class
 * 
 *  @author    nisinaka
 *  @created   1998/09/29 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/19 (by Mitsuhiro Asada)
 *  @updated   2004/12/06 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun675 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPoint.java,v 8.14 2008/02/20 06:30:55 nisinaka Exp $
 */
public abstract class JunPoint extends JunGeometry {

	/**
	 * Answer the neighborhood points of the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point[]
	 * @category Utilities
	 */
	public static Point[] NeighborhoodPoints_(Point aPoint) {
		return Neighborhood8Points_(aPoint);
	}

	/**
	 * Answer the 4 neighborhood points of the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point[]
	 * @category Utilities
	 */
	public static Point[] Neighborhood4Points_(Point aPoint) {
		return new Point[] { new Point(aPoint.x + 1, aPoint.y + 0), new Point(aPoint.x + 0, aPoint.y + 1), new Point(aPoint.x - 1, aPoint.y + 0), new Point(aPoint.x + 0, aPoint.y - 1) };

	}

	/**
	 * Answer the 8 neighborhood points of the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point[]
	 * @category Utilities
	 */
	public static Point[] Neighborhood8Points_(Point aPoint) {
		return new Point[] {
				new Point(aPoint.x + 1, aPoint.y + 0),
				new Point(aPoint.x + 1, aPoint.y + 1),
				new Point(aPoint.x + 0, aPoint.y + 1),
				new Point(aPoint.x - 1, aPoint.y + 1),
				new Point(aPoint.x - 1, aPoint.y + 0),
				new Point(aPoint.x - 1, aPoint.y - 1),
				new Point(aPoint.x + 0, aPoint.y - 1),
				new Point(aPoint.x + 1, aPoint.y - 1) };
	}

	/**
	 * Answer the 24 neighborhood points of the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point[]
	 * @category Utilities
	 */
	public static Point[] Neighborhood24Points_(Point aPoint) {
		ArrayList aList = new ArrayList(24);
		for (int y = -2; y <= 2; y++) {
			for (int x = -2; x <= 2; x++) {
				if (x != 0 || y != 0) {
					aList.add(new Point(aPoint.x + x, aPoint.y + y));
				}
			}
		}
		return (Point[]) aList.toArray(new Point[aList.size()]);
	}

	/**
	 * Answer the x coordinate.
	 * 
	 * @return double
	 * @category accessing
	 */
	public abstract double x();

	/**
	 * Answer the y coordinate.
	 * 
	 * @return double
	 * @category accessing
	 */
	public abstract double y();

	/**
	 * Answer the sum of this point and aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @category arithmetic
	 */
	public abstract JunPoint plus_(JunPoint aPoint);

	/**
	 * Answer the number representing the ordering of the receiver in the
	 * generality hierarchy.
	 * 
	 * @return int
	 * @category coercing
	 */
	public abstract int generality();

	/**
	 * Answer the length from the origin.
	 * 
	 * @return double
	 * @category compatibilities
	 */
	public double r() {
		return this.rho();
	}

	/**
	 * Answer the receiver as Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category converting
	 */
	public abstract Jun2dPoint as2dPoint();

	/**
	 * Answer the receiver as Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category converting
	 */
	public abstract Jun3dPoint as3dPoint();

	/**
	 * Answer the receiver as an array of double.
	 * 
	 * @return double[]
	 * @category converting
	 */
	public abstract double[] asArray();

	/**
	 * Convert the receiver as Point.
	 * 
	 * @return java.awt.Point
	 * @category converting
	 */
	public Point asPoint() {
		return new Point((int) Math.round(this.x()), (int) Math.round(this.y()));
	}

	/**
	 * Answer the length from the origin.
	 * 
	 * @return double
	 * @category functions
	 */
	public double length() {
		return this.rho();
	}

	/**
	 * Answer the plane with the specified normal vector.
	 * 
	 * @param normalVector jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category functions
	 */
	public JunPlane plane_(JunPoint normalVector) {
		return JunPlane.On_normalVector_(this, normalVector);
	}

	/**
	 * Answer the plane with the specified two points.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category functions
	 */
	public JunPlane plane_and_(JunPoint aPoint1, JunPoint aPoint2) {
		return new JunPlane(this, aPoint1, aPoint2);
	}

	/**
	 * Computes a hash code for this object.
	 * 
	 * @return int
	 * @see java.lang.Object#hashCode()
	 * @category hash function
	 */
	public abstract int hashCode();

	/**
	 * Answer the rho as of a polar coordinate.
	 * 
	 * @return double
	 * @category polar coordinates
	 */
	public abstract double rho();

	/**
	 * Answer true if this point is ZERO, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isPoint()
	 * @category testing
	 */
	public boolean isPoint() {
		return true;
	}

	/**
	 * Answer true if the receiver is a zero, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public abstract boolean isZero();

	/**
	 * Answer true if this point is negative, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public abstract boolean negative();

	/**
	 * Answer true if this point is positive, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public abstract boolean positive();

	/**
	 * Answer the value which side of on a plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return int
	 * @category testing
	 */
	public int whichSideOf_(JunPlane aPlane) {
		return aPlane.whichSide_(this);
	}
}
