package jp.co.sra.jun.delaunay.twoD;

import java.awt.Graphics;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * Jun2dDelaunayProcessor class
 * 
 *  @author    Ryouichi Matsuda
 *  @created   2002/01/14 (by Ryouichi Matsuda)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dDelaunayProcessor.java,v 8.14 2008/02/20 06:30:54 nisinaka Exp $
 */
public class Jun2dDelaunayProcessor extends JunAbstractObject {
	// 1.0d-8
	public static final double DECIMAL_8 = 0.00000001d;
	// 1.0d-12
	public static final double DECIMAL_12 = 0.000000000001d;
	protected Jun2dDelaunayList loops;
	protected Jun2dDelaunayList vertices;
	protected Jun2dDelaunayList edges;
	protected Jun2dBoundingBox bounds;
	protected boolean constrained;

	/**
	 * Create a new instance of <code>Jun2dDelaunayProcessor</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public Jun2dDelaunayProcessor() {
		super();
	}

	/**
	 * Create a new instance of <code>Jun2dDelaunayProcessor</code> and initialize it.
	 * 
	 * @param aJun2dBoundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category Instance creation
	 */
	public Jun2dDelaunayProcessor(Jun2dBoundingBox aJun2dBoundingBox) {
		super();
		this.setBounds_(aJun2dBoundingBox);
		this.initialize();
	}

	/**
	 * Create a new instance of <code>Jun2dDelaunayProcessor</code> and initialize it.
	 * 
	 * @param aJun2dBoundingBox jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayProcessor
	 * @category Instance creation
	 */
	public static Jun2dDelaunayProcessor Bounds_(Jun2dBoundingBox aJun2dBoundingBox) {
		return new Jun2dDelaunayProcessor(aJun2dBoundingBox);
	}

	/**
	 * Create a new instance of <code>Jun2dDelaunayProcessor</code> and initialize it.
	 * 
	 * @param anArrayOfJun3dPoint jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayProcessor
	 * @category Instance creation
	 */
	public static Jun2dDelaunayProcessor FromPoints_(JunPoint[] anArrayOfJun3dPoint) {
		double minX = anArrayOfJun3dPoint[0].x();
		double maxX = minX;
		double minY = anArrayOfJun3dPoint[0].y();
		double maxY = minY;
		for (int i = 0; i < anArrayOfJun3dPoint.length; i++) {
			double x = anArrayOfJun3dPoint[i].x();
			double y = anArrayOfJun3dPoint[i].y();
			if (x < minX) {
				minX = x;
			}
			if (maxX < x) {
				maxX = x;
			}
			if (y < minY) {
				minY = y;
			}
			if (maxY < y) {
				maxY = y;
			}
		}
		double marginX = (maxX - minX) / 4.0d;
		double marginY = (maxY - minY) / 4.0d;
		Jun2dDelaunayProcessor delaunayProcessor = new Jun2dDelaunayProcessor(Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(minX - marginX, minY - marginY), new Jun2dPoint(maxX + marginX, maxY + marginY)));
		for (int i = 0; i < anArrayOfJun3dPoint.length; i++) {
			delaunayProcessor.insertPoint_(anArrayOfJun3dPoint[i]);
		}
		return delaunayProcessor;
	}

	/**
	 * Create a new instance of <code>Jun2dDelaunayProcessor</code> and initialize it.
	 * 
	 * @param anArrayOfPoint jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayProcessor
	 * @category Instance creation
	 */
	public static Jun2dDelaunayProcessor FromPolygon_(JunPoint[] anArrayOfPoint) {
		double minX = anArrayOfPoint[0].x();
		double maxX = minX;
		double minY = anArrayOfPoint[0].y();
		double maxY = minY;
		for (int i = 0; i < anArrayOfPoint.length; i++) {
			double x = anArrayOfPoint[i].x();
			double y = anArrayOfPoint[i].y();
			if (x < minX) {
				minX = x;
			}
			if (maxX < x) {
				maxX = x;
			}
			if (y < minY) {
				minY = y;
			}
			if (maxY < y) {
				maxY = y;
			}
		}
		double marginX = (maxX - minX) / 4.0d;
		double marginY = (maxY - minY) / 4.0d;
		Jun2dDelaunayProcessor delaunayProcessor = new Jun2dDelaunayProcessor(Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(minX - marginX, minY - marginY), new Jun2dPoint(maxX + marginX, maxY + marginY)));
		delaunayProcessor.insertPolygon_(anArrayOfPoint);
		return delaunayProcessor;
	}

	/**
	 * Answer the default bounds.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category Defaults
	 */
	public static Jun2dBoundingBox DefaultBounds() {
		return Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(-0.0001, -0.0001), new Jun2dPoint(0.0001, 0.0001));
	}

	/**
	 * Answer the instance of identity dictionary.
	 * 
	 * @return java.util.Hashtable
	 * @category Utilities
	 */
	public static Hashtable _IdentityDictionary() {
		return new Hashtable() {
			Vector keys = _IdentitySet();
			Vector values = new Vector();

			public Object put(Object key, Object obj) {
				int idx = keys.indexOf(key);

				if (idx < 0) {
					keys.addElement(key);
					values.addElement(obj);
				} else {
					values.setElementAt(obj, idx);
				}

				return null;
			}

			public Object get(Object key) {
				int idx = keys.indexOf(key);

				if (idx < 0) {
					return null;
				} else {
					return values.elementAt(idx);
				}
			}

			public Enumeration keys() {
				return keys.elements();
			}
		};
	}

	/**
	 * Answer the instance of identity set.
	 * 
	 * @return java.util.Vector
	 * @category Utilities
	 */
	public static Vector _IdentitySet() {
		return new Vector() {
			public int indexOf(Object elem) {
				for (int i = 0; i < elementCount; i++) {
					if (elementData[i] == elem) {
						return i;
					}
				}

				return -1;
			}

			public void addElement(Object obj) {
				if (indexOf(obj) < 0) {
					super.addElement(obj);
				}
			}
		};
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		if (bounds == null) {
			bounds = this.DefaultBounds();
		}

		vertices = new Jun2dDelaunayList();
		edges = new Jun2dDelaunayList();
		loops = new Jun2dDelaunayList();
		Jun2dDelaunayVertex vertex00 = this.newBoundaryVertexAtX_y_(bounds.origin().x(), bounds.origin().y());
		Jun2dDelaunayVertex vertex10 = this.newBoundaryVertexAtX_y_(bounds.corner().x(), bounds.origin().y());
		Jun2dDelaunayVertex vertex11 = this.newBoundaryVertexAtX_y_(bounds.corner().x(), bounds.corner().y());
		Jun2dDelaunayVertex vertex01 = this.newBoundaryVertexAtX_y_(bounds.origin().x(), bounds.corner().y());
		Jun2dDelaunayLoop l1 = this.newLoop();
		Jun2dDelaunayLoop l2 = this.newLoop();
		Jun2dDelaunayEdge e1 = this.newEdge();
		Jun2dDelaunayEdge e2 = this.newEdge();
		Jun2dDelaunayEdge e3 = this.newEdge();
		Jun2dDelaunayEdge e4 = this.newEdge();
		Jun2dDelaunayEdge e5 = this.newEdge();
		e1.vertex1_(vertex00);
		e1.vertex2_(vertex10);
		e2.vertex1_(vertex10);
		e2.vertex2_(vertex11);
		e3.vertex1_(vertex11);
		e3.vertex2_(vertex01);
		e4.vertex1_(vertex01);
		e4.vertex2_(vertex00);
		e5.vertex1_(vertex00);
		e5.vertex2_(vertex11);
		e1.halfEdge12().next_(e2.halfEdge12());
		e2.halfEdge12().next_(e5.halfEdge21());
		e5.halfEdge21().next_(e1.halfEdge12());
		e3.halfEdge12().next_(e4.halfEdge12());
		e4.halfEdge12().next_(e5.halfEdge12());
		e5.halfEdge12().next_(e3.halfEdge12());
		e1.halfEdge21().next_(e4.halfEdge21());
		e2.halfEdge21().next_(e1.halfEdge21());
		e3.halfEdge21().next_(e2.halfEdge21());
		e4.halfEdge21().next_(e3.halfEdge21());
		e1.loop12_(l1);
		e2.loop12_(l1);
		e5.loop21_(l1);
		e3.loop12_(l2);
		e4.loop12_(l2);
		e5.loop12_(l2);
		l1.halfEdge_(e1.halfEdge12());
		l2.halfEdge_(e4.halfEdge12());
		vertex00.halfEdge_(e1.halfEdge21());
		vertex10.halfEdge_(e2.halfEdge21());
		vertex11.halfEdge_(e3.halfEdge21());
		vertex01.halfEdge_(e4.halfEdge21());
		constrained = false;
	}

	/**
	 * Answer the receiver's inside triangle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint[][]
	 * @category accessing
	 */
	public JunPoint[][] insideTriangles() {
		final Collection triangles = new ArrayList();
		this.insideTrianglesDo_(new StBlockClosure() {
			public Object value_value_value_(Object p1, Object p2, Object p3) {
				triangles.add(new JunPoint[] { (JunPoint) p1, (JunPoint) p2, (JunPoint) p3 });
				return null;
			}
		});
		return (JunPoint[][]) triangles.toArray(new JunPoint[triangles.size()][]);
	}

	/**
	 * Answer the receiver's loop at the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 * @category accessing
	 */
	public Jun2dDelaunayLoop loopAt_(JunPoint aPoint) {
		return this.loopAtX_y_(aPoint.x(), aPoint.y());
	}

	/**
	 * Answer the receiver's triangle at the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public JunPoint[] triangleAt_(JunPoint aPoint) {
		Jun2dDelaunayLoop loop = this.loopAt_(aPoint);
		return (loop != null) ? new JunPoint[] { loop.point1(), loop.point2(), loop.point3() } : null;
	}

	/**
	 * Answer the receiver's triangle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint[][]
	 * @category accessing
	 */
	public JunPoint[][] triangles() {
		final Collection triangles = new ArrayList();
		this.trianglesDo_(new StBlockClosure() {
			public Object value_value_value_(Object p1, Object p2, Object p3) {
				triangles.add(new JunPoint[] { (JunPoint) p1, (JunPoint) p2, (JunPoint) p3 });
				return null;
			}
		});
		return (JunPoint[][]) triangles.toArray(new JunPoint[triangles.size()][]);
	}

	/**
	 * Answer the receiver's vertex at the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category accessing
	 */
	public Jun2dDelaunayVertex vertexAt_(JunPoint aPoint) {
		return this.nearestVertexFromX_y_(aPoint.x(), aPoint.y());
	}

	/**
	 * Display the receiver on aGraphics.
	 * 
	 * @param aGraphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(final Graphics aGraphicsContext) {
		edges.do_(new StBlockClosure() {
			public Object value_(Object edge_) {
				Jun2dDelaunayEdge edge = (Jun2dDelaunayEdge) edge_;
				edge.displayOn_(aGraphicsContext);
				return null;
			}
		});
	}

	/**
	 * Answer the receiver's new boundary vertexes with the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayBoundaryVertex
	 * @category element creation
	 */
	public Jun2dDelaunayBoundaryVertex newBoundaryVertexAtX_y_(double xNumber, double yNumber) {
		return (Jun2dDelaunayBoundaryVertex) vertices.add_(new Jun2dDelaunayBoundaryVertex(xNumber, yNumber));
	}

	/**
	 * Answer the new edge.
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayEdge
	 * @category element creation
	 */
	public Jun2dDelaunayEdge newEdge() {
		return (Jun2dDelaunayEdge) edges.add_(new Jun2dDelaunayEdge());
	}

	/**
	 * Answer the new loop.
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 * @category element creation
	 */
	public Jun2dDelaunayLoop newLoop() {
		return (Jun2dDelaunayLoop) loops.add_(new Jun2dDelaunayLoop());
	}

	/**
	 * Answer the receiver's new vertexes.
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category element creation
	 */
	public Jun2dDelaunayVertex newVertex() {
		return (Jun2dDelaunayVertex) vertices.add_(new Jun2dDelaunayVertex());
	}

	/**
	 * Answer the receiver's new vertexes with the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category element creation
	 */
	public Jun2dDelaunayVertex newVertexAtX_y_(double xNumber, double yNumber) {
		return (Jun2dDelaunayVertex) vertices.add_(new Jun2dDelaunayVertex(xNumber, yNumber));
	}

	/**
	 * Answer the receiver's new vertexes with the specified x, y and z.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @param zNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category element creation
	 */
	public Jun2dDelaunayVertex newVertexAtX_y_z_(double xNumber, double yNumber, double zNumber) {
		return (Jun2dDelaunayVertex) vertices.add_(new Jun2dDelaunayVertex(xNumber, yNumber, zNumber));
	}

	/**
	 * Enumerate convex hull triangles and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void convexHullTrianglesDo_(final StBlockClosure aBlock) {
		loops.do_(new StBlockClosure() {
			public Object value_(Object loop_) {
				Jun2dDelaunayLoop loop = (Jun2dDelaunayLoop) loop_;
				if (loop.hasBoundaryVertex() == false) {
					aBlock.value_value_value_(loop.point1(), loop.point2(), loop.point3());
				}
				return null;
			}
		});
	}

	/**
	 * Enumerate inside triangles and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void insideTrianglesDo_(StBlockClosure aBlock) {
		final Hashtable distances = _IdentityDictionary(); // Loop @ Int
		final StValueHolder activeLoops = new StValueHolder(_IdentitySet()); // Loop

		vertices.do_(new StBlockClosure() {
			public Object value_(Object vertex_) {
				Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;
				if (vertex.isBoundaryVertex()) {
					vertex.loopsDo_(new StBlockClosure() {
						public Object value_(Object loop_) {
							Jun2dDelaunayLoop loop = (Jun2dDelaunayLoop) loop_;
							distances.put(loop, new Integer(0));
							((Vector) activeLoops.value()).addElement(loop);
							return null;
						}
					});
				}
				return null;
			}
		});

		while (!((Vector) activeLoops.value()).isEmpty()) {
			final Vector updatedLoops = _IdentitySet();
			for (int i = 0; i < ((Vector) activeLoops.value()).size(); i++) {
				Jun2dDelaunayLoop loop = (Jun2dDelaunayLoop) ((Vector) activeLoops.value()).elementAt(i);
				final int distance = ((Integer) distances.get(loop)).intValue();
				loop.halfEdgesDo_(new StBlockClosure() {
					public Object value_(Object halfEdge_) {
						Jun2dDelaunayHalfEdge halfEdge = (Jun2dDelaunayHalfEdge) halfEdge_;
						Jun2dDelaunayLoop neighbor = halfEdge.pair().loop();
						if (neighbor != null) {
							int distanceFromMe = halfEdge.edge().isConstrained() ? (distance + 1) : distance;
							Integer currentDistance = (Integer) distances.get(neighbor);
							if ((currentDistance == null) || (distanceFromMe < currentDistance.intValue())) {
								distances.put(neighbor, new Integer(distanceFromMe));
								updatedLoops.addElement(neighbor);
							}
						}
						return null;
					}
				});
			}
			activeLoops.value_(updatedLoops);
		}

		for (Enumeration e = distances.keys(); e.hasMoreElements();) {
			Jun2dDelaunayLoop loop = (Jun2dDelaunayLoop) e.nextElement();
			int distance = ((Number) distances.get(loop)).intValue();
			if ((distance % 2) != 0) {
				aBlock.value_value_value_(loop.point1(), loop.point2(), loop.point3());
			}
		}
	}

	/**
	 * Enumerate triangles and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void trianglesDo_(final StBlockClosure aBlock) {
		loops.do_(new StBlockClosure() {
			public Object value_(final Object loop_) {
				final Jun2dDelaunayLoop loop = (Jun2dDelaunayLoop) loop_;
				aBlock.value_value_value_(loop.point1(), loop.point2(), loop.point3());
				return null;
			}
		});
	}

	/**
	 * Insert the specified edge with two points.
	 * 
	 * @param fromPoint java.awt.Point
	 * @param toPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category functions
	 */
	public void insertEdgeFrom_to_(JunPoint fromPoint, JunPoint toPoint) {
		Jun2dDelaunayVertex vertex1 = this.insertPoint_(fromPoint);
		Jun2dDelaunayVertex vertex2 = this.insertPoint_(toPoint);
		Jun2dDelaunayEdge edge = new Jun2dDelaunayEdge();
		edge.vertex1_(vertex1);
		edge.vertex2_(vertex2);
		this.insertEdge_(edge);
	}

	/**
	 * Insert the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category functions
	 */
	public Jun2dDelaunayVertex insertPoint_(JunPoint aPoint) {
		Jun2dDelaunayVertex vertex = new Jun2dDelaunayVertex(aPoint);
		return this.insertVertex_(vertex);
	}

	/**
	 * Insert the specified points.
	 * 
	 * @param aCollectionOfPoint jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @category functions
	 */
	public void insertPoints_(JunPoint[] aCollectionOfPoint) {
		for (int i = 0; i < aCollectionOfPoint.length; i++) {
			this.insertPoint_(aCollectionOfPoint[i]);
		}
	}

	/**
	 * Insert the specified polygon.
	 * 
	 * @param pointArray jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @category functions
	 */
	public void insertPolygon_(JunPoint[] pointArray) {
		this.insertPolygon_interim_(pointArray, null);
	}

	/**
	 * Insert the specified polygon with interim block.
	 * 
	 * @param pointArray jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @param aBlockOrNil jp.co.sra.smalltalk.StBlockClosure
	 * @category functions
	 */
	public void insertPolygon_interim_(JunPoint[] pointArray, StBlockClosure aBlockOrNil) {
		Jun2dDelaunayVertex vertex2 = this.insertPoint_(pointArray[pointArray.length - 1]);
		for (int i = 0; i < pointArray.length; i++) {
			JunPoint point = pointArray[i];
			Jun2dDelaunayVertex vertex1 = this.insertPoint_(point);
			if ((vertex1 != null) && (vertex2 != null)) {
				Jun2dDelaunayEdge edge = new Jun2dDelaunayEdge();
				edge.vertex1_(vertex1);
				edge.vertex2_(vertex2);
				this.basicInsertEdge_(edge);
				if (aBlockOrNil != null) {
					aBlockOrNil.value_(this);
				}
			}
			vertex2 = vertex1;
		}
	}

	/**
	 * Basic insert the specified edge.
	 * 
	 * @param aJun2dDelaunayEdge jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayEdge
	 * @category topological operations
	 */
	public void basicInsertEdge_(Jun2dDelaunayEdge aJun2dDelaunayEdge) {
		constrained = true;
		this.makeWayFrom_to_(aJun2dDelaunayEdge.vertex1(), aJun2dDelaunayEdge.vertex2());
	}

	/**
	 * Basic insert the specified vertex on the loop.
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param aJun2dDelaunayLoop jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category topological operations
	 */
	public Jun2dDelaunayVertex basicInsertVertex_onLoop_(Jun2dDelaunayVertex aJun2dDelaunayVertex, Jun2dDelaunayLoop aJun2dDelaunayLoop) {
		final Jun2dDelaunayHalfEdge hEdge1 = aJun2dDelaunayLoop.halfEdge();
		Jun2dDelaunayHalfEdge hEdge2 = hEdge1.next();
		Jun2dDelaunayHalfEdge hEdge3 = hEdge2.next();
		Jun2dDelaunayEdge newEdge1 = this.newEdge();
		Jun2dDelaunayEdge newEdge2 = this.newEdge();
		Jun2dDelaunayEdge newEdge3 = this.newEdge();

		final Jun2dDelaunayLoop newLoop1 = this.newLoop();
		final Jun2dDelaunayLoop newLoop2 = this.newLoop();
		newEdge1.vertex1_(hEdge1.vertex());
		newEdge1.vertex2_(aJun2dDelaunayVertex);
		newEdge2.vertex1_(hEdge2.vertex());
		newEdge2.vertex2_(aJun2dDelaunayVertex);
		newEdge3.vertex1_(hEdge3.vertex());
		newEdge3.vertex2_(aJun2dDelaunayVertex);
		aJun2dDelaunayVertex.halfEdge_(newEdge3.halfEdge12());
		hEdge1.next_(newEdge1.halfEdge12());
		newEdge1.halfEdge12().next_(newEdge3.halfEdge21());
		newEdge3.halfEdge21().next_(hEdge1);
		hEdge2.next_(newEdge2.halfEdge12());
		newEdge2.halfEdge12().next_(newEdge1.halfEdge21());
		newEdge1.halfEdge21().next_(hEdge2);
		hEdge3.next_(newEdge3.halfEdge12());
		newEdge3.halfEdge12().next_(newEdge2.halfEdge21());
		newEdge2.halfEdge21().next_(hEdge3);
		hEdge1.do_(new StBlockClosure() {
			public Object value_(final Object hEdge_) {
				final Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				hEdge.loop_(hEdge1.loop());
				return null;
			}
		});
		hEdge2.do_(new StBlockClosure() {
			public Object value_(final Object hEdge_) {
				final Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				hEdge.loop_(newLoop1);
				return null;
			}
		});
		hEdge3.do_(new StBlockClosure() {
			public Object value_(final Object hEdge_) {
				final Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				hEdge.loop_(newLoop2);
				return null;
			}
		});
		hEdge1.loop().halfEdge_(hEdge1);
		hEdge2.loop().halfEdge_(hEdge2);
		hEdge3.loop().halfEdge_(hEdge3);
		vertices.add_(aJun2dDelaunayVertex);

		if (hEdge1.edge().isConstrained() && (hEdge1.areaWith_(aJun2dDelaunayVertex) < DECIMAL_12)) {
			hEdge1.edge().enableArrange();
			hEdge1.next().edge().disableArrange();
			hEdge1.prev().edge().disableArrange();
			hEdge1.edge().basicArrange();
		}

		if (hEdge2.edge().isConstrained() && (hEdge2.areaWith_(aJun2dDelaunayVertex) < DECIMAL_12)) {
			hEdge2.edge().enableArrange();
			hEdge2.next().edge().disableArrange();
			hEdge2.prev().edge().disableArrange();
			hEdge2.edge().basicArrange();
		}

		if (hEdge3.edge().isConstrained() && (hEdge3.areaWith_(aJun2dDelaunayVertex) < DECIMAL_12)) {
			hEdge3.edge().enableArrange();
			hEdge3.next().edge().disableArrange();
			hEdge3.prev().edge().disableArrange();
			hEdge3.edge().basicArrange();
		}

		hEdge1.edge().setDirty();
		hEdge2.edge().setDirty();
		hEdge3.edge().setDirty();

		return aJun2dDelaunayVertex;
	}

	/**
	 * Insert the specified edge.
	 * 
	 * @param aJun2dDelaunayEdge jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayEdge
	 * @category topological operations
	 */
	public void insertEdge_(Jun2dDelaunayEdge aJun2dDelaunayEdge) {
		aJun2dDelaunayEdge.vertex1_(this.insertVertex_(aJun2dDelaunayEdge.vertex1()));
		aJun2dDelaunayEdge.vertex2_(this.insertVertex_(aJun2dDelaunayEdge.vertex2()));
		this.basicInsertEdge_(aJun2dDelaunayEdge);
	}

	/**
	 * Insert the specified vertex.
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category topological operations
	 */
	public Jun2dDelaunayVertex insertVertex_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		double x = aJun2dDelaunayVertex.x();
		double y = aJun2dDelaunayVertex.y();
		Jun2dDelaunayVertex vertex = this.nearestVertexFromX_y_(x, y);
		if (vertex.containsX_y_(x, y)) {
			return vertex;
		}
		Jun2dDelaunayLoop loop = this.loopAtX_y_(x, y);
		if (loop != null) {
			return this.insertVertex_onLoop_(aJun2dDelaunayVertex, loop);
		}
		return null;
	}

	/**
	 * Insert the specified vertex on the specified loop.
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param aJun2dDelaunayLoop jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category topological operations
	 */
	public Jun2dDelaunayVertex insertVertex_onLoop_(Jun2dDelaunayVertex aJun2dDelaunayVertex, Jun2dDelaunayLoop aJun2dDelaunayLoop) {
		final Jun2dDelaunayHalfEdge hEdge1 = aJun2dDelaunayLoop.halfEdge();
		Jun2dDelaunayHalfEdge hEdge2 = hEdge1.next();
		Jun2dDelaunayHalfEdge hEdge3 = hEdge2.next();
		Jun2dDelaunayEdge newEdge1 = this.newEdge();
		Jun2dDelaunayEdge newEdge2 = this.newEdge();
		Jun2dDelaunayEdge newEdge3 = this.newEdge();

		final Jun2dDelaunayLoop newLoop1 = this.newLoop();
		final Jun2dDelaunayLoop newLoop2 = this.newLoop();
		newEdge1.vertex1_(hEdge1.vertex());
		newEdge1.vertex2_(aJun2dDelaunayVertex);
		newEdge2.vertex1_(hEdge2.vertex());
		newEdge2.vertex2_(aJun2dDelaunayVertex);
		newEdge3.vertex1_(hEdge3.vertex());
		newEdge3.vertex2_(aJun2dDelaunayVertex);
		aJun2dDelaunayVertex.halfEdge_(newEdge3.halfEdge12());
		hEdge1.next_(newEdge1.halfEdge12());
		newEdge1.halfEdge12().next_(newEdge3.halfEdge21());
		newEdge3.halfEdge21().next_(hEdge1);
		hEdge2.next_(newEdge2.halfEdge12());
		newEdge2.halfEdge12().next_(newEdge1.halfEdge21());
		newEdge1.halfEdge21().next_(hEdge2);
		hEdge3.next_(newEdge3.halfEdge12());
		newEdge3.halfEdge12().next_(newEdge2.halfEdge21());
		newEdge2.halfEdge21().next_(hEdge3);
		hEdge1.do_(new StBlockClosure() {
			public Object value_(final Object hEdge_) {
				final Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				hEdge.loop_(hEdge1.loop());

				return null;
			}
		});
		hEdge2.do_(new StBlockClosure() {
			public Object value_(final Object hEdge_) {
				final Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				hEdge.loop_(newLoop1);

				return null;
			}
		});
		hEdge3.do_(new StBlockClosure() {
			public Object value_(final Object hEdge_) {
				final Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				hEdge.loop_(newLoop2);

				return null;
			}
		});
		hEdge1.loop().halfEdge_(hEdge1);
		hEdge2.loop().halfEdge_(hEdge2);
		hEdge3.loop().halfEdge_(hEdge3);
		vertices.add_(aJun2dDelaunayVertex);

		if (hEdge1.edge().isConstrained() && (hEdge1.areaWith_(aJun2dDelaunayVertex) < DECIMAL_8)) {
			hEdge1.edge().enableArrange();
			hEdge1.next().edge().disableArrange();
			hEdge1.prev().edge().disableArrange();
		}

		if (hEdge2.edge().isConstrained() && (hEdge2.areaWith_(aJun2dDelaunayVertex) < DECIMAL_8)) {
			hEdge2.edge().enableArrange();
			hEdge2.next().edge().disableArrange();
			hEdge2.prev().edge().disableArrange();
		}

		if (hEdge3.edge().isConstrained() && (hEdge3.areaWith_(aJun2dDelaunayVertex) < DECIMAL_8)) {
			hEdge3.edge().enableArrange();
			hEdge3.next().edge().disableArrange();
			hEdge3.prev().edge().disableArrange();
		}

		hEdge1.edge().setDirty();
		hEdge2.edge().setDirty();
		hEdge3.edge().setDirty();
		hEdge1.edge().recursiveArrange();
		hEdge2.edge().recursiveArrange();
		hEdge3.edge().recursiveArrange();

		return aJun2dDelaunayVertex;
	}

	/**
	 * Make checkpoint with from point and to point.
	 * 
	 * @param baseVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param vertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category topological operations
	 */
	public Jun2dDelaunayVertex makeCheckpointFrom_to_(Jun2dDelaunayVertex baseVertex, Jun2dDelaunayVertex vertex2) {
		if (baseVertex.halfEdgeToward_(vertex2) != null) {
			return vertex2;
		}

		Jun2dDelaunayHalfEdge baseHalfEdge = this.intersectingHalfEdgeFrom_to_(baseVertex, vertex2);
		if (baseHalfEdge == null) {
			return null;
		}

		while (true) {
			baseHalfEdge = baseHalfEdge.pair();
			if (!baseHalfEdge.edge().isArrangeable()) {
				Jun2dDelaunayVertex intersection = baseHalfEdge.edge().intersectionWithLineSegmentFrom_to_(baseVertex, vertex2);
				this.insertVertex_onLoop_(intersection, baseHalfEdge.loop());
				return intersection;
			}

			if (baseHalfEdge.next().vertex().isOnLineSegmentFrom_to_(baseVertex, vertex2)) {
				return vertex2;
			}

			if (baseHalfEdge.next().intersectsWithLineSegmentFrom_to_(baseVertex, vertex2)) {
				baseHalfEdge = baseHalfEdge.next();
			} else {
				if (baseHalfEdge.prev().intersectsWithLineSegmentFrom_to_(baseVertex, vertex2)) {
					baseHalfEdge = baseHalfEdge.prev();
				} else {
					return null;
				}
			}
		}
	}

	/**
	 * Make step with from point and to point.
	 * 
	 * @param vertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param vertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category topological operations
	 */
	public Jun2dDelaunayVertex makeStepFrom_to_(Jun2dDelaunayVertex vertex1, Jun2dDelaunayVertex vertex2) {
		Jun2dDelaunayVertex startVertex = vertex1;
		Jun2dDelaunayVertex goalVertex = this.makeCheckpointFrom_to_(startVertex, vertex2);

		if (goalVertex == null) {
			return null;
		}

		Vector dirties = _IdentitySet();
		Jun2dDelaunayHalfEdge halfEdge;
		for (halfEdge = startVertex.halfEdgeTo_(goalVertex); halfEdge == null; halfEdge = startVertex.halfEdgeTo_(goalVertex)) {
			halfEdge = this.intersectingHalfEdgeFrom_to_(startVertex, goalVertex);
			if (halfEdge == null) {
				for (int i = 0; i < dirties.size(); i++) {
					Jun2dDelaunayEdge dirtyEdge = (Jun2dDelaunayEdge) dirties.elementAt(i);
					dirtyEdge.recursiveArrange();
				}
				return null;
			}

			halfEdge.edge().enableArrange();
			halfEdge.edge().basicArrange();
			halfEdge.edge().setDirty();
			dirties.addElement(halfEdge.edge());
			halfEdge.next().edge().setDirty();
			dirties.addElement(halfEdge.next().edge());
			halfEdge.prev().edge().setDirty();
			dirties.addElement(halfEdge.prev().edge());
			halfEdge.pair().next().edge().setDirty();
			dirties.addElement(halfEdge.pair().next().edge());
			halfEdge.pair().prev().edge().setDirty();
			dirties.addElement(halfEdge.pair().prev().edge());
		}
		halfEdge.edge().disableArrange();

		for (int i = 0; i < dirties.size(); i++) {
			Jun2dDelaunayEdge dirtyEdge = (Jun2dDelaunayEdge) dirties.elementAt(i);
			dirtyEdge.recursiveArrange();
		}

		return goalVertex;
	}

	/**
	 * Make way with from point and to point.
	 * 
	 * @param vertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param vertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category topological operations
	 */
	public void makeWayFrom_to_(Jun2dDelaunayVertex vertex1, Jun2dDelaunayVertex vertex2) {
		Jun2dDelaunayVertex checkpoint = vertex1;
		while (true) {
			Jun2dDelaunayVertex newCheckpoint = this.makeStepFrom_to_(checkpoint, vertex2);
			if ((newCheckpoint == null) || (newCheckpoint == vertex2)) {
				return;
			}
			checkpoint = newCheckpoint;
		}
	}

	/**
	 * Check the loops.
	 * 
	 * @category private
	 */
	protected void checkLoops() {
		loops.do_(new StBlockClosure() {
			public Object value_(final Object loop_) {
				final Jun2dDelaunayLoop loop = (Jun2dDelaunayLoop) loop_;
				if (loop.area() < 0) {
					throw SmalltalkException.Halt("checkLoops");
				}
				return null;
			}
		});
	}

	/**
	 * Answer the edges.
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayList
	 * @category private 
	 */
	protected Jun2dDelaunayList edges() {
		return edges;
	}

	/**
	 * Exhaust search for loop at the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 * @category private
	 */
	protected Jun2dDelaunayLoop exhaustSearchForLoopAtX_y_(final double xNumber, final double yNumber) {
		return (Jun2dDelaunayLoop) loops.do_(new StBlockClosure() {
			public Object value_(final Object loop_) {
				final Jun2dDelaunayLoop loop = (Jun2dDelaunayLoop) loop_;
				if (loop.containsX_y_(xNumber, yNumber)) {
					return loop;
				}
				return null;
			}
		});
	}

	/**
	 * Exhaust search nearest vertex from the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category private
	 */
	protected Jun2dDelaunayVertex exhaustSearchNearestVertexFromX_y_(final double xNumber, final double yNumber) {
		final StValueHolder theVertex = new StValueHolder(); // Jun2dDelaunayVertex
		final StValueHolder min = new StValueHolder(); // double
		theVertex.value_(vertices.first());
		min.value_(((Jun2dDelaunayVertex) theVertex.value()).squaredDistanceFromX_y_(xNumber, yNumber));
		vertices.do_(new StBlockClosure() {
			public Object value_(Object vertex_) {
				final Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;
				double d = vertex.squaredDistanceFromX_y_(xNumber, yNumber);
				if (d < min._doubleValue()) {
					min.value_(d);
					theVertex.value_(vertex);
				}
				return null;
			}
		});
		return (Jun2dDelaunayVertex) theVertex.value();
	}

	/**
	 * Intersecting half edge from the specified vertex to the other vertex.
	 * 
	 * @param vertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param vertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 * @category private
	 */
	protected Jun2dDelaunayHalfEdge intersectingHalfEdgeFrom_to_(final Jun2dDelaunayVertex vertex1, final Jun2dDelaunayVertex vertex2) {
		return (Jun2dDelaunayHalfEdge) vertex1.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				if (hEdge.prev().intersectsWithLineSegmentFrom_to_(vertex1, vertex2)) {
					return hEdge.prev();
				}
				return null;
			}
		});
	}

	/**
	 * Answer the loop with the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 * @category private
	 */
	protected Jun2dDelaunayLoop loopAtX_y_(double xNumber, double yNumber) {
		return this.topologicalSearchForLoopAtX_y_(xNumber, yNumber);
	}

	/**
	 * Answer the margin.
	 * 
	 * @return double
	 * @category private
	 */
	protected double margin() {
		return 1.0d;
	}

	/**
	 * Answer the nearest edge from the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayEdge
	 * @category private
	 */
	protected Jun2dDelaunayEdge nearestEdgeFromX_y_(final double xNumber, final double yNumber) {
		final StValueHolder theEdge = new StValueHolder(); // Jun2dDelaunayEdge
		final StValueHolder min = new StValueHolder(); // double
		theEdge.value_(edges.first());
		min.value_(new Double(((Jun2dDelaunayEdge) edges.first()).squaredDistanceFromX_y_(xNumber, yNumber)));
		edges.do_(new StBlockClosure() {
			public Object value_(Object edge_) {
				Jun2dDelaunayEdge edge = (Jun2dDelaunayEdge) edge_;
				double d = edge.squaredDistanceFromX_y_(xNumber, yNumber);
				if (d < min._doubleValue()) {
					min.value_(d);
					theEdge.value_(edge);
				}
				return null;
			}
		});
		return (Jun2dDelaunayEdge) theEdge.value();
	}

	/**
	 * Answer the nearest vertex from the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category private
	 */
	protected Jun2dDelaunayVertex nearestVertexFromX_y_(double xNumber, double yNumber) {
		if (constrained) {
			return this.exhaustSearchNearestVertexFromX_y_(xNumber, yNumber);
		} else {
			return this.topologicalSearchNearestVertexFromX_y_(xNumber, yNumber);
		}
	}

	/**
	 * Set the receiver's bounds.
	 * 
	 * @param aJun2dBoundingBox jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category private
	 */
	protected void setBounds_(Jun2dBoundingBox aJun2dBoundingBox) {
		bounds = aJun2dBoundingBox;
	}

	/**
	 * Topological search for loop at the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 * @category private
	 */
	protected Jun2dDelaunayLoop topologicalSearchForLoopAtX_y_(final double xNumber, final double yNumber) {
		StBlockClosure areaBlock = new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				Jun2dDelaunayVertex v1 = hEdge.pair().vertex();
				Jun2dDelaunayVertex v2 = hEdge.vertex();
				return new Double(((v1.x() - xNumber) * (v2.y() - yNumber)) - ((v1.y() - yNumber) * (v2.x() - xNumber)));
			}
		};

		Jun2dDelaunayLoop loop = (Jun2dDelaunayLoop) loops.first();

		while (true) {
			Jun2dDelaunayHalfEdge halfEdge1 = loop.halfEdge();
			Jun2dDelaunayHalfEdge halfEdge2 = halfEdge1.next();
			Jun2dDelaunayHalfEdge halfEdge3 = halfEdge2.next();
			double area1 = ((Double) areaBlock.value_(halfEdge1)).doubleValue();
			double area2 = ((Double) areaBlock.value_(halfEdge2)).doubleValue();
			double area3 = ((Double) areaBlock.value_(halfEdge3)).doubleValue();
			if (loop.area() < 0) {
				return null;
			}
			if (!((area1 < 0) || (area2 < 0) || area3 < 0)) {
				return loop;
			}
			loop = halfEdge1.pair().loop();
			double minArea = area1;
			if (area2 < minArea) {
				minArea = area2;
				loop = halfEdge2.pair().loop();
			}
			if (area3 < minArea) {
				loop = halfEdge3.pair().loop();
			}
			if (loop == null) {
				return null;
			}
		}
	}

	/**
	 * Topological search nearest vertex from the specified x and y.
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @category private
	 */
	protected Jun2dDelaunayVertex topologicalSearchNearestVertexFromX_y_(final double xNumber, final double yNumber) {
		Jun2dDelaunayVertex theVertex = (Jun2dDelaunayVertex) vertices.first();
		while (true) {
			final StValueHolder min = new StValueHolder(); // double
			final StValueHolder nextVertex = new StValueHolder(); // Jun2dDelaunayVertex
			min.value_(theVertex.squaredDistanceFromX_y_(xNumber, yNumber));
			nextVertex.value_(null);
			theVertex.neighborsDo_(new StBlockClosure() {
				public Object value_(Object vertex_) {
					Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;
					double d = vertex.squaredDistanceFromX_y_(xNumber, yNumber);
					if (d < min._doubleValue()) {
						min.value_(d);
						nextVertex.value_(vertex);
					}
					return null;
				}
			});

			if (nextVertex.value() == null) {
				return theVertex;
			}
			theVertex = (Jun2dDelaunayVertex) nextVertex.value();
		}
	}
}
