package jp.co.sra.jun.collections.sequences;

import java.awt.Point;
import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;

/**
 * JunMatrix class
 * 
 *  @author    nisinaka
 *  @created   1998/10/09 (by nisinaka)
 *  @updated   2002/07/18 (by nisinaka)
 *  @updated   2007/05/09 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun660 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunMatrix.java,v 8.14 2008/02/20 06:30:53 nisinaka Exp $
 */
public class JunMatrix extends JunSequence {

	/** The size of the matrix's row. */
	protected int rowSize;

	/** The size of the matrix's column. */
	protected int columnSize;

	/** The array which holds the matrix's elements. */
	protected Object[][] _matrix;

	/**
	 * Create a new instance of <code>JunMatrix</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunMatrix() {
		this(0);
	}

	/**
	 * Create a new instance of <code>JunMatrix</code> and initialize it.
	 * 
	 * @param squareSize int
	 * @category Instance creation
	 */
	public JunMatrix(int squareSize) {
		this(squareSize, squareSize);
	}

	/**
	 * Create a new instance of <code>JunMatrix</code> and initialize it.
	 * 
	 * @param squareSize int
	 * @param collection java.lang.Object[]
	 * @category Instance creation
	 */
	public JunMatrix(int squareSize, Object[] collection) {
		this(squareSize);
		this.collection_(collection);
	}

	/**
	 * Create a new instance of <code>JunMatrix</code> and initialize it.
	 * 
	 * @param aRowSize int
	 * @param aColumnSize int
	 * @category Instance creation
	 */
	public JunMatrix(int aRowSize, int aColumnSize) {
		rowSize = aRowSize;
		columnSize = aColumnSize;
		_matrix = new Object[rowSize][columnSize];
	}

	/**
	 * Create a new instance of <code>JunMatrix</code> and initialize it.
	 * 
	 * @param rowSize int
	 * @param columnSize int
	 * @param collection java.lang.Object[]
	 * @category Instance creation
	 */
	public JunMatrix(int rowSize, int columnSize, Object[] collection) {
		this(rowSize, columnSize);
		this.collection_(collection);
	}

	/**
	 * Answer the object at the position specified with rowNumber and columnNumber.
	 * 
	 * @param rowNumber int
	 * @param columnNumber int
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object _at(int rowNumber, int columnNumber) {
		return _matrix[rowNumber][columnNumber];
	}

	/**
	 * Put 'anObject' at the position specified with 'rowNumber' and 'columnNumber'.
	 * 
	 * @param rowNumber int
	 * @param columnNumber int
	 * @param anObject java.lang.Object
	 * @category accessing
	 */
	public void _put(int rowNumber, int columnNumber, Object anObject) {
		_matrix[rowNumber][columnNumber] = anObject;
	}

	/**
	 * Answer the object at the position specified with index number.
	 * 
	 * @param indexInteger int
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object at_(int indexInteger) {
		return this.atPoint_(this.computeLocation_(indexInteger));
	}

	/**
	 * Put 'anObject' at the position specified with 'aPoint'.
	 * 
	 * @param indexInteger int
	 * @param anObject java.lang.Object
	 * @category accessing
	 */
	public void at_put_(int indexInteger, Object anObject) {
		this.atPoint_put_(this.computeLocation_(indexInteger), anObject);
	}

	/**
	 * Put 'anObject' at the all position.
	 * 
	 * @param anObject java.lang.Object
	 * @category accessing
	 */
	public void atAllPut_(Object anObject) {
		for (int rowIndex = 0; rowIndex < this.rowSize(); rowIndex++) {
			for (int columnIndex = 0; columnIndex < this.columnSize(); columnIndex++) {
				this._put(rowIndex, columnIndex, anObject);
			}
		}
	}

	/**
	 * Anser the object at the position specified with point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object atPoint_(Point aPoint) {
		return this._at(aPoint.x, aPoint.y);
	}

	/**
	 * Put 'anObject' at the position specified with 'aPoint'.
	 * 
	 * @param aPoint java.awt.Point
	 * @param anObject java.lang.Object
	 * @category accessing
	 */
	public void atPoint_put_(Point aPoint, Object anObject) {
		this._put(aPoint.x, aPoint.y, anObject);
	}

	/**
	 * Answer the array of objects in the specified row.
	 * 
	 * @param index int
	 * @return java.lang.Object[]
	 * @category accessing
	 */
	public Object[] basicAt_(int index) {
		return this.atRow_(index);
	}

	/**
	 * Set the array of objects in the specified row.
	 * 
	 * @param anInteger int
	 * @param aCollection java.lang.Object[]
	 * @category accessing
	 */
	public void basicAt_put_(int index, Object[] aCollection) {
		this.atRow_put_(index, aCollection);
	}

	/**
	 * Set all elements of the matrix from 'collection'.
	 * 
	 * @param collection java.lang.Object[]
	 * @category accessing
	 */
	public void collection_(Object[] collection) {
		int index = 0;
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				_matrix[i][j] = collection[index];
				index++;
			}
		}
	}

	/**
	 * Answer a point which represents this matrix size.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point matrixSize() {
		return new Point(rowSize, columnSize);
	}

	/**
	 * Answer the object at the position specified with rowNumber and columnNumber.
	 * 
	 * @param rowNumber int
	 * @param columnNumber int
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object row_column_(int rowNumber, int columnNumber) {
		return this._at(rowNumber, columnNumber);
	}

	/**
	 * Put 'anObject' at the position specified with 'rowNumber' and 'columnNumber'.
	 * 
	 * @param rowNumber int
	 * @param columnNumber int
	 * @param anObject java.lang.Object
	 * @category accessing
	 */
	public void row_column_put_(int rowNumber, int columnNumber, Object anObject) {
		this._put(rowNumber, columnNumber, anObject);
	}

	/**
	 * Answer the number of all elements.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return this.columnSize() * this.rowSize();
	}

	/**
	 * Answer the array of objects in the specified column.
	 * 
	 * @param anInteger int
	 * @return java.lang.Object[]
	 * @category accessing column
	 */
	public Object[] atColumn_(int anInteger) {
		Object[] column = new Object[rowSize];
		for (int i = 0; i < rowSize; i++) {
			column[i] = this._at(i, anInteger);
		}
		return column;
	}

	/**
	 * Set the array of objects in the specified column.
	 * 
	 * @param anInteger int
	 * @param aCollection java.lang.Object[]
	 * @category accessing column
	 */
	public void atColumn_put_(int anInteger, Object[] aCollection) {
		for (int rowIndex = 0; rowIndex < this.rowSize(); rowIndex++) {
			this._put(rowIndex, anInteger, aCollection[rowIndex]);
		}
	}

	/**
	 * Answer the size of column.
	 * 
	 * @return int
	 * @category accessing column
	 */
	public int columnSize() {
		return columnSize;
	}

	/**
	 * Answer the array of objects in the specified row.
	 * 
	 * @param anInteger int
	 * @return java.lang.Object[]
	 * @category accessing row
	 */
	public Object[] atRow_(int anInteger) {
		Object[] row = new Object[columnSize];
		for (int j = 0; j < columnSize; j++) {
			row[j] = this._at(anInteger, j);
		}
		return row;
	}

	/**
	 * Set the array of objects in the specified row.
	 * 
	 * @param anInteger int
	 * @param aCollection java.lang.Object[]
	 * @category accessing row
	 */
	public void atRow_put_(int anInteger, Object[] aCollection) {
		for (int columnIndex = 0; columnIndex < this.columnSize(); columnIndex++) {
			this._put(anInteger, columnIndex, aCollection[columnIndex]);
		}
	}

	/**
	 * Answer the size of row.
	 * 
	 * @return int
	 * @category accessing row
	 */
	public int rowSize() {
		return rowSize;
	}

	/**
	 * Convert this matrix as an array.
	 * 
	 * @return java.lang.Object[]
	 * @category converting
	 */
	public Object[] asArray() {
		Object[] anArray = new Object[rowSize * columnSize];
		int index = 0;

		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				anArray[index] = this._at(i, j);
				index++;
			}
		}

		return anArray;
	}

	/**
	 * Create an entire copy of this matrix.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#copy()
	 * @category copying
	 */
	public StObject copy() {
		JunMatrix newMatrix = (JunMatrix) _New(this.species(), new Integer(rowSize), new Integer(columnSize));
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				newMatrix._put(i, j, this._at(i, j));
			}
		}
		return newMatrix;
	}

	/**
	 * Enumerate all the elements and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.collections.sequences.JunSequence#do_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public Object do_(final StBlockClosure aBlock) {
		return this.doIJ_(new StBlockClosure() {
			public Object value_value_value_(Object v, Object i, Object j) {
				return aBlock.value_(v);
			}
		});
	}

	/**
	 * Enumerate all the elements and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object doIJ_(StBlockClosure aBlock) {
		int rowSize = this.rowSize();
		int columnSize = this.columnSize();
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				Object result = aBlock.value_value_value_(_matrix[i][j], new Integer(i), new Integer(j));

				if (result != null) {
					return result;
				}
			}
		}
		return null;
	}

	/**
	 * Enumerate all the values at the specified row and evaluate the block.
	 * 
	 * @param anInteger int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object rowIndex_doJ_(int anInteger, StBlockClosure aBlock) {
		for (int j = 0; j < columnSize; j++) {
			Object result = aBlock.value_value_(_matrix[anInteger][j], new Integer(j));
			if (result != null) {
				return result;
			}
		}

		return null;
	}

	/**
	 * Answer the receiver's horizontal flip matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category matrix functions
	 */
	public JunMatrix horizontalFlip() {
		int rowSize = this.rowSize();
		int columnSize = this.columnSize();
		JunMatrix newMatrix = (JunMatrix) _New(this.getClass(), new Object[] { new Integer(rowSize), new Integer(columnSize) });
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				newMatrix.row_column_put_(i, columnSize - j - 1, this._at(i, j));
			}
		}
		return newMatrix;
	}

	/**
	 * Answer the receiver's mirror matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category matrix functions
	 */
	public JunMatrix mirror() {
		return this.horizontalFlip();
	}

	/**
	 * Answer the receiver's transpose matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category matrix functions
	 */
	public JunMatrix transpose() {
		int rowSize = this.rowSize();
		int columnSize = this.columnSize();
		JunMatrix newMatrix = (JunMatrix) _New(this.getClass(), new Object[] { new Integer(columnSize), new Integer(rowSize) });
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				newMatrix.row_column_put_(j, i, this._at(i, j));
			}
		}
		return newMatrix;
	}

	/**
	 * Answer the receiver's upside down matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category matrix functions
	 */
	public JunMatrix upsideDown() {
		return this.horizontalFlip();
	}

	/**
	 * Answer the receiver's vertical flip matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category matrix functions
	 */
	public JunMatrix verticalFlip() {
		int rowSize = this.rowSize();
		int columnSize = this.columnSize();
		JunMatrix newMatrix = (JunMatrix) _New(this.getClass(), new Object[] { new Integer(rowSize), new Integer(columnSize) });
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				newMatrix.row_column_put_(rowSize - i - 1, j, this._at(i, j));
			}
		}
		return newMatrix;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("Matrix (");
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				Object value = _matrix[i][j];
				aWriter.write((value != null) ? value.toString() : "null");
				if (j < (columnSize - 1)) {
					aWriter.write(" ");
				}
			}
			if (i < (rowSize - 1)) {
				aWriter.write(" , ");
			}
		}
		aWriter.write(")");
	}

	/**
	 * Answer true if this matrix is square, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isSquare() {
		return this.rowSize() == this.columnSize();
	}

	/**
	 * Calculate the product of this matrix and aMatrix.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @return jp.co.sra.jun.collections.sequences.JunMatrix
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category matrix functions
	 */
	public JunMatrix product_(JunMatrix aMatrix) {
		if (columnSize != aMatrix.rowSize) {
			throw SmalltalkException.Error("my column size is not equal to your row size.");
		}

		JunMatrix newMatrix = (JunMatrix) _New(this.species(), new Integer(rowSize), new Integer(aMatrix.columnSize));
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < aMatrix.columnSize; j++) {
				newMatrix._put(i, j, _calculateProductElementAt(i, j, aMatrix));
			}
		}
		return newMatrix;
	}

	/**
	 * Calculate a product element at i & j with aMatrix.
	 * 
	 * @param i int
	 * @param j int
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @return java.lang.Object
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category private
	 */
	protected Object _calculateProductElementAt(int i, int j, JunMatrix aMatrix) {
		throw new SmalltalkException("don't know how to calculate a product element.");
	}

	/**
	 * Compute location with the specified index.
	 * 
	 * @param anInteger int
	 * @return java.awt.Point
	 * @category private
	 */
	protected Point computeLocation_(int anInteger) {
		return new Point((anInteger) / this.columnSize(), (anInteger) % this.columnSize());
	}

}
