/**
 * StPL-windows.c:
 *
 * 	@author:    Hoshi Takanori
 * 	@version:   2.0
 * 	@created:   2001/12/05 (by Hoshi Takanori)
 * 	@updated:   N/A
 *
 * 	$Id: StPL-windows.c,v 2.0 2005/02/17 05:40:17 hoshi Exp $
 */

#include <Windows.h>

#include "jp_co_sra_smalltalk_SystemInterface.h"
#include "version.h"

#define IMAGE_FROM_USER 1
#define IMAGE_OF_AREA 2
#define IMAGE_OF_WHOLE_SCREEN 3
#define RECTANGLE_FROM_USER 4
#define COLOR_SPUIT 5

#define RGB_PIXEL(red, grren, blue) \
		(0xff000000 | ((red) << 16) | ((green) << 8) | (blue))

// ==================================================================

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    getVersion
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_jp_co_sra_smalltalk_SystemInterface_getVersion
  (JNIEnv *env, jclass cls)
{
	return (*env)->NewStringUTF(env, STPL_JNI_VERSION);
}

// ==================================================================

static jint *capture_screen
  (const RECT *rect)
{
	int width, height, depth, padding, row_bytes, x, y;
	HDC bmp_dc, hdc;
	BITMAPINFO bmpinfo;
	unsigned char *bits, *row, red, green, blue;
	HBITMAP hbmp;
	jint *pixels, *p;

	width = rect->right - rect->left;
	height = rect->bottom - rect->top;

	depth = 24;
	padding = ((depth + 15) / 16) * 16;
	row_bytes = (width * depth + padding - 1) / padding * (padding / 8);

	bmp_dc = CreateCompatibleDC(NULL);
	if (bmp_dc == NULL) {
		return NULL;
	}

	bmpinfo.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	bmpinfo.bmiHeader.biWidth = width;
	bmpinfo.bmiHeader.biHeight = height;
	bmpinfo.bmiHeader.biPlanes = 1;
	bmpinfo.bmiHeader.biBitCount = depth;
	bmpinfo.bmiHeader.biCompression = BI_RGB;
	bmpinfo.bmiHeader.biSizeImage = 0;
	bmpinfo.bmiHeader.biXPelsPerMeter = 0;
	bmpinfo.bmiHeader.biYPelsPerMeter = 0;
	bmpinfo.bmiHeader.biClrUsed = 0;
	bmpinfo.bmiHeader.biClrImportant = 0;

	hbmp = CreateDIBSection(bmp_dc, &bmpinfo, DIB_RGB_COLORS, &bits, 0, 0);
	if (hbmp == NULL) {
		DeleteDC(bmp_dc);
		return NULL;
	}

	SelectObject(bmp_dc, hbmp);
	hdc = GetDC(NULL);
	BitBlt(bmp_dc, 0, 0, width, height, hdc, rect->left, rect->top, SRCCOPY);
	ReleaseDC(NULL, hdc);

	GdiFlush();		// Is this necessary?

	pixels = (jint *) malloc(sizeof(jint) * width * height);
	if (pixels == NULL) {
		DeleteDC(bmp_dc);
		DeleteObject(hbmp);
		return NULL;
	}

	for (p = pixels, y = height - 1; y >= 0; --y) {
		row = bits + y * row_bytes;
		for (x = 0; x < width; ++x) {
			blue  = *row++;
			green = *row++;
			red   = *row++;
			*p++ = RGB_PIXEL(red, grren, blue);
		}
	}

	DeleteDC(bmp_dc);
	DeleteObject(hbmp);

	return pixels;
}

// ==================================================================

static const char wndclass_name[] = "StPL_CaptureWindowClass";
static WNDCLASS wndclass;

static BOOLEAN single_pixel;
static BOOLEAN done;
static BOOLEAN pressed;
static int startx, starty;
static RECT current_rect;

static void InvertSelection(HWND hwnd, const RECT *rect)
{
	HDC hdc;
	RECT r;

	hdc = GetDC(hwnd);

	if (rect->right - rect->left <= 2
	 || rect->bottom - rect->top <= 2) {
		InvertRect(hdc, rect);
	} else {
		r = *rect;
		r.bottom = r.top + 1;
		InvertRect(hdc, &r);

		r = *rect;
		r.top = r.bottom - 1;
		InvertRect(hdc, &r);

		r = *rect;
		++r.top;
		--r.bottom;
		r.right = r.left + 1;
		InvertRect(hdc, &r);

		r = *rect;
		++r.top;
		--r.bottom;
		r.left = r.right - 1;
		InvertRect(hdc, &r);
	}

	ReleaseDC(hwnd, hdc);
}

static LRESULT CALLBACK CaptureWindowProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	int x, y;

	switch (msg) {
	case WM_ACTIVATE:
		return MA_NOACTIVATE;		// Is this necessary?

	case WM_LBUTTONDOWN:
		if (pressed == FALSE) {
			SetCapture(hwnd);
			startx = LOWORD(lParam);
			starty = HIWORD(lParam);
			SetRect(&current_rect, startx, starty, startx + 1, starty + 1);
			InvertSelection(hwnd, &current_rect);
			pressed = TRUE;
		}
		return 0;

	case WM_LBUTTONUP:
		if (pressed) {
			InvertSelection(hwnd, &current_rect);
			ReleaseCapture();
			done = TRUE;
		}
		return 0;

	case WM_MOUSEMOVE:
		if (pressed) {
			InvertSelection(hwnd, &current_rect);
			x = LOWORD(lParam);
			y = HIWORD(lParam);
			if (single_pixel) {
				SetRect(&current_rect, x, y, x + 1, y + 1);
			} else {
				if (x >= startx) {
					current_rect.left = startx;
					current_rect.right = x + 1;
				} else {
					current_rect.left = x;
					current_rect.right = startx + 1;
				}
				if (y >= starty) {
					current_rect.top = starty;
					current_rect.bottom = y + 1;
				} else {
					current_rect.top = y;
					current_rect.bottom = starty + 1;
				}
			}
			InvertSelection(hwnd, &current_rect);
		}
		return 0;

	case WM_RBUTTONUP:
	case WM_CLOSE:
	case WM_DESTROY:
		if (pressed) {
			InvertSelection(hwnd, &current_rect);
			ReleaseCapture();
			pressed = FALSE;
		}
		done = TRUE;
		return 0;

	default:
		return DefWindowProc(hwnd, msg, wParam, lParam);
	}
}

static HWND CreateCaptureWindow(HINSTANCE hInstance)
{
	HDC hdc;
	int width, height;
	HWND hwnd;

	wndclass.style = CS_SAVEBITS;
	wndclass.lpfnWndProc = CaptureWindowProc;
	wndclass.cbClsExtra = 0;
	wndclass.cbWndExtra = 0;
	wndclass.hInstance = hInstance;
	wndclass.hIcon = NULL;
	wndclass.hCursor = LoadCursor(NULL, IDC_CROSS);		// Do I need to release?
	wndclass.hbrBackground = NULL;
	wndclass.lpszMenuName = NULL;
	wndclass.lpszClassName = wndclass_name;

	if (! RegisterClass(&wndclass)) {
		return NULL;
	}

	hdc = GetDC(NULL);
	width = GetDeviceCaps(hdc, HORZRES);
	height = GetDeviceCaps(hdc, VERTRES);
	ReleaseDC(NULL, hdc);

	hwnd = CreateWindowEx(WS_EX_TOPMOST,
		wndclass_name, "Capture Window", WS_POPUP,
		0, 0, width, height, NULL, NULL, hInstance, NULL);
	if (hwnd == NULL) {
		UnregisterClass(wndclass_name, hInstance);
		return NULL;
	}

	return hwnd;
}

static int select_rectangle
  (int operation, HWND hwnd, RECT *rect)
{
	MSG msg;

	single_pixel = (operation == COLOR_SPUIT) ? TRUE : FALSE;

	ShowWindow(hwnd, SW_SHOWNA);
	UpdateWindow(hwnd);

	for (done = FALSE, pressed = FALSE; done == FALSE; ) {
		if (GetMessage(&msg, NULL, 0, 0)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		} else {
			PostQuitMessage(0);		// Post it again to terminate Java VM...
		}
	}

	*rect = current_rect;

	return pressed ? 0 : -1;
}

// ==================================================================

static jintArray do_operation
  (JNIEnv *env, int operation, RECT *rect, jintArray dimension)
{
	HINSTANCE hInstance;
	HWND hwnd = NULL;
	HDC hdc;
	jsize size;
	jint *ptr = NULL, dimension_buf[2];
	jintArray array = NULL;

	if (operation == IMAGE_FROM_USER
	 || operation == RECTANGLE_FROM_USER
	 || operation == COLOR_SPUIT) {
		hInstance = GetModuleHandle(NULL);
		hwnd = CreateCaptureWindow(hInstance);
		if (hwnd == NULL) {
			return NULL;
		}
		if (select_rectangle(operation, hwnd, rect) != 0) {
			goto cleanup;
		}
	} else if (operation == IMAGE_OF_WHOLE_SCREEN) {
		hdc = GetDC(NULL);
		rect->left = 0;
		rect->top = 0;
		rect->right = GetDeviceCaps(hdc, HORZRES);
		rect->bottom = GetDeviceCaps(hdc, VERTRES);
		ReleaseDC(NULL, hdc);
	}

	if (operation == IMAGE_FROM_USER
	 || operation == IMAGE_OF_AREA
	 || operation == IMAGE_OF_WHOLE_SCREEN
	 || operation == COLOR_SPUIT) {
		size = (rect->right - rect->left) * (rect->bottom - rect->top);
		ptr = capture_screen(rect);
	} else if (operation == RECTANGLE_FROM_USER) {
		size = 4;
		ptr = (jint *) malloc(sizeof(jint) * size);
		if (ptr != NULL) {
			ptr[0] = rect->left;
			ptr[1] = rect->top;
			ptr[2] = rect->right - rect->left;
			ptr[3] = rect->bottom - rect->top;
		}
	}

cleanup:
	if (hwnd != NULL) {
		DestroyWindow(hwnd);
		UnregisterClass(wndclass_name, hInstance);
	}

	if (ptr != NULL) {
		array = (*env)->NewIntArray(env, size);
		if (array != NULL) {
			(*env)->SetIntArrayRegion(env, array, 0, size, ptr);
		}
		free(ptr);

		if (dimension != NULL) {
			dimension_buf[0] = rect->right - rect->left;
			dimension_buf[1] = rect->bottom - rect->top;
			(*env)->SetIntArrayRegion(env, dimension, 0, 2, dimension_buf);
		}
	}

	return array;
}

// ==================================================================

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageFromUser
 * Signature: ([I)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageFromUser
  (JNIEnv *env, jobject obj, jintArray dimension)
{
	RECT rect;

	return do_operation(env, IMAGE_FROM_USER, &rect, dimension);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageOfArea
 * Signature: (IIII)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageOfArea
  (JNIEnv *env, jobject obj, jint x, jint y, jint width, jint height)
{
	RECT rect;

	SetRect(&rect, x, y, x + width, y + height);
	return do_operation(env, IMAGE_OF_AREA, &rect, NULL);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageOfWholeScreen
 * Signature: ([I)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageOfWholeScreen
  (JNIEnv *env, jobject obj, jintArray dimension)
{
	RECT rect;

	return do_operation(env, IMAGE_OF_WHOLE_SCREEN, &rect, dimension);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilRectangleFromUser
 * Signature: ()[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilRectangleFromUser
  (JNIEnv *env, jobject obj)
{
	RECT rect;

	return do_operation(env, RECTANGLE_FROM_USER, &rect, NULL);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilColorSpuit
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilColorSpuit
  (JNIEnv *env, jobject obj)
{
	RECT rect;
	jintArray array;
	jint pixel = 0;

	array = do_operation(env, COLOR_SPUIT, &rect, NULL);
	if (array != NULL) {
		(*env)->GetIntArrayRegion(env, array, 0, 1, &pixel);
		// I believe that array will be GC'ed...
	}
	return pixel;
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilMousePoint
 * Signature: ()[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilMousePoint
  (JNIEnv *env, jobject obj)
{
	POINT pt;
	jintArray array;
	jint buf[2];

	GetCursorPos(&pt);

	array = (*env)->NewIntArray(env, 2);
	if (array != NULL) {
		buf[0] = pt.x;
		buf[1] = pt.y;
		(*env)->SetIntArrayRegion(env, array, 0, 2, buf);
	}

	return array;
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilSetMousePoint
 * Signature: (II)V
 */
JNIEXPORT void JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilSetMousePoint
  (JNIEnv *env, jobject obj, jint x, jint y)
{
	SetCursorPos(x, y);
} 

// ==================================================================

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilBrowse
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilBrowse
  (JNIEnv *env, jobject obj, jstring url)
{
	const jchar *jstr = (*env)->GetStringChars(env, url, NULL);
	jsize jlen = (*env)->GetStringLength(env, url);
	int len = WideCharToMultiByte(CP_ACP, 0, jstr, jlen, NULL, 0, NULL, NULL);
	char *buf = (char *) malloc(len + 1);

	if (buf != NULL) {
		len = WideCharToMultiByte(CP_ACP, 0, jstr, jlen, buf, len, NULL, NULL);
		buf[len] = '\0';
		ShellExecuteA(NULL, "open", buf, NULL, NULL, SW_SHOWNORMAL);
		free(buf);
	}
	(*env)->ReleaseStringChars(env, url, jstr);
}
