#include <stdlib.h>
#include <GL/glx.h>

#include <dlfcn.h>
#include <jawt.h>
#include <jawt_md.h>

#include "jp_co_sra_gl4jun_GLjInterface.h"
#include "rendering_mode.h"
#include "version.h"

/**
 * glj-x11.c:
 *
 * 	@author:	MATSUDA Ryouichi
 * 	@version:	2.0
 * 	@created:	1999/07/22 (by Hoshi Takanori)
 * 	@updated:	1999/08/12 (by MATSUDA Ryouichi)
 * 	@updated:	2005/03/04 (by Hoshi Takanori)
 *
 * 	$Id: glj-x11.c,v 2.1 2005/03/04 11:43:25 hoshi Exp $
 */

struct JUN_HANDLE {
	int mode;
	Display *display;
	XVisualInfo *visual;
	Window drawable;
	GLXContext context;
	Pixmap pixmap;
	int width;
	int height;
	Window copyWindow;
	int x;
	int y;
	Bool resize;
};

static int singleOffscreenBuffer[] = {
	GLX_RGBA,
	GLX_DEPTH_SIZE,	1,
	GLX_STENCIL_SIZE,	1,
	GLX_RED_SIZE,	1,
	GLX_BLUE_SIZE,	1,
	GLX_GREEN_SIZE,	1,
	None
};

static int doubleOffscreenBuffer[] = {
	GLX_DOUBLEBUFFER,
	GLX_RGBA,
	GLX_DEPTH_SIZE,	1,
	GLX_STENCIL_SIZE,	1,
	GLX_RED_SIZE,	1,
	GLX_BLUE_SIZE,	1,
	GLX_GREEN_SIZE,	1,
	None
};

// ==================================================================
/*
 *
 */
JNIEXPORT jstring JNICALL Java_jp_co_sra_gl4jun_GLjInterface_getVersion
  (JNIEnv *env, jclass cls)
{
	return (*env)->NewStringUTF(env, JUNGL_JNI_VERSION);
}

// ==================================================================
/**
 *
 */
static jboolean rendering2image
  (jint *image, int width, int height, GLenum format)
{
	GLint swapbytes, lsbfirst, rowlength;
	GLint skiprows, skippixels, alignment;
	unsigned char *buffer, *pBuffer;
	unsigned char byte1, byte2, byte3, byte4;
	jint color;
	int x, y;

	// create buffer
	switch (format) {
	case GL_RGB:
		buffer = (unsigned char*)malloc(width * height * 3);
		break;
	case GL_RGBA:
		buffer = (unsigned char*)malloc(width * height * 4);
		break;
	default:
		buffer = NULL;
	}
	if (buffer == NULL) {
		return JNI_FALSE;
	}

	// copy buffer
	glGetIntegerv(GL_UNPACK_SWAP_BYTES, &swapbytes);
	glGetIntegerv(GL_UNPACK_LSB_FIRST, &lsbfirst);
	glGetIntegerv(GL_UNPACK_ROW_LENGTH, &rowlength);
	glGetIntegerv(GL_UNPACK_SKIP_ROWS, &skiprows);
	glGetIntegerv(GL_UNPACK_SKIP_PIXELS, &skippixels);
	glGetIntegerv(GL_UNPACK_ALIGNMENT, &alignment);
	glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_FALSE);
	glPixelStorei(GL_UNPACK_LSB_FIRST, GL_FALSE);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glReadPixels(0, 0, width, height, format, GL_UNSIGNED_BYTE, buffer);
	glPixelStorei(GL_UNPACK_SWAP_BYTES, swapbytes);
	glPixelStorei(GL_UNPACK_LSB_FIRST, lsbfirst);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, rowlength);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, skiprows);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, skippixels);
	glPixelStorei(GL_UNPACK_ALIGNMENT, alignment);

	// convert buffer
	switch (format) {
	case GL_RGB:
		for (y = height - 1; 0 <= y ; y--) {
			pBuffer = buffer + (width * 3 * y);
			for (x = 0; x < width; x++) {
				byte1 = *pBuffer++;
				byte2 = *pBuffer++;
				byte3 = *pBuffer++;
				color = byte3 | byte2 << 8 | byte1 << 16 | 0xFF << 24;
				*image++ = color;
			}
		}
		break;
	case GL_RGBA:
		for (y = height - 1; 0 <= y ; y--) {
			pBuffer = buffer + (width * 4 * y);
			for (x = 0; x < width; x++) {
				byte1 = *pBuffer++;
				byte2 = *pBuffer++;
				byte3 = *pBuffer++;
				byte4 = *pBuffer++;
				color = byte3 | byte2 << 8 | byte1 << 16 | byte4 << 24;
				*image++ = color;
			}
		}
		break;
	default:
		break;
	}

	// release buffer
	free(buffer);

	return JNI_TRUE;
}

JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljReadImage
  (JNIEnv *env, jobject obj, jint jHandle, jintArray jImage, jboolean isRGBA)
{
	struct JUN_HANDLE *handle;
	jint *image;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;
	image = (*env)->GetIntArrayElements(env, jImage, 0);

	// copy buffer
	glFlush();
	if (isRGBA == JNI_TRUE) {
		rendering2image(image, handle->width, handle->height, GL_RGBA);
	} else {
		// hanged up when set RGB.
		rendering2image(image, handle->width, handle->height, GL_RGBA);
		//rendering2image(image, handle->width, handle->height, GL_RGB);
	}

	// release handle
	(*env)->ReleaseIntArrayElements(env, jImage, image, 0);

	return JNI_TRUE;
}

// ==================================================================
/**
 *
 */
static jboolean get_window
  (JNIEnv *env, jobject component, Window *window, VisualID *visualid)
{
	static void *jawt_handle;
	static jboolean (JNICALL *jawt_getawt)(JNIEnv* env, JAWT* awt);
	static jboolean jawt_failed;

	JAWT awt;
	JAWT_DrawingSurface *ds;
	JAWT_DrawingSurfaceInfo *dsi;
	JAWT_X11DrawingSurfaceInfo *dsi_x11;
	jint lock;

	if (jawt_handle == NULL) {
		if (jawt_failed) {
			return JNI_FALSE;
		}

		jawt_handle = dlopen("libjawt.so", RTLD_LAZY);
		if (jawt_handle == NULL) {
			jawt_failed = JNI_TRUE;
			return JNI_FALSE;
		}

		jawt_getawt = dlsym(jawt_handle, "JAWT_GetAWT");
		if (jawt_getawt == NULL) {
			dlclose(jawt_handle);
			jawt_handle = NULL;
			jawt_failed = JNI_TRUE;
			return JNI_FALSE;
		}
	}

	awt.version = JAWT_VERSION_1_3;
	if ((*jawt_getawt)(env, &awt) == JNI_FALSE) {
		return JNI_FALSE;
	}

	ds = awt.GetDrawingSurface(env, component);
	if (ds == NULL) {
		return JNI_FALSE;
	}

	lock = ds->Lock(ds);
	if ((lock & JAWT_LOCK_ERROR) != 0) {
		awt.FreeDrawingSurface(ds);
		return JNI_FALSE;
	}

	dsi = ds->GetDrawingSurfaceInfo(ds);
	if (dsi == NULL) {
		ds->Unlock(ds);
		awt.FreeDrawingSurface(ds);
		return JNI_FALSE;
	}

	dsi_x11 = (JAWT_X11DrawingSurfaceInfo *) dsi->platformInfo;
	*window = dsi_x11->drawable;
	*visualid = dsi_x11->visualID;

	ds->Unlock(ds);
	awt.FreeDrawingSurface(ds);

	return JNI_TRUE;
}

/**
 *
 */
static XVisualInfo *choose_visual
  (Display *display, VisualID visualid, Bool double_buffer)
{
	XVisualInfo template, *visual;
	int nitems, value;

	template.visualid = visualid;
	visual = XGetVisualInfo(display, VisualIDMask, &template, &nitems);
	if (visual == NULL) {
		return NULL;
	}

	glXGetConfig(display, visual, GLX_DOUBLEBUFFER, &value);
	if (value != double_buffer) {
		XFree(visual);
		return NULL;
	}

	return visual;
}

/*
 *
 */
JNIEXPORT jint JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljCreateContext__Ljava_lang_Object_2III
  (JNIEnv *env, jobject obj, jobject component, jint width, jint height, jint mode)
{
	static Display *display;

	Window drawable, copyWindow = None;
	VisualID visualid;
	XVisualInfo *visual;
	GLXContext context;
	Pixmap pixmap = None;
	struct JUN_HANDLE *handle;

	// open our own display...
	if (display == NULL) {
		display = XOpenDisplay(NULL);
	}

	// create resource
	switch (mode) {
	case RENDER_SINGLE_BUFFER:
		if (get_window(env, component, &drawable, &visualid) == JNI_FALSE) {
			return 0;
		}
		visual = choose_visual(display, visualid, False);
		if (visual == NULL) { return 0; }
		context = glXCreateContext(display, visual, NULL, True);
		break;
	case RENDER_DOUBLE_BUFFER:
		if (get_window(env, component, &drawable, &visualid) == JNI_FALSE) {
			return 0;
		}
		visual = choose_visual(display, visualid, True);
		if (visual == NULL) { return 0; }
		context = glXCreateContext(display, visual, NULL, True);
		break;
	case RENDER_DIRECT:
		if (get_window(env, component, &copyWindow, &visualid) == JNI_FALSE) {
			return 0;
		}
		// no break
	case RENDER_IMAGE:
		visual = glXChooseVisual(display, 0, singleOffscreenBuffer);
		if (visual == NULL) {
			visual = glXChooseVisual(display, 0, doubleOffscreenBuffer);
			if (visual == NULL) { return 0; }
		}
		context = glXCreateContext(display, visual, NULL, False);
		pixmap = XCreatePixmap(display, RootWindow(display, visual->screen),
			width, height, visual->depth);
		drawable = glXCreateGLXPixmap(display, visual, pixmap);
		break;
	default:
		return 0;
	}

	// set handle
	handle = malloc(sizeof(struct JUN_HANDLE));
	if (handle == NULL) {
		return 0;
	}
	handle->mode = mode;
	handle->display = display;
	handle->visual = visual;
	handle->drawable = drawable;
	handle->context = context;
	handle->pixmap = pixmap;
	handle->width = width;
	handle->height = height;
	handle->copyWindow = copyWindow;
	handle->x = 0;
	handle->y = 0;
	handle->resize = 0;

	return (jint)handle;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljDeleteContext
  (JNIEnv *env, jobject obj, jint jHandle, jboolean isApplet)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// delete resource
	switch (handle->mode) {
	case RENDER_SINGLE_BUFFER:
	case RENDER_DOUBLE_BUFFER:
		glXDestroyContext(handle->display, handle->context);
		break;
	case RENDER_DIRECT:
	case RENDER_IMAGE:
		glXDestroyContext(handle->display, handle->context);
		glXDestroyGLXPixmap(handle->display, handle->drawable);
		XFreePixmap(handle->display, handle->pixmap);
		break;
	}
	XFree(handle->visual);

	// release handle
	free(handle);

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljFlushBuffer
  (JNIEnv *env, jobject obj, jint jHandle)
{
	struct JUN_HANDLE *handle;
	GC gc;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// flush buffer
	switch (handle->mode) {
	case RENDER_DOUBLE_BUFFER:
		glXSwapBuffers(handle->display, handle->drawable);
		glXMakeCurrent(handle->display, None, NULL);
		break;
	case RENDER_SINGLE_BUFFER:
	case RENDER_IMAGE:
		glFlush();
		glXMakeCurrent(handle->display, None, NULL);
		break;
	case RENDER_DIRECT:
		glFlush();
		glXMakeCurrent(handle->display, None, NULL);
		gc = XCreateGC(handle->display, handle->pixmap, 0, 0);
		XCopyArea(handle->display, handle->pixmap, handle->copyWindow,
			gc, 0, 0, handle->width, handle->height, handle->x, handle->y);
		XFlush(handle->display);
		XFreeGC(handle->display, gc);
		break;
	}

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljMakeCurrent
  (JNIEnv *env, jobject obj, jint jHandle)
{
	struct JUN_HANDLE *handle;
	XVisualInfo *visual;
	Window drawable;
	GLXContext context = NULL;
	Pixmap pixmap = None;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	if (handle->resize == 0) {
		// make current
		glXMakeCurrent(handle->display, handle->drawable, handle->context);
		glViewport(0, 0, handle->width, handle->height);
	} else {
		switch (handle->mode) {
		case RENDER_SINGLE_BUFFER:
		case RENDER_DOUBLE_BUFFER:
			// make current and set viewport
			glXMakeCurrent(handle->display, handle->drawable, handle->context);
			glViewport(0, 0, handle->width, handle->height);
			break;
		case RENDER_IMAGE:
		case RENDER_DIRECT:
			// delete resouce
			glXDestroyContext(handle->display, handle->context);
			glXDestroyGLXPixmap(handle->display, handle->drawable);
			XFreePixmap(handle->display, handle->pixmap);

			// create resource
			visual = handle->visual;
			context = glXCreateContext(handle->display, visual, NULL, False);
			pixmap = XCreatePixmap(handle->display,
				RootWindow(handle->display, visual->screen),
				handle->width, handle->height, visual->depth);
			drawable = glXCreateGLXPixmap(handle->display, visual, pixmap);
			glXMakeCurrent(handle->display, drawable, context);

			// set handle
			handle->drawable = drawable;
			handle->context = context;
			handle->pixmap = pixmap;
			break;
		}
		handle->resize = 0;
	}

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljSetLocation
  (JNIEnv *env, jobject obj, jint jHandle, jint x, jint y)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// set location
	handle->x = x;
	handle->y = y;

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljSetSize
  (JNIEnv *env, jobject obj, jint jHandle, jint width, jint height)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// set handle
	if (handle->width != width || handle->height != height) {
		handle->width = width;
		handle->height = height;
		handle->resize = 1;
	}

	return JNI_TRUE;
}
