#include <windows.h>
#include <GL\gl.h>

#include <jawt.h>
#include <jawt_md.h>

#include "jp_co_sra_gl4jun_GLjInterface.h"
#include "rendering_mode.h"
#include "version.h"

/**
 * glj-windows.c:
 *
 * 	@author:	MATSUDA Ryouichi
 * 	@version:	2.0
 * 	@created:	1999/08/01 (by MATSUDA Ryouichi)
 * 	@updated:	N/A
 *
 * 	$Id: glj-windows.c,v 2.1 2005/03/08 08:58:43 hoshi Exp $
 */

struct JUN_HANDLE {
	int mode;
	HWND hRenderWND;
	HDC hRenderDC;
	HGLRC hRC;
	LPTSTR lpBits;
	int width;
	int height;
	HDC hBitbltDC;
	int x;
	int y;
	BOOLEAN resize;
	// HBITMAP hBitmap;
};

// ==================================================================
/*
 *
 */
JNIEXPORT jstring JNICALL Java_jp_co_sra_gl4jun_GLjInterface_getVersion
  (JNIEnv *env, jclass cls)
{
	return (*env)->NewStringUTF(env, JUNGL_JNI_VERSION);
}

// ==================================================================
/**
 *
 */
static void bitmap2image
  (unsigned char *bitmap, jint *image, int width, int height)
{
	unsigned char *pBitmap, byte1, byte2, byte3;
	jint color;
	int x, y;
	const int bitsPerPixel = 24;
	const int padding = ((bitsPerPixel + 15) / 16) * 16;
	int rowByteSize = (width * bitsPerPixel + padding - 1) / padding * (padding / 8);

	for (y = height - 1; 0 <= y; y--) {
		pBitmap = bitmap + (y * rowByteSize);
		for (x = 0; x < width; x++) {
			byte1 = *pBitmap++;
			byte2 = *pBitmap++;
			byte3 = *pBitmap++;
			color = byte1 | byte2 << 8 | byte3 << 16 | 0xFF << 24;
			*image++ = color;
		}
	}
}

/**
 *
 */
static jboolean rendering2image
  (jint *image, int width, int height, GLenum format)
{
	GLint swapbytes, lsbfirst, rowlength;
	GLint skiprows, skippixels, alignment;
	unsigned char *buffer, *pBuffer;
	unsigned char byte1, byte2, byte3, byte4;
	jint color;
	int x, y;

	// create buffer
	switch (format) {
	case GL_RGB:
		buffer = (unsigned char*)malloc(width * height * 3);
		break;
	case GL_RGBA:
		buffer = (unsigned char*)malloc(width * height * 4);
		break;
	default:
		buffer = NULL;
	}
	if (buffer == NULL) {
		return JNI_FALSE;
	}

	// copy buffer
	glGetIntegerv(GL_UNPACK_SWAP_BYTES, &swapbytes);
	glGetIntegerv(GL_UNPACK_LSB_FIRST, &lsbfirst);
	glGetIntegerv(GL_UNPACK_ROW_LENGTH, &rowlength);
	glGetIntegerv(GL_UNPACK_SKIP_ROWS, &skiprows);
	glGetIntegerv(GL_UNPACK_SKIP_PIXELS, &skippixels);
	glGetIntegerv(GL_UNPACK_ALIGNMENT, &alignment);
	glPixelStorei(GL_UNPACK_SWAP_BYTES, GL_FALSE);
	glPixelStorei(GL_UNPACK_LSB_FIRST, GL_FALSE);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glReadPixels(0, 0, width, height, format, GL_UNSIGNED_BYTE, buffer);
	glPixelStorei(GL_UNPACK_SWAP_BYTES, swapbytes);
	glPixelStorei(GL_UNPACK_LSB_FIRST, lsbfirst);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, rowlength);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, skiprows);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, skippixels);
	glPixelStorei(GL_UNPACK_ALIGNMENT, alignment);

	// convert buffer
	switch (format) {
	case GL_RGB:
		for (y = height - 1; 0 <= y ; y--) {
			pBuffer = buffer + (width * 3 * y);
			for (x = 0; x < width; x++) {
					byte1 = *pBuffer++;
					byte2 = *pBuffer++;
					byte3 = *pBuffer++;
					color = byte3 | byte2 << 8 | byte1 << 16 | 0xFF << 24;
					*image++ = color;
				}
		}
		break;
	case GL_RGBA:
		for (y = height - 1; 0 <= y ; y--) {
			pBuffer = buffer + (width * 4 * y);
				for (x = 0; x < width; x++) {
					byte1 = *pBuffer++;
					byte2 = *pBuffer++;
					byte3 = *pBuffer++;
					byte4 = *pBuffer++;
					color = byte3 | byte2 << 8 | byte1 << 16 | byte4 << 24;
					*image++ = color;
				}
		}
		break;
	default:
		break;
	}

	// release buffer
	free(buffer);

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljReadImage
  (JNIEnv *env, jobject obj, jint jHandle, jintArray jImage, jboolean isRGBA)
{
	struct JUN_HANDLE *handle;
	jint *image;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;
	image = (*env)->GetIntArrayElements(env, jImage, 0);

	// copy buffer
	glFlush();
	if (isRGBA == JNI_TRUE) {
		rendering2image(image, handle->width, handle->height, GL_RGBA);
	} else {
		switch (handle->mode) {
		case RENDER_SINGLE_BUFFER:
		case RENDER_DOUBLE_BUFFER:
#if 0	// hanged up when set RGB.
			rendering2image(imagePtr, handle->width, handle->height, GL_RGB);
#else
			rendering2image(image, handle->width, handle->height, GL_RGBA);
#endif
			break;
		case RENDER_DIRECT:
		case RENDER_IMAGE:
			bitmap2image(handle->lpBits, image, handle->width, handle->height);
			break;
		}
	}

	// release handle
	(*env)->ReleaseIntArrayElements(env, jImage, image, 0);

	return JNI_TRUE;
}

// ==================================================================
/**
 *
 */
static jboolean get_window
  (JNIEnv *env, jobject component, HWND *window)
{
	static HINSTANCE jawt_handle;
	static jboolean (JNICALL *jawt_getawt)(JNIEnv* env, JAWT* awt);
	static jboolean jawt_failed;

	JAWT awt;
	JAWT_DrawingSurface *ds;
	JAWT_DrawingSurfaceInfo *dsi;
	JAWT_Win32DrawingSurfaceInfo *dsi_win;
	jint lock;

	if (jawt_handle == NULL) {
		if (jawt_failed) {
			return JNI_FALSE;
		}

		jawt_handle = LoadLibrary("jawt.dll");
		if (jawt_handle == NULL) {
			jawt_failed = JNI_TRUE;
			return JNI_FALSE;
		}

		jawt_getawt = (void *) GetProcAddress(jawt_handle, "_JAWT_GetAWT@8");
		if (jawt_getawt == NULL) {
			FreeLibrary(jawt_handle);
			jawt_handle = NULL;
			jawt_failed = JNI_TRUE;
			return JNI_FALSE;
		}
	}

	awt.version = JAWT_VERSION_1_3;
	if ((*jawt_getawt)(env, &awt) == JNI_FALSE) {
		return JNI_FALSE;
	}

	ds = awt.GetDrawingSurface(env, component);
	if (ds == NULL) {
		return JNI_FALSE;
	}

	lock = ds->Lock(ds);
	if ((lock & JAWT_LOCK_ERROR) != 0) {
		awt.FreeDrawingSurface(ds);
		return JNI_FALSE;
	}

	dsi = ds->GetDrawingSurfaceInfo(ds);
	if (dsi == NULL) {
		ds->Unlock(ds);
		awt.FreeDrawingSurface(ds);
		return JNI_FALSE;
	}

	dsi_win = (JAWT_Win32DrawingSurfaceInfo *) dsi->platformInfo;
	*window = dsi_win->hwnd;

	ds->Unlock(ds);
	awt.FreeDrawingSurface(ds);

	return JNI_TRUE;
}

/**
 *
 */
static jboolean createBitmap
  (HBITMAP *hBmp, HDC *hDC, LPTSTR *lpBits, int width, int height)
{
	BITMAPINFO bi;

	*hDC = CreateCompatibleDC(0);
	if (*hDC == 0) {
		return JNI_FALSE;
	}

	bi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	bi.bmiHeader.biWidth = width;
	bi.bmiHeader.biHeight = height;
	bi.bmiHeader.biPlanes = 1;
	bi.bmiHeader.biBitCount = 24;
	bi.bmiHeader.biCompression = BI_RGB;
	bi.bmiHeader.biSizeImage = 0;
	bi.bmiHeader.biXPelsPerMeter = 0;
	bi.bmiHeader.biYPelsPerMeter = 0;
	bi.bmiHeader.biClrUsed = 0;
	bi.bmiHeader.biClrImportant = 0;
	*hBmp = CreateDIBSection(*hDC, &bi, DIB_RGB_COLORS, lpBits, 0, 0);
	SelectObject(*hDC, *hBmp);
	GdiFlush();

	return JNI_TRUE;
}

/**
 *
 */
static HGLRC createContext
  (HDC hDC, jint mode)
{
	PIXELFORMATDESCRIPTOR pfd;
	HGLRC hRC;
	int pixelFormat;

	// create pfd
	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	switch (mode) {
	case RENDER_SINGLE_BUFFER:
		pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL;
		break;
	case RENDER_DOUBLE_BUFFER:
		pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
		break;
	case RENDER_IMAGE:
	case RENDER_DIRECT:
		pfd.dwFlags = PFD_DRAW_TO_BITMAP | PFD_SUPPORT_OPENGL;
		break;
	}
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 24;
	pfd.cRedBits = 0;
	pfd.cRedShift = 0;
	pfd.cGreenBits = 0;
	pfd.cGreenShift = 0;
	pfd.cBlueBits = 0;
	pfd.cBlueShift = 0;
	pfd.cAlphaBits = 0;
	pfd.cAlphaShift = 0;
	pfd.cAccumBits = 0;
	pfd.cAccumRedBits = 0;
	pfd.cAccumGreenBits = 0;
	pfd.cAccumBlueBits = 0;
	pfd.cAccumAlphaBits = 0;
	pfd.cDepthBits = 32;
	pfd.cStencilBits = 1;
	pfd.cAuxBuffers = 0;
	pfd.iLayerType = PFD_MAIN_PLANE;
	pfd.bReserved = 0;
	pfd.dwLayerMask = 0;
	pfd.dwVisibleMask = 0;
	pfd.dwDamageMask = 0;

	// create context
	pixelFormat = ChoosePixelFormat(hDC, &pfd);
	SetPixelFormat(hDC, pixelFormat, &pfd);
	hRC = wglCreateContext(hDC);

	return hRC;
}

/*
 *
 */
JNIEXPORT jint JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljCreateContext__Ljava_lang_Object_2III
  (JNIEnv *env, jobject obj, jobject component, jint width, jint height, jint mode)
{
	HWND hRenderWND, hBitbltWND;
	HDC hRenderDC, hBitbltDC;
	HGLRC hRC;
	LPTSTR lpBits;
	struct JUN_HANDLE *handle;

	// create resource
	switch (mode) {
	case RENDER_SINGLE_BUFFER:
	case RENDER_DOUBLE_BUFFER:
		if (get_window(env, component, &hRenderWND) == JNI_FALSE) {
			return 0;
		}
		hRenderDC = GetDC(hRenderWND);
		break;
	case RENDER_DIRECT:
		if (get_window(env, component, &hBitbltWND) == JNI_FALSE) {
			return 0;
		}
		hBitbltDC = GetDC(hBitbltWND);
		// no break
	case RENDER_IMAGE:
		if (createBitmap((HBITMAP*)&hRenderWND, &hRenderDC, &lpBits, width, height) == JNI_FALSE) {
			return 0;
		}
		break;
	}
	hRC = createContext(hRenderDC, mode);

	// set handle
	handle = malloc(sizeof(struct JUN_HANDLE));
	if (handle == NULL) {
		return 0;
	}
	handle->mode = mode;
	handle->hRenderWND = hRenderWND;
	handle->hRenderDC = hRenderDC;
	handle->hRC = hRC;
	handle->lpBits = lpBits;
	handle->width = width;
	handle->height = height;
	handle->hBitbltDC = hBitbltDC;
	handle->x = 0;
	handle->y = 0;
	handle->resize = 0;

	return (jint)handle;
}

// ==================================================================
/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljDeleteContext
  (JNIEnv *env, jobject obj, jint jHandle, jboolean isApplet)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return 0;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// delete resource
	if (!isApplet) {	// for Applet on Windows98 (not NT)
		wglDeleteContext(handle->hRC);
	}
	switch (handle->mode) {
	case RENDER_SINGLE_BUFFER:
	case RENDER_DOUBLE_BUFFER:
		DeleteDC(handle->hRenderDC);
		break;
	case RENDER_DIRECT:
		DeleteDC(handle->hBitbltDC);
		// no break
	case RENDER_IMAGE:
		DeleteDC(handle->hRenderDC);
		DeleteObject((HBITMAP)handle->hRenderWND);
		break;
	}
	free(handle);

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljFlushBuffer
  (JNIEnv *env, jobject obj, jint jHandle)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// flush buffer
	switch (handle->mode) {
	case RENDER_DOUBLE_BUFFER:
		SwapBuffers(handle->hRenderDC);
		wglMakeCurrent(handle->hRenderDC, NULL);
		break;
	case RENDER_SINGLE_BUFFER:
	case RENDER_IMAGE:
		glFlush();
		wglMakeCurrent(handle->hRenderDC, NULL);
		break;
	case RENDER_DIRECT:
		glFlush();
		wglMakeCurrent(handle->hRenderDC, NULL);
		BitBlt(handle->hBitbltDC, handle->x, handle->y,
			handle->width, handle->height,
			handle->hRenderDC, 0, 0, SRCCOPY);
		break;
	}

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljMakeCurrent
  (JNIEnv *env, jobject obj, jint jHandle)
{
	struct JUN_HANDLE *handle;
	HDC hRenderDC;
	HWND hRenderWND;
	LPTSTR lpBits;
	HGLRC hRC;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	if (handle->resize == 0) {
		// make current
		wglMakeCurrent(handle->hRenderDC, handle->hRC);
		glViewport(0, 0, handle->width, handle->height);
	} else {
		switch (handle->mode) {
		case RENDER_SINGLE_BUFFER:
		case RENDER_DOUBLE_BUFFER:
			// make current and set viewport
			wglMakeCurrent(handle->hRenderDC, handle->hRC);
			glViewport(0, 0, handle->width, handle->height);
			break;
		case RENDER_IMAGE:
		case RENDER_DIRECT:
			// delete resouce
			wglDeleteContext(handle->hRC);
			DeleteObject((HBITMAP)handle->hRenderWND);
			DeleteDC(handle->hRenderDC);

			// create resource
			if (createBitmap((HBITMAP*)&hRenderWND, &hRenderDC, &lpBits, handle->width, handle->height) == JNI_FALSE) {
				// error
				printf("JNI err: gljMakeCurrent() - createBitmap()\n");
			}
			hRC = createContext(hRenderDC, handle->mode);
			wglMakeCurrent(hRenderDC, hRC);

			// set handle
			handle->hRenderWND = hRenderWND;
			handle->hRenderDC = hRenderDC;
			handle->hRC = hRC;
			handle->lpBits = lpBits;
			break;
		}
		handle->resize = 0;
	}

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljSetLocation
  (JNIEnv *env, jobject obj, jint jHandle, jint x, jint y)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// set location
	handle->x = x;
	handle->y = y;

	return JNI_TRUE;
}

/*
 *
 */
JNIEXPORT jboolean JNICALL Java_jp_co_sra_gl4jun_GLjInterface_gljSetSize
  (JNIEnv *env, jobject obj, jint jHandle, jint width, jint height)
{
	struct JUN_HANDLE *handle;

	// get handle
	if (jHandle == 0) {
		return JNI_FALSE;
	}
	handle = (struct JUN_HANDLE*)jHandle;

	// set handle
	if (handle->width != width || handle->height != height) {
		handle->width = width;
		handle->height = height;
		handle->resize = 1;
	}

	return JNI_TRUE;
}
