package sample.form;

import jaxcent.*;
import java.util.*;

/**
  * Jaxcent sample.
  *
  * Presents and maintains a shopping cart via the session.
  */

public class ShoppingCart extends jaxcent.JaxcentPage {

    // The list of available items and costs.  More typically, it would be coming from a database!
   
    static String[][] items = {
        { "Partridge", "50" },
        { "Pear Tree", "100" },
        { "Turtle Dove", "150" },
        { "French Hen", "200" },
        { "Collie Bird", "250" },
        { "Golden Ring", "300" },
    };

    // Page items.

    HtmlTable itemsTable = new HtmlTable( this, "itemsTable" );
    HtmlTable shoppingCart = new HtmlTable( this, "shoppingCart" );

    // Items in shopping cart.
    Map dataMap;
    ArrayList itemsInCart;
    boolean headerAdded = false;

    // Add-Button handling class.
    private class AddItemButton extends HtmlButton {
        int index;

        AddItemButton( JaxcentPage page, String id, int itemIndex )
        {
            super( page, id );
            this.index = itemIndex;
        }

        protected void onClick()
        {
            addItemToCart( index );
        }
    };

    // Remove-Button handling class.
    private class RemoveItemButton extends HtmlButton {
        int index;

        RemoveItemButton( JaxcentPage page, String id, int itemIndex )
        {
            super( page, id );
            this.index = itemIndex;
        }

        protected void onClick()
        {
            removeItemFromCart( index );
        }
    };

    // Constructor populates the item table and recreates the shopping cart.
    // Note that for session data, we do not need to wait until onLoad
    // unless we want to retrieve any items from page.

    public ShoppingCart()
    {
        try {
            dataMap = getAllSessionData( false );
            itemsInCart = (ArrayList) dataMap.get( "shopping.cart" );
            if ( itemsInCart == null ) {
                itemsInCart = new ArrayList();
                dataMap.put( "shopping.cart", itemsInCart );
            }
            itemsTable.insertRow( -1, new String[]{ "<B>Item</B>", "<B>Price</B>", "<B>Add</B>" } );
            for ( int i = 0; i < items.length; i++ ) {
                itemsTable.insertRow( -1, new
                      String[]{ items[ i ][ 0 ],                           // Item name
                                "$ " + items[ i ][ 1 ],                    // Item price
                                "<BUTTON id=add_" + i + ">Add</BUTTON>"    // "Add" button
                              } );
                // Attach object to button.
                new AddItemButton( this, "add_" + i, i );
            }
            if ( itemsInCart.size() > 0 ) {
                addShoppingCartHeader();
            }
            for ( int i = 0; i < itemsInCart.size(); i += 2 ) {
                int itemIndex = ((Integer) itemsInCart.get( i )).intValue();
                int itemCount = ((Integer) itemsInCart.get( i+1 )).intValue();
                addShoppingCartItem( itemIndex, itemCount );
            }
        } catch (Jaxception jax) {
            jax.printStackTrace();
        }
    }

    void addShoppingCartHeader() throws Jaxception
    {
        if ( headerAdded )
            return;

        shoppingCart.insertRow( -1, new String[]{ "<B>Quantity</B>", "<B>Item</B>", "<B>Unit Price</B>", "<B>Cost</B>", "<B>Remove</B>" } );
        headerAdded = true;
    }

    void addShoppingCartItem( int itemIndex, int itemCount ) throws Jaxception
    {
        String price = items[ itemIndex ][ 1 ];
        int totalPrice = Integer.parseInt( price ) * itemCount;
        shoppingCart.insertRow( -1,
                new String[]{
                      String.valueOf( itemCount ),                           // Quantity
                      items[ itemIndex ][ 0 ],                               // Item name
                      price,                                                 // Unit price
                      String.valueOf( totalPrice ),                          // Total cost
                      "<BUTTON id=remove_" + itemIndex + ">Remove</BUTTON>"  // "Remove" button
                } );
         new RemoveItemButton( this, "remove_" + itemIndex, itemIndex ); // Attach button-object to button.
    }

    void addItemToCart( int newItemIndex )
    {
        try {
            // Search if item is already in cart.
            for ( int i = 0; i < itemsInCart.size(); i += 2 ) {
                int itemIndex = ((Integer) itemsInCart.get( i )).intValue();
                if ( newItemIndex == itemIndex ) {
                    // Found in cart.
                    int itemCount = ((Integer) itemsInCart.get( i+1 )).intValue();
                    itemCount++;
                    itemsInCart.set( i+1, new Integer( itemCount ));
                    // Update the shopping cart table.
                    int rowIndex = i / 2 + 1;  // There are 2 entries per item in itemsInCart.  Plus 1 for header row.
                    HtmlTableRow row = shoppingCart.getRow( rowIndex );
                    HtmlTableCell quantityCell = row.getCell( 0 );
                    HtmlTableCell totalCostCell = row.getCell( 3 );
                    quantityCell.setInnerHTML( String.valueOf( itemCount ));
                    totalCostCell.setInnerHTML( String.valueOf( itemCount * Integer.parseInt( items[ itemIndex ][ 1 ] )));
                    return;
                }
            }
            // Not found in existing items in cart, add new row.
            if ( itemsInCart.size() == 0 ) {
                addShoppingCartHeader();
            }
            itemsInCart.add( new Integer( newItemIndex ));
            itemsInCart.add( new Integer( 1 ));
            addShoppingCartItem( newItemIndex, 1 );
        } catch (Jaxception jax) {
            jax.printStackTrace();
        }
    }

    void removeItemFromCart( int itemIndex )
    {
        try {
            // Search for item in cart.
            for ( int i = 0; i < itemsInCart.size(); i += 2 ) {
                if ( itemIndex == ((Integer) itemsInCart.get( i )).intValue()) {
                    // Found item, reduce count or remove if count is 1.
                    int itemCount = ((Integer) itemsInCart.get( i+1 )).intValue();
                    itemCount--;
                    if ( itemCount > 0 ) {
                        itemsInCart.set( i+1, new Integer( itemCount ));
                        // Update the shopping cart table.
                        int rowIndex = i / 2 + 1;  // There are 2 entries per item in itemsInCart.  Plus 1 for header row.
                        HtmlTableRow row = shoppingCart.getRow( rowIndex );
                        HtmlTableCell quantityCell = row.getCell( 0 );
                        HtmlTableCell totalCostCell = row.getCell( 3 );
                        quantityCell.setInnerHTML( String.valueOf( itemCount ));
                        totalCostCell.setInnerHTML( String.valueOf( itemCount * Integer.parseInt( items[ itemIndex ][ 1 ] )));
                    } else {
                        // Remove from itemsInCart and from table.
                        itemsInCart.remove( i+1 );
                        itemsInCart.remove( i );
                        int rowIndex = i / 2 + 1;
                        shoppingCart.deleteRow( rowIndex );
                    }
                    return;
                }
            }
        } catch (Jaxception jax) {
            jax.printStackTrace();
        }
    }
}
