/*
 * $Id:TwisterLauncherThread.java 488 2008-01-27 09:21:39Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister.swing;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Stroke;
import java.awt.Toolkit;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.lang.reflect.InvocationTargetException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Properties;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JWindow;
import javax.swing.SwingUtilities;

import net.sf.jame.core.config.DefaultConfigContext;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.media.AlphaEffect;
import net.sf.jame.media.Context;
import net.sf.jame.media.Controller;
import net.sf.jame.media.Effect;
import net.sf.jame.media.EngineColorPaint;
import net.sf.jame.media.Layer;
import net.sf.jame.media.MotionTweenSequence;
import net.sf.jame.media.Movie;
import net.sf.jame.media.Renderer;
import net.sf.jame.media.Sequence;
import net.sf.jame.media.SimpleRenderedImage;
import net.sf.jame.media.SimpleSequence;
import net.sf.jame.media.SimpleShape;
import net.sf.jame.media.SimpleText;
import net.sf.jame.media.Timeline;
import net.sf.jame.media.g3d.Camera;
import net.sf.jame.media.g3d.Environment;
import net.sf.jame.media.g3d.FastRasterizer;
import net.sf.jame.media.g3d.Handler;
import net.sf.jame.media.g3d.Light;
import net.sf.jame.media.g3d.Material;
import net.sf.jame.media.g3d.Part;
import net.sf.jame.media.g3d.RenderPipeline;
import net.sf.jame.media.g3d.Scene;
import net.sf.jame.media.g3d.Screen;
import net.sf.jame.media.g3d.Solid;
import net.sf.jame.media.g3d.SolidFactory;
import net.sf.jame.media.swing.MovieCanvas;
import net.sf.jame.service.AsyncService;
import net.sf.jame.service.ConnectionFactory;
import net.sf.jame.service.DefaultConnectionFactory;
import net.sf.jame.service.Service;
import net.sf.jame.service.ServiceException;
import net.sf.jame.service.ServiceRegistry;
import net.sf.jame.service.Session;
import net.sf.jame.service.spool.extension.SpoolExtensionRuntime;
import net.sf.jame.twister.TwisterConfig;
import net.sf.jame.twister.TwisterConfigBuilder;

import org.apache.log4j.PropertyConfigurator;

/**
 * @author Andrea Medeghini
 */
public class TwisterLauncherThread extends Thread {
	private final Movie movie = new SplashMovie();
	private final TwisterContext context;
	private JFrame frame;

	/**
	 * @param context
	 */
	public TwisterLauncherThread(final TwisterContext context) {
		super("Launcher");
		this.context = context;
	}

	/**
	 * @see java.lang.Thread#run()
	 */
	@Override
	public void run() {
		try {
			// try {
			// UIManager.setLookAndFeel("javax.swing.plaf.metal.MetalLookAndFeel");
			// }
			// catch (Exception x) {
			// x.printStackTrace();
			// }
			// if (SystemTray.isSupported()) {
			// SystemTray tray = SystemTray.getSystemTray();
			// ActionListener exitListener = new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// exit();
			// }
			// };
			// PopupMenu popup = new PopupMenu();
			// MenuItem defaultItem = new MenuItem("Exit");
			// defaultItem.addActionListener(exitListener);
			// popup.add(defaultItem);
			// Image image = Toolkit.getDefaultToolkit().getImage("JAME.png");
			// final TrayIcon trayIcon = new TrayIcon(image, "JAME 6.0", popup);
			// trayIcon.setImageAutoSize(true);
			// try {
			// tray.add(trayIcon);
			// }
			// catch (AWTException e) {
			// e.printStackTrace();
			// }
			if (System.getProperty("jame.splashscreen", "true").equals("true")) {
				final MovieCanvas canvas = new MovieCanvas(new DefaultContext());
				final SplashScreen splashscreen = new SplashScreen(canvas);
				canvas.addMouseListener(new SplashListener());
				SwingUtilities.invokeAndWait(new Runnable() {
					public void run() {
						splashscreen.setVisible(true);
					}
				});
				canvas.getEngine().start();
				try {
					Thread.sleep(10000);
				}
				catch (final InterruptedException e) {
				}
				canvas.getEngine().stop();
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						splashscreen.setVisible(false);
					}
				});
			}
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					try {
						frame = createWindow(context);
						if (frame != null) {
							frame.setVisible(true);
						}
						else {
							context.exit();
						}
					}
					catch (final Exception e) {
						JOptionPane.showMessageDialog(new Frame(), TwisterSwingResources.getInstance().getString("error.failure"), TwisterSwingResources.getInstance().getString("label.error"), JOptionPane.WARNING_MESSAGE);
						e.printStackTrace();
						context.exit();
					}
				}
			});
		}
		catch (final InterruptedException e) {
			e.printStackTrace();
			context.exit();
		}
		catch (final InvocationTargetException e) {
			e.printStackTrace();
			context.exit();
		}
	}

	private JFrame createWindow(final TwisterContext context) throws ExtensionException, ServiceException {
		final JFileChooser fileChooser = new JFileChooser(System.getProperty("user.home"));
		fileChooser.setDialogTitle(TwisterSwingResources.getInstance().getString("label.workspace"));
		fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		fileChooser.setMultiSelectionEnabled(false);
		final Properties properties = new Properties();
		File workspace = null;
		try {
			properties.load(new FileInputStream(System.getProperty("user.home") + "/JAME.properties"));
			if (properties.get("workspace") == null || System.getProperty("jame.workspace") != null) {
				workspace = new File(System.getProperty("user.home") + "/" + System.getProperty("jame.workspace", "JAME-workspace"));
				properties.put("workspace", workspace.getAbsolutePath());
				try {
					properties.store(new FileOutputStream(System.getProperty("user.home") + "/JAME.properties"), null);
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}
			else {
				workspace = new File((String) properties.get("workspace"));
			}
		}
		catch (final Exception x) {
			x.printStackTrace();
			workspace = new File(System.getProperty("user.home") + "/" + System.getProperty("jame.workspace", "JAME-workspace"));
			properties.put("workspace", workspace.getAbsolutePath());
			try {
				properties.store(new FileOutputStream(System.getProperty("user.home") + "/JAME.properties"), null);
			}
			catch (final Exception e) {
				e.printStackTrace();
			}
		}
		ConnectionFactory connectionFactory = null;
		while (workspace != null) {
			connectionFactory = new DefaultConnectionFactory(workspace);
			final Connection connection = null;
			try {
				connectionFactory.createConnection();
				break;
			}
			catch (final Exception e) {
				JOptionPane.showMessageDialog(new Frame(), TwisterSwingResources.getInstance().getString("error.workspace"), TwisterSwingResources.getInstance().getString("label.error"), JOptionPane.WARNING_MESSAGE);
				// fileChooser.setCurrentDirectory(workspace.getParentFile());
				fileChooser.setCurrentDirectory(workspace);
				// fileChooser.setSelectedFile(workspace);
				final int returnVal = fileChooser.showSaveDialog(new JFrame());
				if (returnVal == JFileChooser.APPROVE_OPTION) {
					workspace = fileChooser.getSelectedFile().getAbsoluteFile().getAbsoluteFile();
				}
				else {
					workspace = null;
				}
			}
			finally {
				if (connection != null) {
					try {
						connection.close();
					}
					catch (final SQLException e) {
					}
				}
			}
		}
		if (workspace != null) {
			Properties log4jProperties = new Properties();
			log4jProperties.put("log4j.rootLogger", "INFO, file, console");
			log4jProperties.put("log4j.appender.console", "org.apache.log4j.ConsoleAppender");
			log4jProperties.put("log4j.appender.console.layout", "org.apache.log4j.PatternLayout");
			log4jProperties.put("log4j.appender.console.layout.ConversionPattern", "%d{HH:mm:ss,SSS} %-5p %c - %m%n");
			log4jProperties.put("log4j.appender.file", "org.apache.log4j.FileAppender");
			log4jProperties.put("log4j.appender.file.file", workspace.getAbsolutePath() + "/JAME.log");
			log4jProperties.put("log4j.appender.file.layout", "org.apache.log4j.PatternLayout");
			log4jProperties.put("log4j.appender.file.layout.ConversionPattern", "%d{HH:mm:ss,SSS} %-5p %c - %m%n");
			log4jProperties.put("log4j.logger.net.sf.jame", "INFO");
			log4jProperties.put("log4j.logger.org.apache.derby", "INFO");
			try {
				log4jProperties.load(new FileInputStream("log4j.properties"));
			}
			catch (Exception e) {
				e.printStackTrace();
			}
			PropertyConfigurator.configure(log4jProperties);
			// System.setProperty("derby.stream.error.file", workspace.getAbsolutePath() + "/derby.log");
			final Session session = new Session(connectionFactory);
			final SpoolExtensionRuntime<?> spoolRuntime = ServiceRegistry.getInstance().getSpoolRegistry().getConfigurableExtension("service.spool.local").createExtensionRuntime();
			Service service = new Service(session, workspace);
			final AsyncService asyncService = new AsyncService(service, spoolRuntime.getJobService(service));
			final TwisterConfigBuilder configBuilder = new TwisterConfigBuilder();
			final TwisterConfig config = configBuilder.createDefaultConfig();
			config.setContext(new DefaultConfigContext());
			final int hcells = Integer.getInteger("jame.hcells", 1);
			final int vcells = Integer.getInteger("jame.vcells", 1);
			return new TwisterFrame(context, asyncService, config, hcells, vcells);
		}
		return null;
	}

	private final class SplashListener extends MouseAdapter {
		/**
		 * @see java.awt.event.MouseAdapter#mousePressed(java.awt.event.MouseEvent)
		 */
		@Override
		public void mousePressed(final MouseEvent e) {
			interrupt();
		}
	}

	private class DefaultContext implements Context {
		/**
		 * @see net.sf.jame.media.Context#getMovie()
		 */
		public Movie getMovie() {
			return movie;
		}

		/**
		 * @see net.sf.jame.media.Context#getColor()
		 */
		public Color getColor() {
			return Color.WHITE;
		}

		/**
		 * @see net.sf.jame.media.Context#loop()
		 */
		public boolean loop() {
			return false;
		}

		/**
		 * @see net.sf.jame.media.Context#debug()
		 */
		public boolean debug() {
			return false;
		}

		/**
		 * @see net.sf.jame.media.Context#println(java.lang.String)
		 */
		public void println(final String s) {
		}

		/**
		 * @see net.sf.jame.media.Context#print(java.lang.String)
		 */
		public void print(final String s) {
		}

		/**
		 * @see net.sf.jame.media.Context#exit(int)
		 */
		public void exit(final int code) {
			interrupt();
		}
	}

	private final class SplashScreen extends JWindow {
		private static final long serialVersionUID = 1L;

		/**
		 * @param canvas
		 */
		public SplashScreen(final MovieCanvas canvas) {
			getContentPane().add(canvas);
			pack();
			final Dimension frame_size = getSize();
			final Dimension screen_size = Toolkit.getDefaultToolkit().getScreenSize();
			setLocation((screen_size.width - frame_size.width) / 2, (screen_size.height - frame_size.height) / 2);
		}
	}

	private class SplashMovie extends Movie {
		private final int movieWidth = 400;
		private final int movieHeight = 256;
		private final int movieLength = 150;
		private final int sequenceLength = 20;
		private final int frameRate = 10;

		/**
		 * 
		 */
		public SplashMovie() {
			super("splash", new Timeline());
		}

		/**
		 * @see net.sf.jame.media.Movie#getSize()
		 */
		@Override
		public Dimension getSize() {
			return new Dimension(movieWidth, movieHeight);
		}

		/**
		 * @see net.sf.jame.media.Movie#getCenter()
		 */
		@Override
		public Point2D getCenter() {
			return new Point2D.Float(-movieWidth / 2f, -movieHeight / 2f);
		}

		/**
		 * @see net.sf.jame.media.Movie#load(java.awt.Component)
		 */
		@Override
		public void load(final Component component) {
			try {
				setFrameRate(frameRate);
				final Timeline timeline = getTimeline();
				final Layer layerShape = new Layer();
				final Layer layerTitle = new Layer();
				final Layer layerImage = new Layer();
				final Layer layerCopyright = new Layer();
				final Stroke stroke = new BasicStroke(1f, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 1f, null, 0);
				final Rectangle2D.Float rectangle = new Rectangle2D.Float(0f, 0f, movieWidth - 0f, movieHeight - 0f);
				// EngineGradientPaint gradient = new EngineGradientPaint(new GradientPaint(0, 0, new Color(0xDF, 0xDF, 0xDF, 0xFF), 0, 200, new Color(0xFA, 0xFA, 0xFA, 0xFF)));
				final EngineColorPaint color0 = new EngineColorPaint(new Color(0xA0, 0xA0, 0xAA, 0x00));
				final EngineColorPaint color1 = new EngineColorPaint(new Color(0x50, 0x50, 0x50, 0x70));
				final EngineColorPaint color2 = new EngineColorPaint(new Color(0x30, 0x30, 0x30, 0x70));
				final EngineColorPaint color3 = new EngineColorPaint(new Color(0x60, 0x50, 0x50, 0x70));
				final BufferedImage imageBuffer = new BufferedImage(movieWidth, movieHeight, BufferedImage.TYPE_INT_ARGB);
				final SceneRenderer imageRenderer = new SceneRenderer(component, movieWidth, movieHeight, ((DataBufferInt) imageBuffer.getRaster().getDataBuffer()).getData());
				final SimpleRenderedImage renderedImage1 = new SimpleRenderedImage("image", imageBuffer, imageRenderer);
				final SimpleText textCopyright = new SimpleText("copyright", TwisterSwingResources.getInstance().getString("copyright"), Font.decode("arial-bold-12"), null, (EngineColorPaint) color3.clone(), null);
				final SimpleText textTitle1 = new SimpleText("title", "JAME 6.0", Font.decode("arial-bold-82"), (EngineColorPaint) color2.clone(), (EngineColorPaint) color3.clone(), null);
				final SimpleText textTitle2 = (SimpleText) textTitle1.clone();
				final SimpleShape shape1 = new SimpleShape("shape", rectangle, (EngineColorPaint) color1.clone(), (EngineColorPaint) color0.clone(), stroke);
				shape1.translate(-movieWidth / 2f, -movieHeight / 2f);
				final SimpleShape shape2 = (SimpleShape) shape1.clone();
				final SimpleRenderedImage renderedImage2 = (SimpleRenderedImage) renderedImage1.clone();
				textTitle1.translate(-180, 24);
				textTitle2.translate(-180, 24);
				textCopyright.translate(-10, movieHeight / 2 - 12);
				final Effect effectShape1 = new AlphaEffect("shape1", 0.0f, 0.0f);
				final Effect effectTitle1 = new AlphaEffect("title1", 1.0f, 0.0f);
				final Effect effectImage1 = new AlphaEffect("image1", 1.0f, 0.0f);
				final Effect effectShape2 = new AlphaEffect("shape2", 0.0f, 0.0f);
				final Effect effectTitle2 = new AlphaEffect("title2", 0.0f, 0.0f);
				final Effect effectImage2 = new AlphaEffect("image2", 0.0f, 0.0f);
				final MotionTweenSequence sequenceShape1 = new MotionTweenSequence(shape1, sequenceLength, 0f, 0f, 0f, 0f, 0f, 0f, 0f, effectShape1);
				final MotionTweenSequence sequenceTitle1 = new MotionTweenSequence(textTitle1, sequenceLength, 0f, 0f, 0f, 0f, 0f, 0f, 0f, effectTitle1);
				final MotionTweenSequence sequenceImage1 = new MotionTweenSequence(renderedImage1, sequenceLength, 0f, 0f, 0f, 0f, 0f, 0f, 0f, effectImage1);
				final MotionTweenSequence sequenceShape2 = new MotionTweenSequence(shape2, movieLength - sequenceLength, 0f, 0f, 0f, 0f, 0f, 0f, 0f, effectShape2);
				final MotionTweenSequence sequenceTitle2 = new MotionTweenSequence(textTitle2, movieLength - sequenceLength, 0f, 0f, 0f, 0f, 0f, 0f, 0f, effectTitle2);
				final MotionTweenSequence sequenceImage2 = new MotionTweenSequence(renderedImage2, movieLength - sequenceLength, 0f, 0f, 0f, 0f, 0f, 0f, 0f, effectImage2);
				final SimpleSequence sequenceCopyright = new SimpleSequence(textCopyright, movieLength, null);
				timeline.addLayerFirst(layerShape);
				timeline.addLayerFirst(layerImage);
				timeline.addLayerFirst(layerTitle);
				timeline.addLayerFirst(layerCopyright);
				layerShape.addSequenceLast(sequenceShape1);
				layerTitle.addSequenceLast(sequenceTitle1);
				layerImage.addSequenceLast(sequenceImage1);
				layerShape.addSequenceLast(sequenceShape2);
				layerTitle.addSequenceLast(sequenceTitle2);
				layerImage.addSequenceLast(sequenceImage2);
				layerCopyright.addSequenceFirst(sequenceCopyright);
			}
			catch (final Exception e) {
				e.printStackTrace();
			}
		}

		/**
		 * @see net.sf.jame.media.Movie#flush()
		 */
		@Override
		public void flush() {
		}

		private class SceneRenderer extends Renderer {
			private RenderPipeline pipeline;
			private Camera camera;
			private Scene scene;
			private float ratio;

			/**
			 * @param component
			 * @param bounds_w
			 * @param bounds_h
			 * @param buffer
			 */
			public SceneRenderer(final Component component, final int bounds_w, final int bounds_h, final int[] buffer) {
				try {
					ratio = (float) bounds_w / (float) bounds_h;
					final int[] zbuffer = new int[bounds_w * bounds_h];
					final int[] idbuffer = new int[bounds_w * bounds_h];
					pipeline = new RenderPipeline(new FastRasterizer());
					final Screen screen = new Screen(bounds_w, bounds_h, buffer, zbuffer, idbuffer);
					camera = new Camera("camera", screen, 0.0f, 0.0f, 200.0f, 0.0f, 0.0f, 0.0f, 2f, ratio);
					final Environment environment = new Environment(0x00AFAFAF, null, 0x00FFFFFF);
					scene = new Scene("scene", environment);
					final Light light = new Light("light", -1.0f, +5.0f, +1.0f, 0xFFFFFF, 0xFFFFFF, 200, 2000);
					pipeline.flat_shading = false;
					pipeline.auto_animate = false;
					pipeline.texture_mapping = true;
					scene.addCamera(camera);
					scene.addLight(light);
					final Part cube = SolidFactory.CUBE("cube", 4, 60f, false);
					cube.setMaterial(new Material(null, 0xFF0000, 50, 100));
					cube.setHandler(new SolidHandler(cube));
					scene.addPart(cube);
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}

			/**
			 * @see net.sf.jame.media.Renderer#setFrame(int)
			 */
			@Override
			public void setFrame(final int frame) {
			}

			/**
			 * @see net.sf.jame.media.Animate#nextFrame()
			 */
			public void nextFrame() {
				pipeline.animate();
			}

			/**
			 * @see net.sf.jame.media.Animate#prevFrame()
			 */
			public void prevFrame() {
			}

			/**
			 * @see net.sf.jame.media.Renderer#init()
			 */
			@Override
			public void init() {
			}

			/**
			 * @see net.sf.jame.media.Renderer#kill()
			 */
			@Override
			public void kill() {
			}

			/**
			 * @see net.sf.jame.media.Renderer#reset()
			 */
			@Override
			public void reset() {
			}

			/**
			 * 
			 */
			public void abort() {
			}

			/**
			 * @see net.sf.jame.media.Renderer#build(net.sf.jame.media.Controller, net.sf.jame.media.Movie, net.sf.jame.media.Layer, net.sf.jame.media.Sequence)
			 */
			@Override
			public void build(final Controller engine, final Movie parent, final Layer layer, final Sequence sequence) {
				scene.build();
			}

			/**
			 * @see net.sf.jame.media.Renderer#render()
			 */
			@Override
			public void render() {
				synchronized (pipeline) {
					pipeline.render(camera, scene);
				}
			}

			private class SolidHandler extends Handler {
				private final Solid solido;

				/**
				 * @param solido
				 */
				public SolidHandler(final Solid solido) {
					this.solido = solido;
				}

				/**
				 * @see net.sf.jame.media.g3d.Handler#animate()
				 */
				@Override
				public void animate() {
					solido.rotateSelf((float) Math.PI / 40f, (float) Math.PI / 60f, (float) Math.PI / 100f);
				}
			}
		}
	}
}
