/*
 * $Id:ConfigurableExtensionReferenceNode.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister.util;

import java.io.Serializable;

import net.sf.jame.core.config.ValueChangeEvent;
import net.sf.jame.core.config.ValueChangeListener;
import net.sf.jame.core.extension.ConfigurableExtensionReference;
import net.sf.jame.core.extension.Extension;
import net.sf.jame.core.extension.ExtensionConfig;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.core.tree.DefaultNode;
import net.sf.jame.core.tree.Node;
import net.sf.jame.core.tree.NodeAction;
import net.sf.jame.core.tree.NodeBuilder;
import net.sf.jame.core.tree.NodeEditor;
import net.sf.jame.core.tree.NodeSession;
import net.sf.jame.core.tree.NodeValue;
import net.sf.jame.twister.TwisterRegistry;
import net.sf.jame.twister.TwisterResources;
import net.sf.jame.twister.tree.extension.NodeBuilderExtensionRuntime;

/**
 * @author Andrea Medeghini
 */
public abstract class ConfigurableExtensionReferenceNode<T extends ExtensionConfig> extends DefaultNode {
	protected ConfigElementListener elementListener;
	protected ConfigurableExtensionConfigElement<T> referenceElement;

	/**
	 * Constructs a new filter node.
	 * 
	 * @param nodeId
	 * @param referenceElement
	 */
	public ConfigurableExtensionReferenceNode(final String nodeId, final ConfigurableExtensionConfigElement<T> referenceElement) {
		super(nodeId);
		if (referenceElement == null) {
			throw new IllegalArgumentException("referenceElement is null");
		}
		this.referenceElement = referenceElement;
		elementListener = new ConfigElementListener();
		setNodeValue(createNodeValue(referenceElement.getReference()));
		setNodeLabel(createNodeLabel());
	}

	/**
	 * @param value
	 * @return
	 */
	protected abstract NodeValue<?> createNodeValue(ConfigurableExtensionReference<T> value);

	/**
	 * @see net.sf.jame.core.tree.Node#setSession(net.sf.jame.core.tree.NodeSession)
	 */
	@Override
	public void setSession(final NodeSession session) {
		if (session != null) {
			referenceElement.addChangeListener(elementListener);
		}
		else {
			referenceElement.removeChangeListener(elementListener);
		}
		super.setSession(session);
	}

	/**
	 * @see net.sf.jame.core.tree.Node#nodeAdded()
	 */
	@Override
	protected void nodeAdded() {
		setNodeValue(createNodeValue(referenceElement.getReference()));
	}

	/**
	 * @see net.sf.jame.core.tree.Node#nodeRemoved()
	 */
	@Override
	protected void nodeRemoved() {
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean equals(final Object o) {
		if (o instanceof ConfigurableExtensionReferenceNode) {
			return (referenceElement == ((ConfigurableExtensionReferenceNode) o).referenceElement);
		}
		return false;
	}

	/**
	 * @param value
	 */
	@SuppressWarnings("unchecked")
	protected void createChildNodes(final NodeValue value) {
		removeAllChildNodes();
		if ((value != null) && (value.getValue() != null)) {
			final ExtensionConfig config = ((ExtensionReferenceNodeValue<ConfigurableExtensionReference<T>>) value).getValue().getExtensionConfig();
			try {
				final Extension<NodeBuilderExtensionRuntime> nodeBuilderExtension = TwisterRegistry.getInstance().getNodeBuilderExtension(config.getExtensionId());
				final NodeBuilder builder = nodeBuilderExtension.createExtensionRuntime().createNodeBuilder(config);
				builder.createNodes(this);
			}
			catch (final ExtensionException e) {
				e.printStackTrace();
			}
		}
	}

	@SuppressWarnings("unchecked")
	private ExtensionConfig getNodeValeAsExtensionConfig() {
		if ((getNodeValue() != null) && (getNodeValue().getValue() != null)) {
			return ((ExtensionReferenceNodeValue<ConfigurableExtensionReference<T>>) getNodeValue()).getValue().getExtensionConfig();
		}
		return null;
	}

	private String createNodeLabel() {
		if (getNodeValeAsExtensionConfig() != null) {
			return getNodeValeAsExtensionConfig().getExtensionReference().getExtensionName() + " extension";
		}
		else {
			return "Extension not defined";
		}
	}

	/**
	 * @see net.sf.jame.core.tree.Node#addDescription(java.lang.StringBuilder)
	 */
	@Override
	protected void addDescription(final StringBuilder builder) {
		if (getChildNodeCount() > 0) {
			builder.append(getChildNode(0).getLabel());
		}
		else {
			super.addDescription(builder);
		}
	}

	/**
	 * @return the filterElement
	 */
	public ConfigurableExtensionConfigElement<T> getReferenceElement() {
		return referenceElement;
	}

	/**
	 * @see net.sf.jame.core.tree.Node#updateNode()
	 */
	@Override
	protected void updateNode() {
		setNodeLabel(createNodeLabel());
		super.updateNode();
	}

	/**
	 * @see net.sf.jame.core.tree.Node#updateNode()
	 */
	@Override
	protected void updateChildNodes() {
		createChildNodes(getNodeValue());
	}

	/**
	 * @see net.sf.jame.core.tree.DefaultNode#createNodeEditor()
	 */
	@Override
	protected NodeEditor createNodeEditor() {
		return new ConfigElementNodeEditor(this);
	}

	/**
	 * @see net.sf.jame.core.tree.DefaultNode#getValueAsString()
	 */
	@SuppressWarnings("unchecked")
	@Override
	public String getValueAsString() {
		return ((getNodeValue() != null) && (getNodeValue().getValue() != null)) ? ((ExtensionReferenceNodeValue<ConfigurableExtensionReference<T>>) getNodeValue()).getValue().getExtensionName() : TwisterResources.getInstance().getString("label.undefined");
	}

	protected class ConfigElementNodeEditor extends NodeEditor {
		/**
		 * @param node
		 */
		public ConfigElementNodeEditor(final DefaultNode node) {
			super(node);
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#doSetValue(net.sf.jame.core.tree.NodeValue)
		 */
		@SuppressWarnings("unchecked")
		@Override
		public void doSetValue(final NodeValue value) {
			referenceElement.removeChangeListener(elementListener);
			if (((ExtensionReferenceNodeValue<ConfigurableExtensionReference<T>>) value).getValue() != null) {
				referenceElement.setReference(((ExtensionReferenceNodeValue<ConfigurableExtensionReference<T>>) value).getValue());
			}
			else {
				referenceElement.setReference(null);
			}
			referenceElement.addChangeListener(elementListener);
			setNodeLabel(ConfigurableExtensionReferenceNode.this.createNodeLabel());
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#createChildNode(java.lang.NodeValue)
		 */
		@Override
		protected Node createChildNode(final NodeValue<?> value) {
			return null;
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#getNodeValueType()
		 */
		@Override
		public Class<?> getNodeValueType() {
			return ExtensionReferenceNodeValue.class;
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#createNodeValue(java.io.Serializable)
		 */
		@SuppressWarnings("unchecked")
		@Override
		public NodeValue<?> createNodeValue(final Serializable value) {
			return ConfigurableExtensionReferenceNode.this.createNodeValue((ConfigurableExtensionReference<T>) value);
		}
	}

	protected class ConfigElementListener implements ValueChangeListener, ExtensionConfigElementEvents {
		@SuppressWarnings("unchecked")
		public void valueChanged(final ValueChangeEvent e) {
			cancel();
			switch (e.getEventType()) {
				case EXTENSION_REFERENCE_CHANGED: {
					setNodeValue(new ExtensionReferenceNodeValue<ConfigurableExtensionReference<T>>((ConfigurableExtensionReference<T>) e.getParams()[0]));
					getSession().appendAction(new NodeAction(getNodeClass(), NodeAction.ACTION_SET_VALUE, e.getTimestamp(), getNodePath(), e.getParams()[0] != null ? ((ConfigurableExtensionReference<T>) e.getParams()[0]).clone() : null, e.getParams()[1] != null ? ((ConfigurableExtensionReference<T>) e.getParams()[1]).clone() : null));
					break;
				}
				default: {
					break;
				}
			}
			setNodeLabel(ConfigurableExtensionReferenceNode.this.createNodeLabel());
		}
	}
}
