/*
 * $Id:EncoderDialog.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GraphicsEnvironment;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;

import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.WindowConstants;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;

import net.sf.jame.core.extension.ConfigurableExtension;
import net.sf.jame.core.extension.Extension;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.core.extension.ExtensionNotFoundException;
import net.sf.jame.core.swing.extension.ExtensionComboBoxModel;
import net.sf.jame.core.swing.extension.ExtensionFilter;
import net.sf.jame.core.swing.extension.ExtensionListCellRenderer;
import net.sf.jame.core.swing.util.GUIFactory;
import net.sf.jame.service.AsyncService;
import net.sf.jame.service.ProgressListener;
import net.sf.jame.service.ServiceRegistry;
import net.sf.jame.service.AsyncService.ServiceVoidCallback;
import net.sf.jame.service.encoder.EncoderHook;
import net.sf.jame.service.encoder.extension.EncoderExtensionConfig;
import net.sf.jame.service.encoder.extension.EncoderExtensionRuntime;
import net.sf.jame.service.profile.RenderProfileDataRow;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public class EncoderDialog extends JDialog {
	private static final Logger logger = Logger.getLogger(EncoderDialog.class);
	private static final long serialVersionUID = 1L;
	private static final String STRING_FRAME_TITLE = "encoderFrame.title";
	// private static final String STRING_FRAME_WIDTH = "encoderFrame.width";
	// private static final String STRING_FRAME_HEIGHT = "encoderFrame.height";
	// private static final String STRING_FRAME_ICON = "encoderFrame.icon";
	private final JProgressBar progressBar = new JProgressBar(0, 100);
	private final JFileChooser profileChooser = new JFileChooser(System.getProperty("user.home"));
	private final JButton profileExportButton = GUIFactory.createButton(new ExportAction(), ServiceSwingResources.getInstance().getString("tooltip.export"));
	private final JButton chooseButton = GUIFactory.createButton(new ChooseAction(), ServiceSwingResources.getInstance().getString("tooltip.choose"));
	private final JButton cancelButton = GUIFactory.createButton(new CancelAction(), ServiceSwingResources.getInstance().getString("tooltip.cancel"));
	private final JButton closeButton = GUIFactory.createButton(new CloseAction(), ServiceSwingResources.getInstance().getString("tooltip.close"));
	private final JTextField pathTextField = GUIFactory.createTextField((String) null, ServiceSwingResources.getInstance().getString("tooltip.path"));
	private final ExtensionComboBoxModel model;
	private RenderProfileDataRow profile;
	private boolean isInterrupted;
	private final AsyncService service;
	private final JComboBox combobox;

	/**
	 * @param frame
	 * @param service
	 * @param profile
	 */
	public EncoderDialog(final JFrame frame, final AsyncService service) {
		super(frame);
		this.service = service;
		setResizable(true);
		setUndecorated(false);
		// setCursor(new Cursor(Cursor.HAND_CURSOR));
		setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
		setTitle(ServiceSwingResources.getInstance().getString(EncoderDialog.STRING_FRAME_TITLE));
		// final Rectangle bounds = GraphicsEnvironment.getLocalGraphicsEnvironment().getMaximumWindowBounds();
		// final int width = Math.min(Integer.parseInt(ServiceResources.getInstance().getString(EncoderDialog.STRING_FRAME_WIDTH)), bounds.width);
		// final int height = Math.min(Integer.parseInt(ServiceResources.getInstance().getString(EncoderDialog.STRING_FRAME_HEIGHT)), bounds.height);
		final JPanel filePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		final JPanel encoderPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		final JPanel buttonsPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
		final Dimension textFieldSize = new Dimension(300, 24);
		pathTextField.setPreferredSize(textFieldSize);
		pathTextField.setMinimumSize(textFieldSize);
		pathTextField.setMaximumSize(textFieldSize);
		encoderPanel.setBorder(new TitledBorder(new EmptyBorder(0, 0, 0, 0), ServiceSwingResources.getInstance().getString("label.encoder")));
		filePanel.setBorder(new TitledBorder(new EmptyBorder(0, 0, 0, 0), ServiceSwingResources.getInstance().getString("label.path")));
		model = new ExtensionComboBoxModel(ServiceRegistry.getInstance().getEncoderRegistry(), false);
		combobox = GUIFactory.createComboBox(model, ServiceSwingResources.getInstance().getString("tooltip.encoder"));
		combobox.setRenderer(new ExtensionListCellRenderer());
		encoderPanel.add(combobox);
		buttonsPanel.add(profileExportButton);
		buttonsPanel.add(closeButton);
		filePanel.add(pathTextField);
		filePanel.add(chooseButton);
		final Box panel = Box.createVerticalBox();
		// profileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		profileChooser.setMultiSelectionEnabled(false);
		panel.setBorder(new EmptyBorder(8, 8, 8, 8));
		panel.add(encoderPanel);
		panel.add(Box.createVerticalStrut(8));
		panel.add(filePanel);
		panel.add(Box.createVerticalStrut(8));
		panel.add(buttonsPanel);
		getContentPane().setLayout(new BorderLayout());
		getContentPane().add(panel);
		pack();
		// setSize(new Dimension(width, height));
		final Point p = GraphicsEnvironment.getLocalGraphicsEnvironment().getCenterPoint();
		p.x -= getWidth() / 2;
		p.y -= getHeight() / 2;
		this.setLocation(p);
	}

	/**
	 * @param profile
	 */
	@SuppressWarnings("unchecked")
	public void setProfile(final RenderProfileDataRow profile) {
		this.profile = profile;
		model.reload(ServiceRegistry.getInstance().getEncoderRegistry(), new ExtensionFilter() {
			/**
			 * @see net.sf.jame.core.swing.extension.ExtensionFilter#accept(net.sf.jame.core.extension.Extension)
			 */
			public boolean accept(final Extension<?> extension) {
				try {
					if ((profile.getTotalFrames() == 0) && ((EncoderExtensionRuntime<?>) extension.createExtensionRuntime()).isMovieSupported()) {
						return false;
					}
				}
				catch (final ExtensionException e) {
					e.printStackTrace();
				}
				return true;
			}
		});
		if ((pathTextField.getText() != null) && (pathTextField.getText().trim().length() > 0)) {
			try {
				final Extension<EncoderExtensionRuntime> extension = (Extension<EncoderExtensionRuntime>) model.getRegistry().getExtension("service.encoder" + pathTextField.getText().substring(pathTextField.getText().indexOf(".")));
				model.setSelectedItemByExtensionId(extension.getExtensionId());
			}
			catch (final ExtensionNotFoundException e) {
				if (profile.getTotalFrames() == 0) {
					model.setSelectedItemByExtensionId("service.encoder.jpeg");
				}
				else {
					model.setSelectedItemByExtensionId("service.encoder.mov");
				}
			}
		}
		else {
			if (profile.getTotalFrames() == 0) {
				model.setSelectedItemByExtensionId("service.encoder.jpeg");
			}
			else {
				model.setSelectedItemByExtensionId("service.encoder.mov");
			}
		}
	}

	private class ExportAction extends AbstractAction {
		private static final long serialVersionUID = 1L;

		/**
		 * 
		 */
		public ExportAction() {
			super(ServiceSwingResources.getInstance().getString("action.export"));
		}

		/**
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(final ActionEvent e) {
			if ((pathTextField.getText() != null) && (pathTextField.getText().trim().length() == 0)) {
				JOptionPane.showMessageDialog(EncoderDialog.this, ServiceSwingResources.getInstance().getString("message.missingFile"), ServiceSwingResources.getInstance().getString("label.exportProfile"), JOptionPane.WARNING_MESSAGE);
			}
			else {
				final File file = new File(pathTextField.getText());
				if (file.exists()) {
					if (JOptionPane.showConfirmDialog(EncoderDialog.this, ServiceSwingResources.getInstance().getString("message.confirmOverwrite"), ServiceSwingResources.getInstance().getString("label.exportProfile"), JOptionPane.WARNING_MESSAGE) == JOptionPane.OK_OPTION) {
						final ProgressDialog dialog = new ProgressDialog();
						dialog.setVisible(true);
					}
				}
				else {
					final ProgressDialog dialog = new ProgressDialog();
					dialog.setVisible(true);
				}
			}
		}
	}

	private class ChooseAction extends AbstractAction {
		private static final long serialVersionUID = 1L;

		/**
		 * 
		 */
		public ChooseAction() {
			super(ServiceSwingResources.getInstance().getString("action.choose"));
		}

		/**
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		@SuppressWarnings("unchecked")
		public void actionPerformed(final ActionEvent e) {
			profileChooser.setDialogTitle(ServiceSwingResources.getInstance().getString("label.exportProfile"));
			final int returnVal = profileChooser.showSaveDialog(new JFrame());
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				pathTextField.setText(profileChooser.getSelectedFile().getAbsoluteFile().getPath());
				try {
					final EncoderExtensionRuntime encoder = ((ConfigurableExtension<EncoderExtensionRuntime, EncoderExtensionConfig>) combobox.getSelectedItem()).createExtensionRuntime();
					if (pathTextField.getText() != null && (encoder.isMovieSupported() && !pathTextField.getText().toLowerCase().endsWith(encoder.getMovieSuffix().toLowerCase()))) {
						pathTextField.setText(pathTextField.getText() + encoder.getMovieSuffix().toLowerCase());
					}
					else if (pathTextField.getText() != null && (encoder.isImageSupported() && !pathTextField.getText().toLowerCase().endsWith(encoder.getImageSuffix().toLowerCase()))) {
						pathTextField.setText(pathTextField.getText() + encoder.getImageSuffix().toLowerCase());
					}
				}
				catch (final Exception x) {
					x.printStackTrace();
				}
			}
		}
	}

	private class CancelAction extends AbstractAction {
		private static final long serialVersionUID = 1L;

		/**
		 * 
		 */
		public CancelAction() {
			super(ServiceSwingResources.getInstance().getString("action.cancel"));
		}

		/**
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(final ActionEvent e) {
			isInterrupted = true;
		}
	}

	private class CloseAction extends AbstractAction {
		private static final long serialVersionUID = 1L;

		/**
		 * 
		 */
		public CloseAction() {
			super(ServiceSwingResources.getInstance().getString("action.close"));
		}

		/**
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(final ActionEvent e) {
			setVisible(false);
		}
	}

	private class EncoderProgressListener implements ProgressListener {
		/**
		 * @param percent
		 */
		public void statusChanged(final int percent) {
			progressBar.setValue(percent);
		}
	}

	public class ProgressDialog extends JDialog {
		private static final long serialVersionUID = 1L;

		public ProgressDialog() {
			super(new JFrame());
			progressBar.setValue(0);
			setModal(true);
			setUndecorated(true);
			setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
			final JPanel buttonsPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
			buttonsPanel.add(cancelButton);
			final JPanel panel = new JPanel();
			panel.setLayout(new BorderLayout());
			panel.setBorder(new CompoundBorder(new LineBorder(Color.GRAY), new TitledBorder(new EmptyBorder(40, 40, 40, 40), ServiceSwingResources.getInstance().getString("message.export"))));
			panel.add(progressBar, BorderLayout.CENTER);
			panel.add(buttonsPanel, BorderLayout.SOUTH);
			getContentPane().setLayout(new BorderLayout());
			getContentPane().add(panel);
			pack();
			final Point p = GraphicsEnvironment.getLocalGraphicsEnvironment().getCenterPoint();
			p.x -= getWidth() / 2;
			p.y -= getHeight() / 2;
			this.setLocation(p);
			addWindowListener(new Dialoglistener());
		}

		private class Dialoglistener extends WindowAdapter {
			/**
			 * @see java.awt.event.WindowAdapter#windowOpened(java.awt.event.WindowEvent)
			 */
			@SuppressWarnings("unchecked")
			@Override
			public void windowOpened(final WindowEvent e) {
				try {
					final EncoderExtensionRuntime encoder = ((ConfigurableExtension<EncoderExtensionRuntime, EncoderExtensionConfig>) combobox.getSelectedItem()).createExtensionRuntime();
					isInterrupted = false;
					encoder.setInterruptHook(new EncoderHook() {
						/**
						 * @see net.sf.jame.service.encoder.EncoderHook#isInterrupted()
						 */
						public boolean isInterrupted() {
							return isInterrupted;
						}
					});
					final EncoderProgressListener listener = new EncoderProgressListener();
					if (encoder != null) {
						encoder.addProgressListener(listener);
						service.exportProfile(new ServiceVoidCallback() {
							/**
							 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
							 */
							public void executed() {
								encoder.removeProgressListener(listener);
								setVisible(false);
								dispose();
							}

							/**
							 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
							 */
							public void failed(final Throwable throwable) {
								JOptionPane.showMessageDialog(EncoderDialog.this, ServiceSwingResources.getInstance().getString("message.exportFailed"), ServiceSwingResources.getInstance().getString("label.exportProfile"), JOptionPane.ERROR_MESSAGE);
								EncoderDialog.logger.error("Can't export the profile", throwable);
								encoder.removeProgressListener(listener);
								setVisible(false);
								dispose();
							}
						}, profile, encoder, new File(pathTextField.getText()));
					}
					else {
						setVisible(false);
						dispose();
					}
				}
				catch (final ExtensionException x) {
					JOptionPane.showMessageDialog(EncoderDialog.this, ServiceSwingResources.getInstance().getString("message.exportFailed"), ServiceSwingResources.getInstance().getString("label.exportProfile"), JOptionPane.ERROR_MESSAGE);
					EncoderDialog.logger.error("Can't export the profile", x);
					x.printStackTrace();
				}
			}
		}
	}
}
