/*
 * $Id:DistributedJob.java 491 2008-01-28 21:59:31Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.spool.impl;

import java.awt.RenderingHints;
import java.awt.image.DataBufferInt;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.HashMap;
import java.util.Map;

import net.sf.jame.core.util.Colors;
import net.sf.jame.service.job.RenderJob;
import net.sf.jame.service.spool.DistributedJobInterface;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.twister.ImageTile;
import net.sf.jame.twister.IntegerVector2D;
import net.sf.jame.twister.TwisterClip;
import net.sf.jame.twister.TwisterClipController;
import net.sf.jame.twister.TwisterConfig;
import net.sf.jame.twister.TwisterRuntime;
import net.sf.jame.twister.renderer.DefaultTwisterRenderer;
import net.sf.jame.twister.renderer.OverlayTwisterRenderer;
import net.sf.jame.twister.renderer.Surface;
import net.sf.jame.twister.renderer.TwisterRenderer;
import net.sf.jame.twister.renderer.TwisterRenderingHints;

/**
 * @author Andrea Medeghini
 */
public class DistributedJob implements DistributedJobInterface {
	private final JobListener listener;
	private final String jobId;
	private long lastUpdate;
	private boolean started;
	private boolean aborted;
	private boolean terminated;
	private Thread thread;
	private TwisterClip clip;
	private RenderJob job;
	private byte[] jobData;
	private final File tmpDir;
	private File tmpFile;
	private int firstFrameNumber;

	/**
	 * @param tmpDir
	 * @param jobId
	 * @param listener
	 */
	public DistributedJob(final File tmpDir, final String jobId, final JobListener listener) {
		lastUpdate = System.currentTimeMillis();
		this.listener = listener;
		this.tmpDir = tmpDir;
		this.jobId = jobId;
		try {
			tmpFile = File.createTempFile("tmp_", ".bin", tmpDir);
		}
		catch (final IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getJobId()
	 */
	public String getJobId() {
		return jobId;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getFrameNumber()
	 */
	public int getFrameNumber() {
		return job != null ? job.getFrameNumber() : 0;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#setFrameNumber(int)
	 */
	public void setFrameNumber(final int frameNumber) {
		job.setFrameNumber(frameNumber);
		lastUpdate = System.currentTimeMillis();
		listener.stateChanged(this);
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#setFirstFrameNumber(int)
	 */
	public void setFirstFrameNumber(final int frameNumber) {
		firstFrameNumber = frameNumber;
		job.setFrameNumber(frameNumber);
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#getFirstFrameNumber()
	 */
	public int getFirstFrameNumber() {
		return firstFrameNumber;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#getClip()
	 */
	public TwisterClip getClip() {
		return clip;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#getJob()
	 */
	public RenderJob getJob() {
		return job;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#getJobData()
	 */
	public byte[] getJobData() {
		return jobData;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#setClip(net.sf.jame.twister.TwisterClip)
	 */
	public void setClip(final TwisterClip clip) {
		this.clip = clip;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#setJob(net.sf.jame.service.job.RenderJob)
	 */
	public void setJob(final RenderJob job) {
		this.job = job;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#setJobData(byte[])
	 */
	public void setJobData(final byte[] jobData) {
		this.jobData = jobData;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#getRAF()
	 */
	public RandomAccessFile getRAF() throws IOException {
		return new RandomAccessFile(tmpFile, "rw");
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedJobInterface#getStream()
	 */
	public InputStream getStream() throws IOException {
		return new FileInputStream(tmpFile);
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final StringBuilder builder = new StringBuilder();
		builder.append("id = ");
		builder.append(jobId);
		builder.append(", frameNumber = ");
		builder.append(getFrameNumber());
		return builder.toString();
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getLastUpdate()
	 */
	public long getLastUpdate() {
		return lastUpdate;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#reset()
	 */
	public void reset() {
		started = false;
		aborted = false;
		terminated = false;
		lastUpdate = System.currentTimeMillis();
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#start()
	 */
	public void start() {
		started = true;
		aborted = false;
		terminated = false;
		if (thread == null) {
			thread = new Thread(new RenderTask(), "DistributedJob Task");
			thread.setPriority(Thread.MIN_PRIORITY);
			thread.setDaemon(true);
			thread.start();
		}
		listener.started(this);
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#stop()
	 */
	public void stop() {
		started = false;
		if (thread != null) {
			// thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
		listener.stopped(this);
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#abort()
	 */
	public void abort() {
		aborted = true;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#dispose()
	 */
	public void dispose() {
		if (tmpFile != null) {
			tmpFile.delete();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isStarted()
	 */
	public boolean isStarted() {
		return started;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isAborted()
	 */
	public boolean isAborted() {
		return aborted;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isTerminated()
	 */
	public boolean isTerminated() {
		return terminated;
	}

	private void terminate() {
		listener.terminated(this);
	}

	private class RenderTask implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			final OutputStream os = null;
			try {
				if (clip.getSequenceCount() > 0) {
					final int frameCount = (job.getStopTime() - job.getStartTime()) * job.getFrameRate();
					int frameTimeInMillis = 0;
					final int tx = job.getTileOffsetX();
					final int ty = job.getTileOffsetY();
					final int tw = job.getTileWidth();
					final int th = job.getTileHeight();
					final int iw = job.getImageWidth();
					final int ih = job.getImageHeight();
					final int bw = job.getBorderWidth();
					final int bh = job.getBorderHeight();
					final int sw = tw + 2 * bw;
					final int sh = th + 2 * bh;
					if ((job.getFrameRate() == 0) || (frameCount == 0)) {
						TwisterConfig config = clip.getSequence(0).getInitialConfig();
						if (config == null) {
							config = clip.getSequence(0).getFinalConfig();
						}
						if (config != null) {
							final Surface surface = new Surface(sw, sh);
							surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
							surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
							surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
							surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
							final TwisterRuntime runtime = new TwisterRuntime(config);
							final TwisterRenderer renderer = new DefaultTwisterRenderer(runtime);
							final Map<Object, Object> hints = new HashMap<Object, Object>();
							hints.put(TwisterRenderingHints.KEY_MEMORY, TwisterRenderingHints.MEMORY_LOW);
							renderer.setRenderingHints(hints);
							renderer.setTile(new ImageTile(new IntegerVector2D(iw, ih), new IntegerVector2D(tw, th), new IntegerVector2D(tx, ty), new IntegerVector2D(bw, bh)));
							final TwisterRuntime overlayRuntime = new TwisterRuntime(config);
							final TwisterRenderer overlayRenderer = new OverlayTwisterRenderer(overlayRuntime);
							final Map<Object, Object> overlayHints = new HashMap<Object, Object>();
							overlayHints.put(TwisterRenderingHints.KEY_MEMORY, TwisterRenderingHints.MEMORY_LOW);
							overlayHints.put(TwisterRenderingHints.KEY_TYPE, TwisterRenderingHints.TYPE_OVERLAY);
							overlayRenderer.setRenderingHints(overlayHints);
							overlayRenderer.setTile(new ImageTile(new IntegerVector2D(iw, ih), new IntegerVector2D(tw, th), new IntegerVector2D(tx, ty), new IntegerVector2D(0, 0)));
							renderer.prepareImage(false);
							overlayRenderer.prepareImage(false);
							renderer.render();
							overlayRenderer.render();
							renderer.drawSurface(surface.getGraphics2D());
							overlayRenderer.drawSurface(surface.getGraphics2D());
							final byte[] row = new byte[sw * 4];
							final int[] data = ((DataBufferInt) surface.getImage().getRaster().getDataBuffer()).getData();
							final FileOutputStream fos = new FileOutputStream(tmpFile);
							for (int j = 0, k = 0; k < sh; k++) {
								for (int i = 0; i < row.length; i += 4) {
									row[i + 0] = (byte) ((data[j] & 0x00FF0000) >> 16);
									row[i + 1] = (byte) ((data[j] & 0x0000FF00) >> 8);
									row[i + 2] = (byte) ((data[j] & 0x000000FF) >> 0);
									row[i + 3] = (byte) ((data[j] & 0xFF000000) >> 24);
									j += 1;
								}
								fos.write(row);
								Thread.yield();
							}
							setFrameNumber(0);
							fos.close();
							overlayRenderer.dispose();
							overlayRuntime.dispose();
							renderer.dispose();
							runtime.dispose();
							surface.dispose();
						}
					}
					else if (job.getFrameNumber() < frameCount) {
						final Surface surface = new Surface(sw, sh);
						surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
						surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
						surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
						surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
						final TwisterClipController controller = new TwisterClipController(clip);
						controller.init();
						final TwisterConfig config = controller.getConfig();
						final TwisterRuntime runtime = new TwisterRuntime(config);
						final TwisterRenderer renderer = new DefaultTwisterRenderer(runtime);
						final Map<Object, Object> hints = new HashMap<Object, Object>();
						hints.put(TwisterRenderingHints.KEY_MEMORY, TwisterRenderingHints.MEMORY_LOW);
						renderer.setRenderingHints(hints);
						renderer.setTile(new ImageTile(new IntegerVector2D(iw, ih), new IntegerVector2D(tw, th), new IntegerVector2D(tx, ty), new IntegerVector2D(bw, bh)));
						final byte[] row = new byte[sw * 4];
						final int[] data = ((DataBufferInt) surface.getImage().getRaster().getDataBuffer()).getData();
						tmpFile = File.createTempFile("tmp_", ".bin", tmpDir);
						final FileOutputStream fos = new FileOutputStream(tmpFile);
						int startFrameNumber = 0;
						if (job.getFrameNumber() > 0 && jobData != null) {
							for (int j = 0, k = 0; k < sh; k++) {
								System.arraycopy(jobData, k * sw * 4, row, 0, row.length);
								for (int i = 0; i < row.length; i += 4) {
									final int argb = Colors.color(row[i + 3], row[i + 0], row[i + 1], row[i + 2]);
									data[j] = argb;
									j += 1;
								}
							}
							startFrameNumber = job.getFrameNumber() + 1;
							frameTimeInMillis = job.getStartTime() * 1000 + (job.getFrameNumber() * 1000) / job.getFrameRate();
							controller.redoAction(frameTimeInMillis, false);
							renderer.prepareImage(false);
							renderer.render();
							renderer.loadSurface(surface);
						}
						for (int frameNumber = startFrameNumber; frameNumber < frameCount; frameNumber++) {
							frameTimeInMillis = job.getStartTime() * 1000 + (frameNumber * 1000) / job.getFrameRate();
							controller.redoAction(frameTimeInMillis, false);
							renderer.prepareImage(false);
							renderer.render();
							renderer.drawSurface(surface.getGraphics2D());
							for (int j = 0, k = 0; k < sh; k++) {
								for (int i = 0; i < row.length; i += 4) {
									row[i + 0] = (byte) ((data[j] & 0x00FF0000) >> 16);
									row[i + 1] = (byte) ((data[j] & 0x0000FF00) >> 8);
									row[i + 2] = (byte) ((data[j] & 0x000000FF) >> 0);
									row[i + 3] = (byte) ((data[j] & 0xFF000000) >> 24);
									j += 1;
								}
								fos.write(row);
								Thread.yield();
							}
							setFrameNumber(frameNumber);
							if (((frameNumber + 1) % 20) == 0 && (frameCount - frameNumber - 1) > 10) {
								break;
							}
							if (aborted) {
								break;
							}
						}
						fos.close();
						renderer.dispose();
						runtime.dispose();
						surface.dispose();
					}
				}
			}
			catch (final Exception e) {
				aborted = true;
				e.printStackTrace();
			}
			finally {
				if (os != null) {
					try {
						os.close();
					}
					catch (final IOException e) {
					}
				}
			}
			terminated = true;
			terminate();
		}
	}
}
