/*
 * $Id:CopyProcessorJob.java 491 2008-01-28 21:59:31Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.spool.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;

import net.sf.jame.core.xml.XML;
import net.sf.jame.service.Service;
import net.sf.jame.service.job.RenderJobDataRow;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.service.spool.SpoolJobInterface;
import net.sf.jame.twister.TwisterClip;
import net.sf.jame.twister.TwisterClipXMLImporter;

import org.w3c.dom.Document;

/**
 * @author Andrea Medeghini
 */
public class CopyProcessorJob implements SpoolJobInterface {
	private final JobListener listener;
	private final String jobId;
	private long lastUpdate;
	private boolean started;
	private boolean aborted;
	private boolean terminated;
	private Thread thread;
	private RenderJobDataRow jobDataRow;
	private final Service service;

	/**
	 * @param service
	 * @param jobId
	 * @param listener
	 */
	public CopyProcessorJob(final Service service, final String jobId, final JobListener listener) {
		lastUpdate = System.currentTimeMillis();
		this.listener = listener;
		this.service = service;
		this.jobId = jobId;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getJobId()
	 */
	public String getJobId() {
		return jobId;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getFrameNumber()
	 */
	public int getFrameNumber() {
		return jobDataRow.getFrameNumber();
	}

	/**
	 * @param frameNumber
	 */
	public void setFrameNumber(final int frameNumber) {
		jobDataRow.setFrameNumber(frameNumber);
		lastUpdate = System.currentTimeMillis();
		listener.stateChanged(this);
	}

	/**
	 * @return the jobDataRow
	 */
	public RenderJobDataRow getJobDataRow() {
		return jobDataRow;
	}

	/**
	 * @param jobDataRow the jobDataRow to set
	 */
	public void setJobDataRow(final RenderJobDataRow jobDataRow) {
		this.jobDataRow = jobDataRow;
	}

	/**
	 * @see net.sf.jame.service.spool.SpoolJobInterface#getClip()
	 */
	public TwisterClip getClip() throws IOException {
		try {
			final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
			final InputStream is = service.getClipInputStream(jobDataRow.getClipId());
			final Document doc = XML.loadDocument(is, "twister-clip.xml");
			return importer.importFromElement(doc.getDocumentElement());
		}
		catch (final Exception e) {
			throw new IOException(e.getMessage());
		}
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final StringBuilder builder = new StringBuilder();
		builder.append("id = ");
		builder.append(jobId);
		builder.append(", frameNumber = ");
		builder.append(getFrameNumber());
		return builder.toString();
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getLastUpdate()
	 */
	public long getLastUpdate() {
		return lastUpdate;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#reset()
	 */
	public void reset() {
		started = false;
		aborted = false;
		terminated = false;
		lastUpdate = System.currentTimeMillis();
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#start()
	 */
	public void start() {
		started = true;
		aborted = false;
		terminated = false;
		if (thread == null) {
			thread = new Thread(new RenderTask(), "CopyProcessorJob Task");
			thread.setPriority(Thread.MIN_PRIORITY);
			thread.setDaemon(true);
			thread.start();
		}
		listener.started(this);
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#stop()
	 */
	public void stop() {
		started = false;
		if (thread != null) {
			thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
		listener.stopped(this);
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#abort()
	 */
	public void abort() {
		aborted = true;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#dispose()
	 */
	public void dispose() {
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isStarted()
	 */
	public boolean isStarted() {
		return started;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isAborted()
	 */
	public boolean isAborted() {
		return aborted;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isTerminated()
	 */
	public boolean isTerminated() {
		return terminated;
	}

	private void terminate() {
		listener.terminated(this);
	}

	private class RenderTask implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			RandomAccessFile raf = null;
			RandomAccessFile is = null;
			try {
				final int frameCount = (jobDataRow.getStopTime() - jobDataRow.getStartTime()) * jobDataRow.getFrameRate();
				if (frameCount == 0) {
					final int tx = jobDataRow.getTileOffsetX();
					final int ty = jobDataRow.getTileOffsetY();
					final int tw = jobDataRow.getTileWidth();
					final int th = jobDataRow.getTileHeight();
					final int bw = jobDataRow.getBorderWidth();
					final int bh = jobDataRow.getBorderHeight();
					final int iw = jobDataRow.getImageWidth();
					final int sw = tw + 2 * bw;
					final int sh = th + 2 * bh;
					final byte[] data = new byte[sw * sh * 4];
					is = service.getJobRandomAccessFile(jobDataRow.getJobId());
					is.readFully(data);
					raf = service.getProfileRandomAccessFile(jobDataRow.getProfileId());
					long pos = (ty * iw + tx) * 4;
					for (int j = sw * bh * 4 + bw * 4, k = 0; k < th; k++) {
						raf.seek(pos);
						raf.write(data, j, tw * 4);
						j += sw * 4;
						pos += iw * 4;
						Thread.yield();
					}
					setFrameNumber(0);
					raf.close();
					is.close();
				}
				else if (!jobDataRow.isPostProcess() && jobDataRow.getFrameNumber() < frameCount) {
					final int tx = jobDataRow.getTileOffsetX();
					final int ty = jobDataRow.getTileOffsetY();
					final int tw = jobDataRow.getTileWidth();
					final int th = jobDataRow.getTileHeight();
					final int bw = jobDataRow.getBorderWidth();
					final int bh = jobDataRow.getBorderHeight();
					final int iw = jobDataRow.getImageWidth();
					final int ih = jobDataRow.getImageHeight();
					final int sw = tw + 2 * bw;
					final int sh = th + 2 * bh;
					final byte[] data = new byte[sw * sh * 4];
					is = service.getJobRandomAccessFile(jobDataRow.getJobId());
					raf = service.getProfileRandomAccessFile(jobDataRow.getProfileId());
					int startFrameNumber = 0;
					if (jobDataRow.getFrameNumber() > 0) {
						startFrameNumber = jobDataRow.getFrameNumber() + 1;
					}
					long pos = (startFrameNumber * sw * sh) * 4;
					is.seek(pos);
					for (int frameNumber = startFrameNumber; frameNumber < frameCount; frameNumber++) {
						is.readFully(data);
						pos = (frameNumber * iw * ih + ty * iw + tx) * 4;
						for (int j = sw * bh * 4 + bw * 4, k = 0; k < th; k++) {
							raf.seek(pos);
							raf.write(data, j, tw * 4);
							j += sw * 4;
							pos += iw * 4;
							Thread.yield();
						}
						setFrameNumber(frameNumber);
						if (aborted) {
							break;
						}
					}
					raf.close();
					is.close();
				}
			}
			catch (final Exception e) {
				aborted = true;
				e.printStackTrace();
			}
			finally {
				if (raf != null) {
					try {
						raf.close();
					}
					catch (final IOException e) {
					}
				}
				if (is != null) {
					try {
						is.close();
					}
					catch (final IOException e) {
					}
				}
			}
			terminated = true;
			terminate();
		}
	}
}
