/*
 * $Id:JAIAbstractEncoderRuntime.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.extensions.encoder;

import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.image.DataBuffer;
import java.awt.image.Raster;
import java.awt.image.SinglePixelPackedSampleModel;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;

import javax.media.jai.ImageLayout;
import javax.media.jai.JAI;
import javax.media.jai.PlanarImage;
import javax.media.jai.SourcelessOpImage;
import javax.media.jai.TileComputationListener;
import javax.media.jai.TileRequest;

import net.sf.jame.service.Service;
import net.sf.jame.service.encoder.EncoderContext;
import net.sf.jame.service.encoder.EncoderException;
import net.sf.jame.service.encoder.EncoderHook;
import net.sf.jame.service.encoder.extension.EncoderExtensionConfig;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public abstract class JAIAbstractEncoderRuntime<T extends EncoderExtensionConfig> extends AbstractEncoderExtensionRuntime<T> {
	private static final Logger logger = Logger.getLogger(JAIAbstractEncoderRuntime.class);
	private EncoderHook hook;
	static {
		JAI.setDefaultTileSize(new Dimension(Service.TILE_WIDTH, Service.TILE_HEIGHT));
		JAI.getDefaultInstance().setTileCache(JAI.createTileCache(4096 * 1024));
	}

	/**
	 * @see net.sf.jame.service.encoder.extension.EncoderExtensionRuntime#setInterruptHook(net.sf.jame.service.encoder.EncoderHook)
	 */
	@Override
	public void setInterruptHook(final EncoderHook hook) {
		this.hook = hook;
	}

	/**
	 * @see net.sf.jame.service.encoder.extension.EncoderExtensionRuntime#encode(net.sf.jame.service.encoder.EncoderContext, java.io.File)
	 */
	@Override
	public void encode(final EncoderContext context, File path) throws EncoderException {
		fireStatusChanged(0);
		if (isImageSupported()) {
			long time = System.currentTimeMillis();
			final TileSourceOp src = new TileSourceOp(context);
			if (!path.getAbsolutePath().toLowerCase().endsWith(getImageSuffix())) {
				path = new File(path.getAbsolutePath() + getImageSuffix());
			}
			JAI.create("filestore", src, path.getAbsolutePath(), this.getFormatName(), null);
			time = System.currentTimeMillis() - time;
			if (JAIAbstractEncoderRuntime.logger.isInfoEnabled()) {
				JAIAbstractEncoderRuntime.logger.info("Profile exported: elapsed time " + String.format("%3.2f", time / 1000.0d) + "s");
			}
		}
		else {
			throw new EncoderException("Can't encode the image");
		}
	}

	/**
	 * @return
	 */
	protected abstract String getFormatName();

	@SuppressWarnings("unchecked")
	private class TileSourceOp extends SourcelessOpImage implements TileComputationListener {
		private final EncoderContext context;
		private int computedTilesCount;

		/**
		 * @param profile
		 * @param file
		 */
		public TileSourceOp(final EncoderContext context) {
			super(new ImageLayout(), new HashMap(), new SinglePixelPackedSampleModel(DataBuffer.TYPE_INT, context.getImageWidth(), context.getImageHeight(), (isAlphaSupported() ? new int[] { 0x00FF0000, 0x0000FF00, 0x000000FF, 0xFF000000 } : new int[] { 0x00FF0000, 0x0000FF00, 0x000000FF })), 0, 0, context.getImageWidth(), context.getImageHeight());
			addTileComputationListener(this);
			this.context = context;
		}

		/**
		 * @see javax.media.jai.SourcelessOpImage#computesUniqueTiles()
		 */
		@Override
		public boolean computesUniqueTiles() {
			return true;
		}

		/**
		 * @see javax.media.jai.OpImage#computeRect(javax.media.jai.PlanarImage[], java.awt.image.WritableRaster, java.awt.Rectangle)
		 */
		@Override
		protected void computeRect(final PlanarImage[] sources, final WritableRaster dest, final Rectangle destRect) {
			final int x = destRect.x;
			final int y = destRect.y;
			final int w = destRect.width;
			final int h = destRect.height;
			try {
				if (JAIAbstractEncoderRuntime.this.hook.isInterrupted()) {
					this.tileCancelled(this, null, this, this.XToTileX(x), this.YToTileY(y));
				}
				else {
					final int[] data = this.context.getTileAsIntArray(0, x, y, w, h, isAlphaSupported() ? 4 : 3);
					dest.setPixels(x, y, w, h, data);
					this.computedTilesCount += 1;
					this.tileComputed(this, null, this, this.XToTileX(x), this.YToTileY(y), dest);
				}
			}
			catch (final IOException e) {
				this.tileComputationFailure(this, null, this, this.XToTileX(x), this.YToTileY(y), e);
				e.printStackTrace();
			}
		}

		/**
		 * @see javax.media.jai.TileComputationListener#tileCancelled(java.lang.Object, javax.media.jai.TileRequest[], javax.media.jai.PlanarImage, int, int)
		 */
		public void tileCancelled(final Object eventSource, final TileRequest[] requests, final PlanarImage image, final int tileX, final int tileY) {
		}

		/**
		 * @see javax.media.jai.TileComputationListener#tileComputationFailure(java.lang.Object, javax.media.jai.TileRequest[], javax.media.jai.PlanarImage, int, int, java.lang.Throwable)
		 */
		public void tileComputationFailure(final Object eventSource, final TileRequest[] requests, final PlanarImage image, final int tileX, final int tileY, final Throwable situation) {
		}

		/**
		 * @see javax.media.jai.TileComputationListener#tileComputed(java.lang.Object, javax.media.jai.TileRequest[], javax.media.jai.PlanarImage, int, int, java.awt.image.Raster)
		 */
		public void tileComputed(final Object eventSource, final TileRequest[] requests, final PlanarImage image, final int tileX, final int tileY, final Raster tile) {
			fireStatusChanged((int) Math.rint((100f * this.computedTilesCount) / (getNumXTiles() * getNumYTiles())));
		}
	}
}
