/*
 * $Id:FileCache.java 495 2008-01-29 22:04:56Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;

import net.sf.jame.service.util.ZIPFilenameFilter;

/**
 * @author Andrea Medeghini
 */
public abstract class FileCache {
	private final File workdir;

	/**
	 * @param workdir
	 * @throws IOException
	 */
	public FileCache(final File workdir) throws IOException {
		this.workdir = workdir;
		workdir.mkdirs();
	}

	/**
	 * @return
	 */
	public File[] list() {
		return workdir.listFiles(createFileFilter());
	}

	/**
	 * @param data
	 * @return
	 */
	public abstract String getFileName(int id);

	/**
	 * @return
	 */
	public String getSuffix() {
		return ZIPFilenameFilter.getSuffix();
	}

	/**
	 * @param data
	 * @throws IOException
	 */
	public void create(final int id) throws IOException {
		final File file = getRealFile(id);
		if (!file.exists() && !file.createNewFile()) {
			throw new IOException("Can't create the file: " + file.getName());
		}
	}

	/**
	 * @param data
	 * @throws IOException
	 */
	public void delete(final int id) throws IOException {
		final File file = getRealFile(id);
		if (file.exists() && !file.delete()) {
			throw new IOException("Can't delete the file: " + file.getName());
		}
	}

	/**
	 * 
	 */
	public void deleteAll() {
		final File[] files = list();
		if (files != null) {
			for (final File file : files) {
				if (file.exists()) {
					file.delete();
				}
			}
		}
	}

	/**
	 * @param id
	 * @return
	 * @throws FileNotFoundException
	 */
	public InputStream getInputStream(final int id) throws FileNotFoundException {
		return new FileInputStream(getRealFile(id));
	}

	/**
	 * @param id
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getOutputStream(final int id) throws FileNotFoundException {
		return new FileOutputStream(getRealFile(id), false);
	}

	/**
	 * @param id
	 * @param append
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getOutputStream(final int id, final boolean append) throws FileNotFoundException {
		return new FileOutputStream(getRealFile(id), append);
	}

	/**
	 * @param id
	 * @return
	 * @throws FileNotFoundException
	 */
	public RandomAccessFile getRandomAccessFile(final int id) throws FileNotFoundException {
		return new RandomAccessFile(getRealFile(id), "rw");
	}

	/**
	 * @param data
	 * @return
	 */
	public boolean exists(final int id) {
		final File file = getRealFile(id);
		return file.exists();
	}

	/**
	 * @return
	 */
	protected File getRealFile(final int id) {
		return new File(workdir, getFileName(id) + getSuffix());
	}

	/**
	 * @return
	 */
	protected FilenameFilter createFileFilter() {
		return new ZIPFilenameFilter();
	}
}
