/*
 * $Id:Movie.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.media;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;

public class Movie extends AbstractGraphics implements Animate {
	private Controller engine;
	private Movie parent;
	private Layer layer;
	private Sequence sequence;
	private final Timeline timeline;
	private final String name;
	private int rate = 24;
	private final Point2D center;
	private final AffineTransform transform;

	public Movie(final String name, final Timeline timeline) {
		this.name = name;
		this.timeline = timeline;
		center = getCenter();
		transform = AffineTransform.getTranslateInstance(-center.getX(), -center.getY());
	}

	@Override
	public final Object clone() throws CloneNotSupportedException {
		return new Movie(name + "_copy", (Timeline) timeline.clone());
	}

	@Override
	public final AffineTransform getTransform() {
		return transform;
	}

	@Override
	public Point2D getCenter() {
		return new Point2D.Float(0f, 0f);
	}

	public Dimension getSize() {
		return new Dimension(640, 480);
	}

	public final Timeline getTimeline() {
		return timeline;
	}

	protected Layer getLayer() {
		return layer;
	}

	protected Sequence getSequence() {
		return sequence;
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public Movie getParent() {
		return parent;
	}

	public final Controller getEngine() {
		return engine;
	}

	@Override
	public final void build(final Controller engine, final Movie parent, final Layer layer, final Sequence sequence) {
		this.engine = engine;
		this.parent = parent;
		this.layer = layer;
		this.sequence = sequence;
		timeline.build(engine, this);
	}

	@Override
	public final void init() {
		timeline.init();
	}

	@Override
	public final void kill() {
		timeline.kill();
	}

	@Override
	public final void reset() {
		timeline.reset();
	}

	public int getFrames() {
		return timeline.getFrames();
	}

	public int getFrame() {
		return timeline.getFrame();
	}

	public final void setFrame(final int frame) {
		timeline.setFrame(frame);
	}

	public final void nextFrame() {
		timeline.nextFrame();
	}

	public final void prevFrame() {
		timeline.prevFrame();
	}

	public final int getFrameRate() {
		if (parent != null) {
			return parent.getFrameRate();
		}
		else {
			return rate;
		}
	}

	protected final void setFrameRate(final int rate) {
		this.rate = rate;
	}

	public void load(final Component component) {
	}

	public void flush() {
	}
}
