/*
 * $Id:XaosFractalRenderer.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform 
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is based on code written by Jan Hubicka and Thomas Marsh
 * http://xaos.sf.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.mandelbrot.renderer;

import java.awt.AlphaComposite;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.util.Map;

import net.sf.jame.core.math.Complex;
import net.sf.jame.core.util.Colors;
import net.sf.jame.mandelbrot.fractal.incolouring.IncolouringFormulaRuntimeElement;
import net.sf.jame.mandelbrot.fractal.outcolouring.OutcolouringFormulaRuntimeFormula;
import net.sf.jame.twister.renderer.Surface;

/**
 * @author Andrea Medeghini
 */
public final class XaosFractalRenderer extends AbstractFractalRenderer {
	private static final boolean DUMP = false;
	private static final boolean DUMP_XAOS = false;
	private static final boolean SHOW_REFRESH = false;
	private static final boolean SHOW_SYMETRY = false;
	private static final boolean SHOW_CALCULATE = false;
	private static final boolean SHOW_SOLIDGUESS = false;
	private static final boolean PRINT_REALLOCTABLE = false;
	private static final boolean PRINT_CALCULATE = false;
	private static final boolean PRINT_POSITIONS = false;
	private static final boolean PRINT_MOVETABLE = false;
	private static final boolean PRINT_FILLTABLE = false;
	private static final boolean PRINT_MULTABLE = false;
	private static final boolean PRINT_REGION = false;
	private static final boolean USE_XAOS = true;
	private static final boolean USE_SYMETRY = true;
	private static final boolean USE_SOLIDGUESS = true;
	private static final boolean USE_MULTITHREAD = true;
	private static final int GUESS_RANGE = 4;
	private static final int RANGES = 2;
	private static final int RANGE = 4;
	private static final int STEPS = 8;
	private static final int MASK = 0x7;
	private static final int DSIZE = (XaosFractalRenderer.RANGES + 1);
	private static final int FPMUL = 64;
	private static final int FPRANGE = XaosFractalRenderer.FPMUL * XaosFractalRenderer.RANGE;
	private static final int MAX_PRICE = Integer.MAX_VALUE;
	private static final int NEW_PRICE = XaosFractalRenderer.FPRANGE * XaosFractalRenderer.FPRANGE;
	private static final int[] multable = new int[XaosFractalRenderer.FPRANGE * 2];
	static {
		for (int i = -XaosFractalRenderer.FPRANGE; i < XaosFractalRenderer.FPRANGE; i++) {
			XaosFractalRenderer.multable[XaosFractalRenderer.FPRANGE + i] = i * i;
		}
		if (XaosFractalRenderer.PRINT_MULTABLE) {
			AbstractFractalRenderer.logger.debug("Multable:");
			for (int i = -XaosFractalRenderer.FPRANGE; i < XaosFractalRenderer.FPRANGE; i++) {
				AbstractFractalRenderer.logger.debug("i = " + i + ", i * i = " + XaosFractalRenderer.multable[XaosFractalRenderer.FPRANGE + i]);
			}
		}
	}
	private final RenderingStrategy mandelbrotRenderingStrategy = new MandelbrotRenderingStrategy();
	private final RenderingStrategy juliaRenderingStrategy = new JuliaRenderingStrategy();
	private final boolean isSolidguessEnabled = true;
	private boolean isSolidguessSupported = true;
	private final boolean isSymetryEnabled = true;
	private boolean isVerticalSymetrySupported = true;
	private boolean isHorizontalSymetrySupported = true;
	private boolean useCache = false;
	private boolean isAborted = false;
	private final RendererData renderedData;
	private final PrepareLinesTask prepareLinesTask = new PrepareLinesTask();
	private final PrepareColumnsTask prepareColumnsTask = new PrepareColumnsTask();

	/**
	 * 
	 */
	public XaosFractalRenderer(final int threadPriority) {
		super(threadPriority);
		renderedData = new RendererData();
		prepareLinesTask.start();
		prepareColumnsTask.start();
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer#dispose()
	 */
	@Override
	public void dispose() {
		prepareLinesTask.stop();
		prepareColumnsTask.stop();
		super.dispose();
	}

	/**
	 * @see net.sf.jame.explorer.core.fractal.renderer.AbstractFractalRenderer#free()
	 */
	protected void free() {
		super.free();
		if (renderedData != null) {
			renderedData.free();
		}
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#setRenderingHints(java.util.Map)
	 */
	public void setRenderingHints(final Map<Object, Object> hints) {
	}

	/**
	 * @see net.sf.jame.explorer.core.fractal.renderer.AbstractFractalRenderer#init()
	 */
	protected void init() {
		super.init();
		renderedData.reallocate(getBufferWidth(), getBufferHeight());
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer#setMode(int)
	 */
	public void setMode(final int renderMode) {
		this.renderMode |= renderMode;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer#isTileSupported()
	 */
	@Override
	protected boolean isTileSupported() {
		return true;
	}

	/**
	 * 
	 */
	protected void swapBuffers() {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Swap buffers...");
		}
		renderedData.swap();
	}

	/**
	 * @see net.sf.jame.explorer.core.fractal.renderer.AbstractFractalRenderer#doFractal(boolean)
	 */
	protected void doFractal(final boolean dynamic) {
		isAborted = false;
		updateShift();
		updateRegion();
		updateTransform();
		renderingStrategy.updateParameters();
		// if ((fractal.getRenderingFormula() == null) || (fractal.getRenderingFormula().getFormulaRuntime() == null)) {
		// return;
		// }
		// if ((fractal.getTransformingFormula() == null) || (fractal.getTransformingFormula().getFormulaRuntime() == null)) {
		// return;
		// }
		// if (fractal.getOutcolouringFormulaCount() == 0) {
		// return;
		// }
		// if (fractal.getIncolouringFormulaCount() == 0) {
		// return;
		// }
		// for (int i = 0; i < fractal.getOutcolouringFormulaCount(); i++) {
		// if (fractal.getOutcolouringFormula(i).getFormulaRuntime() == null) {
		// return;
		// }
		// }
		// for (int i = 0; i < fractal.getIncolouringFormulaCount(); i++) {
		// if (fractal.getIncolouringFormula(i).getFormulaRuntime() == null) {
		// return;
		// }
		// }
		if (fractal.getRenderingFormula().getFormulaRuntime() != null) {
			fractal.getRenderingFormula().getFormulaRuntime().prepareForRendering();
		}
		for (int i = 0; i < fractal.getOutcolouringFormulaCount(); i++) {
			if (fractal.getOutcolouringFormula(i).getFormulaRuntime() != null) {
				if (fractal.getOutcolouringFormula(i).isAutoIterations() && fractal.getRenderingFormula().getFormulaRuntime() != null) {
					fractal.getOutcolouringFormula(i).getFormulaRuntime().prepareForRendering(fractal.getRenderingFormula().getFormulaRuntime().getIterations());
				}
				else {
					fractal.getOutcolouringFormula(i).getFormulaRuntime().prepareForRendering(fractal.getOutcolouringFormula(i).getIterations());
				}
			}
		}
		for (int i = 0; i < fractal.getIncolouringFormulaCount(); i++) {
			if (fractal.getIncolouringFormula(i).getFormulaRuntime() != null) {
				if (fractal.getIncolouringFormula(i).isAutoIterations() && fractal.getRenderingFormula().getFormulaRuntime() != null) {
					fractal.getIncolouringFormula(i).getFormulaRuntime().prepareForRendering(fractal.getRenderingFormula().getFormulaRuntime().getIterations());
				}
				else {
					fractal.getIncolouringFormula(i).getFormulaRuntime().prepareForRendering(fractal.getIncolouringFormula(i).getIterations());
				}
			}
		}
		if (XaosFractalRenderer.PRINT_REGION) {
			AbstractFractalRenderer.logger.debug("Region: " + area.toString());
		}
		final boolean refresh = (renderMode & FractalRenderer.MODE_REFRESH) != 0;
		useCache = refresh || !dynamic;
		isSolidguessSupported = XaosFractalRenderer.USE_SOLIDGUESS && isSolidguessEnabled && isSolidGuessSupported();
		isVerticalSymetrySupported = XaosFractalRenderer.USE_SYMETRY && isSymetryEnabled && isVerticalSymetrySupported();
		isHorizontalSymetrySupported = XaosFractalRenderer.USE_SYMETRY && isSymetryEnabled && isHorizontalSymetrySupported();
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Solidguess supported = " + isSolidguessSupported);
			AbstractFractalRenderer.logger.debug("Vertical symetry supported = " + isVerticalSymetrySupported);
			AbstractFractalRenderer.logger.debug("Horizontal symetry supported = " + isHorizontalSymetrySupported);
			AbstractFractalRenderer.logger.debug("Use cache = " + useCache);
		}
		if (XaosFractalRenderer.USE_MULTITHREAD && !XaosFractalRenderer.DUMP_XAOS) {
			prepareLinesTask.execute();
			prepareColumnsTask.executeTask();
			try {
				prepareLinesTask.waitTask();
				prepareColumnsTask.waitTask();
			}
			catch (final InterruptedException e) {
			}
		}
		else {
			prepareLines();
			prepareColumns();
		}
		if (XaosFractalRenderer.PRINT_REALLOCTABLE) {
			AbstractFractalRenderer.logger.debug("ReallocTable:");
			for (final Realloc element : renderedData.reallocX) {
				AbstractFractalRenderer.logger.debug(element.toString());
			}
			AbstractFractalRenderer.logger.debug("ReallocTable:");
			for (final Realloc element : renderedData.reallocY) {
				AbstractFractalRenderer.logger.debug(element.toString());
			}
		}
		swapBuffers();
		move();
		processReallocTable(dynamic, refresh);
		updatePosition();
		renderMode = 0;
	}

	private void prepareLines() {
		final double beginy = area.points[0].i;
		final double endy = area.points[1].i;
		double stepy = 0;
		if (((renderMode & FractalRenderer.MODE_CALCULATE) == 0) && XaosFractalRenderer.USE_XAOS) {
			stepy = XaosFractalRenderer.makeReallocTable(renderedData.reallocY, renderedData.dynamicy, beginy, endy, renderedData.positionY, !useCache);
		}
		else {
			stepy = XaosFractalRenderer.initReallocTableAndPosition(renderedData.reallocY, renderedData.positionY, beginy, endy);
		}
		if (fractal.getRenderingFormula().getFormulaRuntime() != null && fractal.getTransformingFormula().getFormulaRuntime() != null) {
			final double symy = fractal.getRenderingFormula().getFormulaRuntime().getVerticalSymetryPoint();
			if (isVerticalSymetrySupported && fractal.getRenderingFormula().getFormulaRuntime().isVerticalSymetryAllowed() && fractal.getTransformingFormula().getFormulaRuntime().isVerticalSymetryAllowed() && (!((beginy > symy) || (symy > endy)))) {
				XaosFractalRenderer.prepareSymetry(renderedData.reallocY, (int) ((symy - beginy) / stepy), symy, stepy);
			}
		}
	}

	private void prepareColumns() {
		final double beginx = area.points[0].r;
		final double endx = area.points[1].r;
		double stepx = 0;
		if (((renderMode & FractalRenderer.MODE_CALCULATE) == 0) && XaosFractalRenderer.USE_XAOS) {
			stepx = XaosFractalRenderer.makeReallocTable(renderedData.reallocX, renderedData.dynamicx, beginx, endx, renderedData.positionX, !useCache);
		}
		else {
			stepx = XaosFractalRenderer.initReallocTableAndPosition(renderedData.reallocX, renderedData.positionX, beginx, endx);
		}
		if (fractal.getRenderingFormula().getFormulaRuntime() != null && fractal.getTransformingFormula().getFormulaRuntime() != null) {
			final double symx = fractal.getRenderingFormula().getFormulaRuntime().getHorizontalSymetryPoint();
			if (isHorizontalSymetrySupported && fractal.getRenderingFormula().getFormulaRuntime().isHorizontalSymetryAllowed() && fractal.getTransformingFormula().getFormulaRuntime().isHorizontalSymetryAllowed() && (!((beginx > symx) || (symx > endx)))) {
				XaosFractalRenderer.prepareSymetry(renderedData.reallocX, (int) ((symx - beginx) / stepx), symx, stepx);
			}
		}
	}

	private static double initReallocTableAndPosition(final Realloc[] realloc, final double[] position, final double begin, final double end) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Init ReallocTable and position...");
		}
		final double step = (end - begin) / realloc.length;
		double tmpPosition = begin;
		Realloc tmpRealloc = null;
		for (int i = 0; i < realloc.length; i++) {
			tmpRealloc = realloc[i];
			position[i] = tmpPosition;
			tmpRealloc.position = tmpPosition;
			tmpRealloc.recalculate = true;
			tmpRealloc.refreshed = false;
			tmpRealloc.dirty = true;
			tmpRealloc.isCached = false;
			tmpRealloc.plus = i;
			tmpRealloc.symTo = -1;
			tmpRealloc.symRef = -1;
			tmpPosition += step;
		}
		return step;
	}

	private void updatePosition() {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Update position...");
		}
		for (int k = 0; k < renderedData.reallocX.length; k++) {
			renderedData.positionX[k] = renderedData.reallocX[k].position;
		}
		for (int k = 0; k < renderedData.reallocY.length; k++) {
			renderedData.positionY[k] = renderedData.reallocY[k].position;
		}
	}

	private static int price(final int p1, final int p2) {
		return XaosFractalRenderer.multable[(XaosFractalRenderer.FPRANGE + p1) - p2];
	}

	private static void addPrices(final Realloc[] realloc, int r1, final int r2) {
		// if (r1 < r2)
		while (r1 < r2) {
			final int r3 = r1 + ((r2 - r1) >> 1);
			realloc[r3].priority = (realloc[r2].position - realloc[r3].position) * realloc[r3].priority;
			if (realloc[r3].symRef != -1) {
				realloc[r3].priority /= 2.0;
			}
			XaosFractalRenderer.addPrices(realloc, r1, r3);
			// XaosFractalRenderer.addPrices(realloc, r3 + 1, r2);
			r1 = r3 + 1;
		}
	}

	private static void prepareSymetry(final Realloc[] realloc, final int symi, double symPosition, final double step) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Prepare symetry...");
		}
		int i = 0;
		int j = 0;
		double tmp;
		double abs;
		double distance;
		double tmpPosition;
		final int size = realloc.length;
		final int max = size - XaosFractalRenderer.RANGE - 1;
		int min = XaosFractalRenderer.RANGE;
		int istart = 0;
		Realloc tmpRealloc = null;
		Realloc symRealloc = null;
		symPosition *= 2;
		int symj = (2 * symi) - size;
		if (symj < 0) {
			symj = 0;
		}
		distance = step * XaosFractalRenderer.RANGE;
		for (i = symj; i < symi; i++) {
			if (realloc[i].symTo != -1) {
				continue;
			}
			tmpRealloc = realloc[i];
			tmpPosition = tmpRealloc.position;
			tmpRealloc.symTo = (2 * symi) - i;
			if (tmpRealloc.symTo > max) {
				tmpRealloc.symTo = max;
			}
			j = ((tmpRealloc.symTo - istart) > XaosFractalRenderer.RANGE) ? (-XaosFractalRenderer.RANGE) : (-tmpRealloc.symTo + istart);
			if (tmpRealloc.recalculate) {
				while ((j < XaosFractalRenderer.RANGE) && ((tmpRealloc.symTo + j) < (size - 1))) {
					tmp = symPosition - realloc[tmpRealloc.symTo + j].position;
					abs = Math.abs(tmp - tmpPosition);
					if (abs < distance) {
						if (((i == 0) || (tmp > realloc[i - 1].position)) && (tmp < realloc[i + 1].position)) {
							distance = abs;
							min = j;
						}
					}
					else if (tmp < tmpPosition) {
						break;
					}
					j += 1;
				}
			}
			else {
				while ((j < XaosFractalRenderer.RANGE) && ((tmpRealloc.symTo + j) < (size - 1))) {
					if (tmpRealloc.recalculate) {
						tmp = symPosition - realloc[tmpRealloc.symTo + j].position;
						abs = Math.abs(tmp - tmpPosition);
						if (abs < distance) {
							if (((i == 0) || (tmp > realloc[i - 1].position)) && (tmp < realloc[i + 1].position)) {
								distance = abs;
								min = j;
							}
						}
						else if (tmp < tmpPosition) {
							break;
						}
					}
					j += 1;
				}
			}
			tmpRealloc.symTo += min;
			symRealloc = realloc[tmpRealloc.symTo];
			if ((min == XaosFractalRenderer.RANGE) || (tmpRealloc.symTo <= symi) || (symRealloc.symTo != -1) || (symRealloc.symRef != -1)) {
				tmpRealloc.symTo = -1;
				continue;
			}
			if (!tmpRealloc.recalculate) {
				tmpRealloc.symTo = -1;
				if ((symRealloc.symTo != -1) || !symRealloc.recalculate) {
					continue;
				}
				symRealloc.plus = tmpRealloc.plus;
				symRealloc.symTo = i;
				istart = tmpRealloc.symTo - 1;
				symRealloc.recalculate = false;
				symRealloc.refreshed = false;
				symRealloc.dirty = true;
				symRealloc.isCached = false;
				tmpRealloc.symRef = tmpRealloc.symTo;
				symRealloc.position = symPosition - tmpRealloc.position;
			}
			else {
				if (symRealloc.symTo != -1) {
					tmpRealloc.symTo = -1;
					continue;
				}
				tmpRealloc.plus = symRealloc.plus;
				istart = tmpRealloc.symTo - 1;
				tmpRealloc.recalculate = false;
				tmpRealloc.refreshed = false;
				tmpRealloc.dirty = true;
				tmpRealloc.isCached = false;
				symRealloc.symRef = i;
				tmpRealloc.position = symPosition - symRealloc.position;
			}
		}
	}

	private static void prepareMove(final Movetable movetable, final Realloc[] reallocX) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Prepare move...");
		}
		final Movetable.Data[] table = movetable.data;
		Movetable.Data tmpData = null;
		int i = 0;
		int j = 0;
		int s = 0;
		while (i < reallocX.length) {
			if (!reallocX[i].dirty) {
				tmpData = table[s];
				tmpData.to = i;
				tmpData.length = 1;
				tmpData.from = reallocX[i].plus;
				for (j = i + 1; j < reallocX.length; j++) {
					if (reallocX[j].dirty || ((j - reallocX[j].plus) != (tmpData.to - tmpData.from))) {
						break;
					}
					tmpData.length += 1;
				}
				i = j;
				s += 1;
			}
			else {
				i += 1;
			}
		}
		tmpData = table[s];
		tmpData.length = 0;
		if (XaosFractalRenderer.PRINT_MOVETABLE) {
			AbstractFractalRenderer.logger.debug("Movetable:");
			for (i = 0; table[i].length > 0; i++) {
				AbstractFractalRenderer.logger.debug("i = " + i + " " + table[i].toString());
			}
		}
	}

	private static void prepareFill(final Filltable filltable, final Realloc[] reallocX) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Prepare fill...");
		}
		final Filltable.Data[] table = filltable.data;
		Filltable.Data tmpData = null;
		int i = 0;
		int j = 0;
		int k = 0;
		int s = 0;
		int n = 0;
		for (i = 0; i < reallocX.length; i++) {
			if (reallocX[i].dirty) {
				j = i - 1;
				for (k = i + 1; (k < reallocX.length) && reallocX[k].dirty; k++) {
					;
				}
				while ((i < reallocX.length) && reallocX[i].dirty) {
					if ((k < reallocX.length) && ((j < i) || ((reallocX[i].position - reallocX[j].position) > (reallocX[k].position - reallocX[i].position)))) {
						j = k;
					}
					else {
						if (j < 0) {
							break;
						}
					}
					n = k - i;
					tmpData = table[s];
					tmpData.length = n;
					tmpData.from = j;
					tmpData.to = i;
					while (n > 0) {
						reallocX[i].position = reallocX[j].position;
						reallocX[i].dirty = false;
						n -= 1;
						i += 1;
					}
					s += 1;
				}
			}
		}
		tmpData = table[s];
		tmpData.length = 0;
		if (XaosFractalRenderer.PRINT_FILLTABLE) {
			AbstractFractalRenderer.logger.debug("Filltable:");
			for (i = 0; table[i].length > 0; i++) {
				AbstractFractalRenderer.logger.debug("i = " + i + " " + table[i].toString());
			}
		}
	}

	private static double makeReallocTable(final Realloc[] realloc, final Dynamic dynamic, final double begin, final double end, final double[] position, final boolean invalidate) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Make ReallocTable...");
		}
		Realloc tmpRealloc = null;
		Dynamic.Data prevData = null;
		Dynamic.Data bestData = null;
		Dynamic.Data tmpData = null;
		int bestPrice = XaosFractalRenderer.MAX_PRICE;
		int price = 0;
		int price1 = 0;
		int i = 0;
		int y = 0;
		int p = 0;
		int ps = 0;
		int pe = 0;
		int ps1 = 0;
		int yend = 0;
		int flag = 0;
		final int size = realloc.length;
		final double step = (end - begin) / size;
		final double tofix = (size * XaosFractalRenderer.FPMUL) / (end - begin);
		final int[] delta = dynamic.delta;
		delta[size] = Integer.MAX_VALUE;
		for (i = size - 1; i >= 0; i--) {
			delta[i] = (int) ((position[i] - begin) * tofix);
			if (delta[i] > delta[i + 1]) {
				delta[i] = delta[i + 1];
			}
		}
		if (XaosFractalRenderer.DUMP_XAOS) {
			AbstractFractalRenderer.logger.debug("positions (fixed point):");
			for (i = 0; i < size; i++) {
				AbstractFractalRenderer.logger.debug(String.valueOf(delta[i]));
			}
		}
		for (i = 0; i < size; i++) {
			dynamic.swap();
			yend = y - XaosFractalRenderer.FPRANGE;
			if (XaosFractalRenderer.DUMP_XAOS) {
				AbstractFractalRenderer.logger.debug("a0) yend = " + yend);
			}
			if (yend < 0) {
				yend = 0;
			}
			p = ps;
			while (delta[p] < yend) {
				p += 1;
			}
			ps1 = p;
			yend = y + XaosFractalRenderer.FPRANGE;
			if (XaosFractalRenderer.DUMP_XAOS) {
				AbstractFractalRenderer.logger.debug("a1) yend = " + yend);
			}
			if (XaosFractalRenderer.DUMP_XAOS) {
				AbstractFractalRenderer.logger.debug("b0) i = " + i + ", y = " + y + ", ps1 = " + ps1 + ", ps = " + ps + ", pe = " + pe);
			}
			if ((ps != pe) && (p > ps)) {
				if (p < pe) {
					prevData = dynamic.oldBest[p - 1];
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("c0) previous = " + prevData.toString());
					}
				}
				else {
					prevData = dynamic.oldBest[pe - 1];
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("c1) previous = " + prevData.toString());
					}
				}
				price1 = prevData.price;
			}
			else {
				if (i > 0) {
					prevData = dynamic.calData[i - 1];
					price1 = prevData.price;
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("c2) previous = " + prevData.toString());
					}
				}
				else {
					prevData = null;
					price1 = 0;
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("c3) previous = null");
					}
				}
			}
			tmpData = dynamic.calData[i];
			price = price1 + XaosFractalRenderer.NEW_PRICE;
			if (XaosFractalRenderer.DUMP_XAOS) {
				AbstractFractalRenderer.logger.debug("d0) add row/column " + i + ": price = " + price + " (previous price = " + price1 + ")");
			}
			bestData = tmpData;
			bestPrice = price;
			tmpData.price = price;
			tmpData.pos = -1;
			tmpData.previous = prevData;
			if (XaosFractalRenderer.DUMP_XAOS) {
				// Toolbox.println("d1) bestprice = " + bestprice + ", bestdata = " + bestdata.toString());
			}
			if (ps != pe) {
				if (p == ps) {
					if (delta[p] != delta[p + 1]) {
						prevData = dynamic.calData[i - 1];
						price1 = prevData.price;
						price = price1 + XaosFractalRenderer.price(delta[p], y);
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("g0) approximate row/column " + i + " with old row/column " + p + ": price = " + price + " (previous price = " + price1 + ")");
						}
						if (price < bestPrice) {
							tmpData = dynamic.conData[(p << XaosFractalRenderer.DSIZE) + (i & XaosFractalRenderer.MASK)];
							bestData = tmpData;
							bestPrice = price;
							tmpData.price = price;
							tmpData.pos = p;
							tmpData.previous = prevData;
							if (XaosFractalRenderer.DUMP_XAOS) {
								// Toolbox.println("g1) bestprice = " + bestprice + ", bestdata = " + bestdata.toString());
							}
						}
					}
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("g2) store data: p = " + p + ", bestdata = " + bestData.toString());
					}
					dynamic.newBest[p++] = bestData;
				}
				prevData = null;
				price1 = price;
				while (p < pe) {
					if (delta[p] != delta[p + 1]) {
						// if (prevData != dynamic.oldBest[p - 1])
						// {
						prevData = dynamic.oldBest[p - 1];
						price1 = prevData.price;
						price = price1 + XaosFractalRenderer.NEW_PRICE;
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("h0) add row/column " + i + ": price = " + price + " (previous price = " + price1 + ")");
						}
						if (price < bestPrice) {
							tmpData = dynamic.conData[((p - 1) << XaosFractalRenderer.DSIZE) + (i & XaosFractalRenderer.MASK)];
							bestData = tmpData;
							bestPrice = price;
							tmpData.price = price;
							tmpData.pos = -1;
							tmpData.previous = prevData;
							if (XaosFractalRenderer.DUMP_XAOS) {
								AbstractFractalRenderer.logger.debug("h1) store data: p - 1 = " + (p - 1) + ", bestdata = " + bestData.toString());
							}
							dynamic.newBest[p - 1] = bestData;
							if (XaosFractalRenderer.DUMP_XAOS) {
								// Toolbox.println("h2) bestprice = " + bestprice + ", bestdata = " + bestdata.toString());
							}
						}
						price = price1 + XaosFractalRenderer.price(delta[p], y);
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("h3) approximate row/column " + i + " with old row/column " + p + ": price = " + price + " (previous price = " + price1 + ")");
						}
						if (price < bestPrice) {
							tmpData = dynamic.conData[(p << XaosFractalRenderer.DSIZE) + (i & XaosFractalRenderer.MASK)];
							bestData = tmpData;
							bestPrice = price;
							tmpData.price = price;
							tmpData.pos = p;
							tmpData.previous = prevData;
							if (XaosFractalRenderer.DUMP_XAOS) {
								// Toolbox.println("h4) bestprice = " + bestprice + ", bestdata = " + bestdata.toString());
							}
						}
						else if (delta[p] > y) {
							if (XaosFractalRenderer.DUMP_XAOS) {
								AbstractFractalRenderer.logger.debug("h5) store data: p = " + p + ", bestdata = " + bestData.toString());
							}
							dynamic.newBest[p++] = bestData;
							break;
						}
						// }
					}
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("h6) store data: p = " + p + ", bestdata = " + bestData.toString());
					}
					dynamic.newBest[p++] = bestData;
				}
				while (p < pe) {
					if (delta[p] != delta[p + 1]) {
						// if (prevData != dynamic.oldBest[p - 1])
						// {
						prevData = dynamic.oldBest[p - 1];
						price1 = prevData.price;
						price = price1 + XaosFractalRenderer.NEW_PRICE;
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("i0) add row/column " + i + ": price = " + price + " (previous price = " + price1 + ")");
						}
						if (price < bestPrice) {
							tmpData = dynamic.conData[((p - 1) << XaosFractalRenderer.DSIZE) + (i & XaosFractalRenderer.MASK)];
							bestData = tmpData;
							bestPrice = price;
							tmpData.price = price;
							tmpData.pos = -1;
							tmpData.previous = prevData;
							if (XaosFractalRenderer.DUMP_XAOS) {
								AbstractFractalRenderer.logger.debug("i1) store data: p - 1 = " + (p - 1) + ", bestdata = " + bestData.toString());
							}
							dynamic.newBest[p - 1] = bestData;
							if (XaosFractalRenderer.DUMP_XAOS) {
								// Toolbox.println("i2) bestprice = " + bestprice + ", bestdata = " + bestdata.toString());
							}
						}
						price = price1 + XaosFractalRenderer.price(delta[p], y);
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("i3) add row/column " + i + ": price = " + price + " (previous price = " + price1 + ")");
						}
						if (price < bestPrice) {
							tmpData = dynamic.conData[(p << XaosFractalRenderer.DSIZE) + (i & XaosFractalRenderer.MASK)];
							bestData = tmpData;
							bestPrice = price;
							tmpData.price = price;
							tmpData.pos = p;
							tmpData.previous = prevData;
							if (XaosFractalRenderer.DUMP_XAOS) {
								AbstractFractalRenderer.logger.debug("i4) bestprice = " + bestPrice + ", bestdata = " + bestData.toString());
							}
						}
						// }
					}
					if (XaosFractalRenderer.DUMP_XAOS) {
						// Toolbox.println("i5) store data: p = " + p + ", bestdata = " + bestdata.toString());
					}
					dynamic.newBest[p++] = bestData;
				}
				if (p > ps) {
					prevData = dynamic.oldBest[p - 1];
					price1 = prevData.price;
				}
				else {
					prevData = dynamic.calData[i - 1];
					price1 = prevData.price;
				}
				price = price1 + XaosFractalRenderer.NEW_PRICE;
				if (XaosFractalRenderer.DUMP_XAOS) {
					AbstractFractalRenderer.logger.debug("l0) add row/column " + i + ": price = " + price + " (previous price = " + price1 + ")");
				}
				if ((price < bestPrice) && (p > ps1)) {
					tmpData = dynamic.conData[((p - 1) << XaosFractalRenderer.DSIZE) + (i & XaosFractalRenderer.MASK)];
					bestData = tmpData;
					bestPrice = price;
					tmpData.price = price;
					tmpData.pos = -1;
					tmpData.previous = prevData;
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("l1) store data: p - 1 = " + (p - 1) + ", bestdata = " + bestData.toString());
					}
					dynamic.newBest[p - 1] = bestData;
					if (XaosFractalRenderer.DUMP_XAOS) {
						// Toolbox.println("l2) bestprice = " + bestprice + ", bestdata = " + bestdata.toString());
					}
				}
				while (delta[p] < yend) {
					if (delta[p] != delta[p + 1]) {
						price = price1 + XaosFractalRenderer.price(delta[p], y);
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("l3) approximate row/column " + i + " with old row/column " + p + ": price = " + price + " (previous price = " + price1 + ")");
						}
						if (price < bestPrice) {
							tmpData = dynamic.conData[(p << XaosFractalRenderer.DSIZE) + (i & XaosFractalRenderer.MASK)];
							bestData = tmpData;
							bestPrice = price;
							tmpData.price = price;
							tmpData.pos = p;
							tmpData.previous = prevData;
							if (XaosFractalRenderer.DUMP_XAOS) {
								// Toolbox.println("l4) bestprice = " + bestprice + ", bestdata = " + bestdata.toString());
							}
						}
						else if (delta[p] > y) {
							break;
						}
					}
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("l5) store data: p = " + p + ", bestdata = " + bestData.toString());
					}
					dynamic.newBest[p++] = bestData;
				}
				while (delta[p] < yend) {
					if (XaosFractalRenderer.DUMP_XAOS) {
						AbstractFractalRenderer.logger.debug("l6) store data: p = " + p + ", bestdata = " + bestData.toString());
					}
					dynamic.newBest[p++] = bestData;
				}
			}
			else {
				if (delta[p] < yend) {
					if (i > 0) {
						prevData = dynamic.calData[i - 1];
						price1 = prevData.price;
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("e0) previous = " + prevData.toString());
						}
					}
					else {
						prevData = null;
						price1 = 0;
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("e1) previous = null");
						}
					}
					while (delta[p] < yend) {
						if (delta[p] != delta[p + 1]) {
							price = price1 + XaosFractalRenderer.price(delta[p], y);
							if (XaosFractalRenderer.DUMP_XAOS) {
								AbstractFractalRenderer.logger.debug("f0) approximate row/column " + i + " with old row/column " + p + ": price = " + price + " (previous price = " + price1 + ")");
							}
							if (price < bestPrice) {
								tmpData = dynamic.conData[(p << XaosFractalRenderer.DSIZE) + (i & XaosFractalRenderer.MASK)];
								bestData = tmpData;
								bestPrice = price;
								tmpData.price = price;
								tmpData.pos = p;
								tmpData.previous = prevData;
								if (XaosFractalRenderer.DUMP_XAOS) {
									// Toolbox.println("f1) bestprice = " + bestprice + ", bestdata = " + bestdata.toString());
								}
							}
							else if (delta[p] > y) {
								break;
							}
						}
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("f2) store data: p = " + p + ", bestdata = " + bestData.toString());
						}
						dynamic.newBest[p++] = bestData;
					}
					while (delta[p] < yend) {
						if (XaosFractalRenderer.DUMP_XAOS) {
							AbstractFractalRenderer.logger.debug("f3) store data: p = " + p + ", bestdata = " + bestData.toString());
						}
						dynamic.newBest[p++] = bestData;
					}
				}
			}
			ps = ps1;
			ps1 = pe;
			pe = p;
			y += XaosFractalRenderer.FPMUL;
		}
		if ((begin > delta[0]) && (end < delta[size - 1])) {
			flag = 1;
		}
		if ((delta[0] > 0) && (delta[size - 1] < (size * XaosFractalRenderer.FPMUL))) {
			flag = 2;
		}
		if (XaosFractalRenderer.DUMP_XAOS) {
			AbstractFractalRenderer.logger.debug("flag = " + flag);
		}
		if (XaosFractalRenderer.DUMP_XAOS) {
			AbstractFractalRenderer.logger.debug("best table:");
		}
		for (i = size - 1; i >= 0; i--) {
			if (XaosFractalRenderer.DUMP_XAOS) {
				AbstractFractalRenderer.logger.debug("data = " + bestData.toString());
			}
			tmpData = bestData.previous;
			tmpRealloc = realloc[i];
			tmpRealloc.symTo = -1;
			tmpRealloc.symRef = -1;
			if (bestData.pos < 0) {
				tmpRealloc.recalculate = true;
				tmpRealloc.refreshed = false;
				tmpRealloc.dirty = true;
				tmpRealloc.isCached = false;
				tmpRealloc.plus = tmpRealloc.pos;
			}
			else {
				tmpRealloc.plus = bestData.pos;
				tmpRealloc.position = position[bestData.pos];
				if (invalidate) {
					tmpRealloc.isCached = false;
				}
				tmpRealloc.recalculate = false;
				tmpRealloc.refreshed = false;
				tmpRealloc.dirty = false;
			}
			bestData = tmpData;
		}
		XaosFractalRenderer.newPositions(realloc, size, begin, end, step, position, flag);
		return step;
	}

	private static void newPositions(final Realloc[] realloc, final int size, double begin1, final double end1, final double step, final double[] position, final int flag) {
		Realloc tmpRealloc = null;
		double delta = 0;
		double begin = 0;
		double end = 0;
		final int l = size;
		int s = -1;
		int e = -1;
		if (begin1 > end1) {
			begin1 = end1;
		}
		if (XaosFractalRenderer.PRINT_POSITIONS) {
			AbstractFractalRenderer.logger.debug("Positions :");
		}
		while (s < (l - 1)) {
			e = s + 1;
			if (realloc[e].recalculate) {
				while (e < l) {
					if (!realloc[e].recalculate) {
						break;
					}
					e++;
				}
				if (e < l) {
					end = realloc[e].position;
				}
				else {
					end = end1;
				}
				if (s < 0) {
					begin = begin1;
				}
				else {
					begin = realloc[s].position;
				}
				if ((e == l) && (begin > end)) {
					end = begin;
				}
				if ((e - s) == 2) {
					delta = (end - begin) * 0.5;
				}
				else {
					delta = (end - begin) / (e - s);
				}
				switch (flag) {
					case 1: {
						for (s++; s < e; s++) {
							begin += delta;
							tmpRealloc = realloc[s];
							tmpRealloc.position = begin;
							tmpRealloc.priority = 1 / (1 + (Math.abs((position[s] - begin)) * step));
							if (XaosFractalRenderer.PRINT_POSITIONS) {
								AbstractFractalRenderer.logger.debug("pos = " + s + ",position = " + tmpRealloc.position + ",price = " + tmpRealloc.priority);
							}
						}
						break;
					}
					case 2: {
						for (s++; s < e; s++) {
							begin += delta;
							tmpRealloc = realloc[s];
							tmpRealloc.position = begin;
							tmpRealloc.priority = Math.abs((position[s] - begin)) * step;
							if (XaosFractalRenderer.PRINT_POSITIONS) {
								AbstractFractalRenderer.logger.debug("pos = " + s + ",position = " + tmpRealloc.position + ",price = " + tmpRealloc.priority);
							}
						}
						break;
					}
					default: {
						for (s++; s < e; s++) {
							begin += delta;
							tmpRealloc = realloc[s];
							tmpRealloc.position = begin;
							tmpRealloc.priority = 1.0;
							if (XaosFractalRenderer.PRINT_POSITIONS) {
								AbstractFractalRenderer.logger.debug("pos = " + s + ",position = " + tmpRealloc.position + ",price = " + tmpRealloc.priority);
							}
						}
						break;
					}
				}
			}
			s = e;
		}
	}

	private void processReallocTable(final boolean dynamic, final boolean refresh) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Process ReallocTable...");
		}
		if (dynamic || !XaosFractalRenderer.USE_XAOS) {
			int total = 0;
			total = XaosFractalRenderer.initPrices(renderedData.queue, total, renderedData.reallocX);
			total = XaosFractalRenderer.initPrices(renderedData.queue, total, renderedData.reallocY);
			if (XaosFractalRenderer.DUMP) {
				AbstractFractalRenderer.logger.debug("total = " + total);
			}
			if (total > 0) {
				if (total > 1) {
					XaosFractalRenderer.sortQueue(renderedData.queue, 0, total - 1);
				}
				processQueue(total);
			}
			if (XaosFractalRenderer.USE_XAOS) {
				processReallocTable(false, refresh);
			}
		}
		else {
			final int[] position = renderedData.position;
			final int[] offset = renderedData.offset;
			position[0] = 1;
			offset[0] = 0;
			int s = 1;
			int i = 0;
			int j = 0;
			int tocalcx = 0;
			int tocalcy = 0;
			Realloc[] tmpRealloc = null;
			tmpRealloc = renderedData.reallocX;
			for (i = 0; i < tmpRealloc.length; i++) {
				if (tmpRealloc[i].recalculate) {
					tocalcx++;
				}
			}
			tmpRealloc = renderedData.reallocY;
			for (i = 0; i < tmpRealloc.length; i++) {
				if (tmpRealloc[i].recalculate) {
					tocalcy++;
				}
			}
			for (i = 1; i < XaosFractalRenderer.STEPS; i++) {
				position[i] = 0;
			}
			while (s < XaosFractalRenderer.STEPS) {
				for (i = 0; i < XaosFractalRenderer.STEPS; i++) {
					if (position[i] == 0) {
						for (j = i; j < XaosFractalRenderer.STEPS; j++) {
							if (position[j] != 0) {
								break;
							}
						}
						position[offset[s] = (j + i) >> 1] = 1;
						s += 1;
					}
				}
			}
			// for (i = 0; i < position.length; i++)
			// {
			// System.out.println(i + " = " + position[i] + ", " + offset[i]);
			// }
			if (refresh) {
				tmpRealloc = renderedData.reallocY;
				for (final Realloc element : tmpRealloc) {
					if (element.isCached && !element.refreshed) {
						refreshLine(element, renderedData.reallocX, renderedData.reallocY);
					}
				}
				tmpRealloc = renderedData.reallocX;
				for (final Realloc element : tmpRealloc) {
					if (element.isCached && !element.refreshed) {
						refreshColumn(element, renderedData.reallocX, renderedData.reallocY);
					}
				}
			}
			renderedData.oldTime = renderedData.newTime = System.currentTimeMillis();
			for (s = 0; !isAborted && (s < XaosFractalRenderer.STEPS); s++) {
				// AbstractFractalRenderer.logger.debug("step = " + s);
				tmpRealloc = renderedData.reallocY;
				for (i = offset[s]; !isAborted && (i < tmpRealloc.length); i += XaosFractalRenderer.STEPS) {
					if (tmpRealloc[i].recalculate) {
						renderLine(tmpRealloc[i], renderedData.reallocX, renderedData.reallocY);
						tocalcy -= 1;
					}
					else if (!tmpRealloc[i].isCached) {
						renderLine(tmpRealloc[i], renderedData.reallocX, renderedData.reallocY);
					}
					if (isInterrupted()) {
						isAborted = true;
						break;
					}
					Thread.yield();
				}
				tmpRealloc = renderedData.reallocX;
				for (i = offset[s]; !isAborted && (i < tmpRealloc.length); i += XaosFractalRenderer.STEPS) {
					if (tmpRealloc[i].recalculate) {
						renderColumn(tmpRealloc[i], renderedData.reallocX, renderedData.reallocY);
						tocalcx -= 1;
					}
					else if (!tmpRealloc[i].isCached) {
						renderColumn(tmpRealloc[i], renderedData.reallocX, renderedData.reallocY);
					}
					if (isInterrupted()) {
						isAborted = true;
						break;
					}
					Thread.yield();
				}
				renderedData.newTime = System.currentTimeMillis();
				if (!isAborted && ((renderedData.newTime - renderedData.oldTime) > 50) && (s < XaosFractalRenderer.STEPS)) {
					tmpRealloc = renderedData.reallocY;
					for (i = 0; i < tmpRealloc.length; i++) {
						tmpRealloc[i].changeDirty = tmpRealloc[i].dirty;
						tmpRealloc[i].changePosition = tmpRealloc[i].position;
					}
					tmpRealloc = renderedData.reallocX;
					for (i = 0; i < tmpRealloc.length; i++) {
						tmpRealloc[i].changeDirty = tmpRealloc[i].dirty;
						tmpRealloc[i].changePosition = tmpRealloc[i].position;
					}
					percent = (int) (((s + 1) * 100) / (float) XaosFractalRenderer.STEPS);
					fill();
					copy();
					Thread.yield();
					tmpRealloc = renderedData.reallocY;
					for (i = 0; i < tmpRealloc.length; i++) {
						tmpRealloc[i].dirty = tmpRealloc[i].changeDirty;
						tmpRealloc[i].position = tmpRealloc[i].changePosition;
					}
					tmpRealloc = renderedData.reallocX;
					for (i = 0; i < tmpRealloc.length; i++) {
						tmpRealloc[i].dirty = tmpRealloc[i].changeDirty;
						tmpRealloc[i].position = tmpRealloc[i].changePosition;
					}
					renderedData.oldTime = renderedData.newTime;
				}
				// if (isInterrupted())
				// {
				// isAborted = true;
				// break;
				// }
			}
			if (!isAborted) {
				percent = 100;
			}
		}
		fill();
		copy();
		Thread.yield();
	}

	private void move() {
		XaosFractalRenderer.prepareMove(renderedData.moveTable, renderedData.reallocX);
		doMove(renderedData.moveTable, renderedData.reallocY);
	}

	private void fill() {
		if (isVerticalSymetrySupported && isHorizontalSymetrySupported) {
			doSymetry(renderedData.reallocX, renderedData.reallocY);
		}
		XaosFractalRenderer.prepareFill(renderedData.fillTable, renderedData.reallocX);
		doFill(renderedData.fillTable, renderedData.reallocY);
	}

	private void copy() {
		final Graphics2D g2d = getGraphics();
		g2d.setComposite(AlphaComposite.Src);
		g2d.drawImage(renderedData.newBuffer, 0, 0, null);
	}

	private static int initPrices(final Realloc[] queue, int total, final Realloc[] realloc) {
		int i = 0;
		int j = 0;
		for (i = 0; i < realloc.length; i++) {
			if (realloc[i].recalculate) {
				for (j = i; (j < realloc.length) && realloc[j].recalculate; j++) {
					queue[total++] = realloc[j];
				}
				if (j == realloc.length) {
					j -= 1;
				}
				XaosFractalRenderer.addPrices(realloc, i, j);
				i = j;
			}
		}
		return total;
	}

	private static void sortQueue(final Realloc[] queue, final int l, final int r) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Sort queue...");
		}
		final double m = (queue[l].priority + queue[r].priority) / 2.0;
		Realloc t = null;
		int i = l;
		int j = r;
		do {
			while (queue[i].priority > m) {
				i++;
			}
			while (queue[j].priority < m) {
				j--;
			}
			if (i <= j) {
				t = queue[i];
				queue[i] = queue[j];
				queue[j] = t;
				i++;
				j--;
			}
		}
		while (j >= i);
		if (l < j) {
			XaosFractalRenderer.sortQueue(queue, l, j);
		}
		if (r > i) {
			XaosFractalRenderer.sortQueue(queue, i, r);
		}
	}

	private void processQueue(final int size) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Process queue...");
		}
		int i = 0;
		for (i = 0; i < size; i++) {
			if (renderedData.queue[i].line) {
				renderLine(renderedData.queue[i], renderedData.reallocX, renderedData.reallocY);
			}
			else {
				renderColumn(renderedData.queue[i], renderedData.reallocX, renderedData.reallocY);
			}
			if (isInterrupted()) {
				isAborted = true;
				break;
			}
			Thread.yield();
		}
	}

	private void doSymetry(final Realloc[] reallocX, final Realloc[] reallocY) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Do symetry...");
		}
		final int rowsize = getBufferWidth();
		int from_offset = 0;
		int to_offset = 0;
		int i = 0;
		int j = 0;
		for (i = 0; i < reallocY.length; i++) {
			if ((reallocY[i].symTo >= 0) && (!reallocY[reallocY[i].symTo].dirty)) {
				from_offset = reallocY[i].symTo * rowsize;
				System.arraycopy(renderedData.newRGB, from_offset, renderedData.newRGB, to_offset, rowsize);
				if (useCache) {
					System.arraycopy(renderedData.newCacheR, from_offset, renderedData.newCacheR, to_offset, rowsize);
					System.arraycopy(renderedData.newCacheI, from_offset, renderedData.newCacheI, to_offset, rowsize);
					System.arraycopy(renderedData.newCacheTime, from_offset, renderedData.newCacheTime, to_offset, rowsize);
				}
				if (XaosFractalRenderer.SHOW_SYMETRY) {
					for (int k = 0; k < rowsize; k++) {
						renderedData.newRGB[to_offset + k] = Colors.mixColors(renderedData.newRGB[to_offset + k], 0xFFFF0000, 127);
					}
				}
				reallocY[i].dirty = false;
				reallocY[i].isCached = useCache;
			}
			to_offset += rowsize;
			// Thread.yield();
		}
		for (i = 0; i < reallocX.length; i++) {
			if ((reallocX[i].symTo >= 0) && (!reallocX[reallocX[i].symTo].dirty)) {
				to_offset = i;
				from_offset = reallocX[i].symTo;
				final int[] newRGB = renderedData.newRGB;
				final double[] newCacheR = renderedData.newCacheR;
				final double[] newCacheI = renderedData.newCacheI;
				final int[] newCacheTime = renderedData.newCacheTime;
				for (j = 0; j < reallocY.length; j++) {
					newRGB[to_offset] = newRGB[from_offset];
					if (useCache) {
						newCacheR[to_offset] = newCacheR[from_offset];
						newCacheI[to_offset] = newCacheI[from_offset];
						newCacheTime[to_offset] = newCacheTime[from_offset];
					}
					if (XaosFractalRenderer.SHOW_SYMETRY) {
						newRGB[to_offset] = Colors.mixColors(newRGB[to_offset], 0xFFFF0000, 127);
					}
					to_offset += rowsize;
					from_offset += rowsize;
				}
				reallocX[i].dirty = false;
				reallocX[i].isCached = useCache;
			}
			// Thread.yield();
		}
	}

	private void doMove(final Movetable movetable, final Realloc[] reallocY) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Do move...");
		}
		final Movetable.Data[] table = movetable.data;
		Movetable.Data tmpData = null;
		final int rowsize = getBufferWidth();
		int new_offset = 0;
		int old_offset = 0;
		int from = 0;
		int to = 0;
		int i = 0;
		int s = 0;
		for (i = 0; i < reallocY.length; i++) {
			if (!reallocY[i].dirty) {
				s = 0;
				old_offset = reallocY[i].plus * rowsize;
				while ((tmpData = table[s]).length > 0) {
					from = old_offset + tmpData.from;
					to = new_offset + tmpData.to;
					System.arraycopy(renderedData.oldRGB, from, renderedData.newRGB, to, tmpData.length);
					if (useCache) {
						System.arraycopy(renderedData.oldCacheR, from, renderedData.newCacheR, to, tmpData.length);
						System.arraycopy(renderedData.oldCacheI, from, renderedData.newCacheI, to, tmpData.length);
						System.arraycopy(renderedData.oldCacheTime, from, renderedData.newCacheTime, to, tmpData.length);
					}
					s += 1;
				}
			}
			new_offset += rowsize;
			// Thread.yield();
		}
	}

	private void doFill(final Filltable filltable, final Realloc[] reallocY) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Do fill...");
		}
		final Filltable.Data[] table = filltable.data;
		Filltable.Data tmpData = null;
		final int rowsize = getBufferWidth();
		int from_offset = 0;
		int to_offset = 0;
		int from = 0;
		int to = 0;
		int i = 0;
		int j = 0;
		int k = 0;
		int t = 0;
		int s = 0;
		int c = 0;
		int d = 0;
		// double c_xr = 0;
		// double c_xi = 0;
		// int c_time = 0;
		for (i = 0; i < reallocY.length; i++) {
			if (reallocY[i].dirty) {
				j = i - 1;
				for (k = i + 1; (k < reallocY.length) && reallocY[k].dirty; k++) {
					;
				}
				while ((i < reallocY.length) && reallocY[i].dirty) {
					if ((k < reallocY.length) && ((j < i) || ((reallocY[i].position - reallocY[j].position) > (reallocY[k].position - reallocY[i].position)))) {
						j = k;
					}
					else {
						if (j < 0) {
							break;
						}
					}
					to_offset = i * rowsize;
					from_offset = j * rowsize;
					if (!reallocY[j].dirty) {
						s = 0;
						final int[] newRGB = renderedData.newRGB;
						// final double[] newCacheR = renderedData.newCacheR;
						// final double[] newCacheI = renderedData.newCacheI;
						// final int[] newCacheTime = renderedData.newCacheTime;
						while ((tmpData = table[s]).length > 0) {
							from = from_offset + tmpData.from;
							to = from_offset + tmpData.to;
							c = newRGB[from];
							// if (useCache)
							// {
							// c_xr = newCacheR[from];
							// c_xi = newCacheI[from];
							// c_time = newCacheTime[from];
							// }
							for (t = 0; t < tmpData.length; t++) {
								d = to + t;
								newRGB[d] = c;
								// if (useCache)
								// {
								// newCacheR[d] = c_xr;
								// newCacheI[d] = c_xi;
								// newCacheTime[d] = c_time;
								// }
							}
							s += 1;
						}
					}
					System.arraycopy(renderedData.newRGB, from_offset, renderedData.newRGB, to_offset, rowsize);
					// if (useCache)
					// {
					// System.arraycopy(renderedData.newCacheR, from_offset, renderedData.newCacheR, to_offset, rowsize);
					// System.arraycopy(renderedData.newCacheI, from_offset, renderedData.newCacheI, to_offset, rowsize);
					// System.arraycopy(renderedData.newCacheTime, from_offset, renderedData.newCacheTime, to_offset, rowsize);
					// }
					reallocY[i].position = reallocY[j].position;
					// reallocY[i].isCached = false;
					reallocY[i].dirty = false;
					i += 1;
				}
			}
			else {
				s = 0;
				from_offset = i * rowsize;
				final int[] newRGB = renderedData.newRGB;
				// final double[] newCacheR = renderedData.newCacheR;
				// final double[] newCacheI = renderedData.newCacheI;
				// final int[] newCacheTime = renderedData.newCacheTime;
				while ((tmpData = table[s]).length > 0) {
					from = from_offset + tmpData.from;
					to = from_offset + tmpData.to;
					c = newRGB[from];
					// if (useCache)
					// {
					// c_xr = newCacheR[from];
					// c_xi = newCacheI[from];
					// c_time = newCacheTime[from];
					// }
					for (t = 0; t < tmpData.length; t++) {
						d = to + t;
						newRGB[d] = c;
						// if (useCache)
						// {
						// newCacheR[d] = c_xr;
						// newCacheI[d] = c_xi;
						// newCacheTime[d] = c_time;
						// }
					}
					s += 1;
				}
				// reallocY[i].isCached = false;
				reallocY[i].dirty = false;
			}
			// Thread.yield();
		}
	}

	private void renderLine(final Realloc realloc, final Realloc[] reallocX, final Realloc[] reallocY) {
		if (XaosFractalRenderer.PRINT_CALCULATE) {
			AbstractFractalRenderer.logger.debug("Calculate line " + realloc.pos);
		}
		final int rowsize = getBufferWidth();
		final double position = realloc.position;
		final int r = realloc.pos;
		int offset = r * rowsize;
		int i;
		int j;
		int k;
		int n;
		double n_xr = 0;
		double n_xi = 0;
		int n_time = 0;
		int distl = 0;
		int distr = 0;
		int distu = 0;
		int distd = 0;
		int offsetu;
		int offsetd;
		int offsetl;
		int offsetul;
		int offsetur;
		int offsetdl;
		int offsetdr;
		int rend = r - XaosFractalRenderer.GUESS_RANGE;
		final Complex z = new Complex(0, 0);
		final Complex w = new Complex(0, 0);
		final RenderedPoint p = new RenderedPoint();
		final int[] newRGB = renderedData.newRGB;
		final double[] newCacheR = renderedData.newCacheR;
		final double[] newCacheI = renderedData.newCacheI;
		final int[] newCacheTime = renderedData.newCacheTime;
		if (rend < 0) {
			rend = 0;
		}
		for (i = r - 1; (i >= rend) && reallocY[i].dirty; i--) {
			;
		}
		distu = r - i;
		rend = r + XaosFractalRenderer.GUESS_RANGE;
		if (rend >= reallocY.length) {
			rend = reallocY.length - 1;
		}
		for (j = r + 1; (j < rend) && reallocY[j].dirty; j++) {
			;
		}
		distd = j - r;
		if ((!isSolidguessSupported) || (i < 0) || (j >= reallocY.length) || reallocY[i].dirty || reallocY[j].dirty) {
			for (k = 0; k < reallocX.length; k++) {
				if (!reallocX[k].dirty) {
					z.r = renderedData.x0;
					z.i = renderedData.y0;
					w.r = reallocX[k].position;
					w.i = position;
					p.tr = reallocX[k].position;
					p.ti = position;
					newRGB[offset] = renderingStrategy.renderPoint(p, z, w);
					if (useCache) {
						newCacheR[offset] = p.zr;
						newCacheI[offset] = p.zi;
						newCacheTime[offset] = p.time;
					}
					if (XaosFractalRenderer.SHOW_CALCULATE) {
						newRGB[offset] = Colors.mixColors(newRGB[offset], 0xFFFFFF00, 127);
					}
				}
				offset += 1;
			}
		}
		else {
			distr = 0;
			distl = Integer.MAX_VALUE / 2;
			offsetu = offset - (distu * rowsize);
			offsetd = offset + (distd * rowsize);
			for (k = 0; k < reallocX.length; k++) {
				if (!reallocX[k].dirty) {
					if (distr <= 0) {
						rend = k + XaosFractalRenderer.GUESS_RANGE;
						if (rend >= reallocX.length) {
							rend = reallocX.length - 1;
						}
						for (j = k + 1; (j < rend) && reallocX[j].dirty; j++) {
							distr = j - k;
						}
						if (j >= rend) {
							distr = Integer.MAX_VALUE / 2;
						}
					}
					if ((distr < (Integer.MAX_VALUE / 4)) && (distl < (Integer.MAX_VALUE / 4))) {
						offsetl = offset - distl;
						offsetul = offsetu - distl;
						offsetdl = offsetd - distl;
						offsetur = offsetu + distr;
						offsetdr = offsetd + distr;
						n = newRGB[offsetl];
						if (useCache) {
							n_xr = newCacheR[offsetl];
							n_xi = newCacheI[offsetl];
							n_time = newCacheTime[offsetl];
						}
						if ((n == newRGB[offsetu]) && (n == newRGB[offsetd]) && (n == newRGB[offsetul]) && (n == newRGB[offsetur]) && (n == newRGB[offsetdl]) && (n == newRGB[offsetdr])) {
							newRGB[offset] = n;
							if (useCache) {
								newCacheR[offset] = n_xr;
								newCacheI[offset] = n_xi;
								newCacheTime[offset] = n_time;
							}
							if (XaosFractalRenderer.SHOW_SOLIDGUESS) {
								newRGB[offset] = Colors.mixColors(newRGB[offset], 0xFFFF0000, 127);
							}
						}
						else {
							z.r = renderedData.x0;
							z.i = renderedData.y0;
							w.r = reallocX[k].position;
							w.i = position;
							p.tr = reallocX[k].position;
							p.ti = position;
							newRGB[offset] = renderingStrategy.renderPoint(p, z, w);
							if (useCache) {
								newCacheR[offset] = p.zr;
								newCacheI[offset] = p.zi;
								newCacheTime[offset] = p.time;
							}
							if (XaosFractalRenderer.SHOW_CALCULATE) {
								newRGB[offset] = Colors.mixColors(newRGB[offset], 0xFFFFFF00, 127);
							}
						}
					}
					else {
						z.r = renderedData.x0;
						z.i = renderedData.y0;
						w.r = reallocX[k].position;
						w.i = position;
						p.tr = reallocX[k].position;
						p.ti = position;
						newRGB[offset] = renderingStrategy.renderPoint(p, z, w);
						if (useCache) {
							newCacheR[offset] = p.zr;
							newCacheI[offset] = p.zi;
							newCacheTime[offset] = p.time;
						}
						if (XaosFractalRenderer.SHOW_CALCULATE) {
							newRGB[offset] = Colors.mixColors(newRGB[offset], 0xFFFFFF00, 127);
						}
					}
					distl = 0;
				}
				offset += 1;
				offsetu += 1;
				offsetd += 1;
				distr -= 1;
				distl += 1;
			}
		}
		realloc.recalculate = false;
		realloc.refreshed = true;
		realloc.dirty = false;
		realloc.isCached = useCache;
	}

	private void renderColumn(final Realloc realloc, final Realloc[] reallocX, final Realloc[] reallocY) {
		if (XaosFractalRenderer.PRINT_CALCULATE) {
			AbstractFractalRenderer.logger.debug("Calculate column " + realloc.pos);
		}
		final int rowsize = getBufferWidth();
		final double position = realloc.position;
		final int r = realloc.pos;
		int offset = r;
		int rend = r - XaosFractalRenderer.GUESS_RANGE;
		int i;
		int j;
		int k;
		int n;
		double n_xr = 0;
		double n_xi = 0;
		int n_time = 0;
		int distl = 0;
		int distr = 0;
		int distu = 0;
		int distd = 0;
		int offsetl;
		int offsetr;
		int offsetu;
		int offsetlu;
		int offsetru;
		int offsetld;
		int offsetrd;
		int sumu;
		int sumd;
		final Complex z = new Complex(0, 0);
		final Complex w = new Complex(0, 0);
		final RenderedPoint p = new RenderedPoint();
		final int[] newRGB = renderedData.newRGB;
		final double[] newCacheR = renderedData.newCacheR;
		final double[] newCacheI = renderedData.newCacheI;
		final int[] newCacheTime = renderedData.newCacheTime;
		if (rend < 0) {
			rend = 0;
		}
		for (i = r - 1; (i >= rend) && reallocX[i].dirty; i--) {
			;
		}
		distl = r - i;
		rend = r + XaosFractalRenderer.GUESS_RANGE;
		if (rend >= reallocX.length) {
			rend = reallocX.length - 1;
		}
		for (j = r + 1; (j < rend) && reallocX[j].dirty; j++) {
			;
		}
		distr = j - r;
		if ((!isSolidguessSupported) || (i < 0) || (j >= reallocX.length) || reallocX[i].dirty || reallocX[j].dirty) {
			for (k = 0; k < reallocY.length; k++) {
				if (!reallocY[k].dirty) {
					z.r = renderedData.x0;
					z.i = renderedData.y0;
					w.r = position;
					w.i = reallocY[k].position;
					p.tr = position;
					p.ti = reallocY[k].position;
					newRGB[offset] = renderingStrategy.renderPoint(p, z, w);
					if (useCache) {
						newCacheR[offset] = p.zr;
						newCacheI[offset] = p.zi;
						newCacheTime[offset] = p.time;
					}
					if (XaosFractalRenderer.SHOW_CALCULATE) {
						newRGB[offset] = Colors.mixColors(newRGB[offset], 0xFFFFFF00, 127);
					}
				}
				offset += rowsize;
			}
		}
		else {
			distd = 0;
			distu = Integer.MAX_VALUE / 2;
			offsetl = offset - distl;
			offsetr = offset + distr;
			for (k = 0; k < reallocY.length; k++) {
				if (!reallocY[k].dirty) {
					if (distd <= 0) {
						rend = k + XaosFractalRenderer.GUESS_RANGE;
						if (rend >= reallocY.length) {
							rend = reallocY.length - 1;
						}
						for (j = k + 1; (j < rend) && reallocY[j].dirty; j++) {
							distd = j - k;
						}
						if (j >= rend) {
							distd = Integer.MAX_VALUE / 2;
						}
					}
					if ((distd < (Integer.MAX_VALUE / 4)) && (distu < (Integer.MAX_VALUE / 4))) {
						sumu = distu * rowsize;
						sumd = distd * rowsize;
						offsetu = offset - sumu;
						offsetlu = offsetl - sumu;
						offsetru = offsetr - sumu;
						offsetld = offsetl + sumd;
						offsetrd = offsetr + sumd;
						n = newRGB[offsetu];
						if (useCache) {
							n_xr = newCacheR[offsetu];
							n_xi = newCacheI[offsetu];
							n_time = newCacheTime[offsetu];
						}
						if ((n == newRGB[offsetl]) && (n == newRGB[offsetr]) && (n == newRGB[offsetlu]) && (n == newRGB[offsetru]) && (n == newRGB[offsetld]) && (n == newRGB[offsetrd])) {
							newRGB[offset] = n;
							if (useCache) {
								newCacheR[offset] = n_xr;
								newCacheI[offset] = n_xi;
								newCacheTime[offset] = n_time;
							}
							if (XaosFractalRenderer.SHOW_SOLIDGUESS) {
								newRGB[offset] = Colors.mixColors(newRGB[offset], 0xFFFF0000, 127);
							}
						}
						else {
							z.r = renderedData.x0;
							z.i = renderedData.y0;
							w.r = position;
							w.i = reallocY[k].position;
							p.tr = position;
							p.ti = reallocY[k].position;
							newRGB[offset] = renderingStrategy.renderPoint(p, z, w);
							if (useCache) {
								newCacheR[offset] = p.zr;
								newCacheI[offset] = p.zi;
								newCacheTime[offset] = p.time;
							}
							if (XaosFractalRenderer.SHOW_CALCULATE) {
								newRGB[offset] = Colors.mixColors(newRGB[offset], 0xFFFFFF00, 127);
							}
						}
					}
					else {
						z.r = renderedData.x0;
						z.i = renderedData.y0;
						w.r = position;
						w.i = reallocY[k].position;
						p.tr = position;
						p.ti = reallocY[k].position;
						newRGB[offset] = renderingStrategy.renderPoint(p, z, w);
						if (useCache) {
							newCacheR[offset] = p.zr;
							newCacheI[offset] = p.zi;
							newCacheTime[offset] = p.time;
						}
						if (XaosFractalRenderer.SHOW_CALCULATE) {
							newRGB[offset] = Colors.mixColors(newRGB[offset], 0xFFFFFF00, 127);
						}
					}
					distu = 0;
				}
				offset += rowsize;
				offsetl += rowsize;
				offsetr += rowsize;
				distd -= 1;
				distu += 1;
			}
		}
		realloc.recalculate = false;
		realloc.refreshed = true;
		realloc.dirty = false;
		realloc.isCached = useCache;
	}

	private void refreshLine(final Realloc realloc, final Realloc[] reallocX, final Realloc[] reallocY) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Refresh line...");
		}
		final int rowsize = getBufferWidth();
		int offset = realloc.pos * rowsize;
		int k = 0;
		final RenderedPoint p = new RenderedPoint();
		final int[] newRGB = renderedData.newRGB;
		final double[] newCacheR = renderedData.newCacheR;
		final double[] newCacheI = renderedData.newCacheI;
		final int[] newCacheTime = renderedData.newCacheTime;
		if (realloc.isCached && !realloc.refreshed) {
			for (final Realloc tmpRealloc : reallocX) {
				if (tmpRealloc.isCached && !tmpRealloc.refreshed) {
					k = offset;
					p.zr = newCacheR[k];
					p.zi = newCacheI[k];
					p.time = newCacheTime[k];
					p.tr = tmpRealloc.position;
					p.ti = realloc.position;
					newRGB[k] = renderColor(p);
					if (XaosFractalRenderer.SHOW_REFRESH) {
						newRGB[k] = Colors.mixColors(newRGB[k], 0xFF0000FF, 127);
					}
				}
				offset += 1;
			}
			realloc.refreshed = true;
		}
	}

	private void refreshColumn(final Realloc realloc, final Realloc[] reallocX, final Realloc[] reallocY) {
		if (XaosFractalRenderer.DUMP) {
			AbstractFractalRenderer.logger.debug("Refresh column...");
		}
		final int rowsize = getBufferWidth();
		int offset = realloc.pos;
		int k = 0;
		final RenderedPoint p = new RenderedPoint();
		final int[] newRGB = renderedData.newRGB;
		final double[] newCacheR = renderedData.newCacheR;
		final double[] newCacheI = renderedData.newCacheI;
		final int[] newCacheTime = renderedData.newCacheTime;
		if (realloc.isCached && !realloc.refreshed) {
			for (final Realloc tmpRealloc : reallocY) {
				if (tmpRealloc.isCached && !tmpRealloc.refreshed) {
					k = offset;
					p.zr = newCacheR[k];
					p.zi = newCacheI[k];
					p.time = newCacheTime[k];
					p.tr = realloc.position;
					p.ti = tmpRealloc.position;
					newRGB[k] = renderColor(p);
					if (XaosFractalRenderer.SHOW_REFRESH) {
						newRGB[k] = Colors.mixColors(newRGB[k], 0xFF0000FF, 127);
					}
				}
				offset += rowsize;
			}
			realloc.refreshed = true;
		}
	}

	/**
	 * @see net.sf.jame.explorer.core.fractal.renderer.AbstractFractalRenderer#getMandelbrotRenderingStrategy()
	 */
	@Override
	protected RenderingStrategy getMandelbrotRenderingStrategy() {
		return mandelbrotRenderingStrategy;
	}

	/**
	 * @see net.sf.jame.explorer.core.fractal.renderer.AbstractFractalRenderer#createJuliaRenderingStrategy()
	 */
	@Override
	protected RenderingStrategy getJuliaRenderingStrategy() {
		return juliaRenderingStrategy;
	}

	/**
	 * @author Andrea Medeghini
	 */
	public static class Movetable {
		/**
		 * 
		 */
		public Data[] data;

		/**
		 * @param width
		 */
		public Movetable(final int width) {
			data = new Data[width + 1];
			for (int i = 0; i <= width; i++) {
				data[i] = new Data();
			}
		}

		/**
		 * @see java.lang.Object#finalize()
		 */
		public void finalize() throws Throwable {
			data = null;
			super.finalize();
		}

		/**
		 * @author Andrea Medeghini
		 */
		public class Data {
			int length;
			int from;
			int to;

			/**
			 * @see java.lang.Object#toString()
			 */
			public String toString() {
				return "<from = " + from + ", to = " + to + ", length = " + length + ">";
			}
		}
	}

	/**
	 * @author Andrea Medeghini
	 */
	public static class Filltable {
		/**
		 * 
		 */
		public Data[] data;

		/**
		 * @param width
		 */
		public Filltable(final int width) {
			data = new Data[width + 1];
			for (int i = 0; i <= width; i++) {
				data[i] = new Data();
			}
		}

		/**
		 * @see java.lang.Object#finalize()
		 */
		public void finalize() throws Throwable {
			data = null;
			super.finalize();
		}

		/**
		 * @author Andrea Medeghini
		 */
		public class Data {
			int length;
			int from;
			int to;

			/**
			 * @see java.lang.Object#toString()
			 */
			public String toString() {
				return "<from = " + from + ", to = " + to + ", length = " + length + ">";
			}
		}
	}

	/**
	 * @author Andrea Medeghini
	 */
	public static class Dynamic {
		/**
		 * 
		 */
		public int[] delta;
		/**
		 * 
		 */
		public Data[] oldBest;
		/**
		 * 
		 */
		public Data[] newBest;
		/**
		 * 
		 */
		public Data[] calData;
		/**
		 * 
		 */
		public Data[] conData;

		/**
		 * @param size
		 */
		public Dynamic(final int size) {
			delta = new int[size + 1];
			oldBest = new Data[size];
			newBest = new Data[size];
			calData = new Data[size];
			conData = new Data[size << XaosFractalRenderer.DSIZE];
			for (int i = 0; i < size; i++) {
				calData[i] = new Data();
			}
			for (int i = 0; i < (size << XaosFractalRenderer.DSIZE); i++) {
				conData[i] = new Data();
			}
		}

		/**
		 * @see java.lang.Object#finalize()
		 */
		public void finalize() throws Throwable {
			oldBest = null;
			newBest = null;
			calData = null;
			conData = null;
			super.finalize();
		}

		/**
		 * 
		 */
		public void swap() {
			final Dynamic.Data[] tmp_best = newBest;
			newBest = oldBest;
			oldBest = tmp_best;
		}

		/**
		 * @author Andrea Medeghini
		 */
		public class Data {
			Data previous;
			int pos;
			int price;

			/**
			 * @see java.lang.Object#toString()
			 */
			public String toString() {
				return "<price = " + price + ", pos = " + pos + ">";
			}
		}
	}

	/**
	 * @author Andrea Medeghini
	 */
	public static class Realloc {
		/**
		 * 
		 */
		public boolean isCached;
		/**
		 * 
		 */
		public boolean refreshed;
		/**
		 * 
		 */
		public boolean recalculate;
		/**
		 * 
		 */
		public boolean changeDirty;
		/**
		 * 
		 */
		public boolean dirty;
		/**
		 * 
		 */
		public boolean line;
		/**
		 * 
		 */
		public int pos;
		/**
		 * 
		 */
		public int plus;
		/**
		 * 
		 */
		public int symTo;
		/**
		 * 
		 */
		public int symRef;
		/**
		 * 
		 */
		public double changePosition;
		/**
		 * 
		 */
		public double position;
		/**
		 * 
		 */
		public double priority;

		/**
		 * @param line
		 */
		public Realloc(final boolean line) {
			this.line = line;
		}

		/**
		 * @see java.lang.Object#toString()
		 */
		public String toString() {
			return "<pos = " + pos + ", symref = " + symRef + ", symto = " + symTo + ", plus = " + plus + ", dirty = " + dirty + ", recalculate = " + recalculate + ", line = " + line + ", priority = " + priority + ", position = " + position + ", iscached = " + isCached + ">";
		}
	}

	/**
	 * @author Andrea Medeghini
	 */
	public static class RendererData {
		/**
		 * 
		 */
		public BufferedImage newBuffer;
		/**
		 * 
		 */
		public BufferedImage oldBuffer;
		/**
		 * 
		 */
		public double[] newCacheR;
		/**
		 * 
		 */
		public double[] newCacheI;
		/**
		 * 
		 */
		public int[] newCacheTime;
		/**
		 * 
		 */
		public double[] oldCacheR;
		/**
		 * 
		 */
		public double[] oldCacheI;
		/**
		 * 
		 */
		public int[] oldCacheTime;
		/**
		 * 
		 */
		public int[] newRGB;
		/**
		 * 
		 */
		public int[] oldRGB;
		/**
		 * 
		 */
		public double[] positionX;
		/**
		 * 
		 */
		public double[] positionY;
		/**
		 * 
		 */
		public Realloc[] reallocX;
		/**
		 * 
		 */
		public Realloc[] reallocY;
		/**
		 * 
		 */
		public Dynamic dynamicx;
		/**
		 * 
		 */
		public Dynamic dynamicy;
		/**
		 * 
		 */
		public Movetable moveTable;
		/**
		 * 
		 */
		public Filltable fillTable;
		/**
		 * 
		 */
		public Realloc[] queue;
		/**
		 * 
		 */
		public double x0 = 0;
		/**
		 * 
		 */
		public double y0 = 0;
		/**
		 * 
		 */
		public long newTime;
		/**
		 * 
		 */
		public long oldTime;
		/**
		 * 
		 */
		public final int[] position = new int[XaosFractalRenderer.STEPS];
		/**
		 * 
		 */
		public final int[] offset = new int[XaosFractalRenderer.STEPS];

		/**
		 * @see java.lang.Object#finalize()
		 */
		public void finalize() throws Throwable {
			free();
			super.finalize();
		}

		/**
		 * 
		 */
		public void free() {
			positionX = null;
			positionY = null;
			reallocX = null;
			reallocY = null;
			dynamicx = null;
			dynamicy = null;
			moveTable = null;
			fillTable = null;
			queue = null;
			newCacheR = null;
			newCacheI = null;
			newCacheTime = null;
			oldCacheR = null;
			oldCacheI = null;
			oldCacheTime = null;
			if (newBuffer != null) {
				newBuffer.flush();
			}
			newBuffer = null;
			if (oldBuffer != null) {
				oldBuffer.flush();
			}
			oldBuffer = null;
		}

		/**
		 * @param width
		 * @param height
		 */
		public void reallocate(final int width, final int height) {
			positionX = new double[width];
			positionY = new double[height];
			reallocX = new Realloc[width];
			reallocY = new Realloc[height];
			dynamicx = new Dynamic(width);
			dynamicy = new Dynamic(height);
			moveTable = new Movetable(width);
			fillTable = new Filltable(width);
			queue = new Realloc[reallocX.length + reallocY.length];
			for (int i = 0; i < width; i++) {
				reallocX[i] = new Realloc(false);
				reallocX[i].pos = i;
				positionX[i] = 0;
			}
			for (int i = 0; i < height; i++) {
				reallocY[i] = new Realloc(true);
				reallocY[i].pos = i;
				positionY[i] = 0;
			}
			newBuffer = new BufferedImage(width, height, Surface.DEFAULT_TYPE);
			newRGB = ((DataBufferInt) newBuffer.getRaster().getDataBuffer()).getData();
			oldBuffer = new BufferedImage(width, height, Surface.DEFAULT_TYPE);
			oldRGB = ((DataBufferInt) oldBuffer.getRaster().getDataBuffer()).getData();
			newCacheR = new double[width * height];
			newCacheI = new double[width * height];
			newCacheTime = new int[width * height];
			oldCacheR = new double[width * height];
			oldCacheI = new double[width * height];
			oldCacheTime = new int[width * height];
		}

		/**
		 * 
		 */
		public void swap() {
			final int[] tmpRGB = oldRGB;
			final BufferedImage tmpBuffer = oldBuffer;
			oldRGB = newRGB;
			oldBuffer = newBuffer;
			newRGB = tmpRGB;
			newBuffer = tmpBuffer;
			final double[] tmpCacheR = oldCacheR;
			final double[] tmpCacheI = oldCacheI;
			final int[] tmpCacheTime = oldCacheTime;
			oldCacheR = newCacheR;
			oldCacheI = newCacheI;
			oldCacheTime = newCacheTime;
			newCacheR = tmpCacheR;
			newCacheI = tmpCacheI;
			newCacheTime = tmpCacheTime;
		}
	}

	private class MandelbrotRenderingStrategy implements RenderingStrategy {
		/**
		 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer.RenderingStrategy#isVerticalSymetrySupported()
		 */
		public boolean isVerticalSymetrySupported() {
			for (int i = 0; i < fractal.getOutcolouringFormulaCount(); i++) {
				final OutcolouringFormulaRuntimeFormula outcolouringFormula = fractal.getOutcolouringFormula(i);
				if (outcolouringFormula.getFormulaRuntime() != null && !outcolouringFormula.getFormulaRuntime().isVerticalSymetryAllowed()) {
					return false;
				}
			}
			for (int i = 0; i < fractal.getIncolouringFormulaCount(); i++) {
				final IncolouringFormulaRuntimeElement incolouringFormula = fractal.getIncolouringFormula(i);
				if (incolouringFormula.getFormulaRuntime() != null && !incolouringFormula.getFormulaRuntime().isVerticalSymetryAllowed()) {
					return false;
				}
			}
			return true;
		}

		/**
		 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer.RenderingStrategy#isHorizontalSymetrySupported()
		 */
		public boolean isHorizontalSymetrySupported() {
			for (int i = 0; i < fractal.getOutcolouringFormulaCount(); i++) {
				final OutcolouringFormulaRuntimeFormula outcolouringFormula = fractal.getOutcolouringFormula(i);
				if (outcolouringFormula.getFormulaRuntime() != null && !outcolouringFormula.getFormulaRuntime().isHorizontalSymetryAllowed()) {
					return false;
				}
			}
			for (int i = 0; i < fractal.getIncolouringFormulaCount(); i++) {
				final IncolouringFormulaRuntimeElement incolouringFormula = fractal.getIncolouringFormula(i);
				if (incolouringFormula.getFormulaRuntime() != null && !incolouringFormula.getFormulaRuntime().isHorizontalSymetryAllowed()) {
					return false;
				}
			}
			return true;
		}

		/**
		 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer.RenderingStrategy#renderPoint(net.sf.jame.mandelbrot.renderer.RenderedPoint)
		 */
		public int renderPoint(final RenderedPoint p, final Complex px, final Complex pw) {
			if (fractal.getRenderingFormula().getFormulaRuntime() != null && fractal.getTransformingFormula().getFormulaRuntime() != null) {
				fractal.getTransformingFormula().getFormulaRuntime().renderPoint(pw);
				p.xr = px.r;
				p.xi = px.i;
				p.wr = pw.r;
				p.wi = pw.i;
				return XaosFractalRenderer.this.renderPoint(p);
			}
			return 0;
		}

		/**
		 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer.RenderingStrategy#updateParameters()
		 */
		public void updateParameters() {
			if (fractal.getRenderingFormula().getFormulaRuntime() != null) {
				renderedData.x0 = fractal.getRenderingFormula().getFormulaRuntime().getInitialPoint().r;
				renderedData.y0 = fractal.getRenderingFormula().getFormulaRuntime().getInitialPoint().i;
			}
			else {
				renderedData.x0 = 0;
				renderedData.y0 = 0;
			}
		}
	}

	private class JuliaRenderingStrategy implements RenderingStrategy {
		/**
		 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer.RenderingStrategy#isHorizontalSymetrySupported()
		 */
		public boolean isHorizontalSymetrySupported() {
			return false;
		}

		/**
		 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer.RenderingStrategy#isVerticalSymetrySupported()
		 */
		public boolean isVerticalSymetrySupported() {
			return false;
		}

		/**
		 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer.RenderingStrategy#renderPoint(net.sf.jame.mandelbrot.renderer.RenderedPoint)
		 */
		public int renderPoint(final RenderedPoint p, final Complex px, final Complex pw) {
			if (fractal.getRenderingFormula().getFormulaRuntime() != null && fractal.getTransformingFormula().getFormulaRuntime() != null) {
				fractal.getTransformingFormula().getFormulaRuntime().renderPoint(px);
				p.xr = pw.r;
				p.xi = pw.i;
				p.wr = px.r;
				p.wi = px.i;
				return XaosFractalRenderer.this.renderPoint(p);
			}
			return 0;
		}

		/**
		 * @see net.sf.jame.mandelbrot.renderer.AbstractFractalRenderer.RenderingStrategy#updateParameters()
		 */
		public void updateParameters() {
			renderedData.x0 = oldConstant.getX();
			renderedData.y0 = oldConstant.getY();
		}
	}

	private class PrepareLinesTask implements Runnable {
		private Thread thread;
		private boolean dirty;
		private boolean running;

		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				while (running) {
					synchronized (this) {
						if (!dirty) {
							wait();
						}
						prepareLines();
						dirty = false;
						notify();
					}
				}
			}
			catch (final InterruptedException e) {
			}
			synchronized (this) {
				dirty = false;
				notifyAll();
			}
		}

		public void stop() {
			try {
				if (thread != null) {
					running = false;
					thread.interrupt();
					thread.join();
				}
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}

		public void start() {
			if (thread == null) {
				thread = factory.newThread(this);
				thread.setName(thread.getName() + " PrepareLinesTask");
				running = true;
				thread.start();
			}
		}

		public void execute() {
			synchronized (this) {
				dirty = true;
				notify();
			}
		}

		public void waitTask() throws InterruptedException {
			synchronized (this) {
				while (dirty) {
					wait();
				}
			}
		}
	}

	private class PrepareColumnsTask implements Runnable {
		private Thread thread;
		private boolean dirty;
		private boolean running;

		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				while (running) {
					synchronized (this) {
						if (!dirty) {
							wait();
						}
						prepareColumns();
						dirty = false;
						notify();
					}
				}
			}
			catch (final InterruptedException e) {
			}
			synchronized (this) {
				dirty = false;
				notifyAll();
			}
		}

		public void stop() {
			try {
				if (thread != null) {
					running = false;
					thread.interrupt();
					thread.join();
				}
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}

		public void start() {
			if (thread == null) {
				thread = factory.newThread(this);
				thread.setName(thread.getName() + " PrepareColumnsTask");
				running = true;
				thread.start();
			}
		}

		public void executeTask() {
			synchronized (this) {
				dirty = true;
				notify();
			}
		}

		public void waitTask() throws InterruptedException {
			synchronized (this) {
				while (dirty) {
					wait();
				}
			}
		}
	}
}
