/*
 * $Id:MandelbrotImageRuntime.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.mandelbrot.extensions.image;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Polygon;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sf.jame.core.math.Complex;
import net.sf.jame.mandelbrot.MandelbrotFractalManager;
import net.sf.jame.mandelbrot.MandelbrotRuntime;
import net.sf.jame.mandelbrot.fractal.MandelbrotFractalRuntimeElement;
import net.sf.jame.mandelbrot.fractal.rendering.RenderingFormulaRuntimeFormula;
import net.sf.jame.mandelbrot.renderer.RenderedPoint;
import net.sf.jame.mandelbrot.renderer.SimpleFractalRenderer;
import net.sf.jame.mandelbrot.renderer.XaosFractalRenderer;
import net.sf.jame.twister.DoubleVector2D;
import net.sf.jame.twister.ImageTile;
import net.sf.jame.twister.IntegerVector2D;
import net.sf.jame.twister.Rectangle;
import net.sf.jame.twister.View;
import net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime;
import net.sf.jame.twister.renderer.TwisterRenderer;
import net.sf.jame.twister.renderer.TwisterRenderingHints;

/**
 * @author Andrea Medeghini
 */
public class MandelbrotImageRuntime extends ImageExtensionRuntime<MandelbrotImageConfig> {
	// private static final Logger logger = Logger.getLogger(MandelbrotImageRuntime.class);
	private Map<Object, Object> hints = new HashMap<Object, Object>();
	private MandelbrotRuntime mandelbrotRuntime;
	private RendererStrategy rendererStrategy;
	private int lastStatus;
	private ImageTile tile;

	/**
	 * @see net.sf.jame.core.extension.ConfigurableExtensionRuntime#configReloaded()
	 */
	@Override
	public void configReloaded() {
		mandelbrotRuntime = new MandelbrotRuntime(getConfig().getMandelbrotConfig());
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#startRenderer()
	 */
	@Override
	public void startRenderer() {
		if (rendererStrategy != null) {
			rendererStrategy.startRenderer();
		}
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#abortRenderer()
	 */
	@Override
	public void abortRenderer() {
		if (rendererStrategy != null) {
			rendererStrategy.abortRenderer();
		}
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#joinRenderer()
	 */
	@Override
	public void joinRenderer() throws InterruptedException {
		if (rendererStrategy != null) {
			rendererStrategy.joinRenderer();
		}
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#getRenderingStatus()
	 */
	@Override
	public int getRenderingStatus() {
		if (rendererStrategy != null) {
			return rendererStrategy.getRenderingStatus();
		}
		return 0;
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#isDynamic()
	 */
	@Override
	public boolean isDynamic() {
		if (rendererStrategy != null) {
			return rendererStrategy.isDynamic();
		}
		return false;
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#prepareImage(boolean)
	 */
	@Override
	public void prepareImage(final boolean isDynamicRequired) {
		if (rendererStrategy != null) {
			rendererStrategy.prepareImage(isDynamicRequired);
		}
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#drawImage(java.awt.Graphics2D)
	 */
	@Override
	public void drawImage(final Graphics2D g2d) {
		if (rendererStrategy != null) {
			rendererStrategy.drawImage(g2d);
		}
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#drawImage(java.awt.Graphics2D, int, int, int, int)
	 */
	@Override
	public void drawImage(final Graphics2D g2d, final int x, final int y, final int w, final int h) {
		if (rendererStrategy != null) {
			rendererStrategy.drawImage(g2d, x, y, w, h);
		}
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#getImageSize()
	 */
	@Override
	public IntegerVector2D getImageSize() {
		return tile.getImageSize();
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#setTile(net.sf.jame.twister.ImageTile)
	 */
	@Override
	public void setTile(final ImageTile tile) {
		if (this.tile != tile) {
			this.tile = tile;
			if (hints.get(TwisterRenderingHints.KEY_TYPE) == TwisterRenderingHints.TYPE_OVERLAY) {
				rendererStrategy = new OverlayRendererStrategy(tile);
			}
			else {
				rendererStrategy = new DefaultRendererStrategy(tile);
			}
		}
	}

	/**
	 * @see net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime#setRenderingHints(java.util.Map)
	 */
	@Override
	public void setRenderingHints(final Map<Object, Object> hints) {
		this.hints = hints;
	}

	/**
	 * @see net.sf.jame.core.extension.ExtensionRuntime#dispose()
	 */
	@Override
	public void dispose() {
		if (rendererStrategy != null) {
			rendererStrategy.dispose();
			rendererStrategy = null;
		}
		if (mandelbrotRuntime != null) {
			mandelbrotRuntime.dispose();
			mandelbrotRuntime = null;
		}
		super.dispose();
	}

	/**
	 * @see net.sf.jame.core.config.RuntimeElement#isChanged()
	 */
	@Override
	public boolean isChanged() {
		boolean isChanged = (mandelbrotRuntime != null) && (mandelbrotRuntime.changeCount() > 0);
		final int status = (rendererStrategy != null) ? rendererStrategy.getRenderingStatus() : -1;
		isChanged |= (status == TwisterRenderer.STATUS_RENDERING) || (lastStatus == TwisterRenderer.STATUS_RENDERING);
		lastStatus = (rendererStrategy != null) ? rendererStrategy.getRenderingStatus() : -1;
		return super.isChanged() || isChanged;
	}

	private interface RendererStrategy {
		/**
		 * 
		 */
		public void dispose();

		/**
		 * @param isDynamicRequired
		 */
		public void prepareImage(boolean isDynamicRequired);

		/**
		 * @param g2d
		 */
		public void drawImage(final Graphics2D g2d);

		/**
		 * @param g2d
		 * @param x
		 * @param y
		 * @param w
		 * @param h
		 */
		public void drawImage(Graphics2D g2d, int x, int y, int w, int h);

		/**
		 * @return
		 */
		public int getRenderingStatus();

		/**
		 * 
		 */
		public void startRenderer();

		/**
		 * 
		 */
		public void abortRenderer();

		/**
		 * @throws InterruptedException
		 */
		public void joinRenderer() throws InterruptedException;

		/**
		 * @return
		 */
		public boolean isDynamic();
	}

	private class DefaultRendererStrategy implements RendererStrategy {
		private MandelbrotFractalManager fractalManager;
		private int dynamicCount = 0;
		private boolean dirty = true;

		/**
		 * @param tile
		 */
		public DefaultRendererStrategy(final ImageTile tile) {
			if (hints.get(TwisterRenderingHints.KEY_QUALITY) == TwisterRenderingHints.QUALITY_REALTIME) {
				fractalManager = new MandelbrotFractalManager(new XaosFractalRenderer(Thread.MIN_PRIORITY + 2));
			}
			else {
				fractalManager = new MandelbrotFractalManager(new SimpleFractalRenderer(Thread.MIN_PRIORITY + 1));
			}
			fractalManager.setRenderingHints(hints);
			fractalManager.setFractal(mandelbrotRuntime.getMandelbrotFractal());
			loadConfig();
			fractalManager.setTile(tile);
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#startRenderer()
		 */
		public void startRenderer() {
			fractalManager.startRenderer();
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#abortRenderer()
		 */
		public void abortRenderer() {
			fractalManager.abortRenderer();
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#joinRenderer()
		 */
		public void joinRenderer() throws InterruptedException {
			fractalManager.joinRenderer();
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#getRenderingStatus()
		 */
		public int getRenderingStatus() {
			return fractalManager.getRenderingStatus();
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#drawImage(java.awt.Graphics2D)
		 */
		public void drawImage(final Graphics2D g2d) {
			if (tile != null) {
				fractalManager.drawImage(g2d);
				if (dirty) {
					dirty = false;
					try {
						fractalManager.joinRenderer();
						fractalManager.startRenderer();
					}
					catch (final InterruptedException e) {
					}
				}
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#drawImage(java.awt.Graphics2D, int, int, int, int)
		 */
		public void drawImage(final Graphics2D g2d, final int x, final int y, final int w, final int h) {
			if (tile != null) {
				fractalManager.drawImage(g2d, x, y, w, h);
				if (dirty) {
					dirty = false;
					try {
						fractalManager.joinRenderer();
						fractalManager.startRenderer();
					}
					catch (final InterruptedException e) {
					}
				}
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#prepareImage(boolean)
		 */
		public void prepareImage(final boolean isDynamicRequired) {
			if (isDynamicRequired) {
				boolean isChanged = mandelbrotRuntime.isChanged();
				isChanged |= mandelbrotRuntime.isViewChanged();
				if (isChanged) {
					dynamicCount = 0;
					dirty = true;
				}
				if (dynamicCount >= 1) {
					dirty = true;
				}
				if (dirty) {
					fractalManager.abortRenderer();
					loadConfig();
				}
			}
			else {
				loadConfig();
			}
			if (fractalManager.isDynamic()) {
				dynamicCount += 1;
			}
			else {
				dynamicCount = 0;
			}
		}

		private void loadConfig() {
			fractalManager.setView(getConfig().getMandelbrotConfig().getView(), getConfig().getMandelbrotConfig().getConstant(), getConfig().getMandelbrotConfig().getImageMode());
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#dispose()
		 */
		public void dispose() {
			if (fractalManager != null) {
				fractalManager.dispose();
				fractalManager = null;
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#isDynamic()
		 */
		public boolean isDynamic() {
			return dynamicCount > 0;
		}
	}

	private class OverlayRendererStrategy implements RendererStrategy {
		private MandelbrotFractalManager fractalManager;
		private boolean suspended = false;
		private boolean dirty = true;

		/**
		 * @param tile
		 */
		public OverlayRendererStrategy(final ImageTile tile) {
			if (hints.get(TwisterRenderingHints.KEY_QUALITY) == TwisterRenderingHints.QUALITY_REALTIME) {
				fractalManager = new MandelbrotFractalManager(new XaosFractalRenderer(Thread.MIN_PRIORITY + 2));
			}
			else {
				fractalManager = new MandelbrotFractalManager(new SimpleFractalRenderer(Thread.MIN_PRIORITY + 1));
			}
			fractalManager.setRenderingHints(hints);
			fractalManager.setFractal(mandelbrotRuntime.getMandelbrotFractal());
			loadConfig();
			final Rectangle previewArea = getConfig().getMandelbrotConfig().getPreviewArea();
			final int px = (int) Math.rint(tile.getImageSize().getX() * previewArea.getX());
			final int py = (int) Math.rint(tile.getImageSize().getY() * previewArea.getY());
			final int pw = (int) Math.rint(tile.getImageSize().getX() * previewArea.getW());
			final int ph = (int) Math.rint(tile.getImageSize().getY() * previewArea.getH());
			final int tx = (int) Math.rint(tile.getTileOffset().getX());
			final int ty = (int) Math.rint(tile.getTileOffset().getY());
			final int tw = (int) Math.rint(tile.getTileSize().getX());
			final int th = (int) Math.rint(tile.getTileSize().getY());
			// final int iw = (int) Math.rint(tile.getImageSize().getX());
			// final int ih = (int) Math.rint(tile.getImageSize().getY());
			suspended = true;
			if (px < tx + tw && px + pw > tx) {
				if (py < ty + th && py + ph > ty) {
					int ox = 0;
					int oy = 0;
					int ow = pw;
					int oh = ph;
					if (tx > px) {
						ox = tx - px;
					}
					if (ty > py) {
						oy = ty - py;
					}
					if (ox + ow > pw) {
						ow = pw - ox;
					}
					if (oy + oh > ph) {
						oh = ph - oy;
					}
					if (ow > 0 && oh > 0 && ox >= 0 && oy >= 0) {
						fractalManager.setTile(new ImageTile(new IntegerVector2D(pw, ph), new IntegerVector2D(ow, oh), new IntegerVector2D(ox, oy), new IntegerVector2D(0, 0)));
						suspended = false;
					}
				}
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#startRenderer()
		 */
		public void startRenderer() {
			if (!suspended) {
				fractalManager.startRenderer();
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#abortRenderer()
		 */
		public void abortRenderer() {
			if (!suspended) {
				fractalManager.abortRenderer();
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#joinRenderer()
		 */
		public void joinRenderer() throws InterruptedException {
			if (!suspended) {
				fractalManager.joinRenderer();
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#getRenderingStatus()
		 */
		public int getRenderingStatus() {
			return fractalManager.getRenderingStatus();
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#drawImage(java.awt.Graphics2D)
		 */
		public void drawImage(final Graphics2D g2d) {
			if (tile != null) {
				g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.8f));
				if (getConfig().getMandelbrotConfig().getShowPreview() && !suspended) {
					final Rectangle previewArea = getConfig().getMandelbrotConfig().getPreviewArea();
					final int px = (int) Math.rint(tile.getImageSize().getX() * previewArea.getX());
					final int py = (int) Math.rint(tile.getImageSize().getY() * previewArea.getY());
					final int pw = (int) Math.rint(tile.getImageSize().getX() * previewArea.getW());
					final int ph = (int) Math.rint(tile.getImageSize().getY() * previewArea.getH());
					final int tx = (int) Math.rint(tile.getTileOffset().getX());
					final int ty = (int) Math.rint(tile.getTileOffset().getY());
					final int tw = (int) Math.rint(tile.getTileSize().getX());
					final int th = (int) Math.rint(tile.getTileSize().getY());
					if (px < tx + tw && px + pw > tx) {
						if (py < ty + th && py + ph > ty) {
							int ox = 0;
							int oy = 0;
							int ow = pw;
							int oh = ph;
							if (tx > px) {
								ox = tx - px;
							}
							if (ty > py) {
								oy = ty - py;
							}
							if (ox + ow > pw) {
								ow = pw - ox;
							}
							if (oy + oh > ph) {
								oh = ph - oy;
							}
							if (ow > 0 && oh > 0 && ox >= 0 && oy >= 0) {
								g2d.setColor(Color.RED);
								g2d.setClip(tx, ty, tw, th);
								fractalManager.drawImage(g2d, px - tx + ox, py - ty + oy, ow, oh);
								g2d.drawRect(px - tx + ox - 1, py - ty + oy - 1, ow, oh);
								final MandelbrotFractalRuntimeElement fractal = mandelbrotRuntime.getMandelbrotFractal();
								if (fractal != null) {
									final RenderingFormulaRuntimeFormula formula = fractal.getRenderingFormula();
									if (formula != null) {
										if (formula.getFormulaRuntime() != null) {
											final DoubleVector2D scale = formula.getFormulaRuntime().getScale();
											final DoubleVector2D center = formula.getFormulaRuntime().getCenter();
											final DoubleVector2D constant = getConfig().getMandelbrotConfig().getConstant();
											final View view = getConfig().getMandelbrotConfig().getView();
											final double x = view.getPosition().getX();
											final double y = view.getPosition().getY();
											final double z = view.getPosition().getZ();
											final double a = view.getRotation().getZ();
											final double qx = (constant.getX() - center.getX() - x) / (z * scale.getX());
											final double qy = (constant.getY() - center.getY() - y) / (z * scale.getY());
											final double cx = (Math.cos(-a) * qx) + (Math.sin(-a) * qy);
											final double cy = (Math.cos(-a) * qy) - (Math.sin(-a) * qx);
											final int dx = (int) Math.rint(cx * tile.getImageSize().getX());
											final int dy = (int) Math.rint(cy * tile.getImageSize().getX());
											g2d.drawRect(dx + tile.getImageSize().getX() / 2 - tile.getTileOffset().getX() - 2, dy + tile.getImageSize().getY() / 2 - tile.getTileOffset().getY() - 2, 4, 4);
										}
									}
								}
							}
						}
					}
				}
				if (getConfig().getMandelbrotConfig().getShowOrbit()) {
					g2d.setColor(Color.YELLOW);
					final View view = getConfig().getMandelbrotConfig().getView();
					final double x = view.getPosition().getX();
					final double y = view.getPosition().getY();
					final double z = view.getPosition().getZ();
					final double a = view.getRotation().getZ();
					final int tw = tile.getImageSize().getX();
					final int th = tile.getImageSize().getY();
					final MandelbrotFractalRuntimeElement fractal = mandelbrotRuntime.getMandelbrotFractal();
					if (fractal != null) {
						final RenderingFormulaRuntimeFormula formula = fractal.getRenderingFormula();
						if (formula != null) {
							final RenderedPoint rp = new RenderedPoint();
							if (formula.getFormulaRuntime() != null) {
								formula.getFormulaRuntime().prepareForRendering();
								final DoubleVector2D center = formula.getFormulaRuntime().getCenter();
								final DoubleVector2D scale = formula.getFormulaRuntime().getScale();
								rp.xr = formula.getFormulaRuntime().getInitialPoint().r;
								rp.xi = formula.getFormulaRuntime().getInitialPoint().i;
								rp.wr = getConfig().getMandelbrotConfig().getConstantElement().getValue().getX();
								rp.wi = getConfig().getMandelbrotConfig().getConstantElement().getValue().getY();
								final List<Complex> orbit = formula.getFormulaRuntime().renderOrbit(rp);
								final Polygon p = new Polygon();
								for (int i = 0; i < orbit.size(); i++) {
									final Complex pa = orbit.get(i);
									final double tx = (pa.r - center.getX() - x) / (scale.getX() * z) * tw;
									final double ty = (pa.i - center.getY() - y) / (scale.getY() * z) * tw;
									final int px = (int) Math.rint((Math.cos(a) * tx) - (Math.sin(a) * ty)) + tw / 2 - tile.getTileOffset().getX();
									final int py = (int) Math.rint((Math.cos(a) * ty) + (Math.sin(a) * tx)) + th / 2 - tile.getTileOffset().getY();
									p.addPoint(px, py);
								}
								g2d.setClip(0, 0, tw, th);
								g2d.draw(p);
								g2d.setClip(null);
								// if (orbit.size() > 1)
								// {
								// Complex pa = orbit.get(0);
								//									
								// for (int i = 1; i < orbit.size(); i++)
								// {
								// Complex pb = orbit.get(i);
								//										
								// double tx1 = (pa.r - center.r - x) / (scale.r * z) * tw;
								//											
								// double ty1 = (pa.i - center.i - y) / (scale.i * z) * tw;
								//										
								// double tx2 = (pb.r - center.r - x) / (scale.r * z) * tw;
								//										
								// double ty2 = (pb.i - center.i - y) / (scale.i * z) * tw;
								//										
								// int x1 = (int) Math.rint((Math.cos(a) * tx1) - (Math.sin(a) * ty1)) + tw / 2;
								//										
								// int y1 = (int) Math.rint((Math.cos(a) * ty1) + (Math.sin(a) * tx1)) + th / 2;
								//										
								// int x2 = (int) Math.rint((Math.cos(a) * tx2) - (Math.sin(a) * ty2)) + tw / 2;
								//										
								// int y2 = (int) Math.rint((Math.cos(a) * ty2) + (Math.sin(a) * tx2)) + th / 2;
								//										
								// g2d.drawLine(x1, y1, x2, y2);
								//										
								// pa = pb;
								// }
								// }
							}
						}
					}
				}
				if (dirty) {
					dirty = false;
					try {
						fractalManager.joinRenderer();
						if (!suspended) {
							fractalManager.startRenderer();
						}
					}
					catch (final InterruptedException e) {
					}
				}
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#drawImage(java.awt.Graphics2D, int, int, int, int)
		 */
		public void drawImage(final Graphics2D g2d, final int x, final int y, final int w, final int h) {
			this.drawImage(g2d);
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#prepareImage(boolean)
		 */
		public void prepareImage(final boolean isDynamicRequired) {
			if (isDynamicRequired) {
				if (!suspended) {
					if (mandelbrotRuntime.isChanged()) {
						dirty = true;
					}
					if (dirty) {
						fractalManager.abortRenderer();
						loadConfig();
					}
				}
			}
			else {
				loadConfig();
			}
		}

		private void loadConfig() {
			if (getConfig().getMandelbrotConfig().getImageMode() == 0) {
				fractalManager.setMandelbrotMode(1);
			}
			else {
				fractalManager.setMandelbrotMode(0);
			}
			fractalManager.setConstant(getConfig().getMandelbrotConfig().getConstant());
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#dispose()
		 */
		public void dispose() {
			if (fractalManager != null) {
				fractalManager.dispose();
				fractalManager = null;
			}
		}

		/**
		 * @see net.sf.jame.mandelbrot.extensions.image.MandelbrotImageRuntime.RendererStrategy#isDynamic()
		 */
		public boolean isDynamic() {
			return false;
		}
	}
}
