/*
 * $Id:ListConfigElement.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.core.config;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Andrea Medeghini
 */
public class ListConfigElement<T extends ConfigElement> extends AbstractConfigElement implements ListConfigElementEvents {
	private static final long serialVersionUID = 1L;
	private List<T> elements;

	/**
	 * 
	 */
	public ListConfigElement(final String configElementId) {
		this(configElementId, 0);
	}

	/**
	 * 
	 */
	public ListConfigElement(final String configElementId, final int size) {
		super(configElementId);
		elements = new ArrayList<T>(size);
	}

	/**
	 * Returns a element.
	 * 
	 * @param index the index.
	 * @return the layer.
	 */
	public T getElement(final int index) {
		return elements.get(index);
	}

	/**
	 * Returns a element index.
	 * 
	 * @param element the element.
	 * @return the index.
	 */
	public int indexOfElement(final T element) {
		return elements.indexOf(element);
	}

	/**
	 * Returns the number of elements.
	 * 
	 * @return the number of elements.
	 */
	public int getElementCount() {
		return elements.size();
	}

	/**
	 * Adds a element.
	 * 
	 * @param element the to add.
	 */
	public void appendElement(final T element) {
		final int index = elements.size();
		elements.add(element);
		if (checkContext()) {
			element.setContext(getContext());
			fireConfigChanged(new ValueChangeEvent(ListConfigElementEvents.ELEMENT_ADDED, getContext().getTimestamp(), element, index));
		}
	}

	/**
	 * Inserts a element.
	 * 
	 * @param index the index.
	 * @param element the to add.
	 */
	public void insertElementAfter(final int index, final T element) {
		if ((index < 0) || (index > elements.size() - 1)) {
			throw new IllegalArgumentException("index out of bounds");
		}
		if (index < elements.size() - 1) {
			elements.add(index + 1, element);
		}
		else {
			elements.add(element);
		}
		if (checkContext()) {
			element.setContext(getContext());
			fireConfigChanged(new ValueChangeEvent(ListConfigElementEvents.ELEMENT_INSERTED_AFTER, getContext().getTimestamp(), element, index));
		}
	}

	/**
	 * Inserts a element.
	 * 
	 * @param index the index.
	 * @param element the to add.
	 */
	public void insertElementBefore(final int index, final T element) {
		if ((index < 0) || (index > elements.size() - 1)) {
			throw new IllegalArgumentException("index out of bounds");
		}
		elements.add(index, element);
		if (checkContext()) {
			element.setContext(getContext());
			fireConfigChanged(new ValueChangeEvent(ListConfigElementEvents.ELEMENT_INSERTED_BEFORE, getContext().getTimestamp(), element, index));
		}
	}

	/**
	 * Removes a element.
	 * 
	 * @param index the element index to remove.
	 */
	public void removeElement(final int index) {
		final T element = elements.remove(index);
		if (checkContext()) {
			fireConfigChanged(new ValueChangeEvent(ListConfigElementEvents.ELEMENT_REMOVED, getContext().getTimestamp(), element, index));
		}
	}

	/**
	 * Removes a element.
	 * 
	 * @param element the to remove.
	 */
	public void removeElement(final T element) {
		final int index = elements.indexOf(element);
		elements.remove(index);
		if (checkContext()) {
			fireConfigChanged(new ValueChangeEvent(ListConfigElementEvents.ELEMENT_REMOVED, getContext().getTimestamp(), element, index));
		}
	}

	/**
	 * @see net.sf.jame.core.config.AbstractConfigElement#clone()
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ListConfigElement<T> clone() {
		final ListConfigElement<T> element = new ListConfigElement<T>(getClassId());
		for (int i = 0; i < getElementCount(); i++) {
			element.appendElement((T) getElement(i).clone());
		}
		return element;
	}

	/**
	 * @see net.sf.jame.core.config.AbstractConfigElement#setContext(net.sf.jame.core.config.ConfigContext)
	 */
	@Override
	public void setContext(final ConfigContext context) {
		super.setContext(context);
		for (int i = 0; i < getElementCount(); i++) {
			getElement(i).setContext(getContext());
		}
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		final ListConfigElement other = (ListConfigElement) obj;
		if (elements == null) {
			if (other.elements != null) {
				return false;
			}
		}
		else if (!elements.equals(other.elements)) {
			return false;
		}
		return true;
	}

	/**
	 * @see net.sf.jame.core.config.AbstractConfigElement#dispose()
	 */
	@Override
	public void dispose() {
		if (elements != null) {
			elements.clear();
			elements = null;
		}
		super.dispose();
	}
}
