/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;

/**
 * Provides utility methods related to class
 * @author Christos KK Loverdos
 */
public final class ClassUtil
{
    private ClassUtil() {}

    /**
     * Returns a unique name representing this class. The class' loader
     * is taken into account.
     * @param c
     */
    public static String identity(Class c)
    {
        return System.identityHashCode(c.getClassLoader()) + "/" + c.getName();
    }

    /**
     * Returns the context class loader of the current thread.
     */
    public static ClassLoader getContextClassLoader()
    {
        return Thread.currentThread().getContextClassLoader();
    }

    /**
     * Returns the resource, obtained from the context class loader.
     * @param name
     * @return <code>null</code> if not found.
     */
    public static URL getContextResource(String name)
    {
        return getContextClassLoader().getResource(name);
    }

    /**
     * Retrieves either the file or the context resource with the given <code>name</code>, whichever
     * is found first.
     * @param name
     * @return <code>null</code> if not found.
     */
    public static URL getFileOrResource(String name)
    {
        File f = new File(name);
        if(f.isFile())
        {
            try
            {
                return f.toURI().toURL();
            }
            catch(MalformedURLException e)
            {
                return null;
            }
        }
        else
        {
            return getContextResource(name);
        }
    }

    /**
     * Convenience method that calls {@link #getFileOrResource(String)}.
     * @param name
     * @return <code>null</code> if not found.
     * @throws IOException
     */
    public static InputStream getFileOrResourceAsStream(String name) throws IOException
    {
        URL url = getFileOrResource(name);
        if(null == url)
        {
            return null;
        }
        return url.openStream();
    }

    /**
     * Tries to load the class first by using the context class loader and then by calling
     * {@link Class#forName(String)}.
     *
     * @param name
     * @return <code>null</code> if not found.
     */
    public static Class loadClass(String name)
    {
        try
        {
            return getContextClassLoader().loadClass(name);
        }
        catch(ClassNotFoundException e)
        {
        }

        try
        {
            return Class.forName(name);
        }
        catch(ClassNotFoundException e)
        {
        }

        return null;
    }

    public static Object newInstance(Class c)
    {
        if(null == c)
        {
            return null;
        }
        try
        {
            return c.newInstance();
        }
        catch(InstantiationException e)
        {
            return null;
        }
        catch(IllegalAccessException e)
        {
            return null;
        }
    }

    public static Object newInstance(String className)
    {
        return newInstance(loadClass(className));
    }

    public static Object newInstance(String className, ClassLoader cl)
    {
        return newInstance(loadClass(className, cl));
    }

    /**
     * Tries to load the class from the provided class <code>loader</code>.
     * If the <code>loader</code> is <code>null</code>, then {@link #loadClass(String)}
     * is called.
     * @param name
     * @param loader
     * @return the class or <code>null</code> if not found.
     */
    public static Class loadClass(String name, ClassLoader loader)
    {
        if(null == loader)
        {
            return loadClass(name);
        }
        try
        {
            return loader.loadClass(name);
        }
        catch(ClassNotFoundException e)
        {
            return null;
        }
    }

    /**
     * Tries to load the class first by calling {@link #loadClass(String)} and,
     * if this fails, then by using a {@link java.net.URLClassLoader} constructed
     * with the provided <code>urls</code>.
     * @param name
     * @param urls
     * @return <code>null</code> if not found.
     */
    public static Class forName(String name, URL[] urls)
    {
    	try
		{
    		if(Util.emptySafe(urls))
			{
				return loadClass(name);
			}
			else
			{
				return Class.forName(name, false, URLClassLoader.newInstance(urls, ClassLoader.getSystemClassLoader()));
			}
		}
		catch(ClassNotFoundException e)
		{
            return null;
		}
    }

    /**
     * Returns only the name part after the last dot.
     * @param c
     */
    public static String getShortClassName(Class c)
    {
        return getShortClassName(c.getName());
    }

    /**
     * Returns only the name part after the last dot.
     */
    public static String getShortClassName(String name)
    {
        return name.substring(name.lastIndexOf('.') + 1);
    }

    /**
     * Returns only the name part after the last dot.
     */
    public static String getShortClassName(Object o)
    {
        return getShortClassName(o.getClass());
    }

    /**
     * If the class is an array, returns its name with <code>[]</code> appended.
     * @param c
     */
    public static String normalizeClassName(Class c)
    {
        StringBuffer sb = new StringBuffer();
        while(c.isArray())
        {
            sb.append("[]");
            c = c.getComponentType();
        }
        sb.insert(0, c.getName());
        return sb.toString();
    }

    /**
     * If the class is an array, returns its name with <code>[]</code> appended.
     * Names in the form <code>[I</code> and <code>[Ljava.lang.Integer;</code>
     * can be handled correctly.
     */
    public static String normalizeClassName(String name)
    {
        int dim = 0;

        while(name.charAt(dim) == '[')
        {
            dim++;
        }

        name = name.substring(dim);

        if(dim > 0)
        {
            switch(name.charAt(0))
            {
                case 'B':
                    name = "byte";
                    break;
                case 'C':
                    name = "char";
                    break;
                case 'D':
                    name = "double";
                    break;
                case 'F':
                    name = "float";
                    break;
                case 'I':
                    name = "int";
                    break;
                case 'J':
                    name = "long";
                    break;
                case 'S':
                    name = "short";
                    break;
                case 'V':
                    name = "void";
                    break;
                case 'Z':
                    name = "bool";
                    break;
                case 'L':
                    name = name.substring(1, name.length() - 1);
                    break;
            }
        }

        for(int i = 0; i < dim; i++)
        {
            name += "[]";
        }

        return name;
    }
}
