package org.ckkloverdos.type.java;

import org.ckkloverdos.util.ClassUtil;

/**
 * A wrapper around java types with convenient methods for equality testing.
 *
 * In particular, this class works in conjuction with a {@link JavaTypeRegistry},
 * which can handle type aliases.
 *
 * I have developed this little-framework in order to have java types communicated between
 * systems for which only an alias is meaningful. For example, a <code>java.lang.Integer</code>
 * can be represented as an <code>"INTEGER"</code> string in a database and this can be properly
 * mapped to the underlying {@link Integer} VM type by using a {@link JavaTypeRegistry}.
 * 
 *
 * This little-framework has been created owards representation & transformation, not subtyping (at the moment).
 * 
 * @author Christos KK Loverdos
 * @see JavaTypeRegistry
 */
public class JavaType
{
    private Class c;

    public JavaType(Class c)
    {
        this.c = c;
    }

    /**
     * Returns the java class wrapped by this instance.
     */
    public Class getJavaClass()
    {
        return c;
    }

    /**
     * Returns the {@link ClassUtil#getShortClassName(Class) short class name}
     * of the wrapped java class.
     */
    public String getName()
    {
        return ClassUtil.getShortClassName(c);
    }

    /**
     * Returns the {@link Class#getName() class name} of the wrapped java class.
     */
    public String getFullName()
    {
        return c.getName();
    }
    
    /**
     * Returns <code>true</code> iff <code>other</code>
     */
    public boolean is(JavaType other, JavaTypeRegistry r)
    {
        return is(other.getJavaClass(), r);
    }

    /**
     * Returns <code>true</code> iff <code>other</code> wraps the same java class. 
     * @param other
     */
    public boolean is(JavaType other)
    {
        return is(other.getJavaClass());
    }

    /**
     * Returns <code>true</code> iff <code>otherClass</code> equals the java class
     * wrapped by this instance, taking into account any representatives registered
     * with <code>registry</code>.
     *
     * If <code>registry</code> is <code>null</code>, then the result is computed
     * by directly comparing equality between the class wrapped by this instance and
     * <code>otherClass</code>
     * @param otherClass
     * @param registry the {@link JavaTypeRegistry} to checked against, if not <code>null</code>.
     */
    public boolean is(Class otherClass, JavaTypeRegistry registry)
    {
        if(null != registry)
        {
            return registry.is(otherClass, this);
        }
        
        return getJavaClass().equals(otherClass);
    }

    /**
     * Returns <code>true</code> iff <code>otherClass</code> equals the java class
     * wrapped by this instance.
     */
    public boolean is(Class otherClass)
    {
        return getJavaClass().equals(otherClass);
    }

    /**
     * Always returns <code>false</code>. Subclasses should normally delegate
     * to {@link #is(String, JavaTypeRegistry)}, provided that a type registry
     * has been defined elsewhere.
     */
    public boolean is(String alias)
    {
        return false;
    }

    
    /**
     * Checks if the given alias, which must exist in <code>registry</code> is the
     * same type as this instance.
     */
    public boolean is(String alias, JavaTypeRegistry registry)
    {
        return is(registry.getByName(alias), registry);
    }

    /**
     * This instance equals <code>other</code> iff <code>other</code> is strictly of JavaType class
     * and <code>other</code>'s wrapped class equals the wrapped class of this instance.
     * @param other
     */
    public boolean equals(Object other)
    {
        if(this == other)
        {
            return true;
        }
        if(other == null || getClass() != other.getClass())
        {
            return false;
        }

        JavaType javaType = (JavaType) other;

        return c.equals(javaType.c);
    }

    public int hashCode()
    {
        return c.hashCode();
    }

    public String getDescription()
    {
        return c.getName();
    }
    
    public String toString()
    {
        return "JavaType(" + c.getName() + ')';
    }
}
