/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.tuple;

import org.ckkloverdos.util.Util;

/**
 * An abstract class implementing {@link #equals(Object)} and {@link #hashCode()}.
 *
 * A tuple implementation should override <code>TupleSkeleton</code>, so that
 * these two methods have consistent behaviour. 
 *
 * @author Christos KK Loverdos
 */
public abstract class TupleSkeleton implements ITuple
{
    /**
     * Compares for equality. This tuple is equal to <code>obj</code> iff
     * the following conditions simultaneously hold:
     * <ol>
     *  <li> <code>obj</code> is also a {@link org.ckkloverdos.tuple.ITuple}. </li>
     *  <li> the size of the two tuples is the same. </li>
     *  <li> the elements of the two tuples are equal, compared one by one.</li>
     * </ol>
     *
     * Two empty tuples are always equal.
     * 
     * The underlying implementation of the tuples doesnot matter.
     * @param obj
     */
    public boolean equals(Object obj)
    {
        if(obj instanceof ITuple)
        {
            ITuple other = (ITuple) obj;
            if(other.isEmpty())
            {
                return isEmpty();
            }
            else if(isEmpty())
            {
                return other.isEmpty();
            }

            if(size() == other.size())
            {
                for(int i=0; i<size(); i++)
                {
                    Object thisElem = get(i);
                    Object otherElem = other.get(i);

                    if(!Util.equalSafe(thisElem, otherElem))
                    {
                        return false;
                    }
                }

                return true;
            }
        }
        return false;
    }

    /**
     * Returns the hash code value for this tuple.
     * An empty tuple always has a hash code zero.
     * The computed value depends on the tuple elements and not on the
     * tuple implementation.
     */
    public int hashCode()
    {
        if(isEmpty())
        {
            return 0;
        }
        int hash = 7;
        for(int i=0; i<size(); i++)
        {
            Object elem = get(i);
            hash = 31 * hash + (null == elem ? 0 : elem.hashCode());
        }
        return hash;
    }
}
