/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.string;

import org.ckkloverdos.collection.IL;
import org.ckkloverdos.collection.IntArray;
import org.ckkloverdos.collection.L;
import org.ckkloverdos.java.JavaPlatform;
import org.ckkloverdos.log.StdLog;
import org.ckkloverdos.reflect.IReflectiveAccessor;
import org.ckkloverdos.reflect.ReflectUtil;
import org.ckkloverdos.util.Assert;
import org.ckkloverdos.util.ClassUtil;
import org.ckkloverdos.util.Util;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.text.DateFormat;
import java.util.*;

/**
 * Generic class for producing string representations of objects.
 *
 * <p>
 * The purpose of this class is twofold:
 * <ol>
 * <li> To be used in implementing an object's <code>toString()</code> method.
 * <li> To be used whenever a series of name-value pairs should be printed
 * together.
 * </ol>
 *
 * It works in two modes of operation, namely horizontal and vertical.
 * {@link java.util.Collection}s, {@link java.util.Map}s, arrays, primitive values and object graphs
 * are handled in a uniform way.
 *
 * This class can handle circular references and, with a little help from the
 * interested object, can also handle pretty printing for nested hierarchies.
 * The latter is achieved for a class by implementing the {@link IToStringAware}
 * interface.
 *
 * <p/>
 * In case you need to pretty-print a third-party class for which you do not
 * have access to or you should alter the source code in order implement
 * {@link org.ckkloverdos.string.IToStringAware}, then you can register an
 * instance of {@link org.ckkloverdos.string.IToStringAwareProvider}.
 *
 * <h3>Note</h3>
 * When building a <code>toString()</code> method for some object
 * <code>items</code> of class <code>A</code>, <b>do not</b> create other
 * instances of <code>A</code> in order to put them as part of
 * <code>items</code>'s string representation, since this will lead to
 * successive <code>toString()</code> calls on instances of <code>A</code>,
 * resulting to {@link StackOverflowError}.
 *
 * <h3>Examples</h3>
 * <ol>
 * <li> Implementing an object's <code>toString()</code> method.
 *
 * <pre>
 * public class Person
 * {
 * 	private String name;
 * 	private int age;
 *
 * 	public String toString()
 * 	{
 * 		ToString ts = new ToString(this, false);
 * 		ts.add(&quot;name&quot;, name);
 * 		ts.add(&quot;age&quot;, age);
 * 		return ts.toString();
 * 	}
 * }
 * </pre>
 *
 * Usually, objects are part of a larger hierarachy, so in order for them to be
 * pretty-printed correctly, they should conform to the {@link IToStringAware}
 * interface. In this case, all the code tha should go to the
 * <code>toString()</code> method, is now placed in the only method of
 * {@link IToStringAware}. So, the class above should be modified as follows:
 *
 * <pre>
 * public class Person implements IToStringAware
 * {
 * 	private String name;
 * 	private int age;
 *
 * 	public String toString()
 * 	{
 * 		ToString ts = new ToString(this, false);
 * 		toStringAware(ts);
 * 		return ts.toString();
 * 	}
 *
 * 	public void toStringAware(ToString ts)
 * 	{
 * 		ts.addFile(&quot;name&quot;, name);
 * 		ts.addFile(&quot;age&quot;, age);
 * 	}
 * }
 * </pre>
 *
 * The key difference is that while <code>toString()</code> can only be called from
 * an external object, <code>toStringAware()</code>, if it exists, can be detected
 * and subsequently called by an existing <code>ToString</code> instance. This instance
 * could have been created as part of an ancestor object containing a <code>Person</code>
 * object and since the pretty-printing is guaranteed within <code>ToString</code>,
 * the pretty-printing of the hierarachy up to <code>Person</code> is also guaranteed.
 *
 * <li> Common variable pretty-printing.
 *	<p/>
 *	Instead of doing the, rather common and ugly:
 *  <pre>
 *  sop("Invalid info for: customer=" + customer + ", id=" + id + ", address=" + address + ", phone=" + phone);
 *  </pre>
 *
 *	we can do the following:
 *  <pre>
 *  ToString ts = new ToString();
 *  ts.literal("Invalid info for: ")
 *    .add("customer", customer)
 *    .add("id", id)
 *    .add("address", address)
 *    .add("phone", phone);
 *  sop(ts);
 *  </pre>
 *
 * <li>Implementing {@link org.ckkloverdos.string.IToStringAwareProvider} in order
 * to enable pretty-printing for log4j loggers.
 * <pre>
 * import org.apache.log4j.Logger;
 * import org.ckkloverdos.collection.CollectionUtil;
 * import org.ckkloverdos.string.IToStringAwareProvider;
 * import org.ckkloverdos.string.ToString;
 * import java.util.Enumeration;
 * public class LoggerToStringAwareProvider implements IToStringAwareProvider
 * {
 *   public void toStringAware(Object o, ToString ts)
 *   {
 *     Logger log = (Logger) o;
 *     Enumeration appenders = log.getAllAppenders();
 *     ts.add("name", log.getName());
 *     ts.add("appenders", CollectionUtil.toList(appenders));
 *   }
 * }
 * ...
 * ToString.registerProvider(new LoggerToStringAwareProvider(), Logger.class);
 * </pre>
 * <p/>
 * </ol>
 *
 * @author Christos KK Loverdos
 *
 */
public class ToString implements Cloneable
{
    private static final Map providers = new HashMap();
    private static final String[] defaultExludedProperties = {"class"};

    private Object theObject;
    private IdentityHashMap visited;
    private IdentityHashMap markedCircular;
    private boolean multiline;
    private StringBuffer sb;
    private IntArray addedItemsPerLevel;
    private int nestLevel;

    private boolean made;

    private String arrayStart = "[";
    private String arrayEnd   = "]";
    private String mapStart   = "{";
    private String mapEnd     = "}";
    private String typeStart  = "(";
    private String typeEnd    = ")";
    private String indentationString = "  ";
    private int indentationMultiplier = 1;
    private boolean usingIndices = true;
    private String equalsString = "=";
    private String keyValueSeparator = "=";
    private String nullValue = "null";
    private String singleLineItemSeparator = ", ";
    private boolean endIndented = true;
    private boolean stringQuoted = true;
    private char stringQuote = '"';
    private boolean fullTypeNames = false;
    private boolean usingTypeNames = true;
    private String circularRefString = "REF";

    public ToString(Object theObject, boolean multiline, boolean fullTypeNames)
    {
        this.theObject = theObject;
        this.multiline = multiline;
        
        this.visited = new IdentityHashMap();
        this.markedCircular = new IdentityHashMap();
        this.sb = new StringBuffer();
        this.fullTypeNames = fullTypeNames;

        this.addedItemsPerLevel = new IntArray(3);
        this.addedItemsPerLevel.add(0); //

        if(null != theObject)
        {
            addTypeStart(theObject.getClass());
            increaseNestLevel();
        }
    }

    public ToString(Object theObject, boolean multiline)
    {
        this(theObject, multiline, false);
    }
    
    public ToString(Object theObject)
    {
        this(theObject, false);
    }

    public ToString(boolean multiline)
    {
        this(null, multiline);
    }

    public ToString()
    {
        this(null, false);
    }

    public static void registerProvider(IToStringAwareProvider provider, Class c)
    {
        Assert.notNull(provider, "provider must not be null");
        Assert.notNull(c, "class must not be null");
        providers.put(c, provider);
    }

    private static IToStringAwareProvider getProvider(Class c)
    {
        IToStringAwareProvider provider = (IToStringAwareProvider) providers.get(c);
        if(null != provider)
        {
            return provider;
        }

        // find the first super class...
        Iterator iter = providers.keySet().iterator();
        while(iter.hasNext())
        {
            Class otherClass = (Class) iter.next();
            if(otherClass.isAssignableFrom(c))
            {
                return (IToStringAwareProvider) providers.get(otherClass);
            }
        }
        return null;
    }

    protected Object clone() throws CloneNotSupportedException
    {
        return super.clone();
    }

    /**
     * Creates a new instance with the same internal state as this one.
     */
    public ToString save()
    {
        try
        {
            return (ToString) clone();
        }
        catch(CloneNotSupportedException e)
        {
            throw new RuntimeException(e);
        }
    }

    /**
     * Restores the internal state of this instance from another one.
     */
    public ToString restore(ToString ts)
    {
        theObject = ts.theObject;
        visited = ts.visited;
        markedCircular = ts.markedCircular;
        multiline = ts.multiline;
        sb = ts.sb;
        addedItemsPerLevel = ts.addedItemsPerLevel;
        nestLevel = ts.nestLevel;

        made = ts.made;

        arrayStart = ts.arrayStart;
        arrayEnd = ts.arrayEnd;
        mapStart = ts.mapStart;
        mapEnd = ts.mapEnd;
        typeStart = ts.typeStart;
        typeEnd = ts.typeEnd;
        indentationString = ts.indentationString;
        indentationMultiplier = ts.indentationMultiplier;
        usingIndices = ts.usingIndices;
        equalsString = ts.equalsString;
        keyValueSeparator = ts.keyValueSeparator;
        nullValue = ts.nullValue;
        singleLineItemSeparator = ts.singleLineItemSeparator;
        endIndented = ts.endIndented;
        stringQuoted = ts.stringQuoted;
        stringQuote = ts.stringQuote;
        fullTypeNames = ts.fullTypeNames;
        usingTypeNames = ts.usingTypeNames;
        circularRefString = ts.circularRefString;

        return this;
    }

    public String toString()
    {
        if(null != theObject && !made)
        {
            decreaseNestLevel();
            addTypeEnd();
            made = true;
        }
        return sb.toString();
    }

    public String getKeyValueSeparator()
    {
        return keyValueSeparator;
    }

    public ToString setKeyValueSeparator(String nameValueSeparator)
    {
        this.keyValueSeparator = nameValueSeparator;
        return this;
    }

    public String getTypeStart()
    {
        return typeStart;
    }

    public ToString setTypeStart(String typeStart)
    {
        this.typeStart = typeStart;
        return this;
    }

    public String getTypeEnd()
    {
        return typeEnd;
    }

    public ToString setTypeEnd(String typeEnd)
    {
        this.typeEnd = typeEnd;
        return this;
    }

    public String getMapEnd()
    {
        return mapEnd;
    }

    public ToString setMapEnd(String mapEnd)
    {
        this.mapEnd = mapEnd;
        return this;
    }

    public String getMapStart()
    {
        return mapStart;
    }

    public ToString setMapStart(String mapStart)
    {
        this.mapStart = mapStart;
        return this;
    }

    public boolean isUsingTypeNames()
    {
        return usingTypeNames;
    }

    public ToString setUsingTypeNames()
    {
        this.usingTypeNames = true;
        return this;
    }
    
    public ToString setUsingTypeNames(boolean usingTypeNames)
    {
        this.usingTypeNames = usingTypeNames;
        return this;
    }

    public String getCircularRefString()
    {
        return circularRefString;
    }

    public void setCircularRefString(String circularRefString)
    {
        this.circularRefString = circularRefString;
    }

    public boolean isFullTypeNames()
    {
        return fullTypeNames;
    }

    public ToString setFullTypeNames()
    {
        this.fullTypeNames = true;
        return this;
    }

    public ToString setFullTypeNames(boolean fullTypeNames)
    {
        this.fullTypeNames = fullTypeNames;
        return this;
    }

    public char getStringQuote()
    {
        return stringQuote;
    }

    public ToString setStringQuote(char stringQuote)
    {
        this.stringQuote = stringQuote;
        return this;
    }

    public boolean isStringQuoted()
    {
        return stringQuoted;
    }

    public ToString setStringQuoted()
    {
        this.stringQuoted = true;
        return this;
    }

    public ToString setStringQuoted(boolean stringQuoted)
    {
        this.stringQuoted = stringQuoted;
        return this;
    }

    public boolean isEndIndented()
    {
        return endIndented;
    }

    public ToString setEndIndented()
    {
        this.endIndented = true;
        return this;
    }

    public ToString setEndIndented(boolean arrayEndIndented)
    {
        this.endIndented = arrayEndIndented;
        return this;
    }

    public String getSingleLineItemSeparator()
    {
        return singleLineItemSeparator;
    }

    public ToString setSingleLineItemSeparator(String singleLineItemSeparator)
    {
        this.singleLineItemSeparator = singleLineItemSeparator;
        return this;
    }

    public String getNullValue()
    {
        return nullValue;
    }

    public ToString setNullValue(String nullValue)
    {
        this.nullValue = nullValue;
        return this;
    }

    public boolean isUsingIndices()
    {
        return usingIndices;
    }

    public String getEqualsString()
    {
        return equalsString;
    }

    public ToString setEqualsString(String equalsString)
    {
        this.equalsString = equalsString;
        return this;
    }

    public ToString setUsingIndices(boolean useIndices)
    {
        this.usingIndices = useIndices;
        return this;
    }

    private int getAddedItemsForThisLevel()
    {
        return addedItemsPerLevel.get(nestLevel);
    }

    private void increaseNestLevel()
    {
        nestLevel++;
        addedItemsPerLevel.push(0);
    }

    private void decreaseNestLevel()
    {
        nestLevel--;
        addedItemsPerLevel.pop();
    }

    private void postAddItem()
    {
        addedItemsPerLevel.increase(nestLevel);
    }

    private void newLine()
    {
        sb.append(JavaPlatform.LINE_SEPARATOR);
    }

    private void indent()
    {
        for(int l = 0; l < nestLevel; l++)
        {
            for(int i = 0; i < indentationMultiplier; i++)
            {
                sb.append(indentationString);
            }
        }
    }

    public String getIndentationString()
    {
        return indentationString;
    }

    public ToString setIndentationString(String indentationString)
    {
        this.indentationString = indentationString;
        return this;
    }

    public int getIndentationMultiplier()
    {
        return indentationMultiplier;
    }

    public ToString setIndentationMultiplier(int indentationMultiplier)
    {
        this.indentationMultiplier = indentationMultiplier;
        return this;
    }

    public String getArrayStart()
    {
        return arrayStart;
    }

    public void setArrayStart(String arrayStart)
    {
        this.arrayStart = arrayStart;
    }

    public String getArrayEnd()
    {
        return arrayEnd;
    }

    public void setArrayEnd(String arrayEnd)
    {
        this.arrayEnd = arrayEnd;
    }

    public boolean isMultiline()
    {
        return multiline;
    }

    public ToString setSingleline()
    {
        this.multiline = false;
        return this;
    }

    public ToString setMultiline()
    {
        this.multiline = true;
        return this;
    }

    public ToString setMultiline(boolean multiline)
    {
        this.multiline = multiline;
        return this;
    }

    private void addItemSeparator()
    {
        if(multiline)
        {
            int howmanyAtThisLevel = getAddedItemsForThisLevel();
            if(nestLevel > 0 || howmanyAtThisLevel > 0)
            {
                newLine();
                indent();
            }
        }
        else
        {
            int howmanyAtThisLevel = getAddedItemsForThisLevel();
            if(howmanyAtThisLevel > 0)
            {
                sb.append(singleLineItemSeparator);
            }
        }
    }

    private void preAddItem()
    {
        addItemSeparator();
    }

    private String getTypeName(Class c)
    {
        return usingTypeNames ? (fullTypeNames ? c.getName() : ClassUtil.getShortClassName(c)) : "";
    }

    private void addComplexTypeStart(Class c, String startString)
    {
        sb.append(getTypeName(c));
        sb.append(startString);
    }

    private void addComplexTypeEnd(String endString)
    {
        if(endIndented && multiline)
        {
            addItemSeparator();
        }
        sb.append(endString);
    }

    private void addArrayStart(Class c)
    {
        addComplexTypeStart(c, arrayStart);
    }

    private void addArrayEnd()
    {
        addComplexTypeEnd(arrayEnd);
    }

    private void addMapStart(Class c)
    {
        addComplexTypeStart(c, mapStart);
    }

    private void addMapEnd()
    {
        addComplexTypeEnd(mapEnd);
    }

    private void addTypeStart(Class c)
    {
        addComplexTypeStart(c, typeStart);
    }

    private void addTypeEnd()
    {
        addComplexTypeEnd(typeEnd);
    }

    public ToString append(Object value)
    {
        sb.append(value);
        return this;
    }

    public ToString add(String name, Object value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        addValue(value);

        postAddItem();
        return this;
    }

    public ToString addIfNotNull(String name, Object value)
    {
        return addIf(name, value, null != value);
    }

    public ToString addIf(String name, Object value, boolean condition)
    {
        if(condition)
        {
            add(name, value);
        }
        return this;
    }

    public ToString add(String name, Date value, DateFormat format)
    {
        return add(name, format.format(value));
    }

    public ToString add(String name, byte value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        sb.append(value);

        postAddItem();
        return this;
    }

    public ToString add(String name, boolean value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        sb.append(value);

        postAddItem();
        return this;
    }

    public ToString add(String name, char value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        sb.append(value);

        postAddItem();
        return this;
    }

    public ToString add(String name, double value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        sb.append(value);

        postAddItem();
        return this;
    }

    public ToString add(String name, float value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        sb.append(value);

        postAddItem();
        return this;
    }

    public ToString add(String name, int value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        sb.append(value);

        postAddItem();
        return this;
    }

    public ToString add(String name, long value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        sb.append(value);

        postAddItem();
        return this;
    }

    public ToString add(String name, short value)
    {
        preAddItem();

        sb.append(name);
        sb.append(equalsString);
        sb.append(value);

        postAddItem();
        return this;
    }

    public ToString add(Object o)
    {
        preAddItem();

        addValue(o);

        postAddItem();
        return this;
    }

    public ToString addIfNotNull(Object o)
    {
        if(null != o)
        {
            add(o);
        }
        return this;
    }

    public ToString add(byte o)
    {
        preAddItem();

        sb.append(o);

        postAddItem();
        return this;
    }

    public ToString add(boolean o)
    {
        preAddItem();

        sb.append(o);

        postAddItem();
        return this;
    }

    public ToString add(char o)
    {
        preAddItem();

        sb.append(o);

        postAddItem();
        return this;
    }

    public ToString add(double o)
    {
        preAddItem();

        sb.append(o);

        postAddItem();
        return this;
    }

    public ToString add(float o)
    {
        preAddItem();

        sb.append(o);

        postAddItem();
        return this;
    }

    public ToString add(int o)
    {
        preAddItem();

        sb.append(o);

        postAddItem();
        return this;
    }

    public ToString add(long o)
    {
        preAddItem();

        sb.append(o);

        postAddItem();
        return this;
    }

    public ToString add(short o)
    {
        preAddItem();

        sb.append(o);

        postAddItem();
        return this;
    }

    private void addStringValue(String s)
    {
        if(stringQuoted)
        {
            sb.append(stringQuote);
        }
        sb.append(s);
        if(stringQuoted)
        {
            sb.append(stringQuote);
        }
    }

    private void addToStringAwareProviderValue(Object o, IToStringAwareProvider provider)
    {
        addTypeStart(o.getClass());
        increaseNestLevel();
        provider.toStringAware(o, this);
        decreaseNestLevel();
        addTypeEnd();
    }

    private void addToStringAwareValue(IToStringAware o)
    {
        ToString save = null;
        if(o instanceof IPrepareToStringAware)
        {
            save = this.save();
            ((IPrepareToStringAware)o).prepareToStringAware(this);
        }
        addTypeStart(o.getClass());
        increaseNestLevel();
        o.toStringAware(this);
        decreaseNestLevel();
        addTypeEnd();
        if(o instanceof IPrepareToStringAware)
        {
            restore(save);
        }
    }

    public ToString addNotNullReflectiveProperties(Object obj, String[] exclude)
    {
        return addReflectiveProperties(obj, exclude, false);
    }

    public ToString addReflectiveProperties(Object obj, String[] exclude)
    {
        return addReflectiveProperties(obj, exclude, true);
    }

    public ToString addReflectiveProperties(Object obj, String[] exclude, boolean includeNullValues)
    {
        if(null == obj)
        {
            return this;
        }

        exclude = Util.safe(exclude);
        IL _ex = new L(new HashSet(2 * exclude.length)).addAll(exclude).addAll(defaultExludedProperties);

        try
        {
            Class c = obj.getClass();
            BeanInfo bi = Introspector.getBeanInfo(c);
            PropertyDescriptor[] pds = bi.getPropertyDescriptors();
            for(int i = 0; i < pds.length; i++)
            {
                PropertyDescriptor pd = pds[i];
                String name = pd.getName();
                if(_ex.contains(name))
                {
                    continue;
                }
                IReflectiveAccessor acc = ReflectUtil.getPropertyAccessor(c, name);
                Object value = acc.get(obj);

                if(null == value && !includeNullValues)
                {
                    continue;
                }
                addKeyValueItem(name, value);
            }
        }
        catch(IntrospectionException e)
        {
            add("!!ERROR addReflectiveProperties !!", e.getMessage());
        }

        return this;
    }

    public ToString addReflectiveProperties(Object obj)
    {
        return addReflectiveProperties(obj, defaultExludedProperties, true);
    }

    public ToString addReflectiveProperties()
    {
        return addReflectiveProperties(theObject, defaultExludedProperties, true);
    }

    public ToString addNotNullReflectiveProperties(Object obj)
    {
        return addReflectiveProperties(obj, defaultExludedProperties, false);
    }

    public ToString addNotNullReflectiveProperties()
    {
        return addReflectiveProperties(theObject, defaultExludedProperties, false);
    }

    public ToString addValue(Object o)
    {
        if(null == o)
        {
            addNullValue();
        }
        else if(isVisited(o))
        {
            addCircularValue(o);
        }
        else if(o instanceof String)
        {
            addStringValue((String) o);
        }
        else
        {
            setVisited(o);

            IToStringAwareProvider provider = getProvider(o.getClass());
            if(null != provider)
            {
                addToStringAwareProviderValue(o, provider);
            }
            else if(o.getClass().isArray())
            {
                addArrayValue(o);
            }
            else if(o instanceof IToStringAware)
            {
                addToStringAwareValue((IToStringAware) o);
            }
            else if(o instanceof Collection)
            {
                addCollectionValue((Collection) o);
            }
            else if(o instanceof Map)
            {
                addMapValue((Map) o);
            }
            else
            {
                addObjectValue(o);
            }

            setUnvisited(o);
        }

        return this;
    }

    private void addKeyValueItem(Object key, Object value)
    {
        preAddItem();

        sb.append(key);
        sb.append(keyValueSeparator);
        addValue(value);

        postAddItem();
    }

    private void addIndexedItem(int index, Object item)
    {
        preAddItem();

        if(usingIndices)
        {
            sb.append(index);
            sb.append(equalsString);
        }
        addValue(item);

        postAddItem();
    }

    private void addObjectValue(Object o)
    {
        sb.append(String.valueOf(o));
    }

    private void addMapValue(Map o)
    {
        Class c = o.getClass();

        addMapStart(c);
        increaseNestLevel();
        Iterator iter = o.keySet().iterator();
        while(iter.hasNext())
        {
            Object key = iter.next();
            Object value = o.get(key);
            addKeyValueItem(key, value);
        }
        decreaseNestLevel();
        addMapEnd();
    }

    private void addCollectionValue(Collection o)
    {
        Class c = o.getClass();

        addArrayStart(c);
        increaseNestLevel();
        Iterator iter = o.iterator();
        for(int i = 0; iter.hasNext(); i++)
        {
            addIndexedItem(i, iter.next());
        }
        decreaseNestLevel();
        addArrayEnd();
    }

    private void addArrayValue(Object o)
    {
        Class c = o.getClass();
        int length = Array.getLength(o);

        addArrayStart(c.getComponentType());
        increaseNestLevel();
        for(int i = 0; i < length; i++)
        {
            addIndexedItem(i, Array.get(o, i));
        }
        decreaseNestLevel();
        addArrayEnd();
    }

    private int getVisitedPosition(Object o)
    {
        Integer i = (Integer) this.visited.get(o);
        return i.intValue();
    }

    private void setVisited(Object o)
    {
        if(null != o)
        {
            this.visited.put(o, new Integer(sb.length()));
        }
    }

    private void setUnvisited(Object o)
    {
        if(null != o)
        {
            this.visited.remove(o);
        }
    }

    private boolean isVisited(Object o)
    {
        return this.visited.containsKey(o);
    }

    private void addNullValue()
    {
        sb.append(getNullValue());
    }

    private void addCircularValue(Object o)
    {
        int position = getVisitedPosition(o);
        boolean markedAgain = markedCircular.containsKey(o);
        int marker;
        if(!markedAgain)
        {
            marker = markedCircular.size() + 1;
            markedCircular.put(o, new Integer(marker));
        }
        else
        {
            marker = ((Integer) markedCircular.get(o)).intValue();
        }

        String label = "<" + circularRefString + ":" + marker + ">";
        int labelLen = label.length();

        if(!markedAgain)
        {
            sb.insert(position, label);
            Iterator iter = visited.keySet().iterator();
            while(iter.hasNext())
            {
                Object other = iter.next();
                Integer otherPosI = (Integer) visited.get(other);
                int otherPos = otherPosI.intValue();
                if(otherPos > position)
                {
                    visited.put(other, new Integer(otherPos + labelLen));
                }
            }
        }

        sb.append("<@" + circularRefString + ":" + marker + ">");
    }

    private static Integer i(int n)
    {
        return new Integer(n);
    }

    public static void main(String[] args) throws IntrospectionException
    {
        ToString ts = new ToString();
        ts.add(new int[]{1, 2, 3, 4});
        ts.add(new int[]{100, 200, 300, 400});
        StdLog.log(ts);
        StdLog.log("");

        ToString ts2 = new ToString();//.setMultiline();
        ts2.add(new Integer[]{i(1), i(2), i(3), i(4)});
        ts2.add(new Integer[]{i(100), i(200), i(300), i(400)});
        StdLog.log(JavaPlatform.LINE_SEPARATOR + ts2);
        StdLog.log("");

        Object[] arr1 = new Object[]{i(1), i(2), i(3), null, i(4)};
        List list1 = new L("one", "two", arr1, "three").toList();
        Object[] arr2 = new Object[]{
            i(2001),
            new Object[]{
                i(3001),
                arr1
            },
            i(4001),
            "Hello World!",
            list1,
        };
        Map map1 = new HashMap();
        map1.put("__one", arr1);
        map1.put("__two", list1);
        map1.put("__three", map1);

        list1.add(arr2);
        arr1[3] = list1;

        ToString ts3 = new ToString()
            .setMultiline(false)
            .setFullTypeNames(false)
            .setEndIndented(true)
            .setUsingIndices(true);
        ts3.add(arr2)
            .add("hello", new Object())
            .add("world", map1);
        StdLog.log(JavaPlatform.LINE_SEPARATOR + ts3);

        StdLog.log("*************************");
        ToString ts4 = ts3.save()
            .add(arr2)
            .add("hello", new Object())
            .add("world", map1);
        ;
        StdLog.log(JavaPlatform.LINE_SEPARATOR + ts4);

        ToString tmp = new ToString();
        ToString aaa = new ToString(tmp, true);
        aaa.addReflectiveProperties();
        StdLog.log("aaa = " + aaa);
    }
}
