/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.source;

import org.ckkloverdos.string.StringUtil;
import org.ckkloverdos.io.IOUtil;

import java.io.*;

/**
 * Basic, {@link java.io.InputStream}- based implementation of {@link org.ckkloverdos.source.ISource},
 * intended to be subclassed.
 *
 * @author Christos KK Loverdos
 */
public abstract class SourceSkeleton implements ISource
{
    protected String name;
    protected SourceType type;
    protected InputStream is;
    protected boolean valid;

    protected SourceSkeleton() {}

    private void checkName(String name)
    {
        if(null == name)
        {
            throw new NullPointerException("Null name");
        }
    }

    private void checkType(SourceType type)
    {
        if(null == type)
        {
            throw new NullPointerException("Null type");
        }
    }

//    protected SourceSkeleton(String name, SourceType type)
//    {
//        this(name, type, null);
//    }
    
    protected SourceSkeleton(String name, SourceType type, InputStream is)
    {
        checkName(name);
        checkType(type);

        this.name = name;
        this.type = type;
        this.is = is;
        this.valid = null != is;
    }

    public boolean isValid()
    {
        return valid;
    }

    public String getName()
    {
        return name;
    }

    public String getDescription()
    {
        return type + ":" + name;
    }

    public SourceType getType()
    {
        return type;
    }

    public InputStream getInputStream() throws IOException
    {
        return is;
    }

    public boolean isText()
    {
        return type.isText();
    }

    public String getString() throws IOException
    {
        return getString(StringUtil.UTF8);
    }

    public Reader getReader() throws IOException
    {
        return getReader(StringUtil.UTF8);
    }

    public String getString(String encoding) throws IOException
    {
        return IOUtil.readString(is, encoding);
    }

    public Reader getReader(String encoding) throws IOException
    {
        return new InputStreamReader(is, encoding);
    }

    protected void setValid(boolean valid)
    {
        this.valid = valid;
    }

    protected void setInputStream(InputStream is)
    {
        this.is = is;
    }

    protected void setName(String name)
    {
        checkName(name);
        this.name = name;
    }

    protected void setType(SourceType type)
    {
        checkType(type);
        this.type = type;
    }
}