/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.db;

import org.ckkloverdos.db.proxy.DriverInvocationHandler;
import org.ckkloverdos.util.ClassUtil;

import java.lang.reflect.Proxy;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.SQLException;

/**
 * DB-related utility methods.
 * @author Christos KK Loverdos
 */

public final class DbUtil
{
    private DbUtil()
    {
    }

    /**
     * Registers the JDBC driver implemented by the class having the provided <code>className</code>.
     * The context class loader is used to load the class.
     *
     * @param className
     * @throws SQLException
     * @see org.ckkloverdos.db.DbUtil#registerDriver(String, ClassLoader)
     */
    public static Driver registerDriver(String className) throws SQLException
    {
        return registerDriver(className, ClassUtil.getContextClassLoader());
    }

    /**
     * Registers the JDBC driver implemented by the class having the provided <code>className</code>.
     * The provided class loader is used to load the class.
     *
     * <p/>
     * Note that according to the Javadocs, in order for the driver to be accessible by {@link java.sql.DriverManager},
     * the same class loader as the one that loaded the application/applet is used. This causes
     * problems in the case we have dynamically loaded our driver by using, for instance, a
     * {@link org.ckkloverdos.util.CompositeClassLoader}. In order to overcome the inconvenience,
     * the implementation of this method uses dynamic {@link java.lang.reflect.Proxy proxies}.
     *
     * @param className
     * @param cl
     * @throws SQLException
     * @see DriverInvocationHandler
     */
    public static Driver registerDriver(String className, ClassLoader cl) throws SQLException
    {
        Driver driver = (Driver) ClassUtil.newInstance(className, cl);
        if(null == driver)
        {
            return null;
        }

        Driver proxy = (Driver) Proxy.newProxyInstance(ClassLoader.getSystemClassLoader(), new Class[]{Driver.class}, new DriverInvocationHandler(driver));
        DriverManager.registerDriver(proxy);
        return proxy;
    }
}
