/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.collection;

import org.ckkloverdos.filter.IFilter;
import org.ckkloverdos.function.IFilterFunction;
import org.ckkloverdos.function.IFilterProcedure;
import org.ckkloverdos.function.IFunction;
import org.ckkloverdos.function.IProcedure;
import org.ckkloverdos.hint.BinaryHint;
import org.ckkloverdos.reflect.IReflectiveAccessor;
import org.ckkloverdos.reflect.ReflectUtil;
import org.ckkloverdos.string.ToString;
import org.ckkloverdos.tuple.Pair;
import org.ckkloverdos.util.Util;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.lang.reflect.Array;

/**
 * Basic implementation of {@link org.ckkloverdos.collection.IL}.
 *
 * <p/>
 * The list is a wrapper for the underlying backing store, which is either a {@link java.util.Collection} or an
 * array of objects and treats theses two cases uniformly.
 *
 * <p/>
 * Arrays of primitive types are not supported, so this will throw an
 * exception:
 * <pre>
 * L l = new L(new int[]{1, 2, 3});
 * System.out.println(l);
 * </pre>
 * 
 * @author Christos KK Loverdos
 */
public class L implements IL
{
    protected Collection col;
    protected Object[] array;

    private static final IFunction FUNCTION_chopPrefixRE = new IFunction()
    {
        public Object evaluate(Object o, Object hints)
        {
            if(null == o)
            {
                return o;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            Matcher matcher = p.matcher(so);
            if(matcher.find())
            {
                String prefix = matcher.group(1);
                return so.substring(prefix.length());
            }
            else
            {
                return o;
            }
        }
    };

    private static final IFunction FUNCTION_chopSuffixRE = new IFunction()
    {
        public Object evaluate(Object o, Object hints)
        {
            if(null == o)
            {
                return o;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            Matcher matcher = p.matcher(so);
            if(matcher.find())
            {
                String suffix = matcher.group(1);
                return so.substring(0, so.lastIndexOf(suffix));
            }
            else
            {
                return o;
            }
        }
    };

    private static final IFilter FILTER_filterStartsWith = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o || null == hints)
            {
                return false;
            }
            String so = (String) o;
            String s = (String) hints;
            return so.startsWith(s);
        }
    };
    
    private static final IFilter FILTER_filterNotStartsWith = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o || null == hints)
            {
                return true;
            }
            String so = (String) o;
            String s = (String) hints;
            return !so.startsWith(s);
        }
    };

    private static final IFilter FILTER_filterEndsWith = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o || null == hints)
            {
                return false;
            }
            String so = (String) o;
            String s = (String) hints;
            return so.endsWith(s);
        }
    };

    private static final IFilter FILTER_filterNotEndsWith = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o || null == hints)
            {
                return true;
            }
            String so = (String) o;
            String s = (String) hints;
            return !so.endsWith(s);
        }
    };

    private static final IFilter FILTER_filterFindRE = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o)
            {
                return false;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            return p.matcher(so).find();
        }
    };

    private static final IFilter FILTER_filterNotFindRE = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o)
            {
                return true;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            return !p.matcher(so).find();
        }
    };

    private static final IFilter FILTER_filterFindRE_m = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o)
            {
                return false;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            return p.matcher(so).find();
        }
    };

    private static final IFilter FILTER_filterNotFindRE_m = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o)
            {
                return true;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            return !p.matcher(so).find();
        }
    };

    private static final IFilter FILTER_filterMatchesRE = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o)
            {
                return false;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            return p.matcher(so).matches();
        }
    };

    private static final IFilter FILTER_filterNotMatchesRE = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o)
            {
                return true;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            return !p.matcher(so).matches();
        }
    };

    private static final IFilter FILTER_filterMatchesRE_m = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o)
            {
                return false;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            return p.matcher(so).matches();
        }
    };
    
    private static final IFilter FILTER_filterNotMatchesRE_m = new IFilter()
    {
        public boolean accept(Object o, Object hints)
        {
            if(null == o)
            {
                return true;
            }
            String so = (String) o;
            Pattern p = (Pattern) hints;
            return !p.matcher(so).matches();
        }
    };


    /**
     * Constructs a new list. The backing store is a {@link List}.
     */
    public L()
    {
        this.col = new ArrayList();
    }

    /**
     * Constructs a new list. The backing store used depends on the type of <code>o</code>.
     * In particular:
     * <ul>
     *  <li> If <code>o</code> is <code>null</code>, an empty {@link java.util.List} is used.
     *  <li> If <code>o</code> is a {@link java.util.Collection}, it is used as the backing store.
     *  <li> If <code>o</code> is an <code>Object</code> {@link Class#isArray() array}, it used as the backing store.
     *  <li> Otherwise a singleton {@link java.util.Collections#singletonList(Object) list} containing <code>o</code> is used as the backing store.
     * </ul>
     *
     * @param o
     */
    public L(Object o)
    {
        if(null == o)
        {
            col = Collections.EMPTY_LIST;
        }
        else if(o instanceof Collection)
        {
            col = (Collection) o;
        }
        else if(o.getClass().isArray())
        {
            array = (Object[]) o;
        }
        else
        {
            col = Collections.singletonList(o);
        }
    }

    /**
     * Constructs a new list with an <code>Object</code> array containing the two
     * objects as the backing store.
     */
    public L(Object a, Object b)
    {
        this(new Object[]{a, b});
    }

    /**
     * Constructs a new list with an <code>Object</code> array containing the three
     * objects as the backing store.
     */
    public L(Object a, Object b, Object c)
    {
        this(new Object[]{a, b, c});
    }

    /**
     * Constructs a new list with an <code>Object</code> array containing the four
     * objects as the backing store.
     */
    public L(Object a, Object b, Object c, Object d)
    {
        this(new Object[]{a, b, c, d});
    }

    /**
     * Constructs a new list with an <code>Object</code> array containing the five
     * objects as the backing store.
     */
    public L(Object a, Object b, Object c, Object d, Object e)
    {
        this(new Object[]{a, b, c, d, e});
    }

    /**
     * Constructs a new list with the provided <code>collection</code> as the backing store.
     */
    public L(Collection collection)
    {
        this.col = null == collection ? newList() : collection;
    }

    /**
     * Constructs a new list withe the provided <code>set</code> as the backing store.
     * @param set
     */
    public L(Set set)
    {
        this.col = null == set ? newSet() : set;
    }

    /**
     * Constructs a new list from the elements provided by the <code>iterator</code>.
     */
    public L(Iterator iterator)
    {
        this.col = null == iterator ? newList() : newCollection(iterator);
    }

    /**
     * Constructs a new list with the provided array as the backing store.
     */
    public L(Object[] array)
    {
        this.array = null == array ? new Object[0] : array;
    }

    protected SortedSet newSortedSet()
    {
        return new TreeSet();
    }

    protected SortedSet newSortedSet(Comparator c)
    {
        return new TreeSet(c);
    }

    protected Set newSet()
    {
        if(isSortedSet())
        {
            return newSortedSet();
        }
        return new HashSet();
    }

    protected Set newSet(Collection withElements)
    {
        Set s = newSet();
        s.addAll(withElements);
        return s;
    }

    protected Set newSet(Iterator withElements)
    {
        Set s = newSet();
        addAll(s, withElements);
        return s;
    }

    protected SortedSet newSortedSet(Collection withElements)
    {
        SortedSet s = newSortedSet();
        s.addAll(withElements);
        return s;
    }

    protected SortedSet newSortedSet(Collection withElements, Comparator c)
    {
        SortedSet s = newSortedSet(c);
        s.addAll(withElements);
        return s;
    }

    protected Set newSet(Object[] withElements)
    {
        Set s = newSet();
        addAll(s, withElements);
        return s;
    }

    protected SortedSet newSortedSet(Object[] withElements)
    {
        SortedSet s = newSortedSet();
        addAll(s, withElements);
        return s;
    }

    private Collection addAll(Collection c, Iterator other)
    {
        while(other.hasNext())
        {
            c.add(other.next());
        }
        return c;
    }

    private Collection addAll(Collection c, Object[] array)
    {
        for(int i = 0; i < array.length; i++)
        {
            c.add(array[i]);
        }
        return c;
    }

    protected Collection newCollection()
    {
        if(isSet())
        {
            return newSet();
        }
        return newList();
    }

    protected Collection newCollection(Collection withItems)
    {
        Collection c = newCollection();
        c.addAll(withItems);
        return c;
    }

    protected Collection newCollection(Iterator withItems)
    {
        Collection c = newCollection();
        addAll(c, withItems);
        return c;
    }

    protected Collection newCollection(Object[] withItems)
    {
        Collection c = newCollection();
        addAll(c, withItems);
        return c;
    }

    protected List newList()
    {
        if(null != this.col && this.col instanceof LinkedList)
        {
            return new LinkedList();
        }
        return new ArrayList();
    }

    protected List newList(Collection withElements)
    {
        List r = newList();
        r.addAll(withElements);
        return r;
    }

    protected List newList(Iterator withElements)
    {
        List r = newList();
        addAll(r, withElements);
        return r;
    }

    protected List newList(Object[] withElements)
    {
        List r = newList();
        addAll(r, withElements);
        return r;
    }

    /**
     * Returns an iterator over the elements in this list.
     */
    public Iterator iterator()
    {
        if(null != col)
        {
            return col.iterator();
        }
        else
        {
            return new ArrayIterator(array);
        }
    }

    /**
     * Returns an iterator over the elements in this list that are
     * {@link org.ckkloverdos.filter.IFilter#accept(Object,Object) accepted}
     * by the given <code>filter</code>.
     * @param filter
     */
    public Iterator iterator(IFilter filter)
    {
        return iterator(filter, null);
    }

    /**
     * Returns an iterator over the elements in this list that are
     * {@link org.ckkloverdos.filter.IFilter#accept(Object,Object) accepted}
     * by the given <code>filter</code>.
     *
     * The <code>hints</code> parameter is passed directly to the <code>filter</code>.
     * @param filter
     * @param hints
     */
    public Iterator iterator(IFilter filter, Object hints)
    {
        return filter(filter, hints).iterator();
    }

    /**
     * Returns the number of elements in this list.
     */
    public int size()
    {
        if(null != col)
        {
            return col.size();
        }
        else if(null != array)
        {
            return array.length;
        }
        return 0;
    }

    /**
     * Returns true if this list contains the specified element.
     * @param o
     */
    public boolean contains(Object o)
    {
        if(isCollection())
        {
            return this.col.contains(o);
        }
        else
        {
            for(int i = 0; i < array.length; i++)
            {
                if(Util.equalSafe(o, array[i]))
                {
                    return true;
                }
            }
            return false;
        }
    }

    /**
     * Returns <code>true</code> if this list wraps an <code>Object</code> array.
     */
    public boolean isArray()
    {
        return null != this.array;
    }

    /**
     * Returns <code>true</code> if this list wraps a {@link java.util.Collection}.
     */
    public boolean isCollection()
    {
        return null != this.col;
    }

    /**
     * Returns <code>true</code> if this list wraps a {@link java.util.Collection}
     * and more specifically a {@link java.util.Set}.
     */
    public boolean isSet()
    {
        return null != col && col instanceof Set;
    }

    private boolean isSortedSet()
    {
        return null != col && col instanceof SortedSet;
    }

    /**
     * Returns <code>true</code> if this list wraps a {@link java.util.Collection}
     * and more specifically a {@link java.util.List}.
     */
    public boolean isList()
    {
        return null != col && col instanceof List;
    }

    /**
     * Returns an array containing all of the elements in this collection.
     * If this list is already wrapping an <code>Object</code> array, this array
     * is returned.
     */
    public Object[] toArray()
    {
        if(isCollection())
        {
            return col.toArray();
        }
        else
        {
            return array;
        }
    }

    /**
     * Returns an array containing all of the elements in this collection;
     * the runtime type of the returned array is that of the specified class.
     * If this list is already wrapping an <code>Object</code> array, this array
     * is returned and it MUST be of the given type.
     */
    public Object[] toArray(Class arrayClass)
    {
        if(isCollection())
        {
            return col.toArray((Object[]) Array.newInstance(arrayClass.getComponentType(), 0));
        }
        else
        {
            Object newArray;
            if(array.getClass().equals(arrayClass))
            {
                newArray = array;
            }
            else
            {
                Class componentType = arrayClass.getComponentType();
                newArray = Array.newInstance(componentType, array.length);
                System.arraycopy(array, 0, newArray, 0, array.length);
            }
            return (Object[]) newArray;
        }
    }

    /**
     * Convenience method that returns the strings of this list, assuming of course that
     * this list contains strings.
     */
    public String[] toStringArray()
    {
        return (String[]) toArray(String[].class);
    }

    /**
     * Returns a {@link java.util.List} containing the elements of this array.
     * If this list is already wrapping a {@link java.util.List}, then it is returned
     * intact, otherwise a new one is created.
     */
    public List toList()
    {
        if(isCollection())
        {
            if(isList())
            {
                return (List) this.col;
            }
            else
            {
                return newList(this.col);
            }
        }
        else
        {
            return newList(array);
        }
    }

    /**
     * Returns a {@link java.util.Set} containing the unique elements of this array.
     * If this list is already wrapping a {@link java.util.Set}, then it is returned
     * intact, otherwise a new one is created.
     */
    public Set toSet()
    {
        if(isSet())
        {
            return (Set) this.col;
        }
        else if(isCollection())
        {
            return newSet(this.col);
        }
        else
        {
            return newSet(this.array);
        }
    }

    /**
     * Returns a {@link java.util.SortedSet} containing the unique elements of this array.
     * If this list is already wrapping a {@link java.util.SortedSet}, then it is returned
     * intact, otherwise a new one is created.
     */
    public SortedSet toSortedSet()
    {
        if(isCollection())
        {
            if(isSortedSet())
            {
                return (SortedSet) this.col;
            }
            else
            {
                return newSortedSet(this.col);
            }
        }
        else
        {
            return newSortedSet(this.array);
        }
    }

    /**
     * Returns a {@link java.util.Collection} containing the elements of this array.
     * If this list is already wrapping a {@link java.util.Collection}, then it is returned
     * intact, otherwise a new one is created.
     */
    public Collection toCollection()
    {
        if(isCollection())
        {
            return this.col;
        }
        else
        {
            return newList(this.array);
        }
    }

    /**
     * Returns the <code>i</code>-th element of this list.
     * If the underlying backing store is a set, then the order
     * is that of the set iterator.
     * @param i
     */
    public Object get(int i)
    {
        if(isList())
        {
            return ((List) this.col).get(i);

        }
        else if(isArray())
        {
            return array[i];
        }
        else
        {
            if(i < 0 || i >= size())
            {
                throw new IndexOutOfBoundsException("Bad index " + i);
            }
            Iterator iter = iterator();
            for(int index = 0; index != i; index++)
            {
                iter.next();
            }
            return iter.next();
        }
    }

    /**
     * Convenience method that returns the <code>i</code>-th string of this list,
     * assuming that the list contains strings.
     * @param i
     */
    public String getString(int i)
    {
        return (String) get(i);
    }

    /**
     * Returns the first element of this list, retrieved from the {@link #iterator() iterator}.
     * @throws IndexOutOfBoundsException of the list has no elements.
     */
    public Object head()
    {
        if(0 == size())
        {
            throw new IndexOutOfBoundsException("No elements in the list");
        }
        return iterator().next();
    }

    /**
     * Returns the list constructed from this one if we remove the {@link #head()}.
     * If this list has no elements, an empty one is returned.
     */
    public IL tail()
    {
        return takeAfter(0);
    }

    private IL takeAfter(int n)
    {
        if(n < 0 || n > size() - 1)
        {
            return CollectionProxy.newLike(this).toL();
        }

        n++;
        Iterator iter = iterator();
        while(n > 0)
        {
            iter.next();
            n--;
        }

        CollectionProxy proxy = CollectionProxy.newLike(this);
        if(iter.hasNext())
        {
            proxy.beginMassiveAdd();
            while(iter.hasNext())
            {
                proxy.add(iter.next());
            }
            proxy.endMassiveAdd();
        }

        return proxy.toL();
    }

    /**
     * Returns the first <code>n</code> elements of this list, <code>n</code> starting from one.
     * If the list contains less than <code>n</code> elements, all are returned.
     * If <code>n</code> is less than one, an empty list is returned.
     *
     * <h3>Example:</h3>
     * <pre>
     * IL il = new L("zero", "one", "two", "three", "four");
     * StdLog.log("il = " + il.take(3));
     * </pre>
     * prints:
     * <pre>
     * [0="zero", 1="one", 2="two"]
     * </pre>
     *
     * @param n
     */
    public IL take(int n)
    {
        if(n < 1)
        {
            return CollectionProxy.newLike(this).toL();
        }

        Iterator iter = iterator();
        CollectionProxy proxy = CollectionProxy.newLike(this);
        proxy.beginMassiveAdd();
        while(n > 0 && iter.hasNext())
        {
            proxy.add(iter.next());
            n--;
        }
        proxy.endMassiveAdd();

        return proxy.toL();
    }

    /**
     * Returns the first element of this list that is accepted by <code>filter</code>.
     * Elements are retrieved from the {@link #iterator() iterator}.
     * @throws IndexOutOfBoundsException of the list has no elements or no element
     * is accepted by <code>filter</code>.
     */
    public Object head(IFilter filter)
    {
        return head(filter, null);
    }

    /**
     * Returns the first element of this list that is accepted by <code>filter</code>.
     * Elements are retrieved from the {@link #iterator() iterator}.
     * The second parameter <code>hints</code> is directly passed to the <code>filter</code>.
     *
     * @throws IndexOutOfBoundsException of the list has no elements or no element
     * is accepted by <code>filter</code>.
     */
    public Object head(IFilter filter, Object hints)
    {
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(filter.accept(o, hints))
            {
                return o;
            }
        }

        throw new IndexOutOfBoundsException("No element accepted by " + filter);
    }

    /**
     * Returns a new list with all the elements that are accepted by <code>filter</code>.
     * @param filter
     */
    public IL filter(IFilter filter)
    {
        return filter(filter, null);
    }

    /**
     * Returns a new list with all the elements that are accepted by <code>filter</code>.
     * The second parameter <code>hints</code> is directly passed to the <code>filter</code>.
     *
     */
    public IL filter(IFilter filter, Object hints)
    {
        CollectionProxy proxy = CollectionProxy.newLike(this);
        proxy.beginMassiveAdd();
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(filter.accept(o, hints))
            {
                proxy.add(o);
            }
        }
        proxy.endMassiveAdd();
        return proxy.toL();
    }

    /**
     * Returns a new list containing all the elements of this list, mapped by <code>function</code>.
     * @param function
     */
    public IL map(IFunction function)
    {
        return map(function, null);
    }

    /**
     * Returns a new list containing all the elements of this list, mapped by <code>function</code>.
     * The second parameter <code>hints</code> is directly passed to the <code>function</code>.
     */
    public IL map(IFunction function, Object hints)
    {
        CollectionProxy proxy = CollectionProxy.newLike(this);
        proxy.beginMassiveAdd();
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            proxy.add(function.evaluate(o, hints));
        }
        proxy.endMassiveAdd();
        return proxy.toL();
    }

    /**
     * Combines filtering and mapping in one place.
     * @param ff
     */
    public IL filterMap(IFilterFunction ff)
    {
        return filterMap(ff, null);
    }

    /**
     * Combines filtering and mapping in one place.
     * The second parameter <code>hints</code> is directly passed to <code>ff</code>.
     *
     * <p/>
     * If <code>hints</code> is an implementation of {@link org.ckkloverdos.hint.BinaryHint},
     * then {@link org.ckkloverdos.hint.BinaryHint#getHintA()} is passed to {@link IFilterFunction#accept(Object,Object)}
     * and {@link org.ckkloverdos.hint.BinaryHint#getHintB()} is passed to {@link IFilterFunction#evaluate(Object,Object)}.
     *
     * @see org.ckkloverdos.hint.BinaryHint
     * @param ff
     */
    public IL filterMap(IFilterFunction ff, Object hints)
    {
        BinaryHint bh = BinaryHint.fromHint(hints);
        Object hintA = bh.getHintA();
        Object hintB = bh.getHintB();
        
        CollectionProxy proxy = CollectionProxy.newLike(this);
        proxy.beginMassiveAdd();
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(ff.accept(o, hintA))
            {
                proxy.add(ff.evaluate(o, hintB));
            }
        }
        proxy.endMassiveAdd();
        return proxy.toL();
    }

    /**
     * Returns a new list with the JavaBean properties of the given <code>name</code>
     * for each element.
     *
     * <p/>
     * If <code>name</code> ends with <code>()</code>, then the name is treated as a method,
     * which is used to give the new elements.
     *
     * <h3>Example</h3>
     * The code:
     * <pre>
     * IL il = new L("zero", "one", "two", "three", "four");
     * System.out.println(il.selectProperty("length()"));
     * </pre>
     * prints:
     * <pre>
     * [0=4, 1=3, 2=3, 3=5, 4=4]
     * </pre>
     */
    public IL selectProperty(String propertyName)
    {
        boolean isMethodCall = propertyName.endsWith("()");
        if(isMethodCall)
        {
            propertyName = propertyName.substring(0, propertyName.length() - "()".length());
        }

        CollectionProxy proxy = CollectionProxy.newLike(this);
        proxy.beginMassiveAdd();
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(null == o)
            {
                continue;
            }

            Class c = o.getClass();
            IReflectiveAccessor reflectiveAccessor;

            reflectiveAccessor = ReflectUtil.getAccesorForSelect(c, propertyName, isMethodCall);

            proxy.add(reflectiveAccessor.get(o));
        }
        proxy.endMassiveAdd();
        return proxy.toL();
    }

    /**
     * Processes all alements with <code>procedure</code>.
     */
    public void forEach(IProcedure function)
    {
        forEach(function, null);
    }

    /**
     * Processes all alements with <code>procedure</code>.
     * The second parameter <code>hints</code> is directly passed to <code>procedure</code>.
     */
    public void forEach(IProcedure function, Object hints)
    {
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            function.process(o, hints);
        }
    }

    /**
     * Combines filtering and processing in one place.
     * The elements to be processed are those accepted by the filter.
     */
    public void filterForEach(IFilterProcedure function)
    {
        filterForEach(function, null);
    }

    /**
     * Combines filtering and processing in one place.
     * The elements to be processed are those accepted by the filter.
     * The second parameter <code>hints</code> is directly passed to <code>procedure</code>.
     *
     * <p/>
     * If <code>hints</code> is an implementation of {@link org.ckkloverdos.hint.BinaryHint},
     * then {@link org.ckkloverdos.hint.BinaryHint#getHintA()} is passed to {@link org.ckkloverdos.function.IFilterProcedure#accept(Object,Object)}
     * and {@link org.ckkloverdos.hint.BinaryHint#getHintB()} is passed to {@link org.ckkloverdos.function.IFilterProcedure#process(Object,Object)}.
     *
     * @see org.ckkloverdos.hint.BinaryHint
     */
    public void filterForEach(IFilterProcedure function, Object hints)
    {
        BinaryHint bh = BinaryHint.fromHint(hints);
        Object hintA = bh.getHintA();
        Object hintB = bh.getHintB();
        
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(function.accept(o, hintA))
            {
                function.process(o, hintB);
            }
        }
    }

    /**
     * Returns a copy of this list. The underlying backing store (collection or array)
     * is preserved in the new list.
     */
    public IL copy()
    {
        return CollectionProxy.newLike(this).addAll(this).toL();
    }

    /**
     * Adds the element to the list. The addition is made directly at the backing store and
     * no new list is created.
     *
     * <p/>
     * If the backing store is an array, then the component type of the array
     * must be assignable from the class of the new element, unless the new
     * element is <code>null</code>.
     *
     * <h3>Example 1</h3>
     * The code:
     * <pre>
     * IL l = new L("zero", "one", "two", "three", "four");
     * System.out.println(l.add(new Integer(1000)));
     * </pre>
     * will print:
     * <pre>
     * [0="zero", 1="one", 2="two", 3="three", 4="four", 5=1000]
     * </pre>
     *
     * <h3>Example 2</h3>
     * The code:
     * <pre>
     * IL l = new L(new String[]{"zero", "one", "two", "three", "four"});
     * System.out.println(l.add(new Integer(1000)));
     * </pre>
     * will throw an exception: <code>java.lang.IllegalArgumentException: array element type mismatch</code>.
     *
     * <h3>Example 3</h3>
     * The code:
     * <pre>
     * IL l = new L(new Number[] {new Long(1), new Float(2.2)});
     * System.out.println(l.add(new Integer(1000)));
     * </pre>
     * will print:
     * <pre>
     * [0=1, 1=2.2, 2=1000]
     * </pre>
     * @param o
     * @return this list with the new element added.
     */
    public IL add(Object o) // this is destructive
    {
        setFrom(new CollectionProxy(this).add(o));
        return this;
    }

    /**
     * Adds all the elements of the collection to this list.
     * The comments of {@link #add(Object)} apply here as well.
     * @param c
     */
    public IL addAll(Collection c) // this is destructive
    {
        return setFrom(new CollectionProxy(this).addAll(c));
    }

    /**
     * Adds all the elements of the <code>array</code> to this list.
     * The comments of {@link #add(Object)} apply here as well.
     * @param array
     */
    public IL addAll(Object[] array)
    {
        return setFrom(new CollectionProxy(this).addAll(array));
    }

    /**
     * Adds all the elements of the given list to this one.
     * The comments of {@link #add(Object)} apply here as well.
     * @param l
     */
    public IL addAll(IL l)
    {
        return setFrom(new CollectionProxy(this).addAll(l));
    }

    /**
     * Removes the element from the list (only once).
     * The deletion is made directly at the backing store and
     * no new list is created.
     * @param o
     */
    public IL remove(Object o)
    {
        return setFrom(new CollectionProxy(this).remove(o));
    }

    /**
     * Completely removes the element from the list.
     * The deletion is made directly at the backing store and
     * no new list is created.
     * @param o
     */
    public IL clear(Object o)
    {
        return setFrom(new CollectionProxy(this).clear(o));
    }

    /**
     * Sets the elements of this list from those of the collection <code>proxy</code>.
     * This is lightweight and just updates the underlying backing store
     * reference with that of the <code>proxy</code>.
     * @param proxy
     */
    public IL setFrom(CollectionProxy proxy)
    {
        if(proxy.isCollection())
        {
            col = proxy.getCollection();
        }
        else
        {
            array = proxy.getArray();
        }

        return this;
    }

    public void toStringAware(ToString ts)
    {
        ToString ts2 = new ToString().setUsingTypeNames(false);
        if(null != this.col)
        {
            ts2.add(this.col);
        }
        else
        {
            ts2.add(this.array);
        }

        ts.add(ts2);
    }

    /**
     * Returns <code>true</code> iff all the elements can be accepted by the <code>filter</code>.
     * In case the list is empty, <code>false</code> is returned.
     * @param filter
     */
    public boolean all(IFilter filter)
    {
        return all(filter, null);
    }

    /**
     * Returns <code>true</code> iff all the elements can be accepted by the <code>filter</code>.
     * In case the list is empty, <code>false</code> is returned.
     * The second parameter <code>hints</code> is directly passed to <code>filter</code>.
     * @param filter
     * @param hints
     */
    public boolean all(IFilter filter, Object hints)
    {
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(!filter.accept(o, hints))
            {
                return false;
            }
        }

        return size() > 0;
    }

    /**
     * Returns <code>true</code> iff all the elements can be accepted by the <code>filter</code>.
     * In case the list is empty, <code>true</code> is returned.
     * @param filter
     */
    public boolean any(IFilter filter)
    {
        return any(filter, null);
    }

    /**
     * Returns <code>true</code> iff all the elements can be accepted by the <code>filter</code>.
     * In case the list is empty, <code>true</code> is returned.
     * The second parameter <code>hints</code> is directly passed to <code>filter</code>.
     * @param filter
     * @param hints
     */
    public boolean any(IFilter filter, Object hints)
    {
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(filter.accept(o, hints))
            {
                return true;
            }
        }

        return 0 == size();
    }

    /**
     * Returns a set containing the unique elements of this list.
     * If the underlying backing store is already a set, then it is returned as is.
     */
    public IL unique()
    {
        if(isSet())
        {
            return this;
        }

        Set unique = newSet();
        addAll(unique, iterator());
        return new L(unique);
    }

    /**
     * Implements list zipping. The elements of the new list are instances of
     * {@link org.ckkloverdos.tuple.Pair}. The backing store of the returned
     * list is a {@link List}.
     * @param other
     */
    public IL zip(IL other)
    {
        List r = newList();
        Iterator iThis = this.iterator();
        Iterator iOther = other.iterator();

        while(iThis.hasNext() && iOther.hasNext())
        {
            r.add(new Pair(iThis.next(), iOther.next()));
        }

        return new L(r);
    }

    /**
     * Sorts the elements of the list.
     */
    public IL sort()
    {
        if(isSortedSet())
        {
            return this;
        }
        else if(isSet())
        {
            SortedSet ss = newSortedSet(this.col);
            return new L(ss);
        }
        else if(isCollection())
        {
            List l = toList();
            Collections.sort(l);
            return new L(l);
        }
        else
        {
            Object[] o = toArray();
            Arrays.sort(o);
            return new L(o);
        }
    }

    /**
     * Sorts the elements of the list, using the given <code>Comparator</code>.
     */
    public IL sort(Comparator c)
    {
        if(isSet())
        {
            SortedSet ss = newSortedSet(this.col, c);
            return new L(ss);
        }
        else if(isCollection())
        {
            List l = toList();
            Collections.sort(l, c);
            return new L(l);
        }
        else
        {
            Object[] o = toArray();
            Arrays.sort(o, c);
            return new L(o);
        }
    }

    private Collection newCombinedCollection(IL other)
    {
        if(isSet() && other.isSet())
        {
            return newSet();
        }
        return newList();
    }

    /**
     * Returns a new list containg the elements of this list along with the
     * elements of <code>other</code>.
     *
     * <p/>
     * If the underlying backing store is a set for both lists, then the method
     * implements the set-theoretic union operation.
     * @param other
     */
    public IL union(IL other)
    {
        Collection a;
        if(isSet() && other.isSet())
        {
            a = newSet();
            a.addAll(this.col);
        }
        else
        {
            a = newList();
            addAll(a, this.array);
        }
        addAll(a, other.iterator());
        return new L(a);
    }

    /**
     * Returns a new list containg the elements of this list that also
     * belong to the <code>other</code> list.
     *
     * <p/>
     * If the underlying backing store is a set for both lists, then the method
     * implements the set-theoretic intersection operation.
     * @param other
     */
    public IL intersect(IL other)
    {
        Collection r = newCombinedCollection(other);

        int thisSize = this.size();
        int otherSize = other.size();
        IL small = this;
        IL big = other;
        if(thisSize > otherSize)
        {
            small = other;
            big = this;
        }

        for(Iterator iterator = small.iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(big.contains(o))
            {
                r.add(o);
            }
        }

        return new L(r);
    }

    /**
     * Returns a new list containg the elements of this list that do not
     * belong to the <code>other</code> list.
     *
     * <p/>
     * If the underlying backing store is a set for both lists, then the method
     * implements the set-theoretic difference operation.
     * @param other
     */
    public IL minus(IL other)
    {
        Collection r = newCombinedCollection(other);

        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            Object o = iterator.next();
            if(!other.contains(o))
            {
                r.add(o);
            }
        }
        return new L(r);
    }

    public String toString()
    {
        ToString ts = new ToString();
        toStringAware(ts);
        return ts.toString();
    }

    /// + String methods
    /**
     * Chops the <code>prefix</code> from all elements, which are assumed to be strings.
     * <code>null</code>s pass through.
     * @param prefix
     */
    public IL chopPrefix(String prefix)
    {
        CollectionProxy proxy = CollectionProxy.newLike(this);
        proxy.beginMassiveAdd();
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            String item = (String) iterator.next();
            if(null != item)
            {
                if(item.startsWith(prefix))
                {
                    item = item.substring(prefix.length());
                }
            }

            proxy.add(item);
        }
        proxy.endMassiveAdd();

        return proxy.toL();
    }

    /**
     * Chops the regular expression <code>prefix</code> from all elements, which are assumed to be strings.
     * <code>null</code>s pass through.
     * @param prefix
     */
    public IL chopPrefixRE(String prefix)
    {
        Pattern p = Pattern.compile("^(" + prefix + ")");
        return map(FUNCTION_chopPrefixRE, p);
    }

    /**
     * Chops the <code>suffix</code> from all elements, which are assumed to be strings.
     * <code>null</code>s pass through.
     */
    public IL chopSuffix(String suffix)
    {
        CollectionProxy proxy = CollectionProxy.newLike(this);
        proxy.beginMassiveAdd();
        for(Iterator iterator = iterator(); iterator.hasNext();)
        {
            String item = (String) iterator.next();
            if(null != item)
            {
                if(item.endsWith(suffix))
                {
                    item = item.substring(item.lastIndexOf(suffix));
                }
            }

            proxy.add(item);
        }
        proxy.endMassiveAdd();

        return proxy.toL();
    }

    /**
     * Chops the regula expression <code>suffix</code> from all elements, which are assumed to be strings.
     * <code>null</code>s pass through.
     */
    public IL chopSuffixRE(String suffix)
    {
        Pattern p = Pattern.compile("(" + suffix + ")$");
        return map(FUNCTION_chopSuffixRE, p);
    }

    /**
     * Filters all elements that start with <code>s</code>, assuming that elements are strings.
     * <code>null</code>s do not pass through.
     * @param s
     */
    public IL filterStartsWith(String s)
    {
        return filter(FILTER_filterStartsWith, s);
    }

    /**
     * Filters all elements that do not start with <code>s</code>, assuming that elements are strings.
     * <code>null</code>s pass through.
     * @param s
     */
    public IL filterNotStartsWith(String s)
    {
        return filter(FILTER_filterNotStartsWith, s);
    }

    /**
     * Filters all elements that end with <code>s</code>, assuming that elements are strings.
     * <code>null</code>s do not pass through.
     * @param s
     */
    public IL filterEndsWith(String s)
    {
        return filter(FILTER_filterEndsWith, s);
    }

    /**
     * Filters all elements that end with <code>s</code>, assuming that elements are strings.
     * <code>null</code>s pass through.
     * @param s
     */
    public IL filterNotEndsWith(String s)
    {
        return filter(FILTER_filterNotEndsWith, s);
    }

    /**
     * Uses {@link java.util.regex.Matcher#find()}.
     * <code>null</code>s do not pass through.
     * @param re
     */
    public IL filterFindRE(String re)
    {
        Pattern p = Pattern.compile(re);
        return filter(FILTER_filterFindRE, p);
    }

    /**
     * Uses {@link java.util.regex.Matcher#find()}.
     * <code>null</code>s pass through.
     * @param re
     */
    public IL filterNotFindRE(String re)
    {
        Pattern p = Pattern.compile(re);
        return filter(FILTER_filterNotFindRE, p);
    }

    /**
     * Uses {@link java.util.regex.Matcher#find()}.
     * <code>null</code>s do not pass through.
     * @param re
     */
    public IL filterFindRE(String re, int modifiers)
    {
        Pattern p = Pattern.compile(re, modifiers);
        return filter(FILTER_filterFindRE_m, p);
    }

    /**
     * Uses {@link java.util.regex.Matcher#find()}.
     * <code>null</code>s pass through.
     * @param re
     */
    public IL filterNotFindRE(String re, int modifiers)
    {
        Pattern p = Pattern.compile(re, modifiers);
        return filter(FILTER_filterNotFindRE_m, p);
    }

    /**
     * Uses {@link java.util.regex.Matcher#matches()}.
     * <code>null</code>s do not pass through.
     * @param re
     */
    public IL filterMatchesRE(String re)
    {
        Pattern p = Pattern.compile(re);
        return filter(FILTER_filterMatchesRE, p);
    }

    /**
     * Uses {@link java.util.regex.Matcher#matches()}.
     * <code>null</code>s pass through.
     * @param re
     */
    public IL filterNotMatchesRE(String re)
    {
        Pattern p = Pattern.compile(re);
        return filter(FILTER_filterNotMatchesRE, p);
    }

    /**
     * Uses {@link java.util.regex.Matcher#matches()}.
     * <code>null</code>s do not pass through.
     * @param re
     */
    public IL filterMatchesRE(String re, int modifiers)
    {
        Pattern p = Pattern.compile(re, modifiers);
        return filter(FILTER_filterMatchesRE_m, p);
    }

    /**
     * Uses {@link java.util.regex.Matcher#matches()}.
     * <code>null</code>s pass through.
     * @param re
     */
    public IL filterNotMatchesRE(String re, int modifiers)
    {
        Pattern p = Pattern.compile(re, modifiers);
        return filter(FILTER_filterNotMatchesRE_m, p);
    }

    public IL filterStartsWithRE(String re)
    {
        return filterFindRE("^" + re);
    }

    public IL filterNotStartsWithRE(String re)
    {
        return filterNotFindRE("^" + re);
    }

    public IL filterStartsWithRE(String re, int modifiers)
    {
        return filterFindRE("^" + re, modifiers);
    }

    public IL filterNotStartsWithRE(String re, int modifiers)
    {
        return filterNotFindRE("^" + re, modifiers);
    }

    public IL filterEndsWithRE(String re)
    {
        return filterFindRE(re + "$");
    }

    public IL filterNotEndsWithRE(String re)
    {
        return filterNotFindRE(re + "$");
    }

    public IL filterEndsWithRE(String re, int modifiers)
    {
        return filterFindRE(re + "$", modifiers);
    }

    public IL filterNotEndsWithRE(String re, int modifiers)
    {
        return filterNotFindRE(re + "$", modifiers);
    }
    /// - String methods

    public int hashCode()
    {
        int hash = 0;
        Iterator iter = iterator();
        while(iter.hasNext())
        {
            Object o = iter.next();
            hash += hash ^ o.hashCode();
        }
        return hash;
    }

    /// + print methods
    public void print()
    {
        System.out.println(this);
    }

    public void print(PrintStream ps)
    {
        ps.println(this);
    }

    public void print(PrintWriter pw)
    {
        pw.println(this);
    }
    /// - print methods
}
