/*
 * This file is part of PB-Lib v2.0r1 (add-on)
 *
 * Version 1.5r1
 *
 * Copyright (c) 1996 by Branislav L. Slantchev
 * A Product of Silicon Creations, Inc.
 *
 * This source is distributed under the terms and conditions of the
 * GNU General Public License. A copy of the license is included with
 * this distrbiution (see the file 'Copying.Doc').
 *
 * Contact: 73023.262@compuserve.com
*/
#include "class/editline.h"
#include "kbcodes.h"
#include "utils.h"

#ifndef PB_SDK
	#include "proutil.h"
	#include <stdio.h>
	#include <stdlib.h>
	#include <string.h>
    #include <conio.h>
    #include <ctype.h>
    #include <time.h>
    #include <dos.h>

    #define GotoXY(x,y)		gotoxy(x,y)
    #define SetColor(c)		textattr(c)
    #define printf			cprintf
    #undef  putchar
    #define putchar         putch
#else
	#define SetColor		SetFullColor
#endif

#define shiftState()		Boolean(*(uchar far *)MK_FP(0x40, 0x17) & 0x03)

char TEditLine::leftArrow  = '';
char TEditLine::rightArrow = '';

TEditLine::TEditLine(const CRect &bounds, short aMaxLen):
	data(new char [aMaxLen + 1]),
    maxLen(aMaxLen),
    curPos(0),
    firstPos(0),
    startSel(0),
    endSel(0),
    origin(bounds.a.x, bounds.a.y),
    size(bounds.b.x - bounds.a.x, bounds.b.y - bounds.a.y),
    focused(False),
    mustPatch(False)
{                                  // frg /0 bkg/ch sel/0   infg/0  inbg/ch
	static const char *cpTEditLine = "\x1e\x00\x71\x71\x00\x13\x00\x13 ";

	*data = EOS;
	if( size.x < 4 ) size.x = 4;
    setPalette(cpTEditLine);
}

TEditLine::~TEditLine()
{
	if( data ) delete data;
}

char TEditLine::filterOnDraw(char aChar)
{
	return aChar;
}

// return an element from the palette (cast to unsigned character)
uchar TEditLine::getPalette(short pos)
{
	return uchar(palette[pos % sizeof(palette)]);
}

// set a new palette (overrides the default)
void TEditLine::setPalette(const char *aPalette)
{
	memcpy(palette, aPalette, sizeof(palette));
}

// sets the input focus to a new state (and redraws)
void TEditLine::setState(Boolean ssFocused)
{
	focused = ssFocused;
    selectAll();
    draw();
}

Boolean TEditLine::getState()
{
	return focused;
}

void TEditLine::drawView()
{
	Boolean state = focused;
    focused = True;
    draw();
    focused = state;
}

// selects or deselects the entire string
void TEditLine::selectAll()
{
	firstPos = curPos = startSel = 0;
	if( focused ) endSel = strlen(data);
    else endSel = 0;
}

// True if the line can scroll in the direction
Boolean TEditLine::canScroll(Boolean left)
{
	if( left ) return Boolean(firstPos > 0);
    else return Boolean(strlen(data) - firstPos > size.x - 2);
}

// redraws the current line with scrolling indicators if necessary
void TEditLine::draw()
{
	char   buf[256];
    short  leftSel, rightSel;
    size_t slen;
    char   backChar  = (True == focused) ? getPalette(3) : getPalette(9);
    uchar  foreColor = (True == focused) ? getPalette(0) : getPalette(6);
    uchar  backColor = (True == focused) ? getPalette(2) : getPalette(8);

    // construct the string as it would appear (with indicators)
    slen = min(size.x - 2, strlen(data+firstPos));
    memset(buf, backChar, size.x);
	//memcpy(buf, data+firstPos, slen); // replaced by the filtering
    for( size_t i = 0; i < slen; ++i ) buf[i] = filterOnDraw(*(data+firstPos+i));
    buf[size.x - 2] = EOS;

    // check to see if something is selected currently
    if( focused ){
		leftSel  = startSel - firstPos;
        rightSel = endSel - firstPos;
        leftSel  = max(0, leftSel);
		rightSel = min(size.x-2, rightSel);
	}

    // print the left leading character
    GotoXY(origin.x, origin.y);
    SetColor(backColor);
    if( canScroll(True) ) putchar( leftArrow );
    else putchar(backChar);

    SetColor(foreColor);
    buf[slen] = EOS;
    if( !focused || !(leftSel < rightSel) ) cprintf(buf);
    else{	// we might have a selection to highlight
    	short pos = 0;

    	while( pos < leftSel ) putchar(buf[pos++]);
        SetColor(getPalette(4));
        while( pos < rightSel) putchar(buf[pos++]);
        SetColor(foreColor);
        while( pos < slen ) putchar(buf[pos++]);
	}

    // fill with the background if necessary here
    if( slen < size.x - 2 ){
        SetColor(backColor);
        buf[slen] = backChar;
        cprintf(&buf[slen]);
	}

    // print the right trailing char (arrow if necessary)
    SetColor(backColor);
    if( canScroll(False) ) putchar( rightArrow );
    else putchar(backChar);

    // sync the cursor
    GotoXY(origin.x + curPos - firstPos + 1, origin.y);
}

// deletes the current selection
void TEditLine::deleteSelect()
{
    if( endSel > startSel ){
    	strcpy(data+startSel, data+endSel);
        curPos = startSel;
	}
}

// goes to the start of the previous word (to the left)
void TEditLine::leftWord()
{
	if( curPos ) curPos--;
	while( 0 < curPos && !isalnum(data[curPos])) curPos--;
    while( 0 < curPos && isalnum(data[curPos]) ) curPos--;
    if( curPos ) curPos++;
}

// goes to the start of the next word to the right
void TEditLine::rightWord()
{
	size_t len = strlen(data);
    if( curPos < len ) curPos++;
    while( curPos < len && !isalnum(data[curPos])) curPos++;
    while( curPos < len && isalnum(data[curPos]) ) curPos++;
    while( curPos < len && !isalnum(data[curPos])) curPos++;
    if( curPos < len && curPos - firstPos > size.x - 1 ) mustPatch = True;
}

// delete current pos to end of word plus spaces
void TEditLine::zapWord()
{
	short  pos = curPos;
    size_t len = strlen(data);

    while( pos < len && !isspace(data[pos])) pos++;
    while( pos < len && isspace(data[pos]) ) pos++;
    if( pos > curPos ) strcpy(data+curPos, data+pos);
}

// get the string
void TEditLine::getData(void *ptr)
{
	strcpy((char *)ptr, data);
}

void TEditLine::setData(void *ptr)
{
	strncpy(data, (char *)ptr, maxLen);
    data[maxLen] = EOS;
    selectAll();
}

// handle keystrokes
void TEditLine::handle(ushort aKeyCode)
{
	static Boolean extendKey   = False;
    static Boolean extendBlock = False;
    static Boolean mustKeep    = False;

	if( !focused ){
		return;
    }
    else if( kbdCtrlQ == aKeyCode ){	// extended key movements
		extendKey = True;
        return;
    }
    else if( kbdCtrlK == aKeyCode ){	// extended selection movements
    	extendBlock = True;
        return;
	}

	switch( ctrl2Arrow(aKeyCode) ){

    	case kbdLeft     : if( curPos ) curPos--; break;
        case kbdRight    : if( curPos < strlen(data) ) curPos++; break;
        case kbdIns      : insertMode = Boolean(!insertMode); break;
        case kbdCtrlT    : zapWord(); break;
        case kbdCtrlLeft : leftWord(); break;
        case kbdCtrlRight: rightWord();	break;
        case kbdCtrlY    : *data = EOS; curPos = 0; break;

        case 'S': case 's': if( !extendKey ) goto handleChar;
        case kbdHome: curPos = 0; break;

        case 'D': case 'd': if( !extendKey ) goto handleChar;
        case kbdEnd: curPos = strlen(data); break;

        case 'Y': case 'y':
			if( !extendKey ) goto handleChar;
			data[curPos] = EOS;
		break;

        case 'L': case 'l':
			if( !extendBlock ) goto handleChar;
            startSel = 0;
            endSel = strlen(data);
            extendBlock = False;
            mustKeep = True;
        break;

        case 'H': case 'h':
        	if( !extendBlock ) goto handleChar;
            startSel = endSel = 0;
            extendBlock = False;
        break;

        case kbdBack:
        	if( 0 < curPos ){
            	strcpy(data+curPos-1, data+curPos);
                curPos--;
                if( 0 < firstPos ) firstPos--;
			}
        break;

        case kbdDel:
        	if( endSel == startSel && curPos < strlen(data) ){
                startSel = curPos;
                endSel = curPos + 1;
			}
			deleteSelect();
        break;

        default   :
        handleChar:
            if( ' ' <= (aKeyCode & 0x00ff) ){
            	deleteSelect();
				if( !insertMode && curPos < strlen(data) )
                	strcpy(data+curPos, data+curPos+1);
                if( strlen(data) < maxLen ){
                    memmove(data+curPos+1, data+curPos, strlen(data+curPos)+1);
                    data[curPos++] = aKeyCode;
				}
			}
            else return;
	}

    extendKey = False;
    if( mustKeep ){
		mustKeep = False;
        if( startSel > endSel ){
        	short temp = startSel;
            startSel = endSel;
            endSel = temp;
		}
    }
	else startSel = endSel = 0;
	if( firstPos > curPos ) firstPos = curPos;
	firstPos = max(firstPos, curPos - size.x + 2);

    if( mustPatch ){	// make sure the first char is visible
		firstPos++;		// when doing the ctrl+right arrow skips
        mustPatch = False;
    }

	draw();
}

////////////////////////////////////////////////////////////////////////////
// Input a number derived class
////////////////////////////////////////////////////////////////////////////
TEditNumber::TEditNumber(const CRect &r, short aMaxLen, short aOptions):
	TEditLine(r, aMaxLen),
    options(aOptions)
{
	if( options & edit::blank ) strcpy(data, "0");
    if( options & edit::hex ) options |= edit::pos;
}

void TEditNumber::setData(void *ptr)
{
	char buf[20];
    long nData = *(long *)ptr;

    if( options & edit::pos ) nData = labs(nData);
    if( options & edit::hex ) sprintf(buf, "0x%lX", nData);
    else sprintf(buf, "%ld", nData);
    TEditLine::setData(buf);
}

void TEditNumber::getData(void *ptr)
{
	char buf[20];
    long nData;

    TEditLine::getData(buf);
    if( EOS == buf[0] && (options & edit::blank) ) strcpy(data, "0");
    if( 'X' == buf[1] ) nData = strtol(buf, NULL, 16);
    else nData = strtol(buf, NULL, 10);
    *(long *)ptr = nData;
}

void TEditNumber::handle(ushort aKeyCode)
{
	char charCode = aKeyCode & 0x00ff;

    if( ' ' <= aKeyCode && aKeyCode < 127 ){
    	switch( toupper(charCode) ){

        	case '-': case '+':
            	if( options & edit::pos ) aKeyCode = kbdNoKey;
				else if( (0 != curPos) ||
					(insertMode && ('+' == data[0] || '-' == data[0])) )
                {
                	aKeyCode = kbdNoKey;
                }
            break;

            case 'X':
            	if( !(options & edit::hex) ||
					(1 != curPos) || '0' != data[0] )
                {
                	aKeyCode = kbdNoKey;
				}
            break;

            case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
            	if( !(options & edit::hex) ) aKeyCode = kbdNoKey;
            break;

            case '0': case '1': case '2': case '3': case '4':
			case '5': case '6': case '7': case '8': case '9':
            break;

            default: return;
		}
	}

    TEditLine::handle(aKeyCode);
}

////////////////////////////////////////////////////////////////////////////
// Date input class (supports MM/DD/YY, DD/MM/YY formats), can also write
// a background to show the format that is expected. Does some validation
////////////////////////////////////////////////////////////////////////////
const char *TEditDate::euro = "DD/MM/YY";
const char *TEditDate::amer = "MM/DD/YY";
const char *TEditDate::none = "  /  /  ";

TEditDate::TEditDate(const CRect &bounds, short aOptions):
	TEditNumber(bounds, 8, edit::pos),
    options(aOptions)
{
	if( !(options & edit::european) && !(options & edit::american) )
    	options |= edit::american;	// force format

	if( options & edit::show ){
    	if( options & edit::european ) strcpy(data, euro);
    	else if( options & edit::american ) strcpy(data, amer);
	}
    else strcpy(data, none);
}

void TEditDate::getData(void *rec)
{
	strcpy((char*)rec, data);
}

void TEditDate::setData(void *rec)
{
	strncpy(data, (char*)rec, 8);
    data[8] = EOS;
    data[2] = data[5] = '/';
    selectAll();
}

void TEditDate::handle(ushort aKeyCode)
{
    char charCode = aKeyCode & 0x00ff;

    endSel = startSel = 0;
    insertMode = False;

    switch( aKeyCode ){
    	case kbdBack: aKeyCode = kbdLeft; break;
        case kbdDel : case kbdCtrlY: case kbdCtrlT:
			aKeyCode = kbdNoKey;
		break;
	}

    if( '0' <= charCode && '9' >= charCode ){
    	if( 2 == curPos || 5 == curPos ) TEditNumber::handle(kbdRight);
	}

    if( valid(aKeyCode) ) TEditNumber::handle(aKeyCode);

    if( 2 == curPos || 5 == curPos ){
	    if( kbdLeft == aKeyCode || kbdBack == aKeyCode )
			 TEditNumber::handle(kbdLeft);
        else
        	 TEditNumber::handle(kbdRight);
    }
}

// perform data validation
Boolean TEditDate::valid(ushort aCode)
{
	// only check the digits, the TEditNumber will filter the rest
	if( '0' <= aCode && '9' >= aCode ){
    	switch( curPos ){
        	case 0 :
            	if( options & edit::american ){	// this is month
            monthCheck_1:
                	if( '1' < aCode ) return False;
                    else return True;
				}
                else{	// european-style, this is a day number
            dayCheck_1:
                	if( '3' < aCode ) return False;
                    else return True;
				}

            case 1 :
            	if( options & edit::american ){	// second digit of the month
            monthCheck_2:
                	if( data[curPos-1] == '1' && '2' < aCode ) return False;
                    else return True;
				}
                else{	// second digit of the day
            dayCheck_2:
					if( '3' == data[curPos-1] && '1' < aCode ) return False;
                    else return True;
				}

            case 3 :
            	if( options & edit::american ) goto dayCheck_1;
                else goto monthCheck_1;
            case 4 :
            	if( options & edit::american ) goto dayCheck_2;
                else goto monthCheck_2;

            case 6 :	// fall-through, the year can be anything
            case 7 :
            default: return True;	// not an interesting position here
		}
	}
	return True;
}

////////////////////////////////////////////////////////////////////////////
// Class to input filenames (or paths) with/without wildcards
////////////////////////////////////////////////////////////////////////////
TEditFile::TEditFile(const CRect &bounds, short aMaxLen, short aOptions):
	TEditLine(bounds, aMaxLen),
    options(aOptions)
{
	if( !(options & edit::path) ){
		maxLen = min(MAXFILE+MAXEXT-2, aMaxLen);
        if( maxLen != aMaxLen ){
        	delete data;
            data = new char [maxLen + 1];
            *data = EOS;
		}
    }
}

void TEditFile::handle(ushort aKeyCode)
{
    static const char *notValid = "\"=;+,<>|";	// and isspace() too
    static const char *wildCard = "*?[]";		// these are extended!
    static const char *pathSeps = "\\/:";		// path separators (almost)
    int    code = aKeyCode;

    if( isascii(code) &&
		(isspace(code) ||
		strchr(notValid, code) ||
    	(!(options & edit::wildcards) && strchr(wildCard, code)) ||
        (!(options & edit::path) && strchr(pathSeps, code))) )
    {
    	return;	// invalid character
	}

    // check for one period in filenames and one colon in paths
    if( !(options & edit::path) && ('.' == code) && strchr(data, '.') ) return;
    if( (':' == code) && strchr(data, ':') ) return;	// one ':'

    // check to see if the number of characters after period is valid
    // also check for too many characters without a period
    if( !(options & edit::path) && isprint(code) ){
    	char *p = strchr(data, '.');
        if( p && 4 <= strlen(p) ) return;
        else if( !p && '.' != code && 8 == strlen(data) ) return;
	}

    // see if we want to convert it
    if( (options & edit::upper) && isascii(aKeyCode) )
		aKeyCode = toupper(aKeyCode);

    TEditLine::handle(aKeyCode);
}

////////////////////////////////////////////////////////////////////////////
// The line with the combo list of selections (only those accepted)
////////////////////////////////////////////////////////////////////////////
TEditCombo::TEditCombo(const CRect &bounds, short aOptions, TDoubleList *items):
	TEditLine(bounds, 0),
    options(aOptions),
    list(items),
    iter(*list),
    nCompare(0)
{
	if( list->size() ){
	    if( data ) delete data;
	    maxLen = 0;
	    do{
			maxLen = max(maxLen, strlen((char*)iter.get()));
		}while( iter++ );
	    data = new char [maxLen + 1];
	    *data = EOS;

	    iter.begin();
	    if( options & edit::init ) strcpy(data, (char *)iter.get());
    }
}

TEditCombo::~TEditCombo()
{
	if( options & edit::dispose ) delete list;
}

void TEditCombo::getData(void *ptr)
{
	strcpy((char*)ptr, data);
}

void TEditCombo::setData(void *ptr)
{
	if( list->size() ){
		iter.begin();
	    do{
	    	if( 0 == strcmp((char*)ptr, (char*)iter.get()) ) break;
		}while( iter++ );
	    strcpy(data, (char*)iter.get());
    }
}

// implement the incremental search in either direction
void TEditCombo::incrementalSearch(ushort aKeyCode)
{
    char         buf[256], code = aKeyCode & 0x00ff;
    Boolean      found = False;
    TListCursor  temp = iter;

    if( kbdBack == aKeyCode ) if( nCompare ) nCompare--;
    else if( kbdHome == aKeyCode ) nCompare = 0;

	strncpy(buf, (char *)iter.get(), nCompare);
    buf[nCompare] = EOS;
    if( isalnum(code) && nCompare < maxLen ){
    	buf[nCompare++] = code;
        buf[nCompare] = EOS;
	}
    if( nCompare ){
	    temp.begin();
	    do{
	    	if( 0 == strnicmp((char*)temp.get(), buf, nCompare) ){
	        	strcpy(data, (char*)temp.get());
	            iter = temp;
	            found = True;
			    nCompare = min(nCompare, strlen(data));
	            selectAll();
	            endSel = 0;
	        	break;
			}
		}while( temp.next() );
	    if( !found ) nCompare--;
    }
}

void TEditCombo::handle(ushort aKeyCode)
{
    char code = (aKeyCode & 0x00ff);

    if( !list->size() ) return;		// nothing in the list

    if( ' ' == code ){
    	if( shiftState() ){
			if( !iter.prev() ) iter.end();
        }
        else{
			if( !iter.next() ) iter.begin();
        }
        strcpy(data, (char*)iter.get());
        nCompare = 0;
        selectAll();
	}
    else if( isalnum(code) || kbdHome == aKeyCode || kbdBack == aKeyCode){
    	incrementalSearch(aKeyCode);
	}

    draw();
    GotoXY(origin.x + nCompare - firstPos + 1, origin.y);
}

////////////////////////////////////////////////////////////////////////////
// The password input class
////////////////////////////////////////////////////////////////////////////
// the distribution is borrowed brom Snippets
const char TEditPassword::m_Chars[] =
	"aaabbccddeeeeffgghhiiijklllmmmnnnoooppqrrrsssstttuuvwwxyz"
	"1234567890-|,@._"
	"aaabbccddeeeeffgghhiiijklllmmmnnnoooppqrrrsssstttuuvwwxyz";

TEditPassword::TEditPassword(const CRect &bounds, short aMaxLen, short Options):
	TEditLine(bounds, aMaxLen),
    options(Options),
    pwdChar('*')
{
	if( options | edit::proboard ){	// attempt to get the password character
#ifdef PB_SDK
		pwdChar = Config->pwdchar;
#else
    	CONFIG  config;
    	FILE   *fp = pbfopen(fnCONFIG, "rb");

        if( fp ){
        	fread(&config, sizeof(config), 1, fp);
        	fclose(fp);
            pwdChar = config.pwdchar;
		}
#endif
	}
}

char TEditPassword::setPwdChar(char aChar)
{
	char oldChar = pwdChar;
    pwdChar = aChar;
    draw();		// redraw with the new character
    return oldChar;
}

char TEditPassword::filterOnDraw(char )
{
	static Boolean inited = False;

	if( !inited ){
    	srand((unsigned)time(NULL) | 1);
    	inited = True;
	}

    if( options & edit::secure ) return m_Chars[rand() % sizeof(m_Chars)];
    else return pwdChar;
}

#ifndef PB_SDK
	#undef GotoXY
    #undef SetColor
    #undef printf
    #undef putchar
#else
	#undef SetColor
#endif
