/*
 * Copyright (c) 1995-1997 Branislav L. Slantchev
 * A Product of Silicon Creations, Inc.
 *
 * This source is distributed under the terms and conditions of the
 * GNU General Library License. A copy of the license is included with
 * this distrbiution (see the file 'Copying.Pbl').
 *
 * Contact: 73023.262@compuserve.com
*/
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include "comdef.h"
#include "stdmac.h"
#include "compiler.h"
#include "utils.h"
#include "getopt.h"
#include "file.h"
#include "terminal.h"

#define __VERSION__ "v3.1"

typedef struct
{
	ushort size;     // data size in data file
	short  lines;    // # lines in the animation
	ushort delay;    // delay factor b/n steps, 0=static
	short  column;   // start at column#
	long   offset;   // offset of data into the data file
	char   extra[4]; // spare bytes
} ACL_INDEX;

// The header for the data files (.KDT)
typedef struct
{
	char  name[80];     // name of the file, ASCIIZ
	char  idString[10]; // 10 bytes id string (not ASCIIZ)
	short numRecs;      // number of records stored
	char  __rsrvd[58];  // reserved space
} KAF_HEADER;

// The local header in the data file (.KDT)
typedef struct
{
	char   name[26];   // name of the sequence
	ushort dataSize;   // length of the record (excluding the header)
	short  numLines;   // number of lines to animate
	ushort speed;      // default animation speed (0 == static)
	short  maxChars;   // maximum line length (0 == ignore)
	char   __rsrvd[6];
} KAF_RECORD;

enum Cmds {cmAdd, cmDelete, cmExport, cmShow, cmList, cmEdit, cmMerge, cmCopy};
void print_help();
void print_banner();
void add(char*, char*, Boolean, Boolean, int, int);
void list(char*);
void export(char*, int);
void erase(char*, int);
void show(char*, int, int, int, int);
void edit(char*, int, int, int, Boolean);
void merge(char*, char*);
void copy(char*, char*);

int
main(int argc, char *argv[])
{
	Cmds    Command;
	int     opt, nPrompt = -1, nDelay = 80, nColumn = 1, nLine = 24;
	Boolean bAsk = False, bCompat = False;
	char    dataPath[80], *fileSpec = 0;

	strcpy(dataPath, "a.adt");

	print_banner();

	if( 1 == argc || argv[1][0] == '?' || argv[1][1] == '?' )
	{
		print_help();
		return 1;
	}

	switch( tolower(argv[1][0]) )
	{
		case 'a': Command = cmAdd;    break;
		case 'd': Command = cmDelete; break;
		case 'x': Command = cmExport; break;
		case 's': Command = cmShow;   break;
		case 'l': Command = cmList;   break;
		case 'e': Command = cmEdit;   break;
		case 'm': Command = cmMerge;  break;
		case 'k': Command = cmCopy;   break;
		default : fail("Unknown mode: %c\n", argv[1][0]);
	}

	while( EOF != (opt = getopt(argc - 1, &argv[1], "io:#:rp:c:f:y:")) )
	{
		switch( tolower(opt) )
		{
			case 'i': bAsk = True; break;
			case 'o': file_chext(dataPath, optarg, ".adt"); break;
			case '#': nPrompt = atoi(optarg); break;
			case 'r': bCompat = True; break;
			case 'p': nDelay = atoi(optarg); break;
			case 'c': nColumn = atoi(optarg); break;
			case 'f': fileSpec = optarg; break;
			case 'y': nLine = atoi(optarg); break;
		}
	}

	switch( Command )
	{
		case cmAdd:
			add(dataPath, fileSpec, bAsk, bCompat, nDelay, nColumn);
			break;
		case cmList  : list(dataPath); break;
		case cmExport: export(dataPath, nPrompt); break;
		case cmDelete: erase(dataPath, nPrompt); break;
		case cmShow  : show(dataPath, nPrompt, nDelay, nColumn, nLine); break;
		case cmEdit  : edit(dataPath, nPrompt, nDelay, nColumn, bAsk); break;
		case cmMerge : merge(dataPath, fileSpec); break;
		case cmCopy  : copy(dataPath, fileSpec); break;
	}

	return 0;
}

void
print_help()
{
	puts("Usage   : ACL <command> -<switch> [-<switch>...] -f<filespec>");
	puts("Examples: ACL a -f *.ans -r, ACL l -o archive, ACL -x -#12\n");
	puts("<Commands>");
	puts("   a: Add files to archive          d: Delete records from archive");
	puts("   l: List records in archive       s: Show animated record");
	puts("   x: Export records to file        e: Edit record parameters");
	puts("   m: Merge two data files          k: Copy records from a KAF file");
	puts("<Switches>");
	puts("   f: Specify files to add          i: Interactive mode");
	puts("   r: Add RemoteAccess animations   p: Specify millisecond delay");
	puts("   c: Specify starting column       #: Specify record number");
	puts("   o: Use different archive name    y: Specify row for display");
	puts("\nWhen adding, merging or copying, the '-f fileSpec' switch is required.");
	puts("The default archive is named 'a.*', use '-o altname' to change.");
	puts("Use '-i' when adding or editing to specify column and delay.");
}

void
print_banner()
{
	printf("ACL %s Copyright (c) 1995-1997 Branislav L. Slantchev. [%s]\n"
		   "A product of Silicon Creations, Inc. (part of the PB-GNU Project)"
		   "\n\n", __VERSION__, __DATE__);
}

void
add(char *data, char *spec, Boolean ask, Boolean compat, int nDelay, int column)
{
	char        index[80], buf[1024], base[80], path[80];
	ACL_INDEX   rec;
	ushort      dataLen;
	FILE       *in, *dat, *idx;
	int         done, total = 0;
	zDosFile    ffblk;

	if( !spec ) fail("No '-f filespec' specified with 'add' command\n");

	file_chext(index, data, ".aix");
	strcpy(base, file_dirspec(spec));
	dat = fopen(data, "a+b");
	if( !dat ) fail("unable to open data file '%s'\n", data);
	idx = fopen(index, "a+b");
	if( !idx ) fail("unable to open the index file '%s'\n", index);

	fseek(dat, 0L, SEEK_END);
	fseek(idx, 0L, SEEK_END);
	rec.column = column;
	rec.delay = nDelay;
	memset(rec.extra, 0, sizeof(rec.extra));
	done = FindFirst(spec, 0, &ffblk);
	while( !done )
	{
		strcat(strcpy(path, base), ffblk.name);
		printf("\nadding %s...", path);
		in = fopen(path, "rt");
		if( in )
		{
			rec.lines = rec.size = 0;
			rec.offset = ftell(dat);

			if( compat )
			{
				fgets(buf, sizeof(buf), in);
				sscanf(buf, "%*d %*s %*d %d", &rec.column);
			}

			fgets(buf, sizeof(buf), in);
			while( !feof(in) )
			{
				char *p = buf + strlen(buf) - 1;

				if( '\n' == *p ) *p = EOS;
				dataLen = strlen(buf);
				rec.lines++;
				rec.size += dataLen + sizeof(ushort);
				fwrite(&dataLen, sizeof(ushort), 1, dat);
				fwrite(buf, 1, dataLen, dat);
				fgets(buf, sizeof(buf), in);
			}

			if( ask )
			{
				printf("\nDelay (milsecs)? "); gets(buf);
				if( strlen(buf) ) rec.delay = atoi(buf);
				printf("Starting column? "); gets(buf);
				if( strlen(buf) ) rec.column = atoi(buf);
			}

			fwrite(&rec, sizeof(rec), 1, idx);

			fclose(in);
			printf("done (%d lines)", rec.lines);
		}
		total++;
		done = FindNext(&ffblk);
	}

	printf("\n\nAdded %d records.\n", total);
	fclose(dat);
	fclose(idx);

	if( 0 == file_size(data) ) file_remove(data);
	if( 0 == file_size(index)) file_remove(index);
}

void
list(char *data)
{
	char       path[80];
	int        total = 0;
	ACL_INDEX  rec;
	FILE      *fp;

	file_chext(path, data, ".aix");
	fp = fopen(path, "rb");
	if( fp )
	{
		printf(" Recno   Size   Lines   Delay   Column   Offset\n"
			   " -----   ----   -----   -----   ------   --------\n");
		fread(&rec, sizeof(rec), 1, fp);
		while( !feof(fp) )
		{
			total++;
			printf(" %4d   %5d   %5d   %5d    %5d   %08lX\n",
				total, rec.size, rec.lines, rec.delay, rec.column, rec.offset);
			fread(&rec, sizeof(rec), 1, fp);
		}
		fclose(fp);
		printf(" -----------------------------------------------\n"
			   " Total: %d records.\n", total);
	}
}

void
export(char *data, int nPrompt)
{
	char       index[80], xpath[80], buf[1024];
	ACL_INDEX  rec;
	FILE      *idx, *dat, *out;
	int        total, start, end, exported = 0;

	file_chext(index, data, ".aix");
	total = (int)(file_size(index) / sizeof(ACL_INDEX));

	if( 0 > nPrompt || total < nPrompt )
		fail("Prompt number out of range (valid: 0 thru %d)\n", total);

	idx = fopen(index, "rb");
	dat = fopen(data, "rb");

	if( !idx || !dat ) fail("Unable to open the data file(s)\n");

	if( 0 == nPrompt )
	{
		start = 0;
		end = total;
	}
	else
	{
		start = nPrompt - 1;
		end = nPrompt;
	}

	for( int i = start; i < end; ++i )
	{
		fseek(idx, sizeof(ACL_INDEX) * (long)i, SEEK_SET);
		fread(&rec, sizeof(rec), 1, idx);
		sprintf(xpath, "dump%04d.acl", i+1);
		printf("\nExporting to '%s'...", xpath);
		out = fopen(xpath, "wt");
		if( out )
		{
			sprintf(buf, "%d NO 1 %d\n", rec.lines, rec.column);
			fputs(buf, out);
			fseek(dat, rec.offset, SEEK_SET);
			for( int j = 0; j < rec.lines; ++j )
			{
				short datLen;

				fread(&datLen, sizeof(datLen), 1, dat);
				fread(buf, 1, datLen, dat);
				buf[datLen] = '\n'; buf[datLen+1] = EOS;
				fputs(buf, out);
			}
			fclose(out);
			printf("done (%d lines)", rec.lines);
			exported++;
		}
	}

	printf("\n\nExported %d records.\n", exported);
}

void
erase(char *data, int nPrompt)
{
	char       index[80], newIndex[80], newData[80], buf[1024];
	ACL_INDEX  rec;
	int        total, cur = 0;
	FILE      *idx, *dat, *newidx, *newdat;

	file_chext(index, data, ".aix");
	total = (int)(file_size(index) / sizeof(ACL_INDEX));

	if( nPrompt <= 0 || nPrompt > total )
		fail("Prompt number out of range (valid: 1 thru %d)\n", total);

	idx = fopen(index, "rb");
	dat = fopen(data, "rb");
	if( !idx || !dat ) fail("Unable to open the data file(s) '%s'\n", index);

	file_chext(newIndex, index, ".i$$");
	file_chext(newData, data, ".d$$");

	newidx = fopen(newIndex, "wb");
	newdat = fopen(newData, "wb");
	if( !newidx || !newdat )
	{
		fclose(idx);
		fclose(dat);
		fail("Unable to create the temp files!\n");
	}

	printf("Working...");
	fread(&rec, sizeof(rec), 1, idx);
	while( !feof(idx) )
	{
		if( cur != nPrompt - 1 )
		{
			long   offset = ftell(newdat);
			ushort nBytes = rec.size;

			fseek(dat, rec.offset, SEEK_SET);
			while( nBytes > 0 )
			{
				int bytes = fread(buf,1,min(sizeof(buf),(uint)nBytes),dat);
				fwrite(buf, 1, bytes, newdat);
				nBytes -= bytes;
			}
			rec.offset = offset;
			fwrite(&rec, sizeof(rec), 1, newidx);
		}
		fread(&rec, sizeof(rec), 1, idx);
		++cur;
	}

	fclose(idx);
	fclose(dat);
	fclose(newidx);
	fclose(newdat);
	file_remove(data);
	file_remove(index);
	rename(newIndex, index);
	rename(newData, data);

	printf("done!\n");
}

void
show(char *data, int nPrompt, int nDelay, int nColumn, int nLine)
{
	char           index[80], **lines;
	ACL_INDEX      rec;
	Boolean        done = False;
	FILE          *idx, *dat;
	int            total, i;
	zTerminal      terminal;
	ansi_interp    ansi;
	avatar_interp  avatar;
	pcboard_interp pcboard;

	file_chext(index, data, ".aix");
	total = (int)(file_size(index) / sizeof(ACL_INDEX));

	if( 0 >= nPrompt || nPrompt > total )
		fail("Invalid prompt number, must be between 1 and %d\n", total);
	idx = fopen(index, "rb");
	dat = fopen(data, "rb");
	if( !idx || !dat ) fail("Unable to open data file(s): '%s'\n", index);

	fseek(idx, (nPrompt - 1) * (long)sizeof(ACL_INDEX), SEEK_SET);
	fread(&rec, sizeof(rec), 1, idx);
	fclose(idx);

	if( nDelay == 80 ) nDelay  = rec.delay;
	if( nColumn == 1 ) nColumn = rec.column;
	lines = (char **)new char* [rec.lines];
	if( !lines ) fail("Insufficient memory to display the prompt.\n");

	terminal.RegisterHandler(&avatar);
	terminal.RegisterHandler(&ansi);
	terminal.RegisterHandler(&pcboard);

	fseek(dat, rec.offset, SEEK_SET);
	for( i = 0; i < rec.lines; ++i )
	{
		short datLen;

		fread(&datLen, sizeof(short), 1, dat);
		lines[i] = new char [datLen +  1];
		if( !lines[i] ) fail("Not enough memory to complete the operation.\n");
		fread(lines[i], 1, datLen, dat);
		lines[i][datLen] = EOS;
	}
	fclose(dat);
	_setcursortype(_NOCURSOR);
	while( !done )
	{
		for( i = 0; i < rec.lines; ++i )
		{
			gotoxy(nColumn, nLine);
			terminal.handle(lines[i]);
			if( kbhit() )
			{
				getch();
				done = True;
				break;
			}
			delay(nDelay);
		}
	}
	printf("\x1b[0m");
	gotoxy(1, nLine);
	clreol();
	for( i = 0; i < rec.lines; ++i ) delete[] lines[i];
	delete[] lines;
	_setcursortype(_NORMALCURSOR);
}

void
edit(char *data, int nPrompt, int nDelay, int nColumn, Boolean bAsk)
{
	char       index[80], buf[80];
	ACL_INDEX  rec;
	FILE      *fp;
	int        total;

	file_chext(index, data, ".aix");
	total = (int)(file_size(index) / sizeof(ACL_INDEX));
	if( nPrompt <= 0 || nPrompt > total )
		fail("Invalid edit request, prompt must be between 1 and %d\n", total);

	fp = fopen(index, "r+b");
	if( fp )
	{
		fseek(fp, (long)(nPrompt - 1) * sizeof(ACL_INDEX), SEEK_SET);
		fread(&rec, sizeof(rec), 1, fp);
		printf("Current delay: %d, new: ", rec.delay);
		if( bAsk )
		{
			gets(buf);
			rec.delay = atoi(buf);
		}
		else
		{
			rec.delay = nDelay;
			printf("%d\n", nDelay);
		}
		printf("Current column: %d, new: ", rec.column);
		if( bAsk )
		{
			gets(buf);
			rec.column = atoi(buf);
		}
		else
		{
			rec.column = nColumn;
			printf("%d\n", nColumn);
		}
		fseek(fp, (long)(nPrompt - 1) * sizeof(ACL_INDEX), SEEK_SET);
		fwrite(&rec, sizeof(rec), 1, fp);
		fclose(fp);
	}
}

void
merge(char *data, char *newSpec)
{
	char      index[80], base[80], newData[80], newIndex[80], buf[1024];
	ACL_INDEX rec;
	FILE     *idx, *dat, *newidx, *newdat;
	int       done = 0, total = 0, totalNew = 0;
	zDosFile  ffblk;

	file_chext(index, data, ".aix");
	strcpy(base, file_dirspec(newSpec));
	idx = fopen(index, "a+b");
	dat = fopen(data, "a+b");
	if( !idx || !dat ) fail("Unable to open original for merging.\n");

	fseek(dat, 0L, SEEK_END);
	fseek(idx, 0L, SEEK_END);

	done = FindFirst(newSpec, 0, &ffblk);
	while( !done )
	{
		totalNew = 0;
		strcat(strcpy(newData, base), ffblk.name);
		file_chext(newIndex, newData, ".aix");
		printf("\nMerging '%s'...", newIndex);
		newidx = fopen(newIndex, "rb");
		if( !newidx ) goto process_next;
		newdat = fopen(newData, "rb");
		if( !newdat )
		{
			fclose(newidx);
			goto process_next;
		}

		fread(&rec, sizeof(rec), 1, newidx);
		while( !feof(newidx) )
		{
			long offset = ftell(dat);
			ushort nBytes = rec.size;

			fseek(newdat, rec.offset, SEEK_SET);
			while( nBytes > 0 )
			{
				int bytes = fread(buf,1,min(sizeof(buf),(uint)nBytes),newdat);
				fwrite(buf, 1, bytes, dat);
				nBytes -= bytes;
			}
			rec.offset = offset;
			fwrite(&rec, sizeof(rec), 1, idx);
			totalNew++;
			fread(&rec, sizeof(rec), 1, newidx);
		}

		printf("done (%d records)", totalNew);
		total += totalNew;
	process_next:
		done = FindNext(&ffblk);
	}

	fclose(idx);
	fclose(dat);
	if( 0 == file_size(index) ) file_remove(index);
	if( 0 == file_size(data) ) file_remove(data);

	if( total ) printf("\n\nAdded %d records\n", total);
}

void
copy(char *data, char *fileSpec)
{
	KAF_HEADER  kafHdr;
	KAF_RECORD  kafRec;
	ACL_INDEX   aclRec;
	zDosFile    ffblk;
	char        path[MAXPATH], base[MAXPATH];
	int         done = 0;
	int         nTotalFiles = 0, nTotalRecs = 0;
	FILE       *in, *datOut, *idxOut;

	if( !fileSpec ) fail("No '-f' filespec for copy command.\n");

	datOut = fopen(data, "a+b");
	if( !datOut ) fail("Unable to open the ACL data file.\n");
	file_chext(path, data, ".aix");
	idxOut = fopen(path, "a+b");
	if( !idxOut )
	{
		fclose(datOut);
		if( 0 == file_size(data) ) file_remove(data);
		fail("Unable to open the ACL index/header file.\n");
	}

	fseek(datOut, 0L, SEEK_END);
	fseek(idxOut, 0L, SEEK_END);
	memset(aclRec.extra, 0, sizeof(aclRec.extra));
	strcpy(base, file_dirspec(fileSpec));
	done = FindFirst(fileSpec, 0, &ffblk);
	while( !done )
	{
		strcat(strcpy(path, base), ffblk.name);
		in = fopen(path, "rb");
		if( !in ) goto process_next;
		fread(&kafHdr, sizeof(kafHdr), 1, in);
		if( 0 == memcmp(kafHdr.idString, "SCi-DaT", 7) )
		{
			printf("Adding records from '%s'...", path);
			for( int i = 0; i < kafHdr.numRecs; ++i )
			{
				fread(&kafRec, sizeof(kafRec), 1, in);
				aclRec.offset = ftell(datOut);
				aclRec.lines = kafRec.numLines;
				aclRec.delay = kafRec.speed;
				aclRec.column = max((80 - kafRec.maxChars) / 2, 1);
				aclRec.size = kafRec.dataSize;
				fwrite(&aclRec, sizeof(aclRec), 1, idxOut);
				for( int j = 0; j < kafRec.numLines; ++j )
				{
					ushort datLen;
					char   buf[0x1000];
					// the ACL files do not store the NUL terminator
					fread(&datLen, sizeof(datLen), 1, in);
					fread(buf, datLen, 1, in);
					datLen--;
					fwrite(&datLen, sizeof(datLen), 1, datOut);
					fwrite(buf, datLen, 1, datOut);
				}
			}
			printf("done (%d records)\n", kafHdr.numRecs);
		}
		else printf("File '%s' is not a valid KAF.\n", path);
		fclose(in);
		nTotalRecs += kafHdr.numRecs;
		nTotalFiles++;
	process_next:
		done = FindNext(&ffblk);
	}

	fclose(datOut);
	fclose(idxOut);
	if( 0 == file_size(data) ) file_remove(data);
	file_chext(path, data, ".aix");
	if( 0 == file_size(data) ) file_remove(path);

	if( 0 != nTotalRecs )
	{
		printf("\nAdded %d records from %d files.\n", nTotalRecs, nTotalFiles);
	}
}
