
# .-----------------------------------------------------------------------.
# |                                                                       |
# |    m-Chat v1.6 for Mystic BBS Software v1.7.3+ by : grymmjack(gj!)    |
# |                                                                       |
# +-----------------------------------------------------------------------+
# |                                                                       |
# | Features:                                                             |
# |                                                                       |
# |  SysOp/User Colors      Word-Wrap             IRC-like Actions     |
# |  Logging facilty        Magic-Words           Command Recall       |
# |  User Input Filter      File Viewing          Online help          |
# |  Session Information    Session Timer         Custom MCI Parser    |
# |  99% Customizable       Multi-node Friendly   Comes w/MPL Source   |
# |  Forbidden Files List   Easy Setup/Install    Launch with F-Key    |
# |                                                                       |
# `-----------------------------------------------------------------------'

uses CFG
uses USER
uses DIR

# .----------------------------------------------------------------------.
# | m-Chat Program Information Constants                                 |
# `----------------------------------------------------------------------'
var   PROGRAM_WORD      string    # Magic-Word File
var   PROGRAM_FORBIDDEN string    # Forbidden File List (FFL)
const PROGRAM_NAME      = 'm-Chat'
const PROGRAM_VERSION   = 'v1.6'
const PROGRAM_AUTHOR    = 'grymmjack'
const PROGRAM_AUTHOR_C  = 'grymmjack (grymmjack@home.com)'
const PROGRAM_AUTHOR_A  = 'grymmjack'
const PROGRAM_AUTHOR_T  = 'liquid (liquid@darktech.org)'
const PROGRAM_AUTHOR_I  = 'liquid'
const PROGRAM_CONFIG    = 'MCHAT.CFG'
const PROGRAM_HELP      = 'MCHATH.ANS'
const PROGRAM_ABOUT     = 'MCHATA.ANS'
const PROGRAM_INFO      = 'MCHATI.ANS'
const PROGRAM_SPLASH    = 'MCHAT.ANS'
const PROGRAM_LOG       = 'MCHAT.LOG'
      PROGRAM_FORBIDDEN:= 'MCHAT.FFL'
      PROGRAM_WORD     := 'MCHAT.WRD'

# .----------------------------------------------------------------------.
# | Variables for /INFO command                                          |
# `----------------------------------------------------------------------'
var TOTAL_LOCAL_CHARS   longint
var TOTAL_REMOTE_CHARS  longint
var TOTAL_LOCAL_LINES   longint
var TOTAL_REMOTE_LINES  longint
var TOTAL_COMMANDS_RUN  longint
var START_TIME          longint
var TIMER_START_TIME    longint
var TIMER_CHECK_TIME    longint
var TOTAL_TIME          longint
var TOTAL_TIME_MIN      longint
var TOTAL_TIME_SEC      longint
var NOW_LOGGING         boolean
var LOG_FILE            string
var USER_INPUT          boolean
var MAGIC_WORDS         boolean
var LOG_ACTIVITY        boolean

# .----------------------------------------------------------------------.
# | CFGDATA Constants                                                    |
# `----------------------------------------------------------------------'
const LOG_HEADER        = 1
const LOG_FOOTER        = 2
const LOG_ALWAYS        = 3
const DEFAULT_SHOW_PATH = 4
const STR_ERROR         = 5
const STR_NORMAL        = 6
const STR_LOCAL_TEXT    = 7
const STR_REMOTE_TEXT   = 8
const STR_ACTION        = 9
const SYSOP_NAME        = 10
const STR_START         = 11
const STR_END           = 12
const ALLOW_USER_CMDS   = 13
const SHOW_LOG_MSGS     = 14
const STR_DARK_COLOR    = 15 
const STR_NORMAL_COLOR  = 16 
const STR_BRIGHT_COLOR  = 17 
const STR_DIVIDER       = 18
const CMD_PREFIX        = 19
const STR_REGULAR_COLOR = 20
const MULTINODE_ACTIVITY= 21
const STR_BG_COLOR      = 22
const WORDS_ALWAYS      = 23
const STR_PAUSE         = 24

# .----------------------------------------------------------------------.
# | Magic Words array                                                    |
# `----------------------------------------------------------------------'
var CFGMW string array (1..75)

# .----------------------------------------------------------------------.
# | Forbidden Files List array                                           |
# `----------------------------------------------------------------------'
var CFGFFL string array (1..50)

# .----------------------------------------------------------------------.
# | Generic Global-Scope Variables                                       |
# `----------------------------------------------------------------------'
var counter             integer
var loop                integer
var loop2               integer
var tmpstr1             string
var tmpstr2             string
var tmpstr3             string
var tmplong             longint

# .----------------------------------------------------------------------.
# | Setup default CFGDATA array                                          |
# `----------------------------------------------------------------------'
var CFGDATA string array (1..24)
    # Log file Header (|DT = Date, |TI = Time, |CU = Current user, |CH = Chat time)
    CFGDATA(01) := '+++[ |DT @ |TI NOW LOGGING CHAT WITH |CU ]'
    # Log file Footer (|DT = Date, |TI = Time, |CU = Current user, |CH = Chat time)
    CFGDATA(02) := '---[ |DT @ |TI CHAT LOGGING WITH |CU ENDED (|CH SPENT)]'
    # Always log all chatting (0 = No, 1 = Yes)
    CFGDATA(03) := '1'
    # Default path for showing text files with the /TYPE command
    CFGDATA(04) := 'c:\mystic\text\'
    # String: Command error prefix
    CFGDATA(05) := '|08*|04*|12* |14'
    # String: Normal command prefix
    CFGDATA(06) := '|08-|07-|15- |07'
    # String: Local console text color
    CFGDATA(07) := '|15'
    # String: Remote user text color
    CFGDATA(08) := '|07'
    # String: Action prefix
    CFGDATA(09) := '|10* '
    # Sysops name
    CFGDATA(10) := 'sysop'
    # String: Beginning a chat session
    CFGDATA(11) := '|01|09|11 |15' + CFGDATA(SYSOP_NAME) + ' is online|07...|CR|CR'
    # String: Ending a chat session
    CFGDATA(12) := '|CR|CR|01|09|11 |15' + CFGDATA(SYSOP_NAME) + ' is offline|07...|CR|CR|PA'
    # Allow users to issue commands? (0 = No, 1 = Yes)
    CFGDATA(13) := '0'
    # Show Log file notification messages? (0 = No, 1 = Yes)
    CFGDATA(14) := '1'
    # String: Dark color (|DC)
    CFGDATA(15) := '|01'
    # String: Normal color (|NC)
    CFGDATA(16) := '|09'
    # String: Bright color (|BC)
    CFGDATA(17) := '|15'
    # String: Text divider
    CFGDATA(18) := '|16|01- -- ---|09-|01-|09--|01-|09---|01-|09|$D45-|01-|09---|01-|09--|01-|09-|01--- -- -' 
    # Log prefix/postfix for command entries
    CFGDATA(19) := '*'
    # String: Regular color (|RC)
    CFGDATA(20) := '|07'
    # Description for multi-node activity when users are in chat
    CFGDATA(21) := 'Chatting with SysOp'
    # String: Background color (|BG)
    CFGDATA(22) := '|16'
    # Always use Magic-Words? (0 = No, 1 = Yes)
    CFGDATA(23) := '1'
    # Pause String
    CFGDATA(24) := '                                  |NC(|BCpaused|NC)'






# .----------------------------------------------------------------------.
# |                                                                      |
# | *** Functions and Procedures                                         |
# |                                                                      |
# `----------------------------------------------------------------------'

# .----------------------------------------------------------------------.
# | Procedure: populateFFLArray                                          |
# +----------------------------------------------------------------------+
# | This procedure populates the Forbidden Files List array if the file  |
# | MCHAT.FFL exists in the boards DATA directory. The format for this   |
# | file is plain text ascii with one fully qualified path/filename per  |
# | line you may have a maximum of 25 files in your FFL;                 |
# |                                                                      |
# | C:\MYSTIC\LOGS\MCHAT.LOG                                             |
# | /mystic/logs/mchat.log                                               |
# | x:\mystic\logs\sysop.1                                               |
# |                                                                      |
# | NOTE: Wildcards do NOT evaluate                                      |
# `----------------------------------------------------------------------'
#
procedure populateFFL                
begin
  counter := 0
  if fexist(PROGRAM_FORBIDDEN) = true then
    fopen(2, text, reset, PROGRAM_FORBIDDEN)
    while not eof(2)
      counter := counter + 1
      if counter = 50 then
        fclose(2)
        exit
      else
        freadln(2, CFGFFL(counter))
      endif
    wend
    fclose(2)
  endif
pend



# .----------------------------------------------------------------------.
# | Function: isInFFL                                                    |
# +----------------------------------------------------------------------+
# | This function checks to see if the passed argument exists in the FFL |
# | array. If it does it returns true, otherwise it returns false.       |
# `----------------------------------------------------------------------'
#
function isInFFL(filetocheck string) : boolean
begin
  counter := 0
  for counter := 1 to 50
    if filetocheck = CFGFFL(counter) then
      isInFFL := true
      exit
    endif
  fend
  isInFFL := false
pend



# .----------------------------------------------------------------------.
# | Procedure: populateMWL                                               |
# +----------------------------------------------------------------------+
# | This procedure populates the CFGMWL array with the contents of the   |
# | MCHAT.WRD file in the boards DATA directory if it exists.            |
# |                                                                      |
# | The format for the MCHAT.WRD file is in ASCII text and includes one  |
# | word/replacement word/phrase pair per line seperated by a space and  |
# | followed by a carriage return and you may have up to 50 pairs;       |
# |                                                                      |
# | mchat m-Chat                                                         |
# | mystic Mystic BBS Software                                           |
# | grymmjack jedi-pimp: grymmjack                                       |
# `----------------------------------------------------------------------'
#
procedure populateMWL
begin
  counter := 0
  if fexist(PROGRAM_WORD) = true then
    fopen(2, text, reset, PROGRAM_WORD)
    while not eof(2)
      counter := counter + 1
      if counter = 75 then
        fclose(2)
        exit
      else
        freadln(2, CFGMW(counter))
      endif
    wend
    fclose(2)
  endif
pend



# .----------------------------------------------------------------------.
# | Function: getMagicWord                                               |
# +----------------------------------------------------------------------+
# | This function goes through the MagicWord file and looks for a Magic- |
# | -Word matching the passed string argument. If it is found it returns |
# | the replacement word/phrase for the MagicWord entry. Otherwise it    |
# | returns an empty string.                                             |
# `----------------------------------------------------------------------'
#
function getMagicWord(text string) : string
begin
  
  var magicLine string
  var matchWord string
  var magicWord string
  
  if fexist(PROGRAM_WORD) = true then
    for counter := 1 to 75
      magicLine := CFGMW(counter)
      if magicLine = '' then
        getMagicWord := ''
        exit
      else
        matchWord := copy(magicLine, 1, pos(' ', magicLine) - 1)
        magicWord := copy(magicLine, length(matchWord) + 2, length(magicLine))
        if matchWord = text then
          getMagicWord := magicWord
          exit
        endif
      endif
    fend
  endif

  getMagicWord := ''

pend



# .----------------------------------------------------------------------.
# | Procedure: clearScreen                                               |
# +----------------------------------------------------------------------+
# | This procedure clears the screen using the desired background color  |
# | as specified in the configuration file.                              |
# `----------------------------------------------------------------------'
#
procedure clearScreen
begin

  # If we are logging this session make a note that we cleared the screen if activity logging is on
  if LOG_ACTIVITY = true then
    if NOW_LOGGING = true then
      fwriteln(1, CFGDATA(CMD_PREFIX) + ' CLEARED THE SCREEN ' + CFGDATA(CMD_PREFIX))
    endif
  endif

  write(CFGDATA(STR_BG_COLOR) + '|CL')

pend



# .--------------------------------------------------------------------------.
# | Function: parse_MCI                                                      |
# +--------------------------------------------------------------------------+
# | This function goes through the passed string argument character by       |
# | character searching for MCI codes. When it finds one it then replaces the|
# | MCI code text with the values/text the MCI represents.                   |
# +--------------------------------------------------------------------------+
# |                                                                          |
# | m-Chat Proprietary MCI Code Listing:                                     |
# |                                                                          |
# | |DC Dark color           |NC Normal color         |BC Bright color       |
# | |RC Regular color        |BG Background color     |TI Current time       |
# | |DT Current date         |CU Current user         |CH Session duration   |
# | |PS Pause string         |?W Magic-Words ON/OFF   |?I User Input ON/OFF  |
# | |?U User Commands ON/OFF |?L Logging ON/OFF       |@T Session Start Time |
# | |@D Session Start Date   |LF Log File Path        |LS Log FIle Size      |
# | |%C Total Chars/SysOp    |%L Total Lines/SysOp    |#C Total Chars/User   |
# | |#L Total Lines/User     |%# Difference(Score)    |@W Who is Winning     |
# | |CI Total Cmds Issued    |!V m-Chat Version       |!A Path to MCHATA.ANS |
# | |!H Path to MCHATH.ANS   |!S Path to MCHAT.ANS    |!I Path to MCHATI.ANS |
# | |SY SysOp Name           |SP Default Text Path    |?M Log ActMsgs ON/OFF |
# |                                                                          |
# `--------------------------------------------------------------------------'
#
function parse_MCI(originalString string) : string
begin

  var allParsed boolean
  var parsedString string
  var posCurChar integer
  
  # Set position of current character to 0, and the 'done' flag to false
  posCurChar := 0
  allParsed  := false

  repeat
    # Increment the position of the current character
    posCurChar := posCurChar + 1

    # If character is a pipe
    if copy(originalString, posCurChar, 1) = '|' then
      # Check for m-Chat MCI codes
      # Dark color
      if upper(copy(originalString, posCurChar, 3)) = '|DC' then
        parsedString := parsedString + CFGDATA(STR_DARK_COLOR)
        posCurChar := posCurChar + 2
      # Session start time
      elseif upper(copy(originalString, posCurChar, 3)) = '|@T' then
        parsedString := parsedString + datestr(START_TIME, 1)
        posCurChar := posCurChar + 2
      # Session start date
      elseif upper(copy(originalString, posCurChar, 3)) = '|@D' then
        parsedString := parsedString + timestr(START_TIME, 1)
        posCurChar := posCurChar + 2
      # Log file path        
      elseif upper(copy(originalString, posCurChar, 3)) = '|LF' then
        if NOW_LOGGING = true then
          parsedString := parsedString + LOG_FILE               
          posCurChar := posCurChar + 2
        else
          parsedString := parsedString + 'None'
          posCurChar := posCurChar + 2
        endif
     # Log file size       
      elseif upper(copy(originalString, posCurChar, 3)) = '|LS' then
        if NOW_LOGGING = true then
          parsedString := parsedString + int2str(filesize(1))
        else
          parsedSTring := parsedString + '0'
        endif
        posCurChar := posCurChar + 2
     # Total chars/Sysop   
      elseif upper(copy(originalString, posCurChar, 3)) = '|%C' then
        parsedString := parsedString + int2str(TOTAL_LOCAL_CHARS)
        posCurChar := posCurChar + 2
     # Total lines/Sysop   
      elseif upper(copy(originalString, posCurChar, 3)) = '|%L' then
        parsedString := parsedString + int2str(TOTAL_LOCAL_LINES)
        posCurChar := posCurChar + 2
     # Total chars/User
      elseif upper(copy(originalString, posCurChar, 3)) = '|#C' then
        parsedString := parsedString + int2str(TOTAL_REMOTE_CHARS)
        posCurChar := posCurChar + 2
     # Total lines/User   
      elseif upper(copy(originalString, posCurChar, 3)) = '|#L' then
        parsedString := parsedString + int2str(TOTAL_REMOTE_LINES)
        posCurChar := posCurChar + 2
     # Total difference (score)
      elseif upper(copy(originalString, posCurChar, 3)) = '|%#' then
        if TOTAL_LOCAL_CHARS + TOTAL_LOCAL_LINES > TOTAL_REMOTE_CHARS + TOTAL_REMOTE_LINES then
          tmplong := TOTAL_LOCAL_CHARS + TOTAL_LOCAL_LINES - TOTAL_REMOTE_CHARS - TOTAL_REMOTE_LINES
        else
          tmplong := TOTAL_REMOTE_CHARS + TOTAL_REMOTE_LINES - TOTAL_LOCAL_CHARS - TOTAL_REMOTE_LINES
        endif
        parsedString := parsedString + int2str(tmplong)
        posCurChar := posCurChar + 2
     # Whos winning
      elseif upper(copy(originalString, posCurChar, 3)) = '|@W' then
        if TOTAL_LOCAL_CHARS + TOTAL_LOCAL_LINES > TOTAL_REMOTE_CHARS + TOTAL_REMOTE_LINES then
          tmpstr1 := CFGDATA(SYSOP_NAME)
        else
          tmpstr1 := useralias
        endif
        parsedString := parsedString + tmpstr1
        posCurChar := posCurChar + 2
     # Total commands issued
      elseif upper(copy(originalString, posCurChar, 3)) = '|CI' then
        parsedString := parsedString + int2str(TOTAL_COMMANDS_RUN)  
        posCurChar := posCurChar + 2
     # Sysop name          
      elseif upper(copy(originalString, posCurChar, 3)) = '|SY' then
        parsedString := parsedString + CFGDATA(SYSOP_NAME)
        posCurChar := posCurChar + 2
     # Default text path  
      elseif upper(copy(originalString, posCurChar, 3)) = '|SP' then
        parsedString := parsedString + CFGDATA(DEFAULT_SHOW_PATH)
        posCurChar := posCurChar + 2
     # Toggle: Log activity messages
      elseif upper(copy(originalString, posCurChar, 3)) = '|?M' then
        if LOG_ACTIVITY = true then
          parsedString := parsedString + 'ON'
        else
          parsedString := parsedString + 'OFF'
        endif
        posCurChar := posCurChar + 2
     # m-Chat version
      elseif upper(copy(originalString, posCurChar, 3)) = '|!V' then
        parsedString := parsedString + PROGRAM_VERSION
        posCurChar := posCurChar + 2
     # Path to MCHAT.ANS
      elseif upper(copy(originalString, posCurChar, 3)) = '|!S' then
        if fexist(cfgtextpath + 'MCHAT.ANS') = true then
          tmpstr1 := cfgtextpath + 'MCHAT.ANS'
        else
          tmpstr1 := 'MISSING!'
        endif
        parsedString := parsedString + tmpstr1
        posCurChar := posCurChar + 2
     # Path to MCHATA.ANS
      elseif upper(copy(originalString, posCurChar, 3)) = '|!A' then
        if fexist(cfgtextpath + 'MCHATA.ANS') = true then
          tmpstr1 := cfgtextpath + 'MCHATA.ANS'
        else
          tmpstr1 := 'MISSING!'
        endif
        parsedString := parsedString + tmpstr1
        posCurChar := posCurChar + 2
     # Path to MCHATH.ANS
      elseif upper(copy(originalString, posCurChar, 3)) = '|!H' then
        if fexist(cfgtextpath + 'MCHATH.ANS') = true then
          tmpstr1 := cfgtextpath + 'MCHATH.ANS'
        else
          tmpstr1 := 'MISSING!'
        endif
        parsedString := parsedString + tmpstr1
        posCurChar := posCurChar + 2
     # Path to MCHATI.ANS
      elseif upper(copy(originalString, posCurChar, 3)) = '|!I' then
        if fexist(cfgtextpath + 'MCHATI.ANS') = true then
          tmpstr1 := cfgtextpath + 'MCHATI.ANS'
        else
          tmpstr1 := 'MISSING!'
        endif
        parsedString := parsedString + tmpstr1
        posCurChar := posCurChar + 2
      # Normal color
      elseif upper(copy(originalString, posCurChar, 3)) = '|NC' then
        parsedString := parsedString + CFGDATA(STR_NORMAL_COLOR)
        posCurChar := posCurChar + 2
      # Bright color
      elseif upper(copy(originalString, posCurChar, 3)) = '|BC' then
        parsedString := parsedString + CFGDATA(STR_BRIGHT_COLOR)
        posCurChar := posCurChar + 2
      # Background color
      elseif upper(copy(originalString, posCurChar, 3)) = '|BG' then
        parsedString := parsedString + CFGDATA(STR_BG_COLOR)
        posCurChar := posCurChar + 2
      # Regular color
      elseif upper(copy(originalString, posCurChar, 3)) = '|RC' then
        parsedString := parsedString + CFGDATA(STR_REGULAR_COLOR)
        posCurChar := posCurChar + 2
      # Pause String
      elseif upper(copy(originalString, posCurChar, 3)) = '|PS' then
        parsedString := parsedString + CFGDATA(STR_PAUSE)
        posCurChar := posCurChar + 2
      # Current time
      elseif upper(copy(originalString, posCurChar, 3)) = '|TI' then
        parsedString := parsedString + timestr(datetime, 1)
        posCurChar := posCurChar + 2
      # Current date
      elseif upper(copy(originalString, posCurChar, 3)) = '|DT' then
        parsedString := parsedString + datestr(datetime, 1)
        posCurChar := posCurChar + 2
      # Current user
      elseif upper(copy(originalString, posCurChar, 3)) = '|CU' then
        parsedString := parsedString + useralias
        posCurChar := posCurChar + 2
      # Session duration
      elseif upper(copy(originalString, posCurChar, 3)) = '|CH' then
        TIMER_CHECK_TIME := timer
        TOTAL_TIME := TIMER_CHECK_TIME - TIMER_START_TIME
        TOTAL_TIME_MIN := TOTAL_TIME / 60
        TOTAL_TIME_SEC := TOTAL_TIME MOD 60
        parsedString := parsedString + int2str(TOTAL_TIME_MIN) + 'm' + int2str(TOTAL_TIME_SEC) + 's'
        posCurChar := posCurChar + 2
      # Toggle: Magic-Words
      elseif upper(copy(originalString, posCurChar, 3)) = '|?W' then
        if MAGIC_WORDS = true then
          parsedString := parsedString + 'ON'
        else
          parsedString := parsedString + 'OFF'
        endif
        posCurChar := posCurChar + 2
      # Toggle: User Input
      elseif upper(copy(originalString, posCurChar, 3)) = '|?I' then
        if USER_INPUT = true then
          parsedString := parsedString + 'ON'
        else
          parsedString := parsedString + 'OFF'
        endif
        posCurChar := posCurChar + 2
      # Toggle: User Commands
      elseif upper(copy(originalString, posCurChar, 3)) = '|?U' then
        if CFGDATA(ALLOW_USER_CMDS) = '1' then
          parsedString := parsedString + 'ON'
        else
          parsedString := parsedString + 'OFF'
        endif
        posCurChar := posCurChar + 2
      # Toggle: Logging
      elseif upper(copy(originalString, posCurChar, 3)) = '|?L' then
        if NOW_LOGGING = true then
          parsedString := parsedString + 'ON'
        else
          parsedString := parsedString + 'OFF'
        endif
        posCurChar := posCurChar + 2
      else
        parsedString := parsedString + '|'
      endif
    else
      parsedString := parsedString + copy(originalString, posCurChar, 1)
    endif

    if posCurChar >= length(originalString) then
      allParsed := true
      parse_mci := parsedString
    endif
    
  until allParsed = true

pend



# .----------------------------------------------------------------------.
# | Procedure: create_log                                                |
# +----------------------------------------------------------------------+
# | This procedure opens/creates the log file. If the argument passed is |
# | a new file, it creates it, otherwise it appends to the existing file.|
# `----------------------------------------------------------------------'
#
procedure create_log(args string)
begin
  if fexist(args) then
    fopen(1, text, append, args)
    LOG_FILE := args
  else 
    fopen(1, text, rewrite, args)
    LOG_FILE := args
  endif

  # Write the log header to the file
  fwriteln(1, parse_MCI(CFGDATA(LOG_HEADER)))
  fwriteln(1, '')

  # If log file activity messages are enabled then show a message
  if CFGDATA(SHOW_LOG_MSGS) = '1' then
    writeln('|CR' + CFGDATA(STR_NORMAL) + LOG_FILE + ' log file started...')
  endif
pend



# .----------------------------------------------------------------------.
# | Procedure: close_log                                                 |
# +----------------------------------------------------------------------+
# | This procedure closes the open log file.                             |
# `----------------------------------------------------------------------'
#
procedure close_log
begin
  # Write the log footer to the file
  fwriteln(1, '')
  fwriteln(1, parse_mci(CFGDATA(LOG_FOOTER)))
  fwriteln(1, '')
  fwriteln(1, '')
  
  # Close the file
  fclose(1)
  
  # If log file activity messages are enabled then show a message
  if CFGDATA(SHOW_LOG_MSGS) = '1' then
    writeln('|CR' + CFGDATA(STR_NORMAL) + LOG_FILE + ' log file closed...')
  endif
pend



# .----------------------------------------------------------------------.
# | Procedure: write_log                                                 |
# +----------------------------------------------------------------------+
# | This procedure writes to the log file. We do this in procedure form  |
# | because it is more elegant due to the NOW_LOGGING condition that has |
# | to be tested everytime we want to write to the log.                  |
# `----------------------------------------------------------------------'
#
procedure write_log(args string)
begin
  # Only write to the log if we are currently logging. This procedure can
  # be called even if we aren't logging.
  if NOW_LOGGING = true then
    fwriteln(1, stripmci(args))
  endif
pend



# .----------------------------------------------------------------------.
# | Procedure: log                                                       |
# +----------------------------------------------------------------------+
# | This procedure handles the arguments passed by the /LOG command so   |
# | that the sibling procedures; create_log and close_log do not have to |
# | worry about dealing with the syntax of the argument passed to them.  |
# `----------------------------------------------------------------------'
#
procedure log(args string)
begin

  # If no argument is given...
  if args = '' then
    # ..and we are already logging...
    if NOW_LOGGING = true then
      # ...then close the existing log.
      NOW_LOGGING := false
      close_log
    # ...and we aren't logging...
    else
      # ...then display an error message and prompt for an argument.
      writeln('|CR' + CFGDATA(STR_ERROR) + 'log to what file!?')
    endif
    # And lastly, exit the procedure
    exit
  endif
  
  # Okay, an argument exists..
  if upper(copy(args, length(args)-3, 4)) = '.LOG'
    # ...and it has a .LOG extension...
    NOW_LOGGING := true
    if pos(pathchar, args) = 0 then
      # ...but it has no path information so we prefix the BBS log dir.
      create_log(cfglogspath + args)
    else
      # ..and it has path information so we are going to just pass it on.
      create_log(args)
    endif
  else
    # ..but no extension. So lets append the .LOG to the end and pass it on.
    NOW_LOGGING := true
    # ..and no path information so we prefix the BBS log dir.
    if pos(pathchar, args) = 0 then
      #...and then add an extension and then finall pass it to create_log
      create_log(cfglogspath + args + '.LOG')
    else
      # ..but has a path already so we append .LOG extension and pass it on.
      create_log(args + '.LOG')
    endif
  endif

pend


# .----------------------------------------------------------------------.
# | Procedure: viewFile                                                  |
# +----------------------------------------------------------------------+
# | This procedure displays *.ANS, or *.ASC file that it is passed while |
# | also interpretting the m-Chat MCI codes.                             |
# `----------------------------------------------------------------------'
#
procedure viewFile(filetoview string)
begin
  var curLine string
#  var curChar char
  
  if fexist(filetoview) = true then
    fopen(3, text, reset, filetoview)
    while not eof(3)
      freadln(3, curLine)
      writeln(parse_MCI(curLine))
    wend
    fclose(3)    
    
  elseif fexist(filetoview + '.ans') = true then
    fopen(3, text, reset, filetoview + '.ans')
    while not eof(3)
      freadln(3, curLine)
      writeln(parse_MCI(curLine))
    wend
    fclose(3)    
  elseif fexist(filetoview + '.asc') = true then
    fopen(3, text, reset, filetoview + '.asc')
    while not eof(3)
      freadln(3, curLine)
      writeln(parse_MCI(curLine))
    wend
    fclose(3)    
  else
    exit
  endif
  
pend



# .----------------------------------------------------------------------.
# | Procedure: read_config                                               |
# +----------------------------------------------------------------------+
# | This procedure reads the MCHAT.CFG file into the CFGDATA array.      |
# `----------------------------------------------------------------------'
#
procedure read_config
begin

  # If the MCHAT.CFG file already exists process it
  if fexist(cfgdatapath + PROGRAM_CONFIG) then
    counter := 0
    fopen(1, text, reset, cfgdatapath + PROGRAM_CONFIG)
      while not eof(1)
        freadln(1, tmpstr1)
        if copy(tmpstr1, 1, 1) <> '#' then
          if copy(tmpstr1, 1, 1) <> '' then
            if copy(tmpstr1, 1, 1) <> ' ' then
              counter := counter + 1
              CFGDATA(counter) := tmpstr1
            endif
          endif
        endif
      wend
    fclose(1)
  else
    # Otherwise create MCHAT.CFG with the defaults
    dispfile(cfgtextpath + PROGRAM_SPLASH)
    writeln('|CR' + CFGDATA(STR_ERROR) + cfgdatapath + PROGRAM_CONFIG + ' does NOT exist.')
    writeln(CFGDATA(STR_ERROR) + 'creating this file for you. edit it with a text editor')
    writeln(CFGDATA(STR_ERROR) + 'to customize the way ' + PROGRAM_NAME + ' works..')
      fopen(1, text, rewrite, cfgdatapath + 'mchat.cfg')
      fwriteln(1, '# ' + PROGRAM_CONFIG + ' -- Configuration file for ' + PROGRAM_NAME)
      fwriteln(1, '# NOTE: All lines that begin with a "#" are comments...')
      fwriteln(1, '#       All MCI codes are valid as are color codes!')
      fwriteln(1, '')
      fwriteln(1, '# Log file Header (|DT = Date, |TI = Time, |CU = Current user, |CH = Chat time): ')
      fwriteln(1, CFGDATA(LOG_HEADER))
      fwriteln(1, '# Log file Footer (|DT = Date, |TI = Time, |CU = Current user, |CH = Chat time):')
      fwriteln(1, CFGDATA(LOG_FOOTER))
      fwriteln(1, '# Always log all chatting? (0 = No, 1 = Yes):')
      fwriteln(1, CFGDATA(LOG_ALWAYS))
      fwriteln(1, '# Default path for showing text files with the /TYPE command:')
      fwriteln(1, CFGDATA(DEFAULT_SHOW_PATH))
      fwriteln(1, '# String: Command error prefix:')
      fwriteln(1, CFGDATA(STR_ERROR))
      fwriteln(1, '# String: Normal command prefix:')
      fwriteln(1, CFGDATA(STR_NORMAL))
      fwriteln(1, '# String: Local console text color (use two digit mystic color code! (ie: |15) ):')
      fwriteln(1, CFGDATA(STR_LOCAL_TEXT))
      fwriteln(1, '# String: Remote user text color (use two digit mystic color code! (ie: |07) ):')
      fwriteln(1, CFGDATA(STR_REMOTE_TEXT))
      fwriteln(1, '# String: Action prefix (something like |10*  works good!):')
      fwriteln(1, CFGDATA(STR_ACTION))
      fwriteln(1, '# String: Sysops name:')
      fwriteln(1, CFGDATA(SYSOP_NAME))
      fwriteln(1, '# String: Beginning a chat session:')
      fwriteln(1, CFGDATA(STR_START))
      fwriteln(1, '# String: Ending a chat session:')
      fwriteln(1, CFGDATA(STR_END))
      fwriteln(1, '# Allow users to issue commands? (0 = No, 1 = Yes):')
      fwriteln(1, CFGDATA(ALLOW_USER_CMDS))
      fwriteln(1, '# Show Log file notification messages? (0 = No, 1 = Yes):')
      fwriteln(1, CFGDATA(SHOW_LOG_MSGS))
      fwriteln(1, '# String: Dark color (use two digit mystic color code! (ie: |01) ):')
      fwriteln(1, CFGDATA(STR_DARK_COLOR))
      fwriteln(1, '# String: Normal color (use two digit mystic color code! (ie: |09) ):')
      fwriteln(1, CFGDATA(STR_NORMAL_COLOR))
      fwriteln(1, '# String: Bright color (use two digit mystic color code! (ie: |15) ):')
      fwriteln(1, CFGDATA(STR_BRIGHT_COLOR))
      fwriteln(1, '# String: Text divider (remember all mystic MCIs work!):')
      fwriteln(1, CFGDATA(STR_DIVIDER))
      fwriteln(1, '# Log prefix/postfix for command entries:')
      fwriteln(1, CFGDATA(CMD_PREFIX))
      fwriteln(1, '# String: Regular color (use two digit mystic color code! (ie: |07) ):')
      fwriteln(1, CFGDATA(STR_REGULAR_COLOR))
      fwriteln(1, '# Description for multi-node activity when users are in chat:')
      fwriteln(1, CFGDATA(MULTINODE_ACTIVITY))
      fwriteln(1, '# String: Background color (use two digit mystic color code! (ie: |16) ):')
      fwriteln(1, CFGDATA(STR_BG_COLOR))
      fwriteln(1, '# Always use Magic-Words? (0 = No, 1 = Yes):')
      fwriteln(1, CFGDATA(WORDS_ALWAYS))
      fwriteln(1, '# Pause String:')
      fwriteln(1, CFGDATA(STR_PAUSE))

    fclose(1)
    writeln(CFGDATA(STR_NORMAL) + cfgdatapath + PROGRAM_CONFIG + ' created...|CR|PA')
  endif

pend



# .----------------------------------------------------------------------.
# | Procedure: showHelp                                                  |
# +----------------------------------------------------------------------+
# | This procedure displays the m-Chat online help. If the external file |
# | MCHATH.ANS exists in the BBS text directory it is shown. Otherwise,  |
# | a generic help screen is shown in its place.                         |
# `----------------------------------------------------------------------'
#
procedure showHelp
begin
  # Make a note that we showed the help screen in the log if activity logging is on
  if LOG_ACTIVITY = true then
    write_log(CFGDATA(CMD_PREFIX) + ' SHOWED MCHAT HELP ' + CFGDATA(CMD_PREFIX))
  endif
  
  # If the help screen file MCHATH.ANS exists show it, pause, and then draw
  # the MCHAT.ANS splash screen
  if fexist(cfgtextpath + PROGRAM_HELP) then
    dispfile(cfgtextpath + PROGRAM_HELP)
    writeln(parse_MCI(CFGDATA(STR_PAUSE) + '|PN'))
    dispfile(cfgtextpath + PROGRAM_SPLASH)
  else
    # Otherwise draw a generic help screen            
    writeln('')
    writeln(CFGDATA(STR_DIVIDER))
    writeln(parse_mci('|BC' +  padct(PROGRAM_NAME + '|RC ' + PROGRAM_VERSION + ' |NCCommand Help', 80, ' ')))
    writeln(CFGDATA(STR_DIVIDER))
    writeln(parse_mci('|NC/|BCA|DC, |NC/|BCABOUT|DC......: |RCShow m-Chat specific stuff, like credits, etc.'))
    writeln(parse_mci('|NC/|BCC|DC, |NC/|BCCLS|DC........: |RCClear the screen of all text.'))
    writeln(parse_mci('|NC/|BCG|DC, |NC/|BCGOODBYE|DC....: |RCHang up on the user immediately.'))
    writeln(parse_mci('|NC/|BC?|DC, |NC/|BCH|DC, |NC/|BCHELP|DC...: |RCShow m-Chat command help.'))
    writeln(parse_mci('|NC/|BCI|DC, |NC/|BCINFO|DC.......: |RCShow m-Chat information.'))
    writeln(parse_mci('|NC/|BCL|DC, |NC/|BCLOG|DC........: |RCSave input to <logfile>, or stop logging if already on.'))
    writeln(parse_mci('|NC/|BCM|DC, |NC/|BCMSG|DC........: |RCSend a multinode message (/msg # <text>) where # = node #.'))
    writeln(parse_mci('|NC |BC |DC  |NC/|BCME|DC.........: |RCPerform an action (like IRC).'))
    writeln(parse_mci('|NC/|BCT|DC, |NC/|BCTYPE|DC.......: |RCAttempt to display <text file>.'))
    writeln(parse_mci('|NC/|BCW|DC, |NC/|BCWHO|DC........: |RCShow whos online.'))
    writeln(parse_mci('|NC |BC |DC  |NC^|BCL|DC..........: |RCToggle activity logging on/off.'))
    writeln(parse_mci('|NC |BC |DC  |NC^|BCU|DC..........: |RCToggle user-commands on/off.'))
    writeln(parse_mci('|NC |BC |DC  |NC^|BCW|DC..........: |RCToggle magic-words on/off.'))
    writeln(parse_mci('|NC |BC |DC  |NC^|BCX|DC..........: |RCToggle users input on/off.'))
    writeln(parse_mci('|NC |BC |DC  |NC^|BCY|DC..........: |RCYank current line of text.'))
    writeln(CFGDATA(STR_DIVIDER))
  endif
pend



# .----------------------------------------------------------------------.
# | Procedure: showInfo                                                  |
# +----------------------------------------------------------------------+
# | This procedure displays the m-Chat Session Information. It lets the  |
# | chatters know the states of toggles, and other things such as m-Chat |
# | specific details like program version, etc.                          |
# `----------------------------------------------------------------------'
#
procedure showInfo
begin

  # Make a note that we showed the Session Information in the log if activity logging is on
  if LOG_ACTIVITY = true then
    write_log(CFGDATA(CMD_PREFIX) + ' SHOWED MCHAT SESSION INFO ' + CFGDATA(CMD_PREFIX))
  endif
  
  # Display the Session Information

  if fexist(cfgtextpath + PROGRAM_INFO) then  
    viewFile(cfgtextpath + PROGRAM_INFO)
    writeln(parse_MCI(CFGDATA(STR_PAUSE) + '|PN'))
    dispfile(cfgtextpath + PROGRAM_SPLASH)
  else
    clearScreen
    writeln(parse_MCI('|BC' +  padct(PROGRAM_NAME + '|RC ' + PROGRAM_VERSION + ' |NCInformation', 83, ' ')))
    writeln(CFGDATA(STR_DIVIDER))
    writeln(parse_MCI('|BCStarted on|DC.......: |NC|@D @ |@T |DC(|RCspent|DC: |NC(|BC|CH|NC)|DC)'))
    writeln(parse_MCI('|BCLogging|DC..........: |NC|?L |DC(|RC|LF|DC)|NC |LS bytes'))
    writeln('')
    writeln(parse_MCI('|NC[|BCTotals for this chat|NC]|DC---------- --- -- -      |NC[|BCToggles|NC]|DC---------- --- -- -'))
    writeln(parse_MCI('|DC |NC|SY|DC |[X48|RCMagic-Words|DC.....: |NC|?W'))
    writeln(parse_MCI('|RCCharacters Typed|DC.: |NC|%C|[X48|RCUser Input|DC......: |NC|?I'))
    writeln(parse_MCI('|RCLines Entered|DC....: |NC|%L|[X48|RCUser Commands|DC...: |NC|?U'))
    writeln(parse_MCI('|[X48|RCLog Activity|DC....: |NC|?M'))
    writeln(parse_MCI('|DC |NC|UH|DC '))
    writeln(parse_MCI('|BCCharacters Typed|DC.: |NC|#C'))
    writeln(parse_MCI('|BCLines Entered|DC....: |NC|#L'))
    writeln('')
    writeln(parse_MCI('|BCDifference of|DC....: |NC|%# |DC(|RCin |BC|@Ws |RCfavor|DC)'))
    writeln(parse_MCI('|BCCommands Issued|DC..: |NC|CI'))
    writeln('')
    writeln(parse_MCI('|NC[|BC' + PROGRAM_NAME + ' Specific Information|NC]|DC---------- --- ---  -'))
    writeln(parse_MCI('|BC' + padrt(PROGRAM_SPLASH + '|DC', 20, '.') + ': |NC|!S'))
    writeln(parse_MCI('|BC' + padrt(PROGRAM_ABOUT + '|DC', 20, '.') + ': |NC|!A'))
    writeln(parse_MCI('|BC' + padrt(PROGRAM_HELP + '|DC', 20, '.') + ': |NC|!H'))
    writeln(parse_MCI('|BC' + padrt(PROGRAM_INFO + '|DC', 20, '.') + ': |NC|!I'))
    writeln(CFGDATA(STR_DIVIDER))
    if fexist(cfgtextpath + PROGRAM_SPLASH) then
      writeln(parse_MCI(CFGDATA(STR_PAUSE) + '|PN'))
      dispfile(cfgtextpath + PROGRAM_SPLASH)
    endif
  endif

pend



# .----------------------------------------------------------------------.
# | Procedure: showAbout                                                 |
# +----------------------------------------------------------------------+
# | This procedure displays the m-Chat about screen. It contains info,   |
# | and credits regarding m-Chat.                                        |
# `----------------------------------------------------------------------'
#
procedure showAbout
begin

  # Make a note that we showed the about screen in the log if activity logging is on
  if LOG_ACTIVITY = true then
    write_log(CFGDATA(CMD_PREFIX) + ' SHOWED MCHAT ABOUT SCREEN ' + CFGDATA(CMD_PREFIX))
  endif
 
  # If the external file MCHATA.ANS exists display it
  if fexist(cfgtextpath + PROGRAM_ABOUT) then
    dispfile(cfgtextpath + PROGRAM_ABOUT)
    writeln(parse_MCI(CFGDATA(STR_PAUSE) + '|PN'))
    dispfile(cfgtextpath + PROGRAM_SPLASH)
  else
    # Otherwise show a generic about screen.  
    clearScreen
    writeln(CFGDATA(STR_DIVIDER))
    writeln(parse_MCI(padct('|NCAbout |BC' + PROGRAM_NAME + '|RC ' + PROGRAM_VERSION, 86, ' ')))
    writeln('')
    writeln(parse_MCI('|NC(|BCCredits|NC)|DC---------- --- --  -'))
    writeln(parse_MCI('|BCCode|DC......: |NC' + PROGRAM_AUTHOR_C))
    writeln(parse_MCI('|BCArtwork|DC...: |NC' + PROGRAM_AUTHOR_A))
    writeln(parse_MCI('|BCTesting|DC...: |NC' + PROGRAM_AUTHOR_T))
    writeln(parse_MCI('|BCIdeas|DC.....: |NC' + PROGRAM_AUTHOR_I + '|CR'))
    writeln(parse_MCI('|NC(|BCWhy was it made?|NC)|DC---------- --- --  -'))
    writeln(parse_MCI('|RCm-Chat was written in 100% MPL. It was made because Mystic BBSs default single-'))
    writeln(parse_MCI('line chat mode leaves alot to be desired. Hopefully, m-Chat can fill the void'))
    writeln(parse_MCI('and make Mystic BBS chat a little bit more fun for everyone.|CR'))
    writeln(parse_MCI('|NC(|BCMore to Come...|NC)|DC---------- --- --  -'))
    writeln(parse_MCI('|RCIf you enjoy m-Chat, be sure to connect to the following boards to get more'))
    writeln(parse_MCI('good mods. Thanks for using m-Chat!|CR'))
    writeln(parse_MCI('|NC(|BCOfficial m-Chat Support BBSs|NC)|DC---------- --- --  -'))
    writeln(parse_MCI('|BCsector7 BBS|DC.....: |NCsector7bbs.com |DC- |RCsysop|DC: |RCgrymmjack'))
    writeln(parse_MCI('|BCthe dominion BBS|DC: |NCthedominion.darktech.org |DC - |RCsysop|DC: |RCliquid'))
    writeln('')
    writeln(CFGDATA(STR_DIVIDER))
    writeln(parse_MCI(CFGDATA(STR_PAUSE) + '|PN'))
    dispfile(cfgtextpath + PROGRAM_SPLASH)
  endif

pend



# .----------------------------------------------------------------------.
# | Procedure: nodeMsg                                                   |
# +----------------------------------------------------------------------+
# | This procedure sends a message to another node.                      |
# `----------------------------------------------------------------------'
#
procedure nodeMsg(args string)
begin
  var node string
  var message string

  # If no argument is passed then show an error and exit
  if args = '' then
    writeln('|CR' + CFGDATA(STR_ERROR) + 'proper syntax: /msg # <text> (ie: /msg 1 hey liquid!)')
    exit
  endif

  # Extract the node number from the argument
  node := copy(args, 1, pos(' ', args) - 1)
  # Extract the message from the argument
  message := copy(args, pos(node, args) + 2, length(args))

  # Setup the menu command parameter using the gathered data
  tmpstr1 := node + ';' + message

  # Run the menu commnad - Send the message
  menucmd('NS', tmpstr1)
  
  # Log the fact that we did this if activity logging is on
  if LOG_ACTIVITY = true then
    write_log(CFGDATA(CMD_PREFIX) + ' --- MULTINODE MESSAGE TO NODE: ' + node + ' --- ' + CFGDATA(CMD_PREFIX))
    write_log(CFGDATA(CMD_PREFIX) + ' --- MESSAGE: ' + message + ' ' + CFGDATA(CMD_PREFIX))
  endif

  # Show the user we sent the message
  writeln(parse_MCI('|CR' + CFGDATA(STR_NORMAL) + 'multinode message sent to node |BC' + node + '|NC.|DC.'))
pend



# .----------------------------------------------------------------------.
# | Procedure: typeFile                                                  |
# +----------------------------------------------------------------------+
# | This procedure actually displays the file. It is called by showFile. |
# `----------------------------------------------------------------------'
#
procedure typeFile(argz string)
begin
  # Write a linefeed to make sure we are starting on a new line to prevent
  # offset errors when typing ansis and such 
  write('|CR')
  # Display the file
  dispfile(argz)
  # make a note that we /TYPED the file if activity logging is on
  if LOG_ACTIVITY = true then
    write_log(CFGDATA(CMD_PREFIX) + ' TYPED ' + argz +  ' ' + CFGDATA(CMD_PREFIX))
  endif
pend



# .----------------------------------------------------------------------.
# | Procedure: showFile                                                  |
# +----------------------------------------------------------------------+
# | This procedure handles the arguments passed by the /TYPE command so  |
# | that the actual procedure that displays the file doesn't have to get |
# | involved with it and therefore remains simple.                       |
# `----------------------------------------------------------------------'
#
procedure showFile(args string)
begin

  var mysticdat string
  var usersdat string
  var ext string

  # Set the forbiddin files up
  mysticdat := cfgsyspath + 'MYSTIC.DAT'
  usersdat := cfgdatapath + 'USERS.DAT'
  
  # If an argument is not given, prompt for it
  if args = '' then
    writeln('|CR' + CFGDATA(STR_ERROR) + 'usage: /TYPE <file (ie: C:\MYSTIC\TEXT\WELCOME.ANS)>')
    exit
  endif

  # Check the FFL for the file if it is in the list exit dont allow it to be viewed
  if isInFFL(args) = true or isInFFL(upper(args)) = true or isInFFL(lower(args)) then
    writeln('|CR' + CFGDATA(STR_ERROR) + 'you may NOT view ' + args + '. it is forbidden!')
    write_log(CFGDATA(CMD_PREFIX) + ' !!! ATTEMPTED TO TYPE FORBIDDEN FILE: ' + args + ' !!! ' + CFGDATA(CMD_PREFIX))
    exit
  endif

  # Get the file extension if there is one 
  if pos('.', args) > 0 then
    ext := upper(copy(args, length(args) - 3, 4))
  # Check to see if an extension was ommited
  elseif fexist(cfgtextpath + args + '.ANS') then
    typeFile(cfgtextpath + args + '.ANS')
    exit
  elseif fexist(cfgtextpath + args + '.ASC') then
    typeFile(cfgtextpath + args + '.ASC')
    exit
  endif
 
  # Prevent MYSTIC.DAT, USER.DAT, *.EXE and *.OVR from being displayed at all!
  if upper(args) = upper(mysticdat) or upper(args) = upper(usersdat) or upper(ext) = '.EXE' or upper(ext) = '.COM' or upper(ext) = '.OVR' then
    writeln('|CR' + CFGDATA(STR_ERROR) + 'you may NOT view ' + args + '. it is a security risk.')
    exit
  endif

  # If a default show path exists in the config, try to find the file there
  if CFGDATA(DEFAULT_SHOW_PATH) <> '' then
    if fexist(CFGDATA(DEFAULT_SHOW_PATH) + args) = true then
      typeFile(CFGDATA(DEFAULT_SHOW_PATH) + args)
    endif
  endif
  
  # Check to see if the file exists, if it does, display it
  if fexist(args) then
    typeFile(args)
    exit
  # If all else fails, check the default text directory for the file
  elseif fexist(cfgtextpath + args) then
    typeFile(cfgtextpath + args)
    exit
  else    
    # If we can't find the file after all that stuff, show an error
    writeln('|CR' + CFGDATA(STR_ERROR) + 'file not found: ' + args)
  endif

pend






# .----------------------------------------------------------------------.
# |                                                                      |
# | *** Main Program                                                     |
# |                                                                      |
# `----------------------------------------------------------------------'
#
begin
  # .--------------------------------------------------------------------.
  # | Setup the variables                                                |
  # `--------------------------------------------------------------------'
  var done          boolean
  var drawText      boolean
  var addTextToLine boolean
  var addToLog      boolean
  var foundLastWord boolean
  var wordCounter   integer
  var theWord       integer
  var word1         string
  var word2         string
  var intext        char
  var inline        string
  var oldLine       string
  var tmpline       string

  # .--------------------------------------------------------------------.
  # | Get some information we need about the BBS from mystic             |
  # `--------------------------------------------------------------------'
  getthisuser
  getcfg

  # .--------------------------------------------------------------------.
  # | Setup the Magic-Word file & FFL file                               |
  # `--------------------------------------------------------------------'
  tmpstr1 := cfgdatapath + PROGRAM_WORD
  PROGRAM_WORD := tmpstr1
  tmpstr1 := cfgdatapath + PROGRAM_FORBIDDEN
  PROGRAM_FORBIDDEN := tmpstr1
  
  # .--------------------------------------------------------------------.
  # | Populate the CFGDATA() array with values loaded from the MCHAT.CFG.|
  # | file if it exists. Otherwise make it from scratch                  |
  # `--------------------------------------------------------------------'
  read_config

  # .--------------------------------------------------------------------.
  # | Populate the CFGMW() array with values loaded from the MCHAT.WRD   |
  # | file if it exists.                                                 |
  # `--------------------------------------------------------------------'
  populateMWL

  # .--------------------------------------------------------------------.
  # | Populate the CFGFFL() array with values loaded from the MCHAT.FFL  |
  # | file if it exists.                                                 |
  # `--------------------------------------------------------------------'
  populateFFL
  
  # .--------------------------------------------------------------------.
  # | Reset counter variables for Session Information purposes           |
  # `--------------------------------------------------------------------'
  TOTAL_LOCAL_CHARS := 0
  TOTAL_REMOTE_CHARS := 0
  TOTAL_LOCAL_LINES := 0
  TOTAL_REMOTE_LINES := 0
  
  # .--------------------------------------------------------------------.
  # | Turn off Logging, Magic-Words, and User Input Filter by default    |
  # `--------------------------------------------------------------------'
  NOW_LOGGING := false
  MAGIC_WORDS := false
  USER_INPUT := true
  LOG_FILE := ''
 
  # .--------------------------------------------------------------------.
  # | Setup the Session Duration timer                                   |
  # `--------------------------------------------------------------------'
  START_TIME := datetime
  TIMER_START_TIME := timer
  
  # .--------------------------------------------------------------------.
  # | Set the state of logging activity messaages                        |
  # `--------------------------------------------------------------------'
  if CFGDATA(SHOW_LOG_MSGS) = '1' then
    LOG_ACTIVITY := true
  else
    LOG_ACTIVITY := false
  endif

  # .--------------------------------------------------------------------.
  # | If Always Log is enabled then create/append to MCHAT.LOG file      |
  # `--------------------------------------------------------------------'
  if CFGDATA(LOG_ALWAYS) = '1' then
    log(PROGRAM_LOG)
  endif   
  
  # .--------------------------------------------------------------------.
  # | Initialize scrren and environment.                                 |
  # `--------------------------------------------------------------------'
  dispfile(cfgtextpath + PROGRAM_SPLASH)
  menucmd('NA',CFGDATA(MULTINODE_ACTIVITY))
  write(CFGDATA(STR_START))

  # .--------------------------------------------------------------------.
  # | Set the quit flag to false so we dont quit after first loop ;0     |
  # `--------------------------------------------------------------------'
  done := false

  repeat
    # .------------------------------------------------------------------.
    # | Initialize some IN-LOOP variables                                |
    # `------------------------------------------------------------------'
    drawText := true
    addTextToLine := false
    addToLog := true
    theWord := 0
    wordCounter := 0
    foundLastWord := false

    # .------------------------------------------------------------------.
    # | Get a key from either local or remote                            |
    # `------------------------------------------------------------------'
    intext := readkey

    # .------------------------------------------------------------------.
    # | Check to see if the line contains any spaces. If it doesn't then |
    # | we need to log the line and then start a new one, otherwise the  |
    # | Word-Wrap stuff will break and the log file will look ugly!      |
    # `------------------------------------------------------------------'
    if wherex = 80 then
      write_log(inline)
      inline := ''
    endif
    
    # .------------------------------------------------------------------.
    # | Word-Wrap code triggers when the cursor hits column 79. Running  |
    # | the Word-Wrap code at column 80 caused problems because when the |
    # | cursor reaches column 80 it automatically creates a linefeed. So |
    # | we will run our Word-Wrap code on column 79 instead.             |
    # `------------------------------------------------------------------'
    if wherex = 79 then

      # Set the wordCounter to be the length of the current line + 1 so
      # that we can decrement the length without errors once we begin the loop 
      wordCounter := length(inline) + 1

      repeat

        # Decrement the wordCounter 
        wordCounter := wordCounter - 1

        # Use loop variable to check for existence of space at wordCounter
        loop := pos(' ', copy(inline, wordCounter, 1))

        # If a space is found, we have found the beginning of the last word
        # in the line because we started from the end of the line and worked
        # towards the beginning
        if loop = 1 then 
          # So we exit the loop because we found the last word
          foundLastWord := true
        endif

      # We continue searching for the first space until we find it or
      # wordCounter becomes 0 which means we are at the beginning of the
      # string we are checking for spaces.
      until foundLastWord = true or wordCounter = 0

      # If we found a word and the line contained a space then...
      if wordCounter > 0 and pos(' ', inline) > 0 then

        # Set some temporary strings up. tmpstr1 = the text UP TO the last
        # word in the line, and tmpstr2 = the last word in the line.
        tmpstr1 := copy(inline, 1, wordCounter)
        tmpstr2 := copy(inline, wordCounter, length(inline))

        # If the Sysop is the one typing when the Word-Wrap triggers, then
        # we need to draw the text and the Word-Wrap for him and then we also
        # need to increment the total line count for the SysOp.
        if islocalkey then
          write(CFGDATA(STR_LOCAL_TEXT) + '|[X' + int2str(wordCounter) + strrep(' ', length(tmpstr2)) + '|CR' + copy(tmpstr2, 2, length(tmpstr2)))
          TOTAL_LOCAL_LINES := TOTAL_LOCAL_LINES + 1

        # Otherwise we need to do it for the remote user instead
        else
          write(CFGDATA(STR_REMOTE_TEXT) + '|[X' + int2str(wordCounter) + strrep(' ', length(tmpstr2)) + '|CR' + copy(tmpstr2, 2, length(tmpstr2)))
          TOTAL_REMOTE_LINES := TOTAL_REMOTE_LINES + 1
        endif

        # Regardless of who is typing we need to reset the new line we just
        # drew out to be the same as our input like so:
        inline := copy(inline, 1, wordCounter)

        # And due to the fact that my current logic is somewhat screwed up
        # sometimes the wrapped text can begin with a space which isn't really
        # kosher for logging purposes so this is a quick-fix (all be it 
        # pretty lame) to remedy that:
        if copy(inline, 1, 1) = ' ' then
          write_log(copy(inline, 2, length(inline)))
        else
          write_log(inline)
        endif

        # And finally we set our input line to equal the wrapped text.
        inline := tmpstr2
      endif

    endif

    # .------------------------------------------------------------------.
    # | Backspace handler code. When someone hits backspace we have to   |
    # | draw it and then fix our current line of input to remove the last|
    # | letter in it already so that it actually gets erased.            |
    # `------------------------------------------------------------------'
    if intext = chr(8) then
      # Check to make sure this is a legal keypress vs. User Input Filter
      if islocalkey or USER_INPUT = true then
        write('|[D01 ')
        inline := copy(inline, 1, length(inline) - 1)
      endif
    endif

    # .------------------------------------------------------------------.
    # | Movement/Arrow key code. There are a few keys that need to be    |
    # | trapped specially so that they do not get drawn and make a mess  |
    # | of things. Also this is where we have our ^W, ^X, and ^Y code,   |
    # | as well as our Command Recall code for the up arrow key.         |
    # `------------------------------------------------------------------'
    if isarrow = true then

      # HOME   y
      if intext = chr(71) then
        write('|[X01|$D79 |[X01')
        # For the sake of simplicity home is destructive
        inline := ''
      # UP ARROW - Command Recall
      elseif intext = chr(72) then
        if islocalkey or CFGDATA(ALLOW_USER_CMDS) = '1' then
          write(CFGDATA(STR_LOCAL_TEXT) + '|[X01|$D79 |[X01' + inline)
          inline := oldLine
        else
          write(CFGDATA(STR_REMOTE_TEXT) + '|[X01|$D79 |[X01' + inline)
          inline := oldLine
        endif
        drawText := false
      # LEFT ARROW
      elseif intext = chr(75) then
        write('|[D01 ')
        inline := copy(inline, 1, length(inline) - 1)
      # DELETE
      elseif intext = chr(83) then
        write('|[D01 ')
        inline := copy(inline, 1, length(inline) - 1)
      # RIGHT ARROW
      elseif intext = chr(77) then
        write('|[C01')
        intext := ' '
        drawText := true
      # END
      elseif intext = chr(79) then
        drawText := false
      # DOWN ARROW
      elseif intext = chr(80) then
        drawText := false
      endif

      # .------------------------------------------------------------------.
      # | Log activity toggle (CTRL-L)                                     |
      # `------------------------------------------------------------------'
      elseif intext = chr(12) then
        # Check to make sure this is a legal keypress vs. User Input Filter
        if islocalkey or CFGDATA(ALLOW_USER_CMDS) = '1' then
          if LOG_ACTIVITY = true then
            LOG_ACTIVITY := false
            tmpstr1 := 'disabled'
          else
            LOG_ACTIVITY := true
            tmpstr1 := 'enabled'
          endif
          writeln(parse_MCI('|CR' + CFGDATA(STR_NORMAL) + 'logging activity: ' + tmpstr1 + '|BC.|NC.|DC.'))
        endif

      # .------------------------------------------------------------------.
      # | User commands toggle (CTRL-U)                                    |
      # `------------------------------------------------------------------'
      elseif intext = chr(21) then
        # Check to make sure this is a legal keypress vs. User Input Filter
        if islocalkey or CFGDATA(ALLOW_USER_CMDS) = '1' then
          if CFGDATA(ALLOW_USER_CMDS) = '1' then
            CFGDATA(ALLOW_USER_CMDS) := '0' 
            tmpstr1 := 'disabled'
          else
            CFGDATA(ALLOW_USER_CMDS) := '1'
            tmpstr1 := 'enabled'
          endif
          writeln(parse_MCI('|CR' + CFGDATA(STR_NORMAL) + 'user commands: ' + tmpstr1 + '|BC.|NC.|DC.'))
        endif

      # .------------------------------------------------------------------.
      # | Magic-Words toggle (CTRL-W)                                      |
      # `------------------------------------------------------------------'
      elseif intext = chr(23) then
        # Check to make sure this is a legal keypress vs. User Input Filter
        if islocalkey or CFGDATA(ALLOW_USER_CMDS) = '1' then 
          if MAGIC_WORDS = true then
            MAGIC_WORDS := false
            tmpstr1 := 'disabled'
          else
            MAGIC_WORDS := true
            tmpstr1 := 'enabled'
          endif
          writeln(parse_MCI('|CR' + CFGDATA(STR_NORMAL) + 'magic-words: ' + tmpstr1 + '|BC.|NC.|DC.'))
      endif
    
      # .------------------------------------------------------------------.
      # | Toggle user input filter (CTRL-X)                                |
      # `------------------------------------------------------------------'
      elseif intext = chr(24) then
        # Check to make sure this is a legal keypress vs. User Input Filter
        if islocalkey or CFGDATA(ALLOW_USER_CMDS) = '1' then
          if USER_INPUT = true then
            USER_INPUT := false
            tmpstr1 := 'disabled'
          else
            USER_INPUT := true
            tmpstr1 := 'enabled'
          endif
          writeln(parse_MCI('|CR' + CFGDATA(STR_NORMAL) + 'user input: ' + tmpstr1 + '|BC.|NC.|DC.'))
      endif

      # .------------------------------------------------------------------.
      # | Yank current ;ine (CTRL-Y)                                       |
      # `------------------------------------------------------------------'
      elseif intext = chr(25) then
        write('|[X01|$D79 |[X01')
        inline := ''
      endif

      # .------------------------------------------------------------------.
      # | Quit (CTRL-[) or ESCAPE                                          |
      # `------------------------------------------------------------------'
      elseif intext = chr(27) then
        if islocalkey or CFGDATA(ALLOW_USER_CMDS) = '1' then
          drawText := false
          done := true
        endif

    endif

    # .------------------------------------------------------------------.
    # | IRC-like Acion code. If an action is entered we need to draw it  |
    # | based on who presses ENTER. This creates a small inconsistency   |
    # | because it can then be a combined action, but I am satisfied     |
    # | with this anyway.                                                |
    # `------------------------------------------------------------------'
    if upper(copy(inline, 1, 3)) = '/ME' and intext = chr(13) then

      # Check to see if the line contains more than just '/ME'
      if length(inline) > 3 then

        # If the sysop is doing the action then lets draw it for him and
        # increment his total line count
        if islocalkey then
          write('|[X01' + CFGDATA(STR_ACTION) + CFGDATA(SYSOP_NAME) + copy(inline, 4, length(inline)))
          write_log(stripmci(CFGDATA(STR_ACTION)) + CFGDATA(SYSOP_NAME) + copy(inline, 4, length(inline)))
          TOTAL_LOCAL_LINES := TOTAL_LOCAL_LINES + 1

        # Otherwise, we need to do the same for the user
        else
          write('|[X01' + CFGDATA(STR_ACTION) + useralias + copy(inline, 4, length(inline)))
          write_log(stripmci(CFGDATA(STR_ACTION)) + useralias + copy(inline, 4, length(inline)))
          TOTAL_REMOTE_LINES := TOTAL_REMOTE_LINES + 1
        endif

        # Regardless of who typed it we need to reset our line and
        # some other variables so that the linefeed actually gets drawn 
        # but not logged later on in the catch-all linefeed part of the code.
        inline := ''
        addToLog := false
        drawText := true

      # If it was just a '/ME' with nothing else, just increment the
      # counters accordingly and log it. 
      else
        if islocalkey then
          TOTAL_LOCAL_LINES := TOTAL_LOCAL_LINES + 1
        else
          TOTAL_REMOTE_LINES := TOTAL_REMOTE_LINES + 1
        endif
        addToLog := true
        drawText := true
      endif

    endif
    
    # .------------------------------------------------------------------.
    # | /COMMAND handler code. This code parses and relays arguments of  |
    # | any commands issued to their corresponding functions/procedures. |
    # `------------------------------------------------------------------'
    if copy(inline, 1, 1) = '/' and intext = chr(13) then

      # Check to make sure we are following the security configuration
      if CFGDATA(ALLOW_USER_CMDS) = '1' or islocalkey then

        # Modify the total commands run count 
        TOTAL_COMMANDS_RUN := TOTAL_COMMANDS_RUN + 1

        # /ABOUT command
        if upper(copy(inline, 2, 5)) = 'ABOUT' or upper(copy(inline, 2, 1)) = 'A' then
          showAbout

        # /CLS command          
        elseif upper(copy(inline, 2, 3)) = 'CLS' or upper(copy(inline, 2, 1)) = 'C' then
          clearScreen

        # /GOODBYE command
        elseif upper(copy(inline, 2, 7)) = 'GOODBYE' or upper(copy(inline, 2, 1)) = 'G' then
          menucmd('GI','')

        # /HELP command
        elseif upper(copy(inline, 2, 4)) = 'HELP' or upper(copy(inline, 2, 1)) = 'H' or upper(copy(inline, 2, 1)) = '?' then
          showHelp

        # /INFO command       
        elseif upper(copy(inline, 2, 4)) = 'INFO' or upper(copy(inline, 2, 1)) = 'I' then
          showInfo

        # /LOG command
        elseif upper(copy(inline, 2, 3)) = 'LOG' or upper(copy(inline, 2, 1)) = 'L' then
          # If line contains an argument run the procedure with it
          if pos(' ', inline) > 0 then
            log(copy(inline, pos(' ', inline) + 1, length(inline)))
          else
            # Otherwise run with no args and cause the error message to show up
            log('')
          endif

        # /MSG command              
        elseif upper(copy(inline, 2, 3)) = 'MSG' or upper(copy(inline, 2, 1)) = 'M' then
          # If line contains an argument run the procedure with it
          if pos(' ', inline) > 0 then
            nodeMsg(copy(inline, pos(' ', inline) + 1, length(inline)))
          else
            # Otherwise run with no args and cause the syntax message to show up
            nodeMsg('')
          endif

        # /QUIT command
        elseif upper(copy(inline, 2, 4)) = 'QUIT' or upper(copy(inline, 2, 1)) = 'Q' then
          done := true

        # /TYPE command
        elseif upper(copy(inline, 2, 4)) = 'TYPE' or upper(copy(inline, 2, 1)) = 'T' then
          # If line contains an argument run the procedure with it
          if pos(' ', inline) > 0 then
            showFile(copy(inline, pos(' ', inline) + 1, length(inline)))
          else
            # Otherwise call procedure with no argument resulting in an error
            showFile('')
          endif

        # /WHO command           
        elseif upper(copy(inline, 2, 3)) = 'WHO' or upper(copy(inline, 2, 1)) = 'W' then
          clearScreen
          menucmd('NW','')

        # Not a valid command but a command was detected
        else
          addToLog := true
          drawText := false
          write('|CR')

          # If the first character in the line is a space, skip it when we
          # add this line to the log 
          if addToLog = true then
            if copy(inline, 1, 1) = ' ' then
              write_log(copy(inline, 2, length(inline)))
            # Otherwise just write the line to the log 
            else
              write_log(inline)
            endif

          endif

        endif

        # Regardless of what happened in the command handler
        # we need to set some things up because enter was indeed
        # pressed
        oldLine := inline
        inline := ''
        drawText := false
        addToLog := false

      endif

    endif
   
    # .------------------------------------------------------------------.
    # | ENTER key handler code. This code handles logging and reset of   |
    # | the input lines if certain conditions are met.                   |
    # `------------------------------------------------------------------'
    if intext = chr(13) then
    
      if islocalkey or USER_INPUT = true then

        # Log the line of input
        if addToLog = true then
          if copy(inline, 1, 1) = ' ' then
            write_log(copy(inline, 2, length(inline)))
          else
            write_log(inline)
          endif
        endif

        # Draw the linefeed if necessary
        if drawText = true then
          write('|CR')
        endif

        # If the sysop was the one that pressed ENTER update his counter
        if islocalkey then
          TOTAL_LOCAL_LINES := TOTAL_LOCAL_LINES + 1
        else
          # Otherwise update the users counter
          TOTAL_REMOTE_LINES := TOTAL_REMOTE_LINES + 1
        endif
  
        inline := ''
      else
        TOTAL_REMOTE_LINES := TOTAL_REMOTE_LINES
      endif
  
    # .------------------------------------------------------------------.
    # | All other key handler code.                                      |
    # `------------------------------------------------------------------'
    else
      addTextToLine := true
      drawText := true

      # .------------------------------------------------------------------.
      # | SPACE key handler (Magic-Words) code.                            |
      # `------------------------------------------------------------------'
      if intext = chr(32) then
        if MAGIC_WORDS = true then
          inline := inline + intext
          addTextToLine := false
          drawText := false
          wordCounter := length(inline)

          repeat
            wordCounter := wordCounter - 1
            if pos(' ', copy(inline, wordCounter, 1)) = 1 then
              foundLastWord := true
            endif
          until foundLastWord = true or wordCounter = 0

          theWord := wordCounter
          if theWord > 0 then
            word1 := copy(inline, 1, theWord)
          else
            word1 := ''
          endif

          if wordCounter = length(inline) - 1 then
            word1 := ''
            word2 := inline
          else
            word2 := copy(inline, theWord +1 , length(inline) - 1)
          endif
          
          if copy(word2, length(word2), 1) = ' ' then
            word2 := copy(word2, 1, length(word2) - 1)
          endif

          if copy(word1, 1, 1) = ' ' then
            word1 := copy(word1, 2, length(word1))
          endif

          tmpstr3 := getMagicWord(word2)
          
          if tmpstr3 <> '' then
            inline := word1 + tmpstr3 + ' '
              if length(inline) > 9 then
                write(parse_MCI('|[X01|$D' + int2str(length(inline)) + ' |[X01' + inline))
              else
                write(parse_MCI('|[X01|$D0' + int2str(length(inline)) + ' |[X01' + inline))
              endif
          else
              write(' ')
          endif
        endif
      endif

      # Do NOT draw or add anything to the line or log if it was ESC or any of our CTRL-Sequences
      if intext = chr(27) or intext = chr(12) or intext=chr(21) or intext=chr(23) or intext=chr(24) or intext=chr(25) then
        drawText := false
        addTextToLine := false
        addToLog := false
      endif
      
      # Do NOT add text to the line if it was a backspace
      if intext = chr(8) then
        addTextToLine := false
      endif
      
      # Do not draw anything if it was an invisible special char (arrows).
      if isarrow then 
        drawText := false
        addTextToLine := false
      endif
      
      # If it's not local or the User Input Filter is on then don't draw it
      if USER_INPUT = false and not islocalkey then
        addTextToLine := false
        drawText := false
        addToLog := false
      endif
      
      # If it's a carriage return don't draw it
      if intext = chr(13) then
        drawText := false
      endif
      
      # If it's SysOp pressing the key..
      if islocalkey then
        # and the drawText flag is set then draw the text...
        if drawText = true then
          write(CFGDATA(STR_LOCAL_TEXT) + intext)
        endif
        # and increment character count for sysop
        TOTAL_LOCAL_CHARS := TOTAL_LOCAL_CHARS + 1
      else
        # Otherwise do the same stuff for the user if all
        # conditions are met
        if drawText = true and USER_INPUT = true then
          write(CFGDATA(STR_REMOTE_TEXT) + intext)
        endif
        TOTAL_REMOTE_CHARS := TOTAL_REMOTE_CHARS + 1
      endif
      
      # Append the character to the line variable if the flag is set
      if addTextToLine = true then
        inline := inline + intext
      endif
    endif

  # Repeat this sequence until we type /QUIT
  until done = true

  # Show the end of chat string
  write(CFGDATA(STR_END)) 

  # And close the log file
  if NOW_LOGGING = true then
    close_log
  endif

  # Finally hit return to go back to the BBS after calling with the 
  # new Mystic v1.7.3 !bangs in sysop macro F1-F4 keys
  stuffkey(chr(13))

end

