/*͸
   OS/2 Video API Version 1.00  13-01-96    (C) 1996 by CodeLand Australia 
   FORM.C Entry form routines                          All Rights Reserved 
  ;*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#define INCL_DOS
#include <os2.h>

#include "vid.h"
#include "win.h"
#include "menu.h"
#include "form.h"

/**/

/* Function prototypes */
static void pascal adv_column (struct _form_t *cform,int col);
static void pascal after_edit (struct _form_t *cform);
static void pascal back_space (struct _form_t *cform);
static void pascal beg_pos (struct _form_t *cform);
static void pascal call_func (void (*func)(void));
static void pascal check_format (char *format,int *lenfld,int *lenbuf, int *decpos);
static void pascal check_redisp (struct _form_t *cform);
static int pascal closest_column (struct _form_t *cform, struct _field_t *field);
static int pascal cond_update (struct _form_t *cform);
static void pascal deci_left (struct _form_t *cform,char *start);
static void pascal deci_right (struct _form_t *cform,char *start);
static void pascal del_char (struct _form_t *cform);
static void pascal del_field (struct _form_t *cform,char *start,int high);
static void pascal del_left_word (struct _form_t *cform);
static void pascal del_rest (struct _form_t *cform);
static void pascal del_word (struct _form_t *cform);
static void pascal disp_char (struct _form_t *cform,int ch,int advance);
static void pascal disp_field (struct _form_t *cform,char *start,int high);
static void pascal down_field (struct _form_t *cform);
static void pascal end_line (struct _form_t *cform);
static void pascal end_pos (struct _form_t *cform);
static void pascal free_fields (struct _form_t *cform);
static int pascal goto_field (struct _form_t *cform,int which);
static void pascal insert_char (struct _form_t *cform);
static void pascal next_fchar (struct _form_t *cform);
static void pascal next_pos (struct _form_t *cform);
static void pascal next_word (struct _form_t *cform);
static void pascal prev_fchar (struct _form_t *cform);
static void pascal prev_pos (struct _form_t *cform);
static void pascal prev_word (struct _form_t *cform);
static void pascal restore_field (struct _form_t *cform,int high);
static void pascal set_term_key (struct _form_t *cform,unsigned xch);
static void pascal up_field (struct _form_t *cform);
static int pascal validate_field (struct _form_t *cform);

static struct _field_t *first_rec (struct _form_t *cform);
static struct _field_t *last_rec (struct _form_t *cform);
static struct _field_t *prev_rec (struct _form_t *cform);
static struct _field_t *next_rec (struct _form_t *cform);

/**/

/* Field movement definitions */
#define FLD_FR  0
#define FLD_LS  1
#define FLD_PR  2
#define FLD_NR  3
#define FLD_UP  4
#define FLD_DN  5

/**/

/* Array of function pointers to some of the field movement functions */
static struct _field_t *(*funcs[6])(struct _form_t *) =
    { first_rec,last_rec,prev_rec,next_rec };

/**/

SHORT FormBeg (SHORT fieldattr, SHORT textattr)
{
    struct _form_t *form;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Allocate space for new input form record */
    if((form=malloc(sizeof(struct _form_t)))==NULL)
        return (_WinInfo.errno=W_ALLOCERR);

    /* Add new form record to linked list */
    if(_WinInfo.active->form!=NULL) _WinInfo.active->form->next=form;
    form->prev=_WinInfo.active->form; form->next=NULL;
    _WinInfo.active->form=form;

    /* Adjust attributes if monochrome mapping is on */
    textattr=VidMapAttr(textattr);
    fieldattr=_VidInfo.MapAttr?VidRevsAttr(textattr):fieldattr;

    /* Save info in new form record */
    form->field     = NULL;
    form->getkey    = NULL;
    form->termkey   = NULL;
    form->decimal   =
    form->insert    = 0;
    form->textattr  = (UCHAR)textattr;
    form->fieldattr = (UCHAR)fieldattr;

    return (_WinInfo.errno=W_NOERROR); /* Return with no error */
}

/**/

SHORT FormDef (SHORT wrow, SHORT wcol, PCHAR str, PCHAR format, SHORT fconv,
    SHORT mode, SHORT (*validate) (PCHAR), SHORT help)
{
    SHORT lenbuf, lenfld, decpos;
    PCHAR buf;
    struct _field_t *field;

    /* Check and see if FormBeg() was called */
    if(_WinInfo.active->form==NULL) return (_WinInfo.errno=W_NOFRMBEG);

    /* Try to set cursor coordinates - check for active window */
    if(WinGotoXY(wrow,wcol)) return (_WinInfo.errno);

    /* Get length of input field and syntax check format string */
    check_format(format,&lenfld,&lenbuf,&decpos);
    if(!lenbuf) return (_WinInfo.errno=W_INVFORMT);

    /* If field is in update mode, adjust input    */
    /* string size to match "size" of input buffer */
    if(mode) FormStrSetSz(str,lenbuf);

    /* Allocate space for buffer */
    if((buf=malloc(lenbuf+1))==NULL) return (_WinInfo.errno=W_ALLOCERR);

    /* Allocate space for new input field record */
    if((field=malloc(sizeof(struct _field_t)))==NULL)
        return (_WinInfo.errno=W_ALLOCERR);

    /* Add new record to linked list */
    if(_WinInfo.active->form->field!=NULL)
        _WinInfo.active->form->field->next=field;
    field->prev=_WinInfo.active->form->field;
    field->next=NULL;
    _WinInfo.active->form->field=field;

    /* If not updating, then initialize receiving string to spaces */
    if(!mode) { memset(str,' ',lenbuf); *(str+lenbuf)='\0'; }

    /* Initialize temp buffer to string contents */
    strcpy(buf,str);

    /* Save input field record */
    field->wrow     = (UCHAR)wrow;
    field->wcol     = (UCHAR)wcol;
    field->str      = str;
    field->buf      = buf;
    field->format   = format;
    field->fconv    = (UCHAR)fconv;
    field->mode     = (UCHAR)mode;
    field->validate = validate;
    field->lenfld   = lenfld;
    field->lenfmt   = strlen(format);
    field->lenbuf   = lenbuf;
    field->decpos   = (UCHAR)decpos;
    field->help     = help;
    field->redisp   = 0;
    field->before   =
    field->after    = NULL;

    /* Set current field pointer to new field record */
    _WinInfo.active->form->cfield=field;

    /* If numeric field, format accordingly */
    if(fconv=='9') deci_right(_WinInfo.active->form,buf+lenbuf);

    /* Display input field */
    disp_field(_WinInfo.active->form,NULL,0);

    return(_WinInfo.errno=W_NOERROR); /* Return with no error */
}

/**/

SHORT FormRead (VOID)
{
    struct _form_t  *cform;
    struct _field_t *cfield,*temp;
    PCHAR p;
    SHORT valid, ch, edit, done;
    USHORT xch;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Abbreviate pointer to current form */
    cform=_WinInfo.active->form;

    /* Check and see if winpbeg() was called */
    if(cform==NULL) return (_WinInfo.errno=W_NOFRMBEG);

    /* Check for defined input fields */
    if(cform->field==NULL) return (_WinInfo.errno=W_NOINPDEF);

    /* Go to 1st position of first field */
    cform->cfield=first_rec(cform); beg_pos(cform);

    /* Set help category to 1st field's help category */
    _WinInfo.help=cform->cfield->help;

    /* Call 1st field's before function */
    call_func(cform->cfield->before);

    /* If conditional update, then start cursor end of line */
    edit=cond_update(cform);

    /* Initialize Insert mode to OFF */
    cform->insert=0; VidSmCursor();

    /* Initially display field */
    disp_field(cform,NULL,1);

    for(;;) {
        /* Set cursor coordinates */
        WinGotoXY(cform->cwrow,cform->cwcol);

        /* If alternate getkey function was provided, call it. */
        /* Otherwise call the standard CXL getxch() function   */
        /* If alternate getkey function returned "done" flag,  */
        /* then set termination key variable and exit function */
        if(cform->getkey!=NULL) {
            done=0;
            xch=(*(cform->getkey))(&done);
            if(done) {
                set_term_key(cform,xch);
                goto form_done;
            }
        }
        else xch=MenuGetXCh();

        /* Test keypress */
        switch(xch) {
            case 0x011b:    /* Esc */
                /* If Esc checking is on, restore all fields to their */
                /* original values, free input records, and exit      */
                if(_WinInfo.esc) {
                    after_edit(cform);
                    for(cform->cfield=cform->field;cform->cfield!=NULL;
                      cform->cfield=cform->cfield->prev)
                        restore_field(cform,0);
                    free_fields(cform);
                    set_term_key(cform,xch);
                    return (_WinInfo.errno=W_ESCPRESS);
                }
                break;

            case 0x1c0a:    /* Ctrl-Enter */
                set_term_key(cform,xch);
                goto form_done;

            case 0x1c0d:    /* Enter */
                /* Abbreviate pointer to current field */
                cfield=cform->cfield;

                /* If field is numeric, then format it so */
                if(cfield->fconv=='9')
                    deci_right(cform,((cform->pbuf==cfield->buf)) ?
                      (cfield->buf+cfield->lenbuf) : cform->pbuf);

                /* If we are at last field, validate it, */
                /* then copy temp buffers to receiving   */
                /* buffers, free input records, and exit */
                if(cform->cfield==last_rec(cform)) {
                    set_term_key(cform,xch);
form_done:
                    if(validate_field(cform)) break;
                    after_edit(cform);
                    cform->cfield=cform->field;
                    while(cform->cfield!=NULL) {
                        strcpy(cform->cfield->str,cform->cfield->buf);
                        cform->cfield=cform->cfield->prev;
                    }
                    free_fields(cform);
                    return (_WinInfo.errno=W_NOERROR);
                }
                /* Fall through to Tab */

            case 0x0f09:    /* Tab */
                /* If field is valid, display it, go      */
                /* to next field, first position. if      */
                /* conditional update feature is on, then */
                /* position cursor at end of line         */
                if(!goto_field(cform,FLD_NR)) {
                    beg_pos(cform);
                    if((edit=cond_update(cform))==0) continue;
                }
                break;

            case 0x0e08:    /* BackSpace */
                /* If currently at 1st position of 1st field, ignore. */
                /* otherwise, display a destructive backspace         */
                if(!( (cform->pbuf==cform->cfield->buf) &&
                  (cform->cfield==first_rec(cform)) )) back_space(cform);
                break;

            case 0x4700:    /* Home */
                /* Go to first position in field */
                beg_pos(cform);
                break;

            case 0x0f00:    /* Shift-Tab */
                /* If field is valid, display it, go to   */
                /* previous field, first position.  if    */
                /* conditional update feature is on, then */
                /* position cursor at end of line         */
                if(!goto_field(cform,FLD_PR)) {
                    beg_pos(cform);
                    if((edit=cond_update(cform))==0) continue;
                }
                break;

            case 0x4800:    /* UpArrow */
                /* If field is valid, display it, and */
                /* move cursor up to previous field   */
                goto_field(cform,FLD_UP);
                break;

            case 0x4b00:    /* LeftArrow */
                /* Go to previous position in field */
                prev_pos(cform);
                break;

            case 0x4d00:    /* RightArrow */
                /* Go to next position in field */
                next_pos(cform);
                break;

            case 0x4f00:    /* End */
                /* Go to last position in field */
                end_line(cform);
                break;

            case 0x5000:    /* DownArrow */
                /* If field is valid, display it, and */
                /* move cursor down to next field     */
                goto_field(cform,FLD_DN);
                break;

            case 0x5200:    /* Ins */
                /* Toggle Insert mode */
                if(cform->insert) {
                    cform->insert=0;
                    VidSmCursor();
                }
                else { cform->insert=1; VidLgCursor(); }
                break;

            case 0x5300:    /* Del */
                /* Delete character under cursor */
                del_char(cform);
                break;

            case 0x7500:    /* Ctrl-End */
                /* If field is valid, display it, then */
                /* go to last position of last field   */
                if(!goto_field(cform,FLD_LS)) end_pos(cform);
                break;

            case 0x7700:    /* Ctrl-Home */
                /* If field is valid, display it, then */
                /* go to first position of first field */
                if(!goto_field(cform,FLD_FR)) beg_pos(cform);
                break;

            case 0x0e7f:    /* Ctrl-BackSpace */
                /* Delete word to left of cursor */
                del_left_word(cform);
                break;

            case 0x1312:    /* Ctrl-R */
                /* Restore field to original contents */
                restore_field(cform,1);
                break;

            case 0x1414:    /* Ctrl-T */
                /* Delete word to right of cursor */
                del_word(cform);
                break;

            case 0x1615:    /* Ctrl-U */
                /* Delete from cursor to end of field */
                del_field(cform,cform->pbuf,1);
                break;

            case 0x1519:    /* Ctrl-Y */
                /* Delete from cursor to end of last field */
                del_rest(cform);
                break;

            case 0x7300:    /* Ctrl-LeftArrow */
                /* Move cursor to previous word to the left */
                prev_word(cform);
                break;

            case 0x7400:    /* Ctrl-RightArrow */
                /* Move cursor to next word to the right */
                next_word(cform);
                break;

            case 0x342e:    /* Period */
            case 0x532e:    /* Shift-Del (decimal) */
                /* If field is numeric, right justify, space fill */
                /* digits to the left of the decimal point, and   */
                /* move cursor to position after decimal point    */
                if(cform->cfield->fconv=='9') {
                    deci_left(cform,cform->pbuf);
                    disp_field(cform,NULL,1);
                    temp=cform->cfield;
                    while(!cform->decimal&&cform->cfield==temp)
                        next_pos(cform);
                    break;
                }
                /* Fall through to default */

            default:
                /* If conditional update feature is on, check to see if */
                /* an editing key has been pressed.  if it has not,     */
                /* then erase field and position cursor at beginning    */
                if(cform->cfield->mode==2) {
                    if(!edit) {
                        beg_pos(cform);
                        del_field(cform,cform->pbuf,1);
                    }
                }

                /* Break keycode down to just the ASCII code portion. if */
                /* ASCII code is zero, then it must be an extended key   */
                ch=(char)xch;
                /* if(!isprint(ch)) break; */

                /* If lower, mixed, or upper case conversion */
                /* was specified, then do the conversion     */
                if(*cform->pformat=='M' || cform->cfield->fconv=='M')
                    ch=FormToUpLow(cform->cfield->buf,cform->pbuf,ch);
                else if(*cform->pformat=='U' || cform->cfield->fconv=='U')
                    ch=toupper(ch);
                else if(*cform->pformat=='L' || cform->cfield->fconv=='L')
                    ch=tolower(ch);

                /* See if current format string character is the left angle  */
                /* bracket (start of set inclusion).  if so, search through  */
                /* the set of characters for the character matching          */
                /* keypress.  if a matching character wasn't found, then the */
                /* input character is invalid.                               */
                if(*cform->pformat=='<') {
                    valid=0;
                    p=cform->pformat;
                    cform->pformat++;
                    while(*cform->pformat!='>') {
                        if(ch==*cform->pformat) valid=1;
                        cform->pformat++;
                    }
                    if(!valid) cform->pformat=p;
                }

                /* See if current format string character is the left square */
                /* bracket (start of set exclusion).  if so, search through  */
                /* the set of characters for the character matching          */
                /* keypress.  if a matching character wasn't found, then the */
                /* input character is valid.                                 */
                else {
                    if(*cform->pformat=='[') {
                        valid=1;
                        p=cform->pformat;
                        cform->pformat++;
                        while(*cform->pformat!=']') {
                            if(ch==*cform->pformat) valid=0;
                            cform->pformat++;
                        }
                        if(!valid) cform->pformat=p;
                    }

                    /* See if input character is a valid character */
                    /* of the current format character type        */
                    else {
                        valid=FormCValType(ch,*cform->pformat);
                    }
                }

                /* If input character is a space, then even */
                /* if it's not valid, make it valid for now */
                if(ch==' ') valid=1;

                /* If input character is valid, display it */
                if(valid) {

                    /* If at last position of last field, don't wrap-around */
                    if((cform->pbuf==(cform->cfield->buf+cform->cfield->lenbuf
                      -1))&&(cform->cfield==last_rec(cform))) {
                        valid=0;
                        end_pos(cform);
                    }
                    temp=cform->cfield;
                    disp_char(cform,ch,valid);
                    if(!valid&&cform->cfield->fconv=='9') beg_pos(cform);
                    if(temp!=cform->cfield)
                        if((edit=cond_update(cform))==0) continue;

                    /* If field is numeric, and we just passed the decimal  */
                    /* point, right justify, space fill the left side of it */
                    if(valid) {
                        if(cform->cfield->fconv=='9'&&cform->decimal) {
                            deci_left(cform,cform->cfield->buf+
                                      cform->cfield->decpos-1);
                            disp_field(cform,NULL,1);
                        }
                    }
                }
        }

        edit=1; /* Turn on edit mode */
    }
}

/**/

/* Adds before and/or after function pointers to a field  */
SHORT FormFuncBA (void (*before)(void),void (*after)(void))
{
    struct _form_t *form;
    struct _field_t *field;

    /* Abbreviate pointer to current form and field */
    form=_WinInfo.active->form; field=form->field;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check and see if FormBeg() was called */
    if(form==NULL) return (_WinInfo.errno=W_NOFRMBEG);

    /* Check for defined input fields */
    if(field==NULL) return (_WinInfo.errno=W_NOINPDEF);

    /* Update field's record */
    field->before=before; field->after=after;

    return (_WinInfo.errno=W_NOERROR); /* Return normally */
}

/**/

/* Adjusts the size of a string  */
PCHAR FormStrSetSz (PCHAR str, SHORT newsize)
{
    SHORT i, len;

    len=strlen(str);
    if(newsize<len) *(str+newsize)='\0';
    else {
        for(i=len;i<newsize;i++) *(str+i)=' ';
        *(str+i)='\0';
    }

    return str;
}

/**/

/* Converts a char to upper or lower case depending on the prev char */
SHORT FormToUpLow (PCHAR str, PCHAR pos, SHORT ch)
{
    SHORT i;

    switch(*(pos-1)) {                  /* Check previous character */
        case ' ':                       /* See if it is a separator */
        case '-':
        case '_':
        case ',':
        case '.':
        case '/':
            i=toupper(ch);              /* If it is, convert to upper */
            break;
        default:
            if(pos==str)                /* See if at beginning of string */
                i=toupper(ch);          /* If so, convert to upper */
            else
                i=tolower(ch);          /* Otherwise, convert to lower */
    }
    return i;                           /* Return converted character */
}

/**/

/* Determines if a character is a valid for a given Form type */
SHORT FormCValType (SHORT ch, SHORT ctype)
{
    static CHAR bad_filechars[]="*?;,=+<>|/[]\"";
    SHORT valid=0;

    switch(ctype) {
        case '9': /* Is char type = float?     */
            if( ch=='.' || ch=='+' || ch=='-' ) {
                valid=1;
                break;
            }
        case '#': /* Is char type = numeric? */
            if( ch>='0' && ch<='9' ) valid=1;
            break;
        case '%': /* Is char=numeric or space? */
            if( (ch>='0' && ch<='9') || ch==' ' ) valid=1;
            break;
        case 'A':                               /* is char type = alpha?     */
        case 'L':                               /* is type = tolower alpha?  */
        case 'M':                               /* is type = tomixed alpha?  */
        case 'U':                               /* is type = toupper alpha?  */
            if( isalpha(ch) || ch==' ' ) valid=1;
            break;
        case 'D':
            if( isdigit(ch) || ch=='-' || ch=='/' ) valid=1;
            break;
        case 'F': /* Legal file name character */
            if(strchr(bad_filechars,ch)==NULL) valid=1;
            break;
        case 'W': /* File name character w/wildcard */
            if(strchr(bad_filechars+2,ch)==NULL) valid=1;
            break;
        case 'H': /* Is char type = hexadecimal? */
            if( isdigit(ch) || (ch>='A'&&ch<='F') || (ch>='a'&&ch<='f') )
                valid=1;
            break;
        case 'T': /* Is char type = phone num? */
            if( isdigit(ch) || ch=='(' || ch==')' || ch=='-' || ch==' ' )
                valid=1;
            break;
        case 'P': /* Is char type = password?  */
        case 'X': /* Is char type = alphanum?  */
            if( isalnum(ch) || ch==' ' ) valid=1;
            break;
        case 'Y': /* Is char type = Yes/No?    */
            if( ch=='Y' || ch=='N' || ch=='y' || ch=='n' ) valid=1;
            break;
        case '*': /* Is char type = printable? */
            if(isprint(ch)) valid=1;
            break;
        case '?': /* Is char type = anything?  */
            valid=1;
            break;
        default: /* Invalid char type         */
            return -1;
    }

    return valid;
}

/**/

/* Determines if a string is blank */
SHORT FormStrBlank (PCHAR str)
{
    PCHAR p;
    SHORT blank=1;

    for(p=str;*p;p++) {
        if(!isspace(*p)) {
            blank=0;
            break;
        }
    }

    return blank;
}

/**/

/* Shifts a string left */
PCHAR FormStrShL (PCHAR str, SHORT count)
{
    SHORT i, j;

    if(*str) {
        for(j=0;j<count;j++) {
            for(i=0;*(str+i);i++) *(str+i)=*(str+i+1);
            *(str+i-1)=' ';
        }
    }

    return str;
}

/**/

/* Shifts a string right */
PCHAR FormStrShR (PCHAR str, SHORT count)
{
    SHORT i, j, len;

    if(*str) {
        len=strlen(str)-1;
        for(j=0;j<count;j++) {
            for(i=len;i>0;i--) *(str+i)=*(str+i-1);
            *(str)=' ';
        }
    }

    return str;
}

/**/

/* Converts an integer value to CXL field format */
VOID FormCvtIC (PCHAR field, SHORT value, SHORT size)
{
    SHORT len;

    memset(field,' ',size);
    itoa(value,field,10);
    len=strlen(field);
    if(size>len) *(field+len)=' ';
    *(field+size)='\0';
    FormStrRJust(field);
}

/**/

/* Converts a long value to CXL field format */
VOID FormCvtLC (PCHAR field, ULONG value, SHORT size)
{
    SHORT len;

    memset(field,' ',size);
    ltoa(value,field,10);
    len=strlen(field);
    if(size>len) *(field+len)=' ';
    *(field+size)='\0';
    FormStrRJust(field);
}

/**/

/* Right justifies a string */
PCHAR FormStrRJust (PCHAR str)
{
    CHAR *p, *q;

    for(p=str;*p;p++);                  /* Find end of string */
    p--;
    for(q=p;isspace(*q)&&q>=str;q--);   /* Find last non-space character */
    if(p!=q) {
        while(q>=str) { *p--=*q; *q--=' '; }
    }

    return str;
}

/**/

/* Counts occurrences of 1 string inside another  */
SHORT FormStrISOcC (PCHAR str1, PCHAR str2)
{
    PCHAR p;
    SHORT count=0, len;

    len=strlen(str1);
    p=FormStrIInc(str1,str2);
    while(p!=NULL) { count++; p+=len; p=FormStrIInc(str1,p); }

    return count;
}

/**/

/* Trims trailing spaces off of a string */
PCHAR FormStrTrim (PCHAR str)
{
    SHORT i;

    for(i=strlen(str)-1;isspace(str[i])&&i>=0;i--) ;
    str[i+1]='\0';

    return str;
}

/**/

/* Trims leading spaces off of a string */
PCHAR FormStrLTrim (PCHAR str)
{
    CHAR *p,*q;

    p=q=str;
    while(isspace(*p)) p++;
    while(*p) *q++=*p++;
    *q='\0';

    return str;
}

/**/

/* Determines if string1 is included in string2  */
PCHAR FormStrIInc (PCHAR str1, PCHAR str2)
{
    SHORT max;
    PCHAR p;

    max=strlen(str1);

    for(p=str2;*p;p++) if(!strnicmp(str1,p,max)) return p;

    return NULL; /* String1 not found in string2 */
}

/**/

/* This function will start the cursor at the 1st position in */
/* the field and advance it until we are at specified column  */
static void pascal adv_column (struct _form_t *cform,int col)
{
    struct _field_t *temp;

    temp=cform->cfield;
    beg_pos(cform);
    while(cform->cwcol<(UCHAR)col) next_pos(cform);
    if(cform->cfield!=temp) while(cform->cfield!=temp) prev_pos(cform);
}

/**/

/* This function takes care of a couple tasks to do after editing a field */
static void pascal after_edit (struct _form_t *cform)
{
    /* redisplay current field in field attribute */
    disp_field(cform,NULL,0);

    /* if an 'after' function was defined, call it */
    call_func(cform->cfield->after);

    /* turn off Insert mode */
    cform->insert=0;
    VidSmCursor();
}

/**/

/* This function does a destructive backspace */
static void pascal back_space (struct _form_t *cform)
{
    /* decrement buffer pointer */
    cform->pbuf--;

    /* if buffer pointer is less than start of buffer, then validate */
    /* field, display field, and go to previous field, end of line   */
    /* otherwise, decrement column, format string */
    /* pointer, and back up one position          */
    if( cform->pbuf < cform->cfield->buf ) {
        if(!goto_field(cform,FLD_PR)) end_line(cform);
    }
    else {
        cform->cwcol--;
        cform->pformat--;
        prev_fchar(cform);
    }

    /* if insert mode is on, delete character under cursor */
    /* otherwise, replace character with a space           */
    if(cform->insert)
        del_char(cform);
    else {
        WinPrintC(cform->cwrow,cform->cwcol,cform->textattr,' ');
        *cform->pbuf=' ';
    }
}

/**/

/* This function sets current position to the beginning of the field */
static void pascal beg_pos (struct _form_t *cform)
{
    struct _field_t *cfield;

    cfield         = cform->cfield;
    cform->cwrow   = cfield->wrow;
    cform->cwcol   = cfield->wcol;
    cform->pbuf    = cfield->buf;
    cform->pformat = cfield->format;
    next_fchar(cform);
}

/**/

/* This function calls the given function, then checks all */
/* fields to see if any of their redisplay flags have been */
/* set.  If so, then the field will be redisplayed.        */
static void pascal call_func (void (*func)(void))
{
    if(func!=NULL) {
        (*func)();
        check_redisp(_WinInfo.active->form);
    }
}

/**/

/* This function checks syntax of format string, returns length of field */
static void pascal check_format (char *format,int *lenfld,int *lenbuf,int *decpos)
{
    char *p,ch;
    int valid=1,deccount=0;

    *lenbuf=0;
    *lenfld=0;
    *decpos=0;
    p=format;

    /* do while not end of string and string still valid */
    while(*p && valid) {    

        /* test current character */
        switch(*p) {

            case '\"':
            case '\'':

                /* search for matching quote */
                ch=*p++;
                while(*p!=ch) {

                    /* check for premature end of string */
                    if(!*p) {
                        valid=0;
                        break;
                    }
                    (*lenfld)++;
                    p++;
                }
                p++;
                break;

            case ' ':

                /* skip spaces, they are used for readability */
                p++;
                break;

            case '<':   /* left angle bracket (start of inclusion set) */

                /* search for matching angle bracket */
                p++;
                while(*p!='>') {

                    /* check for premature end of string */
                    if(!*p) {
                        valid=0;
                        break;
                    }
                    p++;
                }
                (*lenfld)++;
                (*lenbuf)++;
                p++;
                break;

            case '[':   /* left square bracket (start of exclusion set) */

                /* search for matching square bracket */
                p++;
                while(*p!=']') {

                    /* check for premature end of string */
                    if(!*p) {
                        valid=0;
                        break;
                    }
                    p++;
                }
                (*lenfld)++;
                (*lenbuf)++;
                p++;
                break;

            case '.':   /* decimal point */

                /* mark decimal position */
                *decpos=(*lenbuf)+1;
                (*lenfld)++;
                p++;

                /* make sure there is only 1 decimal point */
                deccount++;
                if(deccount>1) valid=0;
                break;

            default:

                /* see if character is a valid format character type */
                if( FormCValType(0,*p) == -1 ) {
                    valid=0;
                    break;
                }
                (*lenfld)++;
                (*lenbuf)++;
                p++;
        }
    }

    /* if no decimal was found, "position" it at end of field */
    if(!(*decpos)) *decpos=(*lenbuf)+1;
    if(!valid) {
        *lenbuf=0;
        *lenfld=0;
        *decpos=0;
    }
}

/**/

static void pascal check_redisp (struct _form_t *cform)
{
    struct _field_t *test,*cfield;

    /* search all fields in linked list for fields who have had */
    /* their redisplay flag set.  Redisplay all thos that have. */
    for(test=cform->field;test!=NULL;test=test->prev) {
        if(test->redisp) {
            cfield=cform->cfield;
            cform->cfield=test;
            disp_field(cform,NULL,test==cfield?1:0);
            cform->cfield=cfield;
            test->redisp=0;
        }
    }
}

/**/

/* This function returns the given field's window */
/* column closest to the current window column    */
static int pascal closest_column (struct _form_t *cform,struct _field_t *field)
{
    register int cwcol,wcol;
    int tcol,lcol;

    cwcol=cform->cwcol;
    wcol=field->wcol;
    lcol=wcol+field->lenfld-1;
    if( (cwcol>=wcol) && (cwcol<=lcol) ) tcol=cwcol;
    else
        if(wcol<cwcol)
            tcol=lcol;
        else
            tcol=wcol;
    return(tcol);
}

/**/

/* This function prepares for conditional update */
static int pascal cond_update (struct _form_t *cform)
{
    /* if conditional update, then start cursor end of line */
    if(cform->cfield->mode==2) {
        if(!FormStrBlank(cform->pbuf)) end_line(cform);
        return(0);
    }
    return(1);
}

/**/

/* This function will right justify, space fill numbers before the decimal */
static void pascal deci_left (struct _form_t *cform,char *start)
{
    char *p,*q,*left;
    struct _field_t *cfield;

    /* abbreviate pointer to current field */
    cfield=cform->cfield;

    left=cfield->buf+cfield->decpos-2;

    /* blank out from starting position to left side of decimal */
    for(p=start;p<=left;*p++=' ');

    /* search for first space to left of decimal */
    for(p=left;*p!=' '&&p>cfield->buf;p--);

    if(p>cfield->buf) {

        /* while q >= first position in field */
        q=p;
        while(q>=cfield->buf) {
            for(;(*q==' ')&&(q>=cfield->buf);q--);
            while((q>=cfield->buf)&&(*q!=' ')) {
                *p--=*q;
                *q--=' ';
            }
        }
    }

    /* change leading zeros to blanks */
    q=left;
    for(p=cfield->buf;*p==' '&&p<q;p++);
    if(p<q) while(*p=='0'&&p<q) *p++=' ';

    /* check for all blanks to left of decimal */
    if(*left==' ') *left='0';
}

/**/

/* This function will right justify, space fill numbers before the decimal
   point, and left justify, zero fill numbers after the decimal point */
static void pascal deci_right (struct _form_t *cform,char *start)
{
    char *p,*q,*last,*right;
    struct _field_t *cfield;

    /* abbreviate pointer to current field */
    cfield=cform->cfield;

    /* calculate last position and position to right of decimal */
    last=cfield->buf+cfield->lenbuf-1;
    right=cfield->buf+cfield->decpos-1;

    /* blank out from start position to end of field */
    for(p=start;p<=last;*p++=' ');

    /* first do left side of decimal */
    deci_left(cform,start);

    /* search for first space to right of decimal */
    for(p=right;*p!=' '&&p<last;p++);

    if(p<last) {

        /* while q <= last position in field */
        q=p;
        while(q<=last) {
            for(;*q==' '&&q<=last;q++);
            while(q<=last&&*q!=' ') {
                *p++=*q;
                *q++=' ';
            }
        }
    }

    /* change trailing blanks to zeros */
    for(p=last;*p==' '&&p>=right;p--) *p='0';
}

/**/

/* This function deletes a character and shifts following text left */
static void pascal del_char (struct _form_t *cform)
{
    /* shift characters in buffer left 1 position */
    FormStrShL(cform->pbuf,1);

    /* re-display field */
    disp_field(cform,cform->pbuf,1);
}

/**/

/* This function is used to delete characters to the end of the field */
static void pascal del_field (struct _form_t *cform,char *start,int high)
{
    /* blank out field from start position */
    strset(start,' ');

    /* re-display field */
    disp_field(cform,start,high);
}

/**/

/* This functions deletes the word to the left */
static void pascal del_left_word (struct _form_t *cform)
{
    char *buf;
    int back=0;

    /* abbreviate pointer to start of field's buffer */
    buf=cform->cfield->buf;

    /* see if at 1st position in field.  */
    /* if so, just do a normal backspace */
    if(cform->pbuf==buf) {
        back_space(cform);
        return;
    }

    /* start 1 position back */
    prev_pos(cform);
    back++;

    /* skip over any spaces */
    while(cform->pbuf>buf&&*cform->pbuf==' ') {
        prev_pos(cform);
        back++;
    }

    /* find beginning of word under cursor */
    while(cform->pbuf>buf&&*cform->pbuf!=' ') {
        *cform->pbuf=' ';
        prev_pos(cform);
        back++;
    }
    if(cform->pbuf==buf) {
        *cform->pbuf=' ';
    }
    else {
        next_pos(cform);
        back--;
    }

    /* skip over any spaces */
    while(cform->pbuf>buf&&*cform->pbuf==' ') {
        prev_pos(cform);
        back++;
    }
    if(*cform->pbuf!=' ') {
        next_pos(cform);
        back--;
    }

    /* if Insert mode is on, shift text after cursor to the left */
    if(cform->insert&&cform->pbuf>=buf) FormStrShL(cform->pbuf,back);

    /* re-display field */
    disp_field(cform,cform->pbuf,1);
}

/**/

/* This function deletes from current position through rest of the fields */
static void pascal del_rest (struct _form_t *cform)
{
    register struct _field_t *temp;

    /* save pointer to current field */
    temp=cform->cfield;

    /* delete to end of current field */
    del_field(cform,cform->pbuf,1);

    /* while not last field, find next field, and erase it */
    while(cform->cfield!=last_rec(cform)) {
        cform->cfield=next_rec(cform);
        del_field(cform,cform->cfield->buf,0);
    }

    /* restore pointer to current field */
    cform->cfield=temp;
}

/**/

/* This function deletes the word to the right of the cursor */
static void pascal del_word (struct _form_t *cform)
{
    register char *p,*q;

    /* find end of word under cursor */
    p=q=cform->pbuf;
    while(*q && *q!=' ') q++;

    /* find end of spaces */
    while(*q==' ') q++;

    /* move characters from right to left */
    while(*q) *p++=*q++;

    /* delete trailing characters */
    if(*p) del_field(cform,p,1);

    /* display the field */
    disp_field(cform,cform->pbuf,1);
}

/**/

/* This function displays a character and advances to next position */
static void pascal disp_char (struct _form_t *cform,int ch,int advance)
{
    struct _field_t *temp;

    temp=cform->cfield;
    if(cform->insert) insert_char(cform);
    WinPrintC(cform->cwrow,cform->cwcol,cform->textattr,
            (*cform->pformat=='P'||cform->cfield->fconv=='P')?' ':ch);
    *cform->pbuf=(CHAR)ch;
    if(advance) {
        cform->pbuf++;
        if( cform->pbuf >= (cform->cfield->buf+cform->cfield->lenbuf) ) {
            if(!goto_field(cform,FLD_NR)) beg_pos(cform);
        }
        else {
            cform->cwcol++;
            cform->pformat++;
            next_fchar(cform);
        }
        if(temp!=cform->cfield&&cform->cfield->mode==2 &&
            !FormStrBlank(cform->pbuf)) end_line(cform);
    }
}

/**/

/* This function displays the field of the specified window record */
static void pascal disp_field (struct _form_t *cform,char *start,int high)
{
    register struct _field_t *cfield;
    register int ccol;
    int crow,attr,disp;
    char *pf,*pb,ch;

    /* initialize position */
    cfield = cform->cfield;
    crow   = cfield->wrow;
    ccol   = cfield->wcol;
    pb     = cfield->buf;
    pf     = cfield->format;

    /* set starting point */
    if(start==NULL) start=cfield->buf;

    while(*pf) {

        /* select character attribute depending on input hilite flag */
        attr=high?cform->textattr:cform->fieldattr;

        /* see if we are at starting point yet */
        disp=(pb>=start)?1:0;

        /* test current format character */
        switch(*pf) {

            case '\"':
            case '\'':

                /* display all text inside quotes */
                ch=*pf++;
                while(*pf!=ch) {
                    if(disp) WinPrintC(crow,ccol,attr,*pf);
                    ccol++;
                    pf++;
                }
                break;
            
            case ' ':

                /* skip readability spaces */
                break;

            case '<':

                /* find right angle bracket, and    */
                /* display current buffer character */
                while(*pf!='>') pf++;
                if(disp) WinPrintC(crow,ccol,attr,*pb);
                ccol++;
                pb++;
                break;

            case '[':

                /* find right square bracket, and   */
                /* display current buffer character */
                while(*pf!=']') pf++;
                if(disp) WinPrintC(crow,ccol,attr,*pb);
                ccol++;
                pb++;
                break;

            case '.':

                /* display decimal point */
                if(disp) WinPrintC(crow,ccol,attr,*pf);
                ccol++;
                break;

            default:

                /* if current format character is a password character, */
                /* or field is a password field, then display a blank.  */
                /* otherwise, display the current buffer character.     */
                if(disp)
                    WinPrintC(crow,ccol,attr,(*pf=='P'||cfield->fconv=='P')
                      ?' ':*pb);
                ccol++;
                pb++;
        }
        pf++;
    }
}

/**/

/* This function sets the cursor at next field down from current field */
static void pascal down_field (struct _form_t *cform)
{
    struct _field_t *best,*temp;
    int tcol,brow,bcol,tdist,bdist,trow,crow,ccol;

    /* initialize best record to NULL */
    best = NULL;
    brow = bcol = 32767;

    /* find the closest field downwards */
    crow=(int)cform->cwrow;
    ccol=(int)cform->cwcol;
    for(temp=cform->field;temp!=NULL;temp=temp->prev) {
        trow=(int)temp->wrow;
        tcol=closest_column(cform,temp);
        if(trow>crow) {
            tdist=abs(ccol-tcol);
            bdist=abs(ccol-bcol);
            if((trow<brow)||((trow==brow&&tdist<bdist))) {
                best  = temp;
                brow  = trow;
                bcol  = tcol;
            }
        }
    }

    /* if a field was found, then set current field to found */
    /* field, and advance to the matching column position    */
    if(best!=NULL) {
        cform->cfield=best;
        adv_column(cform,bcol);
    }
}

/**/

/* This function sets current position to end of line in field */
static void pascal end_line (struct _form_t *cform)
{
    char *temp;

    /* save current position */
    temp=cform->pbuf;

    /* initialize to last position in field */
    end_pos(cform);

    /* if field is empty, stay on last position of field */
    if(FormStrBlank(cform->cfield->buf) || *cform->pbuf!=' ') return;

    /* search backwards for non-space or beginning of field */
    while(*cform->pbuf==' ') {
        if(cform->pbuf<=cform->cfield->buf) {
            end_pos(cform);
            return;
        }
        prev_pos(cform);
    }
    next_pos(cform);

    /* if position didn't change go to end of field */
    if(temp==cform->pbuf) end_pos(cform);
}

/**/

/* This function sets current position to the end of the field */
static void pascal end_pos (struct _form_t *cform)
{
    struct _field_t *cfield;

    cfield         = cform->cfield;
    cform->cwrow   = cfield->wrow;
    cform->cwcol   = (UCHAR)cfield->wcol + (UCHAR)cfield->lenfld - (UCHAR)1;
    cform->pbuf    = cfield->buf    + cfield->lenbuf - 1;
    cform->pformat = cfield->format + cfield->lenfmt - 1;
    prev_fchar(cform);
}

/**/

/* This function frees memory held by all of the input definitions */
static void pascal free_fields (struct _form_t *cform)
{
    struct _form_t  *form;
    struct _field_t *field;

    /* do while more fields */
    while(cform->field!=NULL) {

        /* free temp string buffer */
        free(cform->field->buf);

        /* free input field record and update linked list */
        field=cform->field->prev;
        free(cform->field);
        cform->field=field;
        if(cform->field!=NULL) cform->field->next=NULL;
    }

    /* free input form record and update linked list */
    form=cform->prev;
    free(cform);
    _WinInfo.active->form=form;
    if(_WinInfo.active->form!=NULL) _WinInfo.active->form->next=NULL;
}

/**/

/* This function will validate the current field, then move the cursor */
/* to a new field, calling the after and before functions if defined.  */
static int pascal goto_field (struct _form_t *cform,int which)
{
    int err;

    /* validate current field */
    if((err=validate_field(cform))!=0) return(err);

    /* display field, call 'after' function, and turn off Insert mode */
    after_edit(cform);

    /* call function which does the move to the new field */
    if(which==FLD_UP)       up_field(cform);
    else if(which==FLD_DN)  down_field(cform);
    else                    cform->cfield=(*funcs[which])(cform);

    /* set help category */
    _WinInfo.help=cform->cfield->help;

    /* display field in current field attribute */
    disp_field(cform,NULL,1);

    /* if a 'before' function was defined, call it */
    call_func(cform->cfield->before);

    /* return normally */
    return(0);
}

/**/

/* This function inserts a character, shifting text right one position */
static void pascal insert_char (struct _form_t *cform)
{
    /* shift characters right 1 place starting with current position */
    FormStrShR(cform->pbuf,1);

    /* re-display the field */
    disp_field(cform,cform->pbuf,1);
}

/**/

/* This function finds next non-text format character in format string */
static void pascal next_fchar (struct _form_t *cform)
{
    char ch;
    int done=0;

    cform->decimal=0;

    /* do while no format character */
    while(!done) {

        /* test current character in format string */
        switch(*cform->pformat) {

            case '\"':
            case '\'':

                /* find matching quote */
                ch=*cform->pformat++;
                while(ch!=*cform->pformat) {
                    cform->cwcol++;
                    cform->pformat++;
                }
                cform->pformat++;
                break;

            default:

                /* if currrently inside an angle or square */
                /* bracket list, find end of list          */
                if(*(cform->pformat-1)=='<') {
                    while(*cform->pformat!='>') cform->pformat++;
                    cform->pformat++;
                }
                else {
                    if(*(cform->pformat-1)=='[') {
                        while(*cform->pformat!=']') cform->pformat++;
                        cform->pformat++;
                    }
                    else {

                        /* skip readability spaces */
                        if(*cform->pformat==' ') {
                            cform->pformat++;
                        }
                        else {

                            /* check for decimal point */
                            if(*cform->pformat=='.') {
                                cform->pformat++;
                                cform->cwcol++;
                                cform->decimal=1;
                            }
                            else {

                                /* must be at next format character */
                                done=1;
                            }
                        }
                    }
                }
        }
    }
}

/**/

/* This function advances the cursor to the next position in field */
static void pascal next_pos (struct _form_t *cform)
{
    /* increment buffer pointer */
    cform->pbuf++;

    /* if buffer pointer is greater than last position in field     */
    /* and field is valid, then display field and go to next field, */
    /* first position. otherwise, advance to next position          */
    if( cform->pbuf >= (cform->cfield->buf+cform->cfield->lenbuf) ) {
        if(!goto_field(cform,FLD_NR)) beg_pos(cform);
    }
    else {
        cform->cwcol++;
        cform->pformat++;
        next_fchar(cform);
    }
}

/**/

/* This function advances the cursor to the next word */
static void pascal next_word (struct _form_t *cform)
{
    next_pos(cform);
    while(*cform->pbuf!=' ' &&
          cform->pbuf!=(cform->cfield->buf+cform->cfield->lenbuf-1))
        next_pos(cform);
    while(*cform->pbuf==' ' &&
          cform->pbuf!=(cform->cfield->buf+cform->cfield->lenbuf-1))
        next_pos(cform);
    if(cform->pbuf==(cform->cfield->buf+cform->cfield->lenbuf-1))
        next_pos(cform);
}

/**/

/* This function finds previous non-text format character in format string */
static void pascal prev_fchar (struct _form_t *cform)
{
    char ch;
    int done=0;

    /* do while format character not found */
    while(!done) {

        /* test current format character */
        switch(*cform->pformat) {

            case '\"':
            case '\'':

                /* find matching quote */
                ch=*cform->pformat--;
                while(ch!=*cform->pformat) {
                    cform->cwcol--;
                    cform->pformat--;
                }
                cform->pformat--;
                break;

            case ' ':

                /* skip readability spaces */
                cform->pformat--;
                break;

            case '>':

                /* find left angle bracket */
                while(*cform->pformat!='<') cform->pformat--;
                done=1;
                break;

            case ']':

                /* find left square bracket */
                while(*cform->pformat!='[') cform->pformat--;
                done=1;
                break;

            case '.':

                /* skip decimal point   */
                cform->pformat--;
                cform->cwcol--;
                break;

            default:

                /* must be previous format character */
                done=1;
        }
    }
}

/**/

/* This function positions the cursor to the previous position in field */
static void pascal prev_pos (struct _form_t *cform)
{
    /* decrement buffer pointer */
    cform->pbuf--;

    /* if buffer pointer is less than first position in field and   */
    /* field is valid, then display field and go to previous field, */
    /* last position. otherwise, advance to next position           */
    if( cform->pbuf < cform->cfield->buf ) {
        if(!goto_field(cform,FLD_PR)) end_pos(cform);
    }
    else {
        cform->cwcol--;
        cform->pformat--;
        prev_fchar(cform);
    }
}

/**/

/* This function moves the cursor to the beginning of the previous word */
static void pascal prev_word (struct _form_t *cform)
{
    prev_pos(cform);
    while(*cform->pbuf==' ' && cform->pbuf!=cform->cfield->buf)
        prev_pos(cform);
    while(*cform->pbuf!=' ' && cform->pbuf!=cform->cfield->buf)
        prev_pos(cform);
    if(cform->pbuf!=cform->cfield->buf) next_pos(cform);
}

/**/

/* This function restores a field to its contents before editing */
static void pascal restore_field (struct _form_t *cform,int high)
{
    struct _field_t *cfield;

    /* abbreviate pointer to current field */
    cfield=cform->cfield;

    strcpy(cfield->buf,cfield->str);
    if(cfield->fconv=='9') deci_right(cform,cfield->buf+cfield->lenbuf);
    disp_field(cform,NULL,high);
}

/**/

/* This function sets the termination key variable, if defined */
static void pascal set_term_key (struct _form_t *cform,unsigned xch)
{
    if(cform->termkey!=NULL) *(cform->termkey)=xch;
}

/**/

/* This function searches for the previous field up from current field */
static void pascal up_field (struct _form_t *cform)
{
    struct _field_t *best,*temp;
    int brow,bcol,trow,tcol,tdist,bdist,crow,ccol;

    /* initialize best record to NULL */
    best  = NULL;
    brow  = -1;
    bcol  = 32767;

    /* find the closest field upwards */
    crow=(int)cform->cwrow;
    ccol=(int)cform->cwcol;
    for(temp=cform->field;temp!=NULL;temp=temp->prev) {
        trow=(int)temp->wrow;
        tcol=closest_column(cform,temp);
        if(trow<crow) {
            tdist=abs(ccol-tcol);
            bdist=abs(ccol-bcol);
            if((trow>brow)||((trow==brow&&tdist<bdist))) {
                best=temp;
                brow=trow;
                bcol=tcol;
            }
        }
    }

    /* if a field was found, then set current field to found */
    /* field, and advance to the matching column position    */
    if(best!=NULL) {
        cform->cfield=best;
        adv_column(cform,bcol);
    }
}

/**/

/* This function validates the field buffer using the format string */
static int pascal validate_field (struct _form_t *cform)
{
    char *pb,*pf,ch;
    struct _field_t *cfield;
    struct _form_t  *temp;
    int valid=1,i,pos;

    /* abbreviate pointer to current field */
    cfield=cform->cfield;

    /* if field is numeric, then format it so */
    if(cfield->fconv=='9') deci_right(cform,cfield->buf+cfield->lenbuf);

    pb=cfield->buf;
    pf=cfield->format;

    /* if string is blank, don't do the built-in validation. */
    /* As far as we're concerned, a blank field is okay.     */
    if(!FormStrBlank(pb)) {

        /* do while not end of format string and field still valid */
        while(*pf && valid) {

            /* test character in the field buffer with */
            /* current format character for validity.  */
            switch(*pf) {

                case ' ':

                    /* skip readability spaces */
                    pf++;
                    break;

                case '\"':
                case '\'':

                    /* skip quoted text in format string */
                    ch=*pf++;
                    while(ch!=*pf) pf++;
                    pf++;
                    break;

                case '<':

                    /* check current character in field buffer for */
                    /* inclusion in the list of valid characters.  */
                    pf++;
                    valid=0;
                    while(*pf!='>') {
                        if(*pb==*pf) valid=1;
                        pf++;
                    }
                    if(valid) {
                        pf++;
                        pb++;
                    }
                    break;

                case '[':


                    /* check current character in field buffer for */
                    /* exclusion in the list of valid characters.  */
                    pf++;
                    valid=1;
                    while(*pf!=']') {
                        if(*pb==*pf) valid=0;
                        pf++;
                    }
                    if(valid) {
                        pf++;
                        pb++;
                    }
                    break;

                case '.':

                    /* skip decimal point */
                    pf++;
                    break;

                default:

                    /* see if current field character is a valid */
                    /* type of the current format character      */
                    valid=FormCValType(*pb,*pf);
                    if(*pb==' '&&cfield->fconv=='9') valid=1;
                    if(valid) {
                        pf++;
                        pb++;
                    }
            }
        }
    }

    /* if the field passes validation so far and a user-defined */
    /* function was specified, save current environment, call   */
    /* the user function, then restore the environment          */
    if(valid) {
        if(cfield->validate!=NULL) {
            temp=cform;
            pos=( (*(cform->cfield->validate)) (cform->cfield->buf) );
            cform=temp;
            cfield=cform->cfield;

            /* check to see if any of the fields' redisp flags have been set */
            check_redisp(cform);

            /* test return value from user-defined function. */
            /* if not zero, then field is in error           */
            if(pos) {
                beg_pos(cform);
                goto validate_error;
            }
        }
    }
    else {

        /* advance to error position in field */
        beg_pos(cform);
        pos=(int)( ((unsigned long) pb ) - ((unsigned long) cfield->buf ) + 1);
validate_error:
        if(pos>cfield->lenbuf) pos=1;
        for(i=1;i<pos;i++) next_pos(cform);

        /* return error code to caller */
        return(pos);
    }

    /* return with no error (field is valid) */
    return(0);
}

/**/

/* This function finds the first defined field */
static struct _field_t *first_rec (struct _form_t *cform)
{
    struct _field_t *temp;

    for(temp=cform->field;temp->prev!=NULL;temp=temp->prev) ;
    return(temp);
}

/**/

/* This function finds the last defined field */
static struct _field_t *last_rec (struct _form_t *cform)
{
    return(cform->field);
}

/**/

/* This function sets the current field to the prev field in linked list */
static struct _field_t *prev_rec (struct _form_t *cform)
{
    struct _field_t *field;

    field=cform->cfield->prev;
    if(field==NULL) field=last_rec(cform);
    return(field);
}

/**/

/* This function sets the current field to the next field in the linked list */
static struct _field_t *next_rec (struct _form_t *cform)
{
    struct _field_t *field;

    field=cform->cfield->next;
    if(field==NULL) field=first_rec(cform);
    return(field);
}

/**/

