/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* nodes.c - Node handling - who, node messages, etc. */

#include <stdio.h>
#include <string.h>

#include "os.h"
#include "memory.h"
#include "nodes.h"
#include "ask_str.h"
#include "bbs_func.h"
#include "userbase.h"
#include "logfile.h"
#include "output.h"
#include "files.h"
#include "mareas.h"
#include "menutype.h"
#include "language.h"
#include "config.h"

char **node_doing;

static int Fwho;

NODE_REC noderec;

static SAVED_NODEMSG *first_msg, *last_msg;
static unsigned nodemsgs;

/* Save node message */
void add_message(char *str)
{
    SAVED_NODEMSG *node;

    if (nodemsgs == MAX_NODEMSGS_IN_MEMORY)
    {
        /* Remove first (oldest) node message */
        node = first_msg->next;
        _free(first_msg);
        first_msg = node;
        nodemsgs--;
    }

    /* Get memory for node message text */
    node = (SAVED_NODEMSG *) _malloc(sizeof(SAVED_NODEMSG));
    if (node == NULL) return;

    strcpy(node->text, str);
    node->next = NULL;

    /* Update linked list */
    if (first_msg == NULL) first_msg = node;
    if (last_msg != NULL) last_msg->next = node;
    last_msg = node;
    nodemsgs++;
}

void close_node_messages(void)
{
    SAVED_NODEMSG *node;

    while (first_msg != NULL)
    {
        node = first_msg->next;
        _free(first_msg);
        first_msg = node;
    }
}

/* Display last ## node messages */
void display_old_nodemsgs(void)
{
    SAVED_NODEMSG *node;

    output("\r\n@MON@");
    node = first_msg;
    while (node != NULL)
    {
        output(node->text);
        if (!outtext("\r\n")) break;
        node = node->next;
    }
}

void display_who(void)
{
    NODE_REC oldnode;
    int node;

    memcpy(&oldnode,&noderec,sizeof(oldnode));
    output(lang[LANG_WHO_HEADER]);
    for (node=1; node<=nodes; node++)
    {
        if (read_nodefile(node,&noderec) == 0)
        {
            /* Could not read record */
            noderec.doing = DOING_DOWN;
            noderec.BPS = 0;
            strcpy(noderec.name, "-");
            strcpy(noderec.city, "-");
        }

        /* Just to make sure noderec won't crash program */
        noderec.name[sizeof(noderec.name)-1] = '\0';
        noderec.city[sizeof(noderec.name)-1] = '\0';
        if (noderec.doing > WHO_DOINGS) noderec.doing = DOING_NOTHING;

        if (noderec.doing == DOING_DOWN || noderec.doing == DOING_WAITING_CALLER)
            output(lang[LANG_WHO_ENTRY_GONE], node);
        else
            output(lang[LANG_WHO_ENTRY_HERE], node);
    }

    memcpy(&noderec,&oldnode,sizeof(oldnode));
}

void send_nodemsg(char *data)
{
    int F;
    unsigned num;
    char str[256],tmp[256];
    NODE_REC tmpnode;

    /* Ask node number */
    str[0] = '\0';
    if (!ask_string(lang[LANG_ASK_NODENUM], str, sizeof(str)-1, 0, &data)) return;
    if (sscanf(str, "%d", &num) != 1) num = 0;

    if (num < 1 || num > nodes)
    {
        /* Invalid node number */
        output(lang[LANG_INVALID_NODENUM]);
        return;
    }

    if (!read_nodefile(num,&tmpnode) || tmpnode.doing == DOING_DOWN || tmpnode.doing == DOING_WAITING_CALLER)
    {
        output(lang[LANG_NOONE_IN_THERE]);
        return;
    }

    /* Ask text */
    if (data != NULL && data[0] != '\0')
        strcpy(str, data);
    else
    {
        str[0] = '\0';
        if (!ask_string(lang[LANG_ASK_NODETEXT], str, sizeof(str)-1, 0, NULL)) return;
    }

    if (str[0] == '\0') return; /* Aborted */

    sprintf(tmp,"%snodemsg.%d",common_path,num);
    F = FileOpen(tmp, O_WRONLY | O_BINARY, SH_DENYRW);
    if (F == -1)
    {
        /* Doesn't exist, create. */
        F = FileCreate(tmp, CREATE_MODE);
        if (F == -1)
        {
            output("\r\nCouldn't send node message!\r\n");
            write_log("Couldn't create file '%s'", tmp);
        }
        else
        {
            FileMode(F,O_BINARY);
        }
    }

    if (num == nodenum)
        sprintf(tmp, "\x01%s%s\n", lang[LANG_NODEMSG_YOURSELF], str);
    else
        sprintf(tmp, "\x01@X0FLinjalta %d: %s\n", nodenum, str);

    /* Write to node file */
    if (!FileLock(F))
    {
        write_log("send_nodemsg() : Can't lock file: %snodemsg.%d",common_path,num);
        output("\r\nCouldn't send node message!\r\n");
    }
    else
    {
        FileSeek(F,0,SEEK_END);
        FileWrite(F,tmp,strlen(tmp));

        FileUnlock(F);
        FileClose(F);
    }
}

int display_nodefile(char *fname)
{
    unsigned long areanum,oldarea;
    char str[256],*strp;
    int F;
    int ret;

    ret = 0;

    /* Open node file */
    F = FileOpen(fname, O_RDWR | O_BINARY, SH_DENYNO);
    if (F == -1) return 0;
    if (!FileLock(F))
    {
        write_log("display_nodefile() : Can't lock file: %s",fname);
        FileClose(F);
        return 0;
    }

    while (_fgets(str,sizeof(str),F) != NULL)
    {
        switch (str[0])
        {
            case NODETEXT_NORMAL:
                /* Node message */
                output("\r\n%s",str+1);
                add_message(str+1);
                ret = 1;
                break;
            case NODETEXT_MSGWRITTEN:
                /* Message written in area # */
                strp = strchr(str+1,';');
                if (strp == NULL) break;

                /* Get area number */
                *strp++ = '\0';
                if (sscanf(str+1,"%lu",&areanum) != 1 || areanum == 0) break;

                /* Display messages if area is selected */
                oldarea = current_marea;
                if (read_marea_record(areanum) && marea.flags & MAREA_FLAG_SELECTED)
                {
                    output("\r\n%s",strp);
                    add_message(strp);
                    ret = 1;
                }
                read_marea_record(oldarea);
                break;
        }
    }

    FileTrunc(F,0); /* Truncate node file */
    FileUnlock(F);
    FileClose(F);

    return ret;
}

int open_nodefile(void)
{
    char tmp[256];
    unsigned num,recs;

    node_doing = &lang[LANG_NODEDOING_1];

    /* Open nodes.bbs */
    sprintf(tmp,"%snodes.bbs",common_path);
    Fwho = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYNO);
    if (Fwho == -1)
    {
        /* Not found, create it.. */
        Fwho = FileCreate(tmp, CREATE_MODE);
        if (Fwho == -1) return 0; /* Couldn't open it. */
        FileMode(Fwho,O_BINARY);
    }

    recs = (unsigned) (FileSeek(Fwho,0,SEEK_END)/sizeof(noderec));
    memset(&noderec,0,sizeof(noderec));

    for (num=recs; num<nodenum-1; num++)
        FileWrite(Fwho,&noderec,sizeof(noderec));

    first_msg = NULL;
    nodemsgs = 0;
    return 1;
}

void close_nodefile(void)
{
    FileClose(Fwho);
}

void update_nodefile(int no_change)
{
    if (!no_change)
    {
        strcpy(noderec.name,usrsub.name);
        strcpy(noderec.city,usrsub.city);
        noderec.BPS = bpsrate;
    }
    FileSeek(Fwho,(nodenum-1)*sizeof(NODE_REC),SEEK_SET);
    FileWrite(Fwho,&noderec,sizeof(NODE_REC));
}

int read_nodefile(unsigned node, NODE_REC *noderec)
{
    FileSeek(Fwho,(node-1)*sizeof(NODE_REC),SEEK_SET);
    return FileRead(Fwho,noderec,sizeof(NODE_REC)) == sizeof(NODE_REC);
}

void send_nodetext(char *text, int node)
{
    char tmp[256];
    int F;
    NODE_REC tmpnode;

    if (node == 0)
    {
        if (nodenum != 1) send_nodetext(text,1);
        if (nodenum != 2) send_nodetext(text,2);
        if (nodenum != 3) send_nodetext(text,3);
        return;
    }

    if (!read_nodefile(node,&tmpnode) || tmpnode.doing == DOING_DOWN || tmpnode.doing == DOING_WAITING_CALLER) return;

    sprintf(tmp,"%snodemsg.%d",common_path,node);
    F = FileOpen(tmp, O_WRONLY | O_BINARY, SH_DENYRW);
    if (F == -1)
    {
        /* Doesn't exist, create it. */
        F = FileCreate(tmp, CREATE_MODE);
        if (F == -1) write_log("Can't create file '%s'",tmp);
    }

    if (F != -1)
    {
        if (!FileLock(F))
            write_log("send_nodetext() : Can't lock file: %s",tmp);
        else
        {
            FileSeek(F,0,SEEK_END);
            FileWrite(F,text,strlen(text)); FileWrite(F,"\n",1);
            FileUnlock(F);
            FileClose(F);
        }
    }
}
