/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* msg_jam.c - JAM message base handling */

#include <stdio.h>
#include <string.h>
#include <time.h>

#include "os.h"
#include "memory.h"
#include "files.h"
#include "timeslic.h"
#include "crc32.h"
#include "userbase.h"
#include "msg_jam.h"
#include "jam.h"
#include "logfile.h"
#include "mareas.h"
#include "config.h"

#define TMP_BUF_SIZE 4096

LINE_REC *msg_first_linerec;
LINE_REC *msg_linerec;
unsigned long msg_current_line;

FILE_BUF *msg_Fjhr, *msg_Fjdx, *msg_Fjdt;

char msg_from[36];
char msg_to[36];
char msg_subj[81];
unsigned long msg_date;
unsigned msg_flags;

unsigned long msg_msgnum;
unsigned long msg_replyto,msg_reply1st,msg_replynext;

char msg_msgid_kludge[81];
char msg_reply_kludge[81];
char msg_orig_net[100];
char msg_dest_net[100];
char msg_charset_kludge[81];
char msg_intl_kludge[81];
char msg_origin_line[200];

unsigned short msg_zone,msg_net,msg_node,msg_point;

unsigned msg_firstmsg,msg_messages;

unsigned long msg_txtpos,msg_txtlen;
unsigned long msg_txtlines;

unsigned char *msg_dat_buf;
unsigned long msg_dat_start,msg_dat_end;

static char msgtext[256];

int msg_init(void)
{
    msg_Fjhr = NULL; msg_Fjdx = NULL; msg_Fjdt = NULL;
    msg_dat_start = 1; msg_dat_end = 0;

    return 1;
}

void msg_deinit(void)
{
}

int msg_open_area(char *fname, char create)
{
    char tmp[256];
    JAMHDRINFO jamhdr;

    msg_messages = 0;

    msg_first_linerec = NULL;
    msg_dat_start = 1; msg_dat_end = 0;

    if (msg_Fjhr != NULL) write_log("msg_open_area(): File was open: msg_Fjhr\n");
    if (msg_Fjdx != NULL) write_log("msg_open_area(): File was open: msg_Fjdx\n");
    if (msg_Fjdt != NULL) write_log("msg_open_area(): File was open: msg_Fjdt\n");

    /* Open message base files */
    sprintf(tmp,"%s"EXT_HDRFILE, fname); msg_Fjhr = FileBufOpen(tmp, O_RDWR | O_BINARY, SH_DENYNO);
    sprintf(tmp,"%s"EXT_IDXFILE, fname); msg_Fjdx = FileBufOpen(tmp, O_RDWR | O_BINARY, SH_DENYNO);
    sprintf(tmp,"%s"EXT_TXTFILE, fname); msg_Fjdt = FileBufOpen(tmp, O_RDWR | O_BINARY, SH_DENYNO);

    if (create && (msg_Fjhr == NULL && msg_Fjdx == NULL && msg_Fjdt == NULL))
    {
        /* Create message base files */
        sprintf(tmp,"%s"EXT_HDRFILE, fname); msg_Fjhr = FileBufCreate(tmp, CREATE_MODE);
        if (msg_Fjhr == NULL) write_log("msg_open_area() : Can't create file %s", tmp);
        sprintf(tmp,"%s"EXT_IDXFILE, fname); msg_Fjdx = FileBufCreate(tmp, CREATE_MODE);
        if (msg_Fjdx == NULL) write_log("msg_open_area() : Can't create file %s", tmp);
        sprintf(tmp,"%s"EXT_TXTFILE, fname); msg_Fjdt = FileBufCreate(tmp, CREATE_MODE);
        if (msg_Fjdt == NULL) write_log("msg_open_area() : Can't create file %s", tmp);

        if (msg_Fjhr != NULL)
        {
            FileMode(msg_Fjhr->handle, O_BINARY);
            FileMode(msg_Fjdx->handle, O_BINARY);
            FileMode(msg_Fjdt->handle, O_BINARY);

            if (!FileLock(msg_Fjhr->handle))
                write_log("Warning: msg_open_area() : Can't lock file: %s"EXT_HDRFILE, fname);

            memset(&jamhdr, 0, sizeof(jamhdr));
            strcpy(jamhdr.Signature, HEADERSIGNATURE);
            jamhdr.DateCreated = time(NULL);
            jamhdr.PasswordCRC = 0xffffffff;
            if (FileBufWrite(msg_Fjhr, &jamhdr, sizeof(jamhdr)) != sizeof(jamhdr))
            {
                write_log("msg_open_area() : Can't write to file %s"EXT_HDRFILE, fname);
                FileBufClose(msg_Fjhr);
                msg_Fjhr = NULL;
            }

            FileBufUnlock(msg_Fjhr);
        }
    }

    if (msg_Fjhr == NULL || msg_Fjdx == NULL || msg_Fjdt == NULL)
    {
        FileBufClose(msg_Fjhr);
        FileBufClose(msg_Fjdx);
        FileBufClose(msg_Fjdt);
        return 0;
    }

    /* Get number of messages */
    msg_messages = (unsigned) (FileBufSeek(msg_Fjdx,0,SEEK_END)/sizeof(JAMIDXREC));

    /* Get first message number */
    msg_firstmsg = 1;

    return 1;
}

void msg_close_area(void)
{
    /* Close message base files */
    if (msg_Fjhr != NULL) { FileBufClose(msg_Fjhr); msg_Fjhr = NULL; }
    if (msg_Fjdx != NULL) { FileBufClose(msg_Fjdx); msg_Fjdx = NULL; }
    if (msg_Fjdt != NULL) { FileBufClose(msg_Fjdt); msg_Fjdt = NULL; }
}

int msg_read_message(unsigned long num)
{
    JAMSUBFIELD *subfield;
    unsigned len, sublen, subgot;
    JAMIDXREC *jamidx;
    JAMHDR *msgrec;

    if (num == 0) return 0;

    /* Read .JHR file position from .JDX file */
    FileBufSeek(msg_Fjdx, (num-1)*sizeof(*jamidx), SEEK_SET);
    if (FileBufRead(msg_Fjdx, (void **) &jamidx, sizeof(*jamidx)) != sizeof(*jamidx)) return 0;
    if (jamidx->HdrOffset == 0xffffffff) return 0;

    /* Read message record from .JHR file */
    FileBufSeek(msg_Fjhr, jamidx->HdrOffset, SEEK_SET);
    if (FileBufRead(msg_Fjhr, (void **) &msgrec, sizeof(*msgrec)) != sizeof(*msgrec)) return 0;

    if (strcmp(msgrec->Signature, HEADERSIGNATURE) != 0) return 0;
    if (msgrec->Revision != CURRENTREVLEV) return 0;
    if (msgrec->Attribute & MSG_DELETED) return 0;

    msg_flags = (msgrec->Attribute&MSG_PRIVATE) ? MSG_FLAG_PRIVATE : 0;

    msg_date = msgrec->DateWritten;
    msg_msgnum = msgrec->MsgNum;
    msg_replyto = msgrec->ReplyTo;
    msg_reply1st = msgrec->Reply1st;
    msg_replynext = msgrec->ReplyNext;
    msg_txtpos = msgrec->TxtOffset;
    msg_txtlen = msgrec->TxtLen;

    msg_msgid_kludge[0] = 0;
    msg_charset_kludge[0] = 0;

    /* Read subfields */
    if (msgrec->SubfieldLen > 0)
    {
        if (msgrec->SubfieldLen > sizeof(msg_Fjhr->buffer))
        {
            write_log("msgrec.SubfieldLen > 16384 !! Can't read enough data!!");
            printf("msgrec.SubfieldLen > 16384 !! Can't read enough data!!\n");
        }

        /* Read subfields into memory */

        sublen = (unsigned) msgrec->SubfieldLen;
        FileBufRead(msg_Fjhr, (void **) &subfield, sublen);

        subgot = 0;
        while (subgot < sublen)
        {
            len = (unsigned) subfield->DatLen;

            switch (subfield->LoID)
            {
                case JAMSFLD_SENDERNAME:
                    if (len > sizeof(msg_from)-1) len = sizeof(msg_from)-1;
                    msg_from[len] = '\0';
                    memcpy(msg_from,subfield->Buffer,len);
                    break;
                case JAMSFLD_RECVRNAME:
                    if (len > sizeof(msg_to)-1) len = sizeof(msg_to)-1;
                    msg_to[len] = '\0';
                    memcpy(msg_to,subfield->Buffer,len);
                    break;
                case JAMSFLD_SUBJECT:
                    if (len > sizeof(msg_subj)-1) len = sizeof(msg_subj)-1;
                    msg_subj[len] = '\0';
                    memcpy(msg_subj,subfield->Buffer,len);
                    break;
                case JAMSFLD_MSGID:
                    if (len > sizeof(msg_msgid_kludge)-1) len = sizeof(msg_msgid_kludge)-1;
                    msg_msgid_kludge[len] = '\0';
                    memcpy(msg_msgid_kludge,subfield->Buffer,len);
                    break;
                case JAMSFLD_FTSKLUDGE:
                    if (strncmp(subfield->Buffer,"CHARSET: ",9) == 0)
                    {
                        len -= 9;
                        if (len > sizeof(msg_charset_kludge)-1) len = sizeof(msg_charset_kludge)-1;
                        msg_charset_kludge[len] = '\0';
                        memcpy(msg_charset_kludge,subfield->Buffer+9,len);
                    }
                    break;
                case JAMSFLD_OADDRESS:
                    if (len > sizeof(msg_orig_net)-1) len = sizeof(msg_orig_net)-1;
                    msg_orig_net[len] = '\0';
                    memcpy(msg_orig_net,subfield->Buffer,len);
                    break;
                case JAMSFLD_DADDRESS:
                    if (len > sizeof(msg_dest_net)-1) len = sizeof(msg_dest_net)-1;
                    msg_dest_net[len] = '\0';
                    memcpy(msg_dest_net,subfield->Buffer,len);
                    break;
            }

            subgot += sizeof(JAMBINSUBFIELD)+subfield->DatLen;
            subfield = (JAMSUBFIELD *) ((char *) subfield+sizeof(JAMBINSUBFIELD)+subfield->DatLen);
        }
    }

    FileBufSeek(msg_Fjdt, msg_txtpos, SEEK_SET);
    return 1;
}

void add_subfield(char *subdata, char *fielddata, unsigned subtype, JAMHDR *msgrec)
{
    JAMBINSUBFIELD subrec;

    if (fielddata[0] == '\0') return;

    subrec.LoID = (unsigned short) subtype;
    subrec.HiID = 0;
    subrec.DatLen = strlen(fielddata);

    memcpy(subdata,&subrec,sizeof(subrec));
    memcpy(subdata+sizeof(subrec),fielddata,(unsigned) subrec.DatLen);
    msgrec->SubfieldLen += subrec.DatLen+sizeof(subrec);
}

int msg_enter_msg(char *txtfile)
{
    int Ftxt, Fbase;
    size_t readed;
    unsigned long fjhr_pos, msg_msgnum;
    char *subbuf, *buffer, line[80];
    JAMHDRINFO *jamhdr;
    JAMIDXREC *jamidx;
    JAMHDR *msgrec;

    subbuf = (char *) _malloc(4096);
    if (subbuf == NULL) return 0;

    /* Lock message base */
    if (!FileLock(msg_Fjhr->handle))
    {
        write_log("enter_msg() : Can't lock message base");
        _free(subbuf);
        return 0;
    }

    /* Read message base header */
    FileBufSeek(msg_Fjhr, 0, SEEK_SET);
    if ((readed = FileBufRead(msg_Fjhr, (void **) &jamhdr, sizeof(*jamhdr))) != sizeof(*jamhdr))
    {
        if (readed == -1)
            write_log("Can't read message base header - .jhr file not open!?");
        else
            write_log("Can't read message base header");
        FileUnlock(msg_Fjhr->handle);
        _free(subbuf);
        return 0;
    }

    /* Read message number of previous message */
    if (FileBufSeek(msg_Fjdx, 0, SEEK_END) == 0)
    {
        /* No messages in message base */
        jamidx->HdrOffset = FileBufSeek(msg_Fjhr,0,SEEK_END);
    }
    else
    {
        for (;;)
        {
            FileBufSeek(msg_Fjdx, -sizeof(*jamidx), SEEK_CUR);
            if (FileBufRead(msg_Fjdx, (void **) &jamidx, sizeof(*jamidx)) != sizeof(*jamidx))
            {
                jamidx->HdrOffset = FileBufSeek(msg_Fjhr, 0, SEEK_END);
                break;
            }
            if (jamidx->HdrOffset != 0xffffffff) break;

            if (FileBufSeek(msg_Fjdx, -sizeof(*jamidx), SEEK_CUR) == 0)
            {
                /* No messages in message base */
                jamidx->HdrOffset = FileBufSeek(msg_Fjhr, 0, SEEK_END);
                break;
            }
        }
    }

    FileBufSeek(msg_Fjhr, jamidx->HdrOffset, SEEK_SET);
    if (FileBufRead(msg_Fjhr, (void **) &msgrec, sizeof(*msgrec)) != sizeof(*msgrec))
    {
        msg_msgnum = 1;
        msgrec = (JAMHDR *) msg_Fjhr->buffer;
    }
    else
        msg_msgnum = msgrec->MsgNum+1;

    /* Open message text */
    Ftxt = FileOpenIgn(txtfile, O_RDONLY | O_BINARY, SH_DENYNO);
    if (Ftxt == -1)
    {
        write_log("Can't open file '%s'", txtfile);
        FileUnlock(msg_Fjhr->handle);
        _free(subbuf);
        return 0;
    }

    /* Create message header */
    memset(msgrec, 0, sizeof(*msgrec));
    strcpy(msgrec->Signature, HEADERSIGNATURE);
    msgrec->Revision = CURRENTREVLEV;
    msgrec->DateWritten = msg_date;
    msgrec->MsgNum = msg_msgnum;
    msgrec->PasswordCRC = 0xffffffff;
    msgrec->Attribute = MSG_LOCAL;
    if (msg_flags & MSG_FLAG_PRIVATE) msgrec->Attribute |= MSG_PRIVATE;
    if (msg_flags & MSG_FLAG_SENT) msgrec->Attribute |= MSG_SENT;
    if (msg_flags & MSG_FLAG_RECEIVED) msgrec->Attribute |= MSG_READ;
    msgrec->ReplyTo = msg_replyto;

    msgrec->TxtOffset = FileBufSeek(msg_Fjdt, 0, SEEK_END);
    msgrec->TxtLen = FileSeek(Ftxt, 0, SEEK_END);

    /* Create subfields */
    add_subfield(subbuf+msgrec->SubfieldLen, msg_from, JAMSFLD_SENDERNAME, msgrec);
    add_subfield(subbuf+msgrec->SubfieldLen, msg_orig_net, JAMSFLD_OADDRESS, msgrec);
    add_subfield(subbuf+msgrec->SubfieldLen, msg_to, JAMSFLD_RECVRNAME, msgrec);
    add_subfield(subbuf+msgrec->SubfieldLen, msg_dest_net, JAMSFLD_DADDRESS, msgrec);
    add_subfield(subbuf+msgrec->SubfieldLen, msg_subj, JAMSFLD_SUBJECT, msgrec);
    add_subfield(subbuf+msgrec->SubfieldLen, msg_msgid_kludge, JAMSFLD_MSGID, msgrec);
    add_subfield(subbuf+msgrec->SubfieldLen, msg_reply_kludge, JAMSFLD_REPLYID, msgrec);
    add_subfield(subbuf+msgrec->SubfieldLen, msg_charset_kludge, JAMSFLD_FTSKLUDGE, msgrec);

    /* Write message header */
    fjhr_pos = FileBufSeek(msg_Fjhr, 0, SEEK_END);

    if (FileBufWrite(msg_Fjhr, msgrec, sizeof(*msgrec)) != sizeof(*msgrec))
    {
        write_log("Can't write message header");
        FileUnlock(msg_Fjhr->handle);
        _free(subbuf);
        return 0;
    }

    /* Write subfields */
    if (FileBufWrite(msg_Fjhr, subbuf, (unsigned) msgrec->SubfieldLen) != (int) msgrec->SubfieldLen)
    {
        write_log("Can't write subfields");
        FileUnlock(msg_Fjhr->handle);
        _free(subbuf);
        return 0;
    }
    _free(subbuf);

    /* Update .JDX file */
    FileBufSeek(msg_Fjdx, 0, SEEK_END);
    jamidx->UserCRC = lo_crc32(msg_to);
    jamidx->HdrOffset = fjhr_pos;
    if (FileBufWrite(msg_Fjdx, jamidx, sizeof(*jamidx)) != sizeof(*jamidx))
    {
        write_log("Can't write message index");
        FileBufUnlock(msg_Fjhr);
        return 0;
    }

    /* Update .JDT file */
    buffer = (char *) _malloc(TMP_BUF_SIZE);
    if (buffer != NULL)
    {
        FileSeek(Ftxt, 0, SEEK_SET);
        FileBufSeek(msg_Fjdt, msgrec->TxtOffset, SEEK_SET);
        while ((readed = FileRead(Ftxt, buffer, TMP_BUF_SIZE)) > 0)
        {
            if (FileBufWrite(msg_Fjdt, buffer, readed) != (int) readed)
            {
                write_log("Can't write message text");
                _free(buffer);
                FileUnlock(msg_Fjhr->handle);
                return 0;
            }
        }
        _free(buffer);
    }
    FileClose(Ftxt);

    /* Update ModCounter */
    jamhdr->ModCounter++;
    FileBufSeek(msg_Fjhr, 0, SEEK_SET);
    FileBufWrite(msg_Fjhr, jamhdr, sizeof(*jamhdr));

    msg_messages = (unsigned) (FileBufSeek(msg_Fjdx, 0, SEEK_END) / sizeof(*jamidx));

    /* Unlock message base */
    FileUnlock(msg_Fjhr->handle);

    if ((marea_rec->flags & MAREA_FLAG_ECHO) == 0 && (marea_rec->flags & MAREA_FLAG_NETMAIL) == 0)
        return 1;

    /* Write areatag to JAM base file */
    Fbase = FileOpen(jam_base_file, O_RDWR | O_BINARY, SH_DENYRW);
    if (Fbase == -1)
    {
        Fbase = FileCreate(jam_base_file, CREATE_MODE);
        if (Fbase == -1) return 1;
    }

    if (!FileLock(Fbase))
    {
        write_log("enter_msg() : Can't lock message JAM base file");
        FileClose(Fbase);
        return 1;
    }

    while (_fgets(line, sizeof(line), Fbase) != NULL)
    {
        if (stricmp(line, marea_rec->name) == 0)
        {
            /* Tag found, no need to add it again */
            FileUnlock(Fbase);
            FileClose(Fbase);
            return 1;
        }
    }

    sprintf(line, "%s\n", marea_rec->name);
    FileWrite(Fbase, line, strlen(line));

    FileUnlock(Fbase);
    FileClose(Fbase);

    return 1;
}

int msg_init_msgtext(void)
{
    unsigned long last_line,totalread;
    size_t readed;

    unsigned pos,ptr_pos,num;

    char buf[256];
    int spos,origin,reformat,no_quote,start;
    unsigned chars,tens,line_length;

    LINE_REC *linep;

    /* If old reply is still in memory, delete it */
    if (msg_first_linerec != NULL) msg_deinit_msgtext();

    msg_txtlines = 0; last_line = 0; tens = 0; chars = 0; spos = 0;
    msg_origin_line[0] = '\0'; line_length = user.ScreenWidth;

    /* Init first line */
    msg_linerec = (LINE_REC *) _malloc(sizeof(LINE_REC));
    msg_linerec->prev = NULL;
    msg_linerec->next = NULL;
    msg_linerec->line_type = LINE_TYPE_NORMAL;
    msg_linerec->ptr = msg_txtpos;
    msg_linerec->len = 0;

    msg_first_linerec = msg_linerec;

    totalread = 0; readed = 0;
    start = 1; origin = 0; reformat = /*setup.justify != JUSTIFY_NONE*/0; no_quote = 0;
    while (totalread+readed < msg_txtlen)
    {
        totalread += readed;

        /* How much to read? */
        if (msg_txtlen-totalread < BUF_SIZE)
            readed = (size_t) (msg_txtlen-totalread);
        else
            readed = BUF_SIZE;

        if ((msg_dat_start > msg_txtpos+totalread) || (msg_dat_end < msg_txtpos+readed-1+totalread))
        {
            /* Read text from file */
            msg_dat_start = FileBufSeek(msg_Fjdt, msg_txtpos+totalread, SEEK_SET);
            readed = FileBufRead(msg_Fjdt, (void **) &msg_dat_buf, readed);
            msg_dat_end = msg_dat_start+readed-1;
        }
        else
        {
            /* Text already in memory */
            readed = (size_t) (msg_dat_end-msg_txtpos+1-totalread);
        }

        if (readed >= msg_txtlen-totalread) readed = (size_t) (msg_txtlen-totalread);

        pos = (unsigned) (msg_txtpos+totalread-msg_dat_start);
        ptr_pos = (unsigned) -1;

        for (; pos<readed+(msg_txtpos+totalread-msg_dat_start); pos++)
        {
            ptr_pos++;
            switch (msg_dat_buf[pos])
            {
                case 10:
                    /* Do nothing to LFs */
                    tens++;
                    break;
                case 13:
                    /* CR found */

#if 0
                    /* Check spaces and CR */
                    if (reformat && spos > 0)
                    {
                        /* Copy next 10 characters to 'tmp' */
                        if (pos+13 < BUF_SIZE)
                        {
                            /* They're in memory, great. */
                            if (msg_dat_buf[pos+1] == 10)
                                memcpy(tmp,msg_dat_buf+pos+2,10);
                            else
                                memcpy(tmp,msg_dat_buf+pos+1,10);
                            tmp[10] = '\0';
                        }
                        else
                        {
                            /* Gotta read them from file. */
                            FileBufSeek(msg_Fjdt, msg_txtpos+totalread+pos+1, SEEK_SET);
                            if (FileBufRead(msg_Fjdt, tmp, 1) != 1)
                            {
                                tmp[0] = '\0';
                            }
                            else
                            {
                                if (tmp[0] == 10)
                                    tmp[FileBufRead(msg_Fjdt, tmp, 11)] = '\0';
                                else
                                    tmp[FileBufRead(msg_Fjdt, tmp+1, 10)+1] = '\0';
                            }
                            num = (unsigned) (totalread+pos+1);
                            if (num >= msg_txtlen)
                            {
                                /* It's next message text in here already.. */
                                tmp[0] = '\0';
                            }
                            else if (num+10 > msg_txtlen)
                            {
                                tmp[msg_txtlen-num] = '\0';
                            }
                        }

                        if (tmp[0] != ' ' && tmp[0] != 13)
                        {
                            /* Check quoting */
                            for (num=0; num<QUOTE_MAX && tmp[num] != '\0'; num++)
                            {
                                if ((tmp[num] == '"') || (tmp[num] == '\'') || (tmp[num] == '<')) break;
                                if (tmp[num] == '>')
                                {
                                    num = 0;
                                    break;
                                }
                            }

                            if (num)
                            {
                                /* Origin, tagline, etc. */
                                if ((strncmp(tmp," * Origin:",10)) && (strncmp(tmp,"--- ",4)) && (strncmp(tmp,"... ",4)) && (strncmp(tmp,"~~~ ",4)) && (strncmp(tmp,"\xff@SUBJECT:",10)))
                                {
                                    if ((!origin) || (strncmp(tmp,"SEEN-BY:",8)))
                                    {
                                        buf[spos] = ' ';
                                        chars++; spos++;
                                        break;
                                    }
                                }
                            }
                        }
                    }
#endif

                    /*reformat = setup.justify != JUSTIFY_NONE;*/
                    no_quote = 0;

                    /* Remove trailing spaces */
                    while (chars > 0 && (buf[chars-1] == ' ' || buf[chars-1] == 0)) chars--;
                    buf[spos] = '\0';

                    if (spos == 3 && start && (strcmp(buf,"---") == 0 || strcmp(buf,"...") == 0 || strcmp(buf,"~~~") == 0))
                    {
                        /* Update line type */
                        msg_linerec->line_type = buf[0] == '.' ? LINE_TYPE_TAG : LINE_TYPE_TEAR;
                    }

                    /* Add line to linked list */

                    if (chars > 0) last_line = msg_txtlines;
                    msg_txtlines++;

                    /* Allocate memory for new line */
                    linep = (LINE_REC *) _malloc(sizeof(LINE_REC));

                    /* Set last line length and pointer to next line */
                    msg_linerec->len = (char) (chars+tens);
                    msg_linerec->next = linep;

                    if (linep == NULL) goto __end; /* Out of memory! */

                    linep->prev = msg_linerec;
                    linep->next = NULL;
                    linep->len = 0;
                    msg_linerec = linep;

                    if (origin && start && strncmp(buf," * Origin:",10) == 0)
                    {
                        /* This was origin line, save it for later use */
                        strcpy(msg_origin_line, buf);
                    }

                    msg_linerec->line_type = LINE_TYPE_NORMAL;
                    msg_linerec->ptr = msg_txtpos+totalread+ptr_pos+1;
                    chars = 0; tens = 0; spos = 0; start = 1;
                    break;
                case 141:
                    /* Soft-CR */
                    tens++;
                    break;
                default:
                    /* Normal character */
                    if ((chars >= line_length && reformat) || chars+1 >= user.ScreenWidth)
                    {
                        /* Too long line, split it */
                        if (msg_dat_buf[pos] == ' ')
                        {
                            /* Great, we can split it right here! */
                            spos = 0;

                            /* Remove trailing spaces */
                            while (chars > 0 && (buf[chars-1] == ' ' || buf[chars-1] == 0)) chars--;

                            if (chars > 0) last_line = msg_txtlines;
                            msg_txtlines++;

                            /* Allocate memory for new line */
                            linep = (LINE_REC *) _malloc(sizeof(LINE_REC));

                            /* Set last line length and pointer to next line */
                            msg_linerec->len = (char) (chars+tens);
                            msg_linerec->next = linep;

                            if (linep == NULL) goto __end; /* Out of memory! */

                            linep->prev = msg_linerec;
                            linep->next = NULL;
                            linep->line_type = LINE_TYPE_NORMAL;
                            linep->len = 0;
                            msg_linerec = linep;

                            msg_linerec->ptr = msg_txtpos+totalread+ptr_pos+1;
                            chars = 0; tens = 0; start = 0;
                            continue;
                        }

                        /* Search split position */
                        num = spos-1;
                        while (num > 0 && buf[num] != ' ') num--;

                        if (num > 0)
                        {
                            /* Found split character */
                            chars = num;

                            /* Remove trailing spaces */
                            while (chars > 0 && buf[chars-1] == ' ') chars--;

                            if (chars > 0) last_line = msg_txtlines;
                            msg_txtlines++;

                            /* Allocate memory for new line */
                            linep = (LINE_REC *) _malloc(sizeof(LINE_REC));

                            /* Set last line length and pointer to next line */
                            msg_linerec->next = linep;
                            msg_linerec->len = (char) (chars+tens);

                            if (linep == NULL) goto __end; /* Out of memory! */

                            linep->prev = msg_linerec;
                            linep->next = NULL;
                            linep->line_type = LINE_TYPE_NORMAL;
                            linep->len = 0;
                            msg_linerec = linep;

                            msg_linerec->ptr = msg_txtpos+totalread+ptr_pos+2-(spos-num+1);
                            chars = 0; tens = 0; start = 0;

                            memmove(buf,buf+num+1,spos-num); spos -= num+1;
                            chars = spos;
                        }
                        else
                        {
                            /* Couldn't find split pos so just split somewhere */
                            spos = 0;

                            /* Remove trailing spaces */
                            while (chars > 0 && buf[chars-1] == ' ') chars--;

                            if (chars>0) last_line = msg_txtlines;
                            msg_txtlines++;

                            /* Allocate memory for new line */
                            linep = (LINE_REC *) _malloc(sizeof(LINE_REC));

                            /* Set last line length and pointer to next line */
                            msg_linerec->next = linep;
                            msg_linerec->len = (char) (chars+tens);

                            if (linep == NULL) goto __end; /* Out of memory! */

                            linep->prev = msg_linerec;
                            linep->next = NULL;
                            linep->line_type = LINE_TYPE_NORMAL;
                            linep->len = 0;
                            msg_linerec = linep;

                            msg_linerec->ptr = msg_txtpos+totalread+ptr_pos;
                            chars = 0; tens = 0; start = 0;
                        }
                    }

                    /* Add to buffer */
                    buf[spos++] = msg_dat_buf[pos];
                    chars++;

                    /* Check origin */
                    if (spos == 10 && start && strncmp(buf," * Origin:",10) == 0)
                    {
                        msg_linerec->line_type = LINE_TYPE_ORIGIN;
                        origin = 1;
                        reformat = 0;
                    }

                    if (spos == 1 && start && msg_dat_buf[pos] == ' ')
                    {
                        /* Line starts with space, don't reformat this line */
                        reformat = 0;
                    }
                    if (spos == 4 && start && (strncmp(buf,"--- ",4) == 0 || strncmp(buf,"... ",4) == 0 || strncmp(buf,"~~~ ",4) == 0))
                    {
                        /* Tear or tag, don't reformat this line */
                        reformat = 0;
                        msg_linerec->line_type = buf[0] == '.' ? LINE_TYPE_TAG : LINE_TYPE_TEAR;
                    }
                    if (msg_linerec->line_type == LINE_TYPE_NORMAL && (msg_dat_buf[pos] == '_' || msg_dat_buf[pos] == '*'))
                    {
                        msg_linerec->line_type = LINE_TYPE_SPECIAL;
                    }
                    if (spos < QUOTE_MAX && !no_quote && start)
                    {
                        if (msg_dat_buf[pos] == '"' || msg_dat_buf[pos] == '\'' || msg_dat_buf[pos] == '<') no_quote = 1;
                        if (msg_dat_buf[pos] == '>')
                        {
                            /* Quoted line, don't reformat */
                            reformat = 0;
                            msg_linerec->line_type = LINE_TYPE_QUOTE;
                        }
                    }
            }
        }
    }

    /* Remove trailing spaces */
    while (chars > 0 && buf[chars-1] == ' ') chars--;

    if (chars > 0) last_line = msg_txtlines;
    msg_linerec->len = (char) (chars+tens);
    msg_txtlines++;

    if (spos == 3 && start && (strcmp(buf,"---") == 0 || strcmp(buf,"...") == 0 || strcmp(buf,"~~~") == 0))
    {
        /* Update line type */
        msg_linerec->line_type = buf[0] == '.' ? LINE_TYPE_TAG : LINE_TYPE_TEAR;
    }

__end:
    msg_txtlines = last_line+1;

    msg_linerec = msg_first_linerec;
    msg_current_line = 1;
    return 1;
}

void msg_deinit_msgtext(void)
{
    while (msg_first_linerec != NULL)
    {
        msg_linerec = msg_first_linerec->next;
        _free(msg_first_linerec);
        msg_first_linerec = msg_linerec;
    }
}

char *msg_read_textline(unsigned long line)
{
    int nro,slen;
    long maxsize;

    /* Go to right position */
    while (line < msg_current_line)
    {
        if (msg_linerec->prev == NULL)
        {
            msgtext[0] = '\0';
            return msgtext;
        }
        msg_linerec = msg_linerec->prev;
        msg_current_line--;
    }

    while (line > msg_current_line)
    {
        if (msg_linerec->next == NULL)
        {
            msgtext[0] = '\0';
            return msgtext;
        }
        msg_linerec = msg_linerec->next;
        msg_current_line++;
    }

    /* Read from file */
    if ((msg_dat_start > msg_linerec->ptr) || (msg_dat_end < msg_linerec->ptr+msg_linerec->len))
    {
        FileBufSeek(msg_Fjdt, msg_linerec->ptr, SEEK_SET);
        msg_dat_start = msg_linerec->ptr;
        if (BUF_SIZE > msg_txtlen)
            maxsize = msg_txtlen;
        else
            maxsize = BUF_SIZE;
        msg_dat_end = msg_dat_start+FileBufRead(msg_Fjdt, (void **) &msg_dat_buf, (unsigned) maxsize)-1;
    }
    memcpy(msgtext,msg_dat_buf+msg_linerec->ptr-msg_dat_start,msg_linerec->len);
    msgtext[msg_linerec->len] = '\0';

    nro = 0;
    slen = msg_linerec->len;

    while (nro < slen)
    {
        switch ((unsigned char) msgtext[nro])
        {
            case 13:
                msgtext[nro] = 32;
                break;
            /*case 141:
                if (opt.rflags & TREAT_I_NORMAL) break;*/
            case 10:
                memmove(msgtext+nro,msgtext+nro+1,slen-nro-1);
                slen--;
                nro--;
                break;
        }
        nro++;
    }
    while ((slen > 0) && (msgtext[slen-1] == 32)) slen--;
    msgtext[slen] = '\0';

    return msgtext;
}

unsigned msg_read_msgtext(unsigned long msg_txtpos, char **buffer, unsigned bufsize)
{
    unsigned to_read;

    to_read = (unsigned) (msg_txtlen-msg_txtpos);
    if (to_read == 0) return 0;
    if (to_read > bufsize) to_read = bufsize;

    return FileBufRead(msg_Fjdt, (void **) buffer, to_read);
}

int msg_update_message(unsigned long num)
{
    JAMHDRINFO *jamhdr;
    JAMIDXREC *jamidx;
    JAMHDR *msgrec;

    /* Lock message base */
    if (!FileLock(msg_Fjhr->handle))
    {
        write_log("update_message() : Can't lock message base");
        return 0;
    }

    /* Read message base header */
    FileBufSeek(msg_Fjhr, 0, SEEK_SET);
    if (FileBufRead(msg_Fjhr, (void **) &jamhdr, sizeof(*jamhdr)) != sizeof(*jamhdr))
    {
        write_log("Can't read message header");
        FileUnlock(msg_Fjhr->handle);
        return 0;
    }

    /* Read index */
    FileBufSeek(msg_Fjdx, (num-1)*sizeof(*jamidx), SEEK_SET);
    if (FileBufRead(msg_Fjdx, (void **) &jamidx, sizeof(*jamidx)) != sizeof(*jamidx))
    {
        write_log("Can't read message index");
        FileUnlock(msg_Fjhr->handle);
        return 0;
    }

    /* Read message header */
    FileBufSeek(msg_Fjhr, jamidx->HdrOffset, SEEK_SET);
    if (FileBufRead(msg_Fjhr, (void **) &msgrec, sizeof(*msgrec)) != sizeof(*msgrec)) msgrec->MsgNum = 0;

    /* Update flags */
    if (msg_flags & MSG_FLAG_PRIVATE) msgrec->Attribute |= MSG_PRIVATE;
    if (msg_flags & MSG_FLAG_SENT) msgrec->Attribute |= MSG_SENT;
    if (msg_flags & MSG_FLAG_RECEIVED) msgrec->Attribute |= MSG_READ;
    if (msg_flags & MSG_FLAG_DELETED) msgrec->Attribute |= MSG_DELETED;

    /* Write message header */
    FileBufSeek(msg_Fjhr, jamidx->HdrOffset, SEEK_SET);
    if (FileBufWrite(msg_Fjhr, msgrec, sizeof(*msgrec)) != sizeof(*msgrec))
    {
        write_log("Can't write message header");
        FileBufUnlock(msg_Fjhr);
        return 0;
    }

    /* Update ModCounter */
    jamhdr->ModCounter++;
    FileBufSeek(msg_Fjhr, 0, SEEK_SET);
    FileBufWrite(msg_Fjhr, jamhdr, sizeof(*jamhdr));

    msg_messages = (unsigned) (FileBufSeek(msg_Fjdx, 0, SEEK_END)/sizeof(*jamidx));

    /* Unlock message base */
    FileUnlock(msg_Fjhr->handle);

    return 1;
}
