// Filename:   menu.h
// Contents:   the menu system object
// Author: Greg Shaw
// Created:    7/22/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _MENU_H_
#define _MENU_H_

#include "bbshdr.h"             // file that includes everything

// a couple of key transfer identifiers (for out-of-band transmission)
#define    HIT_RETURN  -1       // he hit return in a menu
#define    HIT_WRONG_CHAR  -2   // he hit a key, but it wasn't valid
#define HIT_NOTHING    -3       // he hit nothing

extern User user;               // need this for access level, flags, timelimit, etc.
extern Chat chatobj;		// chat calls

// Structure:  MenuItem
// Contents:   the pertinent information for each menu item.
//     command type (number)
//     misc text
//     access level
//     access level modifier
//     flags
//     flags modifier
// Author: Greg Shaw
// Created:    7/22/93

typedef struct menuitem
{
	int    com_type;            // command type
	char   key;                 // hot key
	char   misc[256];           // miscellaneous text
	int    acl;                 // access level of command
	char   acl_mod;             // access level modifier
	int    flags;               // flags (32)
	char   flags_mod;           // flags modifier
	struct menuitem *next;      // next item in list
} MenuItem;


// Structure:  Menuline
// Contents:   a single line of menu text
// Author: Greg Shaw
// Created:    1/2/95

typedef struct menuline
{
	char   *line;               // line of text
	struct menuline    *next;   // next item in list
} MenuLine;


// Structure:      MacroDef
// Contents:       a macro definition
// Author:     Greg Shaw
// Created:        1/2/95

typedef struct macrodef
{
	char   id;                  // macro identifier (char)
	char   *macro;              // macro definition
	struct macrodef    *next;   // next item in list
} MacroDef;


// Object: Menu
// Purpose:    Encapsulate the menu operations for the BBS.  Everything
//     relating to how the bbs (aka user) operates the menu system
//     is contained within this object.
// Attributes: path - path to menu
//         1..n of struct item (see above for item description)
// Methods:    open - open a menu file (read commands from file)
//     run - run the (already read) menu.  Return selected command
//         number
//     dump - dump contents of menu object to screen (for debugging)
// Author: Greg Shaw
// Created:    7/22/93

class Menu:public language      // inherited from bbs interface
{
	// path to menu
	char	menu_path[MAXPATHLEN+1];
	time_t	user_logon;      // time of user logon (for timeleft prompt)
	MacroDef	*macros;         // macro list
	MenuItem	*items;      // menu items
	MenuLine	*lines;          // lines of text for menu
	int	macro_sub(char **line);     // do macro substitution
	int	clean_storage(void);        // delete dynamic memory
	void	dump();                // dump menu object contents
	public:
		Menu();                 // constructor
		~Menu();                // destructor
		// build menu from file(s)
	int	build(char *path, int top);
	MenuItem	*run(void);    // run the menu
	char	valid_char(void);  // get char (if available) and check against items
};


#endif                          // _MENU_H_






