// Filename:	door.C
// Contents:	the doorsys (switch information) file read/write structures
// Author:	Gregory Shaw
// Created:	6/15/95

#ifndef _DOOR_C_
#define _DOOR_C_

#include "bbshdr.h"

extern User user;		// user interface functions

// Function:	bbs_read_doorsys
// Purpose:	read the door information from the external application
// Input:	<doorsys file>
// Output:	the user object is updated if something has changed in a 
//			writable area.  (timelimit, etc)
// Author:	Gregory Shaw
// Created:	6/20/95

int Door::bbs_read_doorsys(void)
{
	FILE	*doorfile;
	char	tmpstr[255];
	char	color[100];
	char    *ptr;
	int	x;
	int	dumx;
	CardRec *ucard;
	unsigned int	udumx;

	sprintf(tmpstr,"%s/spool/door/%s.doorsys",getenv("BBSDIR"),user.logname());
	if (doorfile = bopen(tmpstr,"r"), doorfile != NULL)
	{
		// ok.  got the file.  Now parse the dumb thing
		// skip the first 14 lines (they're read-only)
		for (x=0; x<14; x++)
			skipline(doorfile);
		// ACL
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%d",&udumx) == 1)
		{
			if (udumx != user.u_acl())
			{
				user.change_acl(dumx,1);	// permanently change ACL
				user.change_acl(dumx,0);	// permanently change ACL
				sprintf(tmpstr,"door: Updated access (%d) for %s",dumx,user.logname());
				ap_log(tmpstr);
			}
		}
		// skip next five
		for (x=0; x<5; x++)
			skipline(doorfile);
		// pagelength
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%d",&dumx) == 1)
		{
			if (dumx != user.lin())
			{
				user.change_lines(dumx);	
				// permanently change lines
				sprintf(tmpstr,"door: Updated lines (%d) for %s",dumx,user.logname());
				ap_log(tmpstr);
			}
		}
		// skip next three
		for (x=0; x<3; x++)
			skipline(doorfile);
		// expdate (skip for now -- it will be added later)
			skipline(doorfile);
		// skip next two
		for (x=0; x<2; x++)
			skipline(doorfile);
		// uploads
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%d",&dumx) == 1)
		{
			if (dumx != user.u_uploads())
			{
				user.change_ul(dumx);	// change uploads
				sprintf(tmpstr,"door: Updated uploads (%d) for %s",dumx,user.logname());
				ap_log(tmpstr);
			}
		}
		// downloads
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%d",&dumx) == 1)
		{
			if (dumx != user.u_downloads())
			{
				user.change_dl(dumx);	// change downloads
				sprintf(tmpstr,"door: Updated downloads (%d) for %s",dumx,user.logname());
				ap_log(tmpstr);
			}
		}
		// daily_K and daily_k_max (not used yet - skip)
		for (x=0; x<2; x++)
			skipline(doorfile);
		// skip next 4
		for (x=0; x<4; x++)
			skipline(doorfile);
		// alias name
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%s",tmpstr) == 1)
		{
			if (strcmp(tmpstr,user.user_alias()))
			{	// different, change
				user.change_alias(tmpstr);	// change alias
				sprintf(tmpstr,"door: Updated alias (%s) for %s",tmpstr,user.logname());
				ap_log(tmpstr);
			}
		}
		// skip next 2	
		for (x=0; x<2; x++)
			ptr = scanline(doorfile);
		// ansi
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%s",tmpstr) == 1)
		{
			if (strchr(tmpstr,'Y') != NULL)
			{
				if (strcmp(user.terminal(),"ansi"))
				{ // not ansi - set to ansi
					user.change_terminal("ansi");
					sprintf(tmpstr,"door: Updated terminal for %s",user.logname());
					ap_log(tmpstr);
				}
			}
			// Note: I'm only changing it if they've selected
			// ansi within the external.  If not, it is not
			// updated (which could be a problem)
		}
		// skip next 2
		for (x=0; x<2; x++)
			skipline(doorfile);
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%d",&dumx) == 1)
		{
			// this could result in an error (credit drops due
			// to a time bank, for instance)
			if (dumx > user.credit())
			{
				user.inc_credit(dumx-user.credit());// change credit
				sprintf(tmpstr,"door: Updated credited time (%d) for %s",dumx,user.logname());
				ap_log(tmpstr);
			}
		}
		// skip next 3
		for (x=0; x<3; x++)
			skipline(doorfile);
		// maxfilesday (skipped for now)
			skipline(doorfile);
		// filestoday	(skipped for now)
			skipline(doorfile);
		// upload_k
			skipline(doorfile);
		// download_k
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%d",&udumx) == 1)
		{
			if (udumx != user.kused)
			{
				user.kused = udumx;	// change  kused
				sprintf(tmpstr,"door: Updated kused (%d) for %s",udumx,user.logname());
				ap_log(tmpstr);
			}
		}
		// command (color:editor)
		ptr = scanline(doorfile);
		if (sscanf(ptr,"%s:%s",color,tmpstr) == 2)
		{
			ucard = user.usercard();
			if (strcmp(color,ucard->colr))
			{	// different color
				strcpy(ucard->colr,color);
				sprintf(tmpstr,"door: Updated card (%s) for %s",color,user.logname());
				ap_log(tmpstr);
			}
			if (strcmp(tmpstr,user.editorname()))
			{	// different editor
				user.change_editor(tmpstr);
				sprintf(tmpstr,"door: Updated editor (%s) for %s",tmpstr,user.logname());
				ap_log(tmpstr);
			}
		}
		// skip next two
		for (x=0; x<2; x++)
			skipline(doorfile);
		// all done!

	}
	else
	{
		ap_log("Unable to open doorsys file for read.");
		return(-1);
	}
	bclose(doorfile);
	delete_doorsys();	// delete file
	return(0);
}
// Function:	delete_doorsys
// Purpose:	delete the doorsys file
// Input:	none
// Output:	file is deleted (errors ignored)
// Author:	Gregory Shaw
// Created:	6/22/95

int Door::delete_doorsys(void)
{
	char tmpstr[255];

	sprintf(tmpstr,"%s/spool/door/%s.doorsys",getenv("BBSDIR"),user.logname());
	if (unlink(tmpstr))
	{
		sprintf(tmpstr,"Unable to delete doorsys for %s",user.logname());
		ap_log(tmpstr);
	}
	return(0);
}


// Function:	init_door
// Purpose:	init the object
// Input:	input from system and user
// Output:	info is initialized
// Author:	Gregory Shaw
// Created:	6/15/95

int Door::init_door(void)
{
        struct tm *tmdata;
	CardRec *usercard;
	time_t	now;

	strcpy(info.comport,"COM0:");	// always local (no funky processing)
	info.baud = 38400;		// no way to tell on locked modems
	info.parity = 8;		// this isn't especially smart
					// but I don't know of anybody that
					// uses 7bit anymore.
	
	info.node = nodenum();
	info.DTErate = 38400;	// no way to tell on locked modems
	info.screen = 'Y';	// always use screen
	info.printer = 'N';	// no printer
	info.pagebell = 'N';	// bell doesn't work on console 
	info.calleralarm = 'N';	// I have no idea what this is
	sprintf(info.fullname,"%s %s",user.fname,user.lname);
	user.callingfrom(info.callfrom);	// get location information
	strcpy(info.homephone,"000-000-0000");	// << will fill in later
	strcpy(info.workphone,user.terminal()); 
	strcpy(info.password,user.user_lang());
	info.seclevel = user.u_acl();
	info.timeson = user.u_logins();
	now = user.last_logon();
	tmdata = localtime(&now);
	sprintf(info.lastcall,"%.2d/%.2d/%.2d",tmdata->tm_mon,tmdata->tm_mday,tmdata->tm_year);
	info.secsremaining = user.timeleft()*60;
	info.minsremaining = user.timeleft();
	strcpy(info.graphicmode,"GR");
	info.pagelength = user.lin();
	info.usermode = 'N';	// novice mode
	strcpy(info.confs,"0");	// no conferences
	info.confexit = 0;	// no conference exit
	now = user.account_expiration();
	tmdata = localtime(&now);
	sprintf(info.expdate,"%.2d/%.2d/%.2d",tmdata->tm_mon,tmdata->tm_mday,tmdata->tm_year);
	info.recnum = 1;	// not supported
	info.defprotocol = 'Z';	// zmodem default
	info.uploads = user.u_uploads();
	info.downloads = user.u_downloads();
	usercard = user.usercard();
	info.daily_k = user.kused;	// max k per day
	info.daily_k_max = usercard->kbytes;
	strcpy(info.birth,"00/00/00");	// not used (yet)
	strcpy(info.mainpath,getenv("BBSDIR"));	// main bbs directory 
	strcpy(info.genpath,"none");	// not used? (I don't know what this is)
	strcpy(info.sysopname,sysop());
	strcpy(info.aliasname,user.user_alias());
	strcpy(info.eventtime,"00:00");	// no event, just a fake out
	info.ecc = 'Y';	// usually error correcting...
	if (!strcmp(user.terminal(),"ansi"))
		info.ansi = 'Y';
	else
		info.ansi = 'N';
	info.recordlock = 'Y';
	info.defaultcolor = def_foreground();
	info.credit = user.credit();
	strcpy(info.newfiles,info.lastcall); 
	now = user.user_logon();
	tmdata = localtime(&now);
	sprintf(info.calltime,"%.2d:%.2d",tmdata->tm_hour,tmdata->tm_min); 
	now = user.last_logon();
	tmdata = localtime(&now);
	sprintf(info.lastcalltime,"%.2d:%.2d",tmdata->tm_hour,tmdata->tm_min); 
	info.maxfilesday = usercard->max_files;
	info.filestoday = user.u_downloads();	// this may need to go to 0
	info.upload_k = 0;
	info.download_k = user.kused;
	sprintf(info.comment,"%s:%s",usercard->colr,user.editorname());
	info.doorsopened = 0;
	info.messagesleft = 0;
	return(0);
};

// Function:	nodenum
// Purpose:	compute the node number from the tty name
// Input:	none
// Output:	a node number
// Author:	Gregory Shaw
// Created:	6/15/95
// Algorithm:	
//		if (ttyname = ttyS*)
//			nodenum = *;
//		else	(pty)
//			nodenum = (tty* - ttyp0) * 64 + (ttyX# - ttyX) 

int Door::nodenum(void)
{
	char ttystr[50];
	char *c,*s;
	int node;

	s = 0;
	strcpy(ttystr,ttyname(fileno(stdout)));
        while (c = strchr(ttystr,'/'), c != NULL)
        {
                s = c+1;
                *c = ' ';
        }
	if (strchr(s,'S') != NULL)	// standard dialup
	{
		sscanf(&s[4],"%d",&node);
	}
	else // pty
	{
		sscanf(&s[4],"%d",&node);
		node += (s[3] - 'p') * 64+64;
	}
	return(node);
}

// Function:	read_doorsys
// Purpose:	read the door information from the external application
// Input:	<doorsys file>
// Output:	
//			writable area.  (timelimit, etc)
// Author:	Gregory Shaw
// Created:	6/20/95

int Door::read_doorsys(void)
{
	FILE	*doorfile;
	char	tmpstr[255];

	sprintf(tmpstr,"%s/spool/door/%s.doorsys",getenv("BBSDIR"),user.logname());
	if (doorfile = bopen(tmpstr,"r"), doorfile != NULL)
	{
		fscanf(doorfile,"%s\r\n",info.comport);
		fscanf(doorfile,"%d\r\n",&info.baud);
		fscanf(doorfile,"%d\r\n",&info.parity);
		fscanf(doorfile,"%d\r\n",&info.node);
		fscanf(doorfile,"%d\r\n",&info.DTErate);
		fscanf(doorfile,"%c\r\n",&info.screen);
		fscanf(doorfile,"%c\r\n",&info.printer);
		fscanf(doorfile,"%c\r\n",&info.pagebell);
		fscanf(doorfile,"%c\r\n",&info.calleralarm);
		// eat the whole line as input
		strcpy(info.fullname,scanline(doorfile));
		strcpy(info.callfrom,scanline(doorfile));
		fscanf(doorfile,"%s\r\n",info.homephone);
		fscanf(doorfile,"%s\r\n",info.workphone);
		fscanf(doorfile,"%s\r\n",info.password);
		fscanf(doorfile,"%d\r\n",&info.seclevel);
		fscanf(doorfile,"%d\r\n",&info.timeson);
		fscanf(doorfile,"%s\r\n",info.lastcall);
		fscanf(doorfile,"%ld\r\n",&info.secsremaining);
		fscanf(doorfile,"%d\r\n",&info.minsremaining);
		fscanf(doorfile,"%s\r\n",info.graphicmode);
		fscanf(doorfile,"%d\r\n",&info.pagelength);
		fscanf(doorfile,"%c\r\n",&info.usermode);
		fscanf(doorfile,"%s\r\n",info.confs);
		fscanf(doorfile,"%d\r\n",&info.confexit);
		fscanf(doorfile,"%s\r\n",info.expdate);
		fscanf(doorfile,"%d\r\n",&info.recnum);
		fscanf(doorfile,"%c\r\n",&info.defprotocol);
		fscanf(doorfile,"%d\r\n",&info.uploads);
		fscanf(doorfile,"%d\r\n",&info.downloads);
		fscanf(doorfile,"%d\r\n",&info.daily_k);
		fscanf(doorfile,"%d\r\n",&info.daily_k_max);
		fscanf(doorfile,"%s\r\n",info.birth);
		fscanf(doorfile,"%s\r\n",info.mainpath);
		fscanf(doorfile,"%s\r\n",info.genpath);
		fscanf(doorfile,"%s\r\n",info.sysopname);
		fscanf(doorfile,"%s\r\n",info.aliasname);
		fscanf(doorfile,"%s\r\n",info.eventtime);
		fscanf(doorfile,"%c\r\n",&info.ecc);
		fscanf(doorfile,"%c\r\n",&info.ansi);
		fscanf(doorfile,"%c\r\n",&info.recordlock);
		fscanf(doorfile,"%d\r\n",(int *)&info.defaultcolor);
		fscanf(doorfile,"%d\r\n",&info.credit);
		fscanf(doorfile,"%s\r\n",info.newfiles);
		fscanf(doorfile,"%s\r\n",info.calltime);
		fscanf(doorfile,"%s\r\n",info.lastcalltime);
		fscanf(doorfile,"%d\r\n",&info.maxfilesday);
		fscanf(doorfile,"%d\r\n",&info.filestoday);
		fscanf(doorfile,"%d\r\n",&info.upload_k);
		fscanf(doorfile,"%d\r\n",&info.download_k);
		fscanf(doorfile,"%s\r\n",info.comment);
		fscanf(doorfile,"%d\r\n",&info.doorsopened);
		fscanf(doorfile,"%d\r\n",&info.messagesleft);
		// all done!
	}
	else
	{
		ap_log("Unable to open doorsys file for read.");
		return(-1);
	}
	bclose(doorfile);
	return(0);
}

// Function:	scanline
// Purpose:	scan an entire line into a buffer
// Input:	file - the file pointer for the file
// Output:	(none)
// Author:	Gregory Shaw
// Created:	7/8/95

char *Door::scanline(FILE *infile)
{
	static char tmpstr[256];	// buffer storage
	char *c;


	c = tmpstr;
	while (*c = fgetc(infile), !iseol(*c) && !feof(infile))
		c++;
	fgetc(infile);	// skip \r right after \n (for compatability)
	*c = 0;
	return(tmpstr);
}

// Function:	skipline
// Purpose:	skip a line in the doorsys file.
// Input:	doorsys file
// Output:	(none)
// Author:	Gregory Shaw
// Created:	6/22/95

int Door::skipline(FILE *infile)
{
	register char c;


	while (c = fgetc(infile), !iseol(c) && !feof(infile));
	c = fgetc(infile);	// skip \r right after \n (for compatability)
	return(feof(infile));
}

// Function:	write_doorsys
// Purpose:	write the doorway information to the doorsys file
// Input:	door information
//		init - whether the door information record has been
//			initialized 
// Output:	the doorsys file
// Author:	Gregory Shaw
// Created:	6/22/95

int Door::write_doorsys(int init)
{
	FILE	*doorfile;
	char	tmpstr[255];


	if (!init)
		init_door();	// setup information
	sprintf(tmpstr,"%s/spool/door/%s.doorsys",getenv("BBSDIR"),user.logname());
	if (doorfile = bopen(tmpstr,"w"), doorfile != NULL)
	{
		// now dump the info object to a file
		fprintf(doorfile,"%s\r\n",info.comport);
		fprintf(doorfile,"%d\r\n",info.baud);
		fprintf(doorfile,"%d\r\n",info.parity);
		fprintf(doorfile,"%d\r\n",info.node);
		fprintf(doorfile,"%d\r\n",info.DTErate);
		fprintf(doorfile,"%c\r\n",info.screen);
		fprintf(doorfile,"%c\r\n",info.printer);
		fprintf(doorfile,"%c\r\n",info.pagebell);
		fprintf(doorfile,"%c\r\n",info.calleralarm);
		fprintf(doorfile,"%s\r\n",info.fullname);
		fprintf(doorfile,"%s\r\n",info.callfrom);
		fprintf(doorfile,"%s\r\n",info.homephone);
		fprintf(doorfile,"%s\r\n",info.workphone);
		fprintf(doorfile,"%s\r\n",info.password);
		fprintf(doorfile,"%d\r\n",info.seclevel);
		fprintf(doorfile,"%d\r\n",info.timeson);
		fprintf(doorfile,"%s\r\n",info.lastcall);
		fprintf(doorfile,"%ld\r\n",info.secsremaining);
		fprintf(doorfile,"%d\r\n",info.minsremaining);
		fprintf(doorfile,"%s\r\n",info.graphicmode);
		fprintf(doorfile,"%d\r\n",info.pagelength);
		fprintf(doorfile,"%c\r\n",info.usermode);
		fprintf(doorfile,"%s\r\n",info.confs);
		fprintf(doorfile,"%d\r\n",info.confexit);
		fprintf(doorfile,"%s\r\n",info.expdate);
		fprintf(doorfile,"%d\r\n",info.recnum);
		fprintf(doorfile,"%c\r\n",info.defprotocol);
		fprintf(doorfile,"%d\r\n",info.uploads);
		fprintf(doorfile,"%d\r\n",info.downloads);
		fprintf(doorfile,"%d\r\n",info.daily_k);
		fprintf(doorfile,"%d\r\n",info.daily_k_max);
		fprintf(doorfile,"%s\r\n",info.birth);
		fprintf(doorfile,"%s\r\n",info.mainpath);
		fprintf(doorfile,"%s\r\n",info.genpath);
		fprintf(doorfile,"%s\r\n",info.sysopname);
		fprintf(doorfile,"%s\r\n",info.aliasname);
		fprintf(doorfile,"%s\r\n",info.eventtime);
		fprintf(doorfile,"%c\r\n",info.ecc);
		fprintf(doorfile,"%c\r\n",info.ansi);
		fprintf(doorfile,"%c\r\n",info.recordlock);
		fprintf(doorfile,"%d\r\n",info.defaultcolor);
		fprintf(doorfile,"%d\r\n",info.credit);
		fprintf(doorfile,"%s\r\n",info.newfiles);
		fprintf(doorfile,"%s\r\n",info.calltime);
		fprintf(doorfile,"%s\r\n",info.lastcalltime);
		fprintf(doorfile,"%d\r\n",info.maxfilesday);
		fprintf(doorfile,"%d\r\n",info.filestoday);
		fprintf(doorfile,"%d\r\n",info.upload_k);
		fprintf(doorfile,"%d\r\n",info.download_k);
		fprintf(doorfile,"%s\r\n",info.comment);
		fprintf(doorfile,"%d\r\n",info.doorsopened);
		fprintf(doorfile,"%d\r\n",info.messagesleft);
	}
	else
	{
		ap_log("Unable to open doorsys file for write.");
		return(-1);
	}
	bclose(doorfile);
	return(0);
}

#endif // _DOOR_C_
