// Filename:   chat.h
// Contents:   The information and object definitions for the chat subsystem
// Author: Gregory Shaw
// Created:    4/9/95

#ifndef _CHAT_H_
#define  _CHAT_H_

#include "bbshdr.h"

// miscellaneous defines
                                // disconnect port after 10 seconds
#define PORT_EXPIRATION_TIME   10
#define CHECK_BROADCAST_TIME   5// check for broadcasts every 5 seconds
                                // give up on private chat after 25 seconds
#define PRIVATE_TIMEOUT        30
#define NO_BROADCAST   "none"   // no broadcast message
#define NO_INFO        "none"   // no information message
#define BROADCAST_POLL "poll"   // poll for message
                                // attention message for private chat
#define SPECIAL_BROADCAST "Aa1133tz"
#define PRIVATE_YES "at8yes28"  // 'yes I'll chat'
#define PRIVATE_NO "at8no28"    // 'no way, hoser'
#define MAX_ROOM_NAME  31
#define MAX_ACL_LIST  32	// 32 names max in an ACL
#define MAX_COLOR_TIME	60*5	// any color pair not used in 5 minutes is toast

// uncomment below for normal (no lines between user output)
// #define CHAT_LINEFEED	"\n\r"

// uncomment below for double spaced output for each user in chat
#define CHAT_LINEFEED	"\n\r\n\r"	// should chat output be double spaced?

// Structure:  RoomName
// Contents:   data structure used to associate room numbers with names
// Author:     Gregory Shaw
// Created:    4/23/95

typedef struct room
{
	int room;                   // room number
	char name[MAX_ROOM_NAME];   // name of room
} RoomName;


// Structure:  BCUser
// Contents:   data structure used to track who has received a broadcast
// Author:     Gregory Shaw
// Created:    4/18/95

typedef struct bcuser
{
                                // user
	char login[MAX_LOGIN_LENGTH];
	struct bcuser *next;        // next
} BCUser;


// Structure:  BroadCast
// Contents:   data structure used to track broadcasts
// Author:     Gregory Shaw
// Created:    4/17/95

typedef struct broad
{
	char   *to;                 // message recipient
	char   *msg;                // broadcast message
	BCUser *sent;               // list of users who have received broadcast
	time_t when_received;       // when message received
	struct broad *next;         // next guy
} BroadCast;


// Structure:  ClientInfo
// Contents:   data structures used in the daemon to track chat usage
// Author: Gregory Shaw
// Created:    4/9/95

typedef struct clientinfo
{
	long   mess_sent;           // messages sent by this user
	time_t connected;           // date the user connected
	char   online;              // is port online and ready?
                                // login name
	char   login_name[MAX_LOGIN_LENGTH];
                                // alias
	char   alias[MAX_ALIAS_LENGTH];
	int    socket;              // what socket number?
	bbsipc *port;               // port assigned to client
	struct clientinfo *next;    // next client in list
} ClientInfo;


// Structure:  RoomInfo
// Contents:   room status information
// Author: Gregory Shaw
// Created:    4/9/95

typedef struct
{
	int    room;                // room number
	int    users;               // number of users
	int    movement;            // how many comings/goings?
	float  avg_participants;    // avg number of participants
	float  avg_lurkers;         // avg number of non-participants
	char   *output_message;     // output message
                                // target person (private)
	char   target[MAX_LOGIN_LENGTH];
	ClientInfo *list;           // client list
} RoomInfo;



// Structure:   colorpair
// Purpose:     Serve as a structure for tracking the color pairs in use
//              while in chat
// Author:      Greg Shaw
// Cretaed:     9/21/95

struct colorpair
{
        short   fore;   // foreground
        short   back;   // background
        short   pair;   // pair (used by ncurses)
        time_t  used;   // when last used
        char    cpair;  // quick checking number
        struct colorpair *next; // next in list
};

typedef struct colorpair ColorPair;





#ifndef _SERVER_                // servers don't know squat about BBS objects

// Object: chat
// Purpose:    encapsulate the client chat interface
// Attributes: chat_room - room currently connected to
//         kill_list - list of users that this user doesn't want to
//         hear from
//             broadcast_msg - storage for a broadcast message
// chatipc - chat server interface object
// broadipc - broadcast interface object
// Methods:    admin_info - return administration information
//     broadcast - send a broadcast to all BBS users
//     check_broadcast - check for a BBS broadcast
//     club_connect - connect to a club (checking password, as
//         appropriate)
//     disconnect - disconnect from the current room
//     dynamic_connect - connect to a dynamically created room
//     edit_kill - edit the chat kill file
//     private - connect to a person for private chat
//     room_connect - connect to a room
//     room_create - create a room
//     room_info - information on all rooms
//     room_users - list the users in a particular room
// Author: Gregory Shaw
// Created:    4/9/95

class Chat: public language
{
public:
	int    chat_room;           // room connected to
	int    room_names;          // number of room names available
                                // users we don't like
	short	num_pairs;	// number of color pairs
	char   *killusers[MAX_CHAT_KILL_USERS];
                                // broadcast from the SysOp
	char   broadcast_message[255];
	bbsipc chatipc;             // socket to server
	bbsipc broadipc;            // socket to server
	bbsipc pchatipc;            // socket to server
                                // names and numbers of rooms
	ColorPair *pair_list;	// list of color pairs
	RoomName roomnames[MAX_CHAT_ROOMS];
//public:
	Chat();                 // constructor
	~Chat();                // destructor
                                // display administration info
	void   admin_info(int server);
                                // check for private request
	int check_private(char *msg, int in_curses, int respond);
                                // send broadcast message
	int    broadcast(int oneuser);
                                // send broadcast message
	int    broadcast_direct(char *login, char *msg);
                                // change club password
	int change_club_password(int room);
	int chatting(void);         // chat user interface
                                // check for broadcast
	char   *check_broadcast(void);
                                // connect to a club room
	void  clear_color_list(void);	// nuke the color pair list
	int   club_connect(int room);
                                // return line from clublist
	char    *club_line(int club);
	void    disconnect(void);   // disconnect from chat system
                                // connect to a dynamically created room
	void    dynamic_connect(void);
	void    edit_kill(void);    // edit the kill file
                                // connect to another person privately
	short	find_color_pair(short fore, short back);	// find a particular color pair
	int    private_connect(char *name, int sysop);
                                // connect to pchat server
	int    private_room_connect(char *name);
	int    read_kill_file(void);// read the kill file
	// connect to a room
	int    room_connect(int room);
	int    room_create(void);   // create a new room
	int    room_info(void);     // display information on all rooms
	int    room_users(int room);// display information about the
	int    save_kill_file(void);// save the kill file
	// current room
};


#endif                          // _SERVER_


#endif                          // _CHAT_H_




