// Filename:   bbsinfo.C
// Contents:   the methods for the bbsinfo object
// Author:     Greg Shaw
// Created:        7/11/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _BBSINFO_C_
#define _BBSINFO_C_

#include "bbshdr.h"

// Function:	constructor
// Purpose:	initialize the bbsinfo cache
// Input:	none
// Output:	all cache information is initialized
// Author:	Greg Shaw
// Created:	1/1/96

bbsinfo::bbsinfo()
{
	cmail = -1;          // checkmail
	cchat = -1;          // credit chat
	cuploads = -1;       // credit uploads
	dcard = -1;          // default card color
	dfore = -1;          // default foreground
	dback = -1;          // default background
	dattr = -1;          // default attribute
	itime = -1;          // inactivity timeout
	lmaccess = -1;       // log menu accesses?
	lmext = -1;          // log externals?
	mcheck = -1;         // mail check time 
	mfiles = -1;         // max files per batch
	slast = -1;          // show last 5 users to log on?
	stime = -1;          // show timeleft in prompt?
	scoms = -1;          // show commands in prompt?
	sfor = -1;           // show fortune on logon?
	tlfud = -1;          // timelimit fudge?
	waitt = -100;          // wait time
	logmaili = -1;		// log mail import?
	mfor = -1;
	mk = 1;     // maxk downloaded per session
	mkd = 1;    // maxk downloaded per day
	rato = -1000.0;           // download/upload ratio
	sys = NULL;           // logon name of sysop
	spager = NULL;        // system pager name
	talkp = NULL;         // talk program name
	dterm = NULL;         // default terminal type
	mspool = NULL;        // mail spool directory
	vprog = NULL;         // validation program
	lhost = NULL;         // log host
	chost = NULL;         // chat host
	whost = NULL;         // watch host
	dhost = NULL;		// database host
	mhost = NULL;		// mail host
	rsent = NULL;		// registration sentence
	rkey = NULL;		// registration key
	bname = NULL;		// bbs name
}

// Function:	desctructor
// Purpose:	free up any allocated memory
// Input:	none
// Output:	all dynamically allocated memory is freed
// Author:	Greg Shaw
// Created:	1/1/96

bbsinfo::~bbsinfo()
{
	if (sys != NULL)
		free(sys);
	sys = NULL;
	if (spager != NULL)
		free(spager);
	spager = NULL;
	if (talkp != NULL)
		free(talkp);
	talkp = NULL;
	if (dterm != NULL)
		free(dterm);
	dterm = NULL;
	if (mspool != NULL)
		free(mspool);
	mspool = NULL;
	if (vprog != NULL)
		free(vprog);
	vprog = NULL;
	if (lhost != NULL)
		free(lhost);
	lhost = NULL;
	if (chost != NULL)
		free(chost);
	chost = NULL;
	if (whost != NULL)
		free(whost);
	whost = NULL;
}

// Function:   findline
// Purpose:        find a line containing a particular word from the bbsinfo
//             file.
// Input:      word - a string containing the word to look for
// Output:     line - the line (if found)
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::findline(char **line, char *word)
{
	FILE   *infile;             // bbs info file
	char   *cnt;                // line byte counter
	char   *in_line;            // a line from the file
	char   found;               // loop boolean
	char   *bbsdir;             // bbsdir string (from environment)
	static char   tmpstr[255];         // temporary string

	// get bbs dir
	if (bbsdir = getenv("BBSDIR"), bbsdir == NULL)
		return(-1);
	// get filename
	sprintf(tmpstr,"%s/config/bbsinfo",bbsdir);
	if (infile = fopen(tmpstr,"r"), infile == NULL)
		return(-1);
	// now look for word in file
	found = 0;
	while (!found && !feof(infile))
	{
		in_line = getline(infile);
		if (cnt = strchr(in_line,'#'), cnt != NULL)
			*cnt = 0;	// everything after # is ignored
		if (strstr(in_line,word) != NULL)
		{
			strcpy(tmpstr,in_line);
			found++;
		}
	}
	*line = tmpstr;
	fclose(infile);
	if (!found)
		return(-1);
	else
		return(0);
};

// Function:	bbs_name
// Purpose:	return the name of the BBS
// Input:	none
// Output:	the name of the BBS
// Author:	Greg Shaw
// Created:	6/30/96

char *bbsinfo::bbs_name(void)
{
	char *line;
	char *c,*t;

	if (bname != NULL)
		return(bname);
	if (findline(&line,"BBS_NAME") != 0)
		return(NULL);
	if (bname = (char *) malloc(120), bname == NULL)
		return(NULL);
	// skip to the end of the above word
	for (c = line; !isspace(*c) && *c != 0; c++);
	while (*c != 0 && isspace(*c))
		c++;
	// now copy up to end of line
	t = bname;
	while (*c != 0 && !iseol(*c))
		*t++ = *c++;
	*t = 0;
	return(bname);
}

// Function:   bbs_uid
// Purpose:    return the uid of the bbs administrator
// Input:  none
// Output: the bbs administrator's uid
// Author: Greg Shaw
// Created:    4/25/94

uid_t bbsinfo::bbs_uid(void)
{
	struct passwd *bbspw;

	if (bbspw = getpwnam("bbs"), bbspw == NULL)
		return(1);
	return(bbspw->pw_uid);
};


// Function:   bbs_gid
// Purpose:    return the gid of the bbs administrator
// Input:  none
// Output: the bbs administrator's gid
// Author: Greg Shaw
// Created:    4/25/94

gid_t bbsinfo::bbs_gid(void)
{
	struct passwd *bbspw;

	if (bbspw = getpwnam("bbs"), bbspw == NULL)
		return(1);
	return(bbspw->pw_gid);
};


// Function:   cardinfo
// Purpose:        return the card information relating to a particular card
// Input:      type - the type of card:
//             0 - red
//             1 - blue
//             2 - green
//             3 - white
//             4 - grey
//             5 - pink
//             6 - yellow
//             7 - rose
//             8 - voilet
//             9 - azure
//            10 - brown
//            11 - peach
//            12 - black
// Output:     a CardRec structure pointer filled in with information from
//             bbsinfo file.  (or NULL for error)
// Author:     Greg Shaw
// Created:        7/27/93

CardRec *bbsinfo::cardinfo(int type)
{
	char   *line;
	CardRec    *card;

	if (card = (CardRec *)malloc(sizeof(CardRec)), card == NULL)
		return(NULL);
	switch(type)
	{
		case 0:
			if (findline(&line,"REDCARD") != 0)
				return(NULL);
			strcpy(card->colr,"red");
			break;
		case 1:
			if (findline(&line,"BLUECARD") != 0)
				return(NULL);
			strcpy(card->colr,"blue");
			break;
		case 2:
			if (findline(&line,"GREENCARD") != 0)
				return(NULL);
			strcpy(card->colr,"green");
			break;
		case 3:
			if (findline(&line,"WHITECARD") != 0)
				return(NULL);
			strcpy(card->colr,"white");
			break;
		case 4:
			if (findline(&line,"GREYCARD") != 0)
				return(NULL);
			strcpy(card->colr,"grey");
			break;
		case 5:
			if (findline(&line,"PINKCARD") != 0)
				return(NULL);
			strcpy(card->colr,"pink");
			break;
		case 6:
			if (findline(&line,"YELLOWCARD") != 0)
				return(NULL);
			strcpy(card->colr,"yellow");
			break;
		//             7 - rose
		case 7:
			if (findline(&line,"ROSECARD") != 0)
				return(NULL);
			strcpy(card->colr,"rose");
			break;
		//             8 - voilet
		case 8:
			if (findline(&line,"VIOLETCARD") != 0)
				return(NULL);
			strcpy(card->colr,"violet");
			break;
		//             9 - azure
		case 9:
			if (findline(&line,"AZURECARD") != 0)
				return(NULL);
			strcpy(card->colr,"azure");
			break;
		//            10 - brown
		case 10:
			if (findline(&line,"BROWNCARD") != 0)
				return(NULL);
			strcpy(card->colr,"brown");
			break;
		//            11 - peach
		case 11:
			if (findline(&line,"PEACHCARD") != 0)
				return(NULL);
			strcpy(card->colr,"peach");
			break;
		//            12 - black
		case 12:
			if (findline(&line,"BLACKCARD") != 0)
				return(NULL);
			strcpy(card->colr,"black");
			break;
		default:
			return(NULL);
	}
	if (sscanf(line,"%*s %d %d %x %d %d %d %d %d %d %d %d",&card->acl, 
		&card->tl, &card->flags,
		&card->kbytes,
		&card->max_kbytes,
		&card->days_valid,
		&card->max_time,
		&card->months_valid,
		&card->max_files,
		&card->start_time,
		&card->end_time) != 11)
		return(NULL);
	return(card);
};

// Function:   chat_avail
// Purpose:        return true if sysop is available for chat
// Inputs:     tim - the hour and time currently
// Outputs:        true if available
// Author:     Greg Shaw
// Created:        7/25/93

int bbsinfo::chat_avail(int tim)
{
	char *line;
	int    start;
	int end;

	if (findline(&line,"CHATON") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&start) != 1)
		return(0);
	if (findline(&line,"CHATOFF") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&end) != 1)
		return(0);
	if (tim > start && tim < end)
		return(1);
	else
		return(0);
};


// Function:   chathost
// Purpose:    return the host where the chat daemon is running
// Input:      none
// Output:     a string that contains the chat host is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:    4/9/95

char *bbsinfo::chathost(void)
{
	char *line;

	if (chost != NULL)
		return(chost);
	if (findline(&line,"CHATHOST") != 0)
		return(NULL);
	if (chost = (char *) malloc(30), chost == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",chost) != 1)
		return(NULL);
	return(chost);
};


// Function:   checkmail
// Purpose:    return true if the BBS is to check for mail
// Input:  none
// Output: 1 for check for mail, 0 otherwise
// Author: Greg Shaw
// Created:    8/10/93

int bbsinfo::checkmail(void)
{
	char *line;

	if (cmail != -1)
		return(cmail);
	if (findline(&line,"CHECKMAIL") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&cmail) != 1)
		return(0);
	return(cmail);
};


// Function:   creditchat
// Purpose:        return true if chat time with sysop is credited
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::creditchat(void)
{
	char *line;

	if (cchat != -1)
		return(cchat);
	if (findline(&line,"CREDITCHAT") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&cchat) != 1)
		return(-1);
	return(cchat);
};


// Function:   credituploads
// Purpose:        return true if upload time is credited
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/13/93

int bbsinfo::credituploads(void)
{
	char *line;

	if (cuploads != -1)
		return(cuploads);
	if (findline(&line,"CREDITUPLOADS") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&cuploads) != 1)
		return(-1);
	return(cuploads);
};

// Function:   dbhost
// Purpose:    return the host where the database is running
// Input:      none
// Output:     a string that contains the database host is returned
// Author:     Greg Shaw
// Created:    3/6/96

char *bbsinfo::dbhost(void)
{
	char *line;

	if (dhost != NULL)
		return(dhost);
	if (findline(&line,"DATABASEHOST") != 0)
		return(NULL);
	if (dhost = (char *) malloc(30), dhost == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",dhost) != 1)
		return(NULL);
	return(dhost);
};



// Function:   def_attribute
// Purpose:    return the default attribute
// Input:      none
// Output:     a string that contains the default termtype is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::def_attribute(void)
{
	char *line;

	if (dattr != -1)
		return(dattr);
	if (findline(&line,"DEFATTRIBUTE") != 0)
		return(0);
	// gotcha!
	if (strstr(line,"normal") != NULL)
		dattr = 1;
	// gotcha!
	if (strstr(line,"bold") != NULL)
		dattr = 2;
	// gotcha!
	if (strstr(line,"blink") != NULL)
		dattr = 3;
	// gotcha!
	if (strstr(line,"inverse") != NULL)
		dattr = 4;
	return(dattr);
};

// Function:   def_card
// Purpose:    return the default card color for the user
// Input:      none
// Output:     the default acl
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::def_card(void)
{
	char *line;

	if (dcard != -1)
		return(dcard);
	if (findline(&line,"DEFCARD") != 0)
		return(0);
	dcard=colortonum(line);
	return(dcard);
};




// Function:   def_foreground
// Purpose:    return the default foreground color to use
// Input:      none
// Output:     a string that contains the default foreground is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::def_foreground(void)
{
	char *line;

	if (dfore != -1)
		return(dfore);
	if (findline(&line,"DEFFOREGROUND") != 0)
		return(0);
	if (strstr(line,"black") != NULL)
		dfore = 1;
	if (strstr(line,"red") != NULL)
		dfore = 2;
	if (strstr(line,"green") != NULL)
		dfore = 3;
	if (strstr(line,"yellow") != NULL)
		dfore = 4;
	if (strstr(line,"blue") != NULL)
		dfore = 5;
	if (strstr(line,"magenta") != NULL)
		dfore = 6;
	if (strstr(line,"cyan") != NULL)
		dfore = 7;
	if (strstr(line,"white") != NULL)
		dfore = 8;
	return(dfore);
};


// Function:   def_background
// Purpose:    return the default background color to use
// Input:      none
// Output:     a string that contains the default background is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::def_background(void)
{
	char *line;

	if (dback != -1)
		return(dback);
	if (findline(&line,"DEFBACKGROUND") != 0)
		return(0);
	if (strstr(line,"black") != NULL)
		dback = 1;
	if (strstr(line,"red") != NULL)
		dback = 2;
	if (strstr(line,"green") != NULL)
		dback = 3;
	if (strstr(line,"yellow") != NULL)
		dback = 4;
	if (strstr(line,"blue") != NULL)
		dback = 5;
	if (strstr(line,"magenta") != NULL)
		dback = 6;
	if (strstr(line,"cyan") != NULL)
		dback = 7;
	if (strstr(line,"white") != NULL)
		dback = 8;
	return(dback);
};


// Function:   def_termtype
// Purpose:        return the default terminal type of a new user
// Input:      none
// Output:     a string that contains the default termtype is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::def_termtype(void)
{
	char *line;

	if (findline(&line,"DEFTERM") != 0)
		return(NULL);
	if (dterm = (char *) malloc(30), dterm == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",dterm) != 1)
		return(NULL);
	return(dterm);
};

// Function:   inactivity_timeout
// Purpose:    return the time to wait before logging out an inactive user
// Input:  none
// Output: true/false
// Author: Greg Shaw
// Created:    8/24/93

int bbsinfo::inactivity_timeout(void)
{
	char *line;
	
	if (itime != -1)
		return(itime);
	if (findline(&line,"INACTIVITY") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&itime) != 1)
		return(-1);
	return(itime);
};

// Function:   log_externals
// Purpose:    return whether to log external calls
// Input:      none
// Output:     1 for log, 0 for no
// Author:     Greg Shaw
// Created:    12/31/94

int bbsinfo::log_externals(void)
{
	char *line;

	if (lmext != -1)
		return(lmext);
	if (findline(&line,"LOG_EXTERNALS") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&lmext) != 1)
		return(0);
	return(lmext);
};

// Function:   log_mail_import
// Purpose:    return whether to log mail importing (from sendmail)
// Input:      none
// Output:     1 for log, 0 for no
// Author:     Greg Shaw
// Created:    5/17/96

int bbsinfo::log_mail_import(void)
{
	char *line;

	if (logmaili != -1)
		return(logmaili);
	if (findline(&line,"LOG_MAIL_IMPORT") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&logmaili) != 1)
		return(0);
	return(logmaili);
};



// Function:   log_menu_accesses
// Purpose:    return whether to log menu movement
// Input:      none
// Output:     1 for log, 0 for no
// Author:     Greg Shaw
// Created:    12/31/94

int bbsinfo::log_menu_accesses(void)
{
	char *line;

	if (lmaccess != -1)
		return(lmaccess);
	if (findline(&line,"LOG_MENU_ACCESSES") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&lmaccess) != 1)
		return(0);
	return(lmaccess);
};



// Function:   loghost
// Purpose:        return the host where the error logger is running
// Input:      none
// Output:     a string that contains the loghost is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::loghost(void)
{
	char *line;

	if (lhost != NULL)
		return(lhost);
	if (findline(&line,"LOGHOST") != 0)
		return(NULL);
	if (lhost = (char *) malloc(30), lhost == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",lhost) != 1)
		return(NULL);
	return(lhost);
};



// Function:   mailchecktime
// Purpose:    return the number of seconds that should elapse between
//     mail checks
// Input:  none
// Output: a string that contains the mail spool dir
// Author: Greg Shaw
// Created:    4/24/94

int bbsinfo::mailchecktime(void)
{
	char *line;

	if (mcheck != -1)
		return(mcheck);
	if (findline(&line,"MAILCHECK") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&mcheck) != 1)
		return(0);
	return(mcheck);
};


// Function:   mailforward
// Purpose:    should the BBS forward all private email into the BBS?
// Input:      none
// Output:     1 for forward, 0 for not
// Author:     Greg Shaw
// Created:    5/25/96

int bbsinfo::mailforward(void)
{
	char *line;

	if (mfor != -1)
		return(mfor);
	if (findline(&line,"MAIL_FORWARD") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&mfor) != 1)
		return(0);
	return(mfor);
}

// Function:   mailhost
// Purpose:    return the host where email should be sent
// Input:      none
// Output:     a string that contains the mail host is returned
// Author:     Greg Shaw
// Created:    3/14/96

char *bbsinfo::mailhost(void)
{
	char *line;

	if (mhost != NULL)
		return(mhost);
	if (findline(&line,"MAILHOST") != 0)
		return(NULL);
	if (mhost = (char *) malloc(30), mhost == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",mhost) != 1)
		return(NULL);
	return(mhost);
};





// Function:   mailspool
// Purpose:    return the mail spool file directory for the system
// Input:      none
// Output:     a string that contains the mail spool dir
// Author:     Greg Shaw
// Created:        8/10/93

char *bbsinfo::mailspool(void)
{
	char *line;

	if (mspool != NULL)
		return(mspool);
	if (findline(&line,"MAILSPOOL") != 0)
		return(NULL);
	if (mspool = (char *) malloc(20), mspool == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",mspool) != 1)
		return(NULL);
	return(mspool);
};


// Function:	maxfiles
// Purpose:	return the maximum number of files the user can download in one batch.
// Input:	none
// Output:	true/false
// Author:	Greg Shaw
// Created:	8/18/95

int bbsinfo::maxfiles(void)
{
	char *line;

	if (mfiles != -1)
		return(mfiles);
	if (findline(&line,"MAXFILES") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&mfiles) != 1)
		return(0);
	return(mfiles);
};


// Function:   maxk
// Purpose:    return the maximum number of Kbytes the user can download at
//     one time.
// Input:  none
// Output: true/false
// Author: Greg Shaw
// Created:    8/24/93

unsigned int bbsinfo::maxk(void)
{
	char *line;

	if (mk != 1)
		return(mk);
	if (findline(&line,"MAXK") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&mk) != 1)
		return(0);
	return(mk);
};


// Function:   maxk_day
// Purpose:    return the maximum number of Kbytes the user can download at
//     		in one day.
// Input:  	none
// Output: 	see above
// Author: 	Greg Shaw
// Created:    	8/18/95

unsigned int bbsinfo::maxk_day(void)
{
	char *line;

	if (mkd != 1)
		return(mkd);
	if (findline(&line,"MAXK_DAY") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&mkd) != 1)
		return(0);
	return(mkd);
};


// Function:   reg_sentence
// Purpose:    return the registration sentence found in the bbsinfo file.
// Input:      none
// Output:     a string that contains the registration sentence
// Author:     Greg Shaw
// Created:    6/30/96

char *bbsinfo::reg_sentence(void)    // registration sentence
{
	char *line;
	char *c,*t;

	if (rsent != NULL)
		return(rsent);
	if (findline(&line,"REG_SENTENCE") != 0)
		return(0);
	if (rsent = (char *) malloc(strlen(line)), rsent == NULL)
		return(NULL);
	// skip to the end of the above word
	for (c = line; *c != 0 && !isspace(*c); c++);
	while (*c != 0 && isspace(*c))
		c++;
	t = rsent;
	while (*c != 0 && !iseol(*c))
		*t++ = *c++;
	*t = 0;
	return(rsent);
}

// Function:   reg_sentence
// Purpose:    return the registration sentence found in the bbsinfo file.
// Input:      none
// Output:     a string that contains the registration sentence
// Author:     Greg Shaw
// Created:    6/30/96

char *bbsinfo::reg_key(void) // key
{
	char *line;

	if (rkey != NULL)
		return(rkey);
	if (findline(&line,"REG_KEY") != 0)
		return(0);
	if (rkey = (char *) malloc(strlen(line)), rkey == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",rkey) != 1)
		return(NULL);
	return(rkey);
}

// Function:   ratio
// Purpose:        return the upload to download ratio
// Input:      none
// Output:     a string that contains the loghost is returned
// Author:     Greg Shaw
// Created:        7/11/93

float bbsinfo::ratio(void)
{
	char *line;

	if (rato != -1000.0)
		return(rato);
	if (findline(&line,"RATIO") != 0)
		return(0.0);
	if (sscanf(line,"%*s%f",&rato) != 1)
		return(0.0);
	return(rato);
};


// Function:   showcoms
// Purpose:        return true if command prompt should show the keys for
//             the valid commands
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/27/93

int bbsinfo::showcoms(void)
{
	char *line;

	if (scoms != -1)
		return(scoms);
	if (findline(&line,"SHOWVALIDKEYS") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&scoms) != 1)
		return(0);
	return(scoms);
};


// Function:   showfortune
// Purpose:    return true if the bbs should give the user a 'fortune cookie'
// Input:  none
// Output: true/false
// Author: Greg Shaw
// Created:    8/22/93

int bbsinfo::showfortune(void)
{
	char *line;

	if (sfor != -1)
		return(sfor);
	if (findline(&line,"SHOWFORTUNE") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&sfor) != 1)
		return(0);
	return(sfor);
};

// Function:   showlast
// Purpose:        return true if command prompt should show the amount of
//             last user has left.
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/27/93

int bbsinfo::showlast(void)
{
	char *line;

	if (slast != -1)
		return(slast);
	if (findline(&line,"SHOWLAST") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&slast) != 1)
		return(0);
	return(slast);
};



// Function:   showtime
// Purpose:        return true if command prompt should show the amount of
//             time user has left.
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/27/93

int bbsinfo::showtime(void)
{
	char *line;

	if (stime != -1)
		return(stime);
	if (findline(&line,"SHOWTIMELEFT") != 0)
		return(0);
	if (sscanf(line,"%*s%d",&stime) != 1)
		return(0);
	return(stime);
};



// Function:   sys_pager
// Purpose:        return the default 'pager' program to be used by the
//             bbs to display files to the user
// Input:      none
// Output:     a string that contains the 'pager' program
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::sys_pager(void)
{
	char *line;

	if (spager != NULL)
		return(spager);
	if (spager = (char *) malloc(20), spager == NULL)
		return(NULL);
	// look for a PAGER environment variable that will override the
	// system default
	if (line = getenv("PAGER"), line != NULL)	
	{
		if (strlen(line) < 20)
			strcpy(spager,line);
	}
	else
	{
		if (findline(&line,"SYSTEMPAGER") != 0)
			return(NULL);
		if (sscanf(line,"%*s%s",spager) != 1)
			return(NULL);
	}
	return(spager);
};



// Function:   sysop
// Purpose:        return the name of the BBS main sysop
// Input:      none
// Output:     a string that contains the 'pager' program
// Notes:      it should be possible to pass in some variable to tell
//             which sysop the user 'wants' in a multiple sysop system
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::sysop(void)
{
	char *line;

	if (findline(&line,"SYSOP") != 0)
		return(NULL);
	if (sys = (char *) malloc(20), sys == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",sys) != 1)
		return(NULL);
	return(sys);
};


// Function:   talkprog
// Purpose:    return the 'talk' program for the system
// Input:  none
// Output: a string that contains the talkprog is returned
//         note: it should be freed on exit.
// Author: Greg Shaw
// Created:    7/11/93

char *bbsinfo::talkprog(void)
{
	char *line;

	if (findline(&line,"TALKPROG") != 0)
		return(NULL);
	if (talkp  = (char *) malloc(30), talkp == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",talkp) != 1)
		return(NULL);
	return(talkp);
};

// Function:   timelimit_fudge
// Purpose:    return the amount of additional time (percentage) to give a
//     user when his time has expired and he is downloading
// Input:  none
// Output: int (percent)
// Author: Greg Shaw
// Created:    10/17/94

int bbsinfo::timelimit_fudge(void)
{
	char *line;

	if (findline(&line,"FUDGETIMELIMIT") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&tlfud) != 1)
		return(0);
	return(tlfud);
};




// Function:   validation_program
// Purpose:    return the name of the program to be used for validation
// Input:      none
// Output:     name of validation program 
// Author:     Greg Shaw
// Created:    8/18/95


char *bbsinfo::validation_program(void)
{
	char *line;

	if (vprog != NULL)
		return(vprog);
	if (findline(&line,"VALIDATION") != 0)
		return(NULL);
	if (vprog = (char *) malloc(strlen(line)), vprog == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",vprog) != 1)
		return(NULL);
	return(vprog);
};



// Function:   waittime
// Purpose:        return the amount of time between calls allowed (hours)
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/13/93

int bbsinfo::waittime(void)
{
	char *line;

	if (waitt != -100)
		return(waitt);
	if (findline(&line,"WAITTIME") != 0)
		return(-1);
	if (sscanf(line,"%*s%d",&waitt) != 1)
		return(-1);
	if (waitt<0)
		waitt=0;
	return(waitt);
};


// Function:   watchost
// Purpose:        return the host where the sysop 'watch' program runs
// Input:      none
// Output:     a string that contains the watchhost is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::watchhost(void)
{
	char *line;

	if (whost != NULL)
		return(whost);
	if (findline(&line,"WATCHHOST") != 0)
		return(NULL);
	if (whost = (char *) malloc(30), whost == NULL)
		return(NULL);
	if (sscanf(line,"%*s%s",whost) != 1)
		return(NULL);
	return(whost);
};


#endif                          // _BBSINFO_C_







